#include "TStoreTypeQBoxLayout.h"
#include "TBoxLayoutWindow.h"
#include <TStoreProperty.h>
#include <TFileIO.h>

#include <qbutton.h>
#include <qcombo.h>
#include <qdir.h>
#include <qfile.h>
#include <qfiledlg.h>

#include <kmsgbox.h>

#include <dlfcn.h>
#include <assert.h>



TStoreType *createStoreTypeQBoxLayout(TStoreType *parent, TLayoutWindow *lw) {
	return (new TStoreTypeQBoxLayout(parent, lw));
}


TStoreTypeQBoxLayout::TStoreTypeQBoxLayout(TStoreType *p, TLayoutWindow *lw) : TStoreTypeQLayout(p, lw) {
	initMetaObject();
	addProperties();

	QBoxLayout *widget;
	if (parent()) {
		QString str = nextName(parent()->object(), storeTypeName());

		widget = new QBoxLayout((QWidget*)parent()->object(), 
								QBoxLayout::LeftToRight);
		CHECK_PTR(widget);
		widget->setName(str);
	}
	else {
		widget = new QBoxLayout(lw, QBoxLayout::LeftToRight, -1, -1, "QBoxLayout1");
	}
	
	TBoxLayoutData *data = new TBoxLayoutData;
	data->direction = QBoxLayout::LeftToRight;
	data->border = 0;
	data->layoutName = widget->name();
	
	addData("QBoxLayout", data);
		
	if (object())
		delete object();
	object(widget);
}


bool TStoreTypeQBoxLayout::isCreatable() {
	return TRUE;
}




// creates a list of all of the properties available to this type,
// including all of the properties from the parent
//
// There are 3 types of properties, ptText, ptList, and ptButton.
//   ptText is the most common and allows the user to type an arbitrary piece
//          of text (such as a name, or a number)
//   ptList allows the user to select from a list of options, usefull
//          for bools and enums
//   ptButton pops up a dialog with more questions, such as a file dialog which 
//          is used to select a bitmap file.
void TStoreTypeQBoxLayout::addProperties() {
	TStoreProperty *sp = NULL;
  
	sp = new TStoreProperty("Layout", ptButton,
							SETFUNC(&QBoxLayoutSetLayoutFunc),
							GETFUNC(&QBoxLayoutGetLayoutFunc),
							0,
							ACTIVEFUNC(&QBoxLayoutActivateLayoutFunc), TRUE);
	CHECK_PTR(sp);
	fProperties.inSort(sp);

	sp = new TStoreProperty("Border", ptText,
							SETFUNC(&QBoxLayoutSetBorderFunc),
							GETFUNC(&QBoxLayoutGetBorderFunc), 
							0,
							0, FALSE);
	CHECK_PTR(sp);
	fProperties.inSort(sp);

	sp = new TStoreProperty("Direction", ptList,
							SETFUNC(&QBoxLayoutSetDirectionFunc),
							GETFUNC(&QBoxLayoutGetDirectionFunc),
							INITFUNC(&QBoxLayoutInitDirectionFunc), 
							0, TRUE);
	CHECK_PTR(sp);
	fProperties.inSort(sp);
}




// The name of this type
QString TStoreTypeQBoxLayout::storeTypeName() {
	return "QBoxLayout";
}



// the headerfile for this type
QString TStoreTypeQBoxLayout::headerFile() {
	return "qlayout.h";
}




// The declaration line for the header file we write out
void TStoreTypeQBoxLayout::writeHFile(QTextStream &os, QString var) {
	if (var!="")
		os << "\tQBoxLayout *" << var << ";" << endl;
}



// The constructor line for the file we write out
void TStoreTypeQBoxLayout::writeCFileConstructor(QTextStream &os, QString var, QString parent) {
	TLayoutDataItem *layoutItem = (TLayoutDataItem*) findAppliedData("QLayout");

	if (layoutItem && layoutItem->parentLayoutTree) 
		os << "\t" << var << " = new QBoxLayout (";
	else
		os << "\t" << var << " = new QBoxLayout ( " << parent << ", " ;

	os<< "QBoxLayout::" << QBoxLayoutGetDirectionFunc();
	os << ", " << QBoxLayoutGetBorderFunc();
	os << ");" << endl;
}


// Any properties that this type has, are written to the C file we generate
void TStoreTypeQBoxLayout::writeCFileProperties(QTextStream &os, QString var) {
	// make sure that the parent properties are written as well
	TStoreTypeQLayout::writeCFileProperties(os, var);
  
	TLayoutDataItem *layoutData = (TLayoutDataItem*) findAppliedData("QLayout");
	if (!layoutData || !layoutData->parentLayoutTree) {
		//cout << "No parent, so dump layout tree " << endl;
		writePropertiesLayoutTree(os, var);
		
		TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");
		if (data->items.count() > 0)
			os << "\t" << var << "->actifsvate();" << endl;
	}
}



void TStoreTypeQBoxLayout::writePropertiesLayoutTree(QTextStream &os, QString var) {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	//cout << "repopulateLayout: " << data->layoutName << endl;
	QListIterator<TBoxLayoutDataItem> it(data->items);
	for ( ; it.current() ; ++it) {
		TBoxLayoutDataItem *item = it.current();

		switch (item->type) {
		case TBoxLayoutDataItem::Widget:
			//cout << "AddingWidget: " << item->objectTree->object()->name() << ", " << item->size
			//   << ", " << item->alignment << endl;
			os << "\t" << var << "->addWidget( " << item->objectTree->validName() 
			   << ", " << item->size << ", ";
			switch(item->alignment) {
			case AlignLeft:
				os << "AlignLeft);" << endl; break;
			case AlignRight:
				os << "AlignRight);" << endl; break;
			case AlignTop:
				os << "AlignTop);" << endl; break;
			case AlignBottom:
				os << "AlignBottom);" << endl; break;
			case AlignCenter:
				os << "AlignCenter);" << endl; break;

			};
			break;
		case TBoxLayoutDataItem::Layout:
		{
			TStoreTypeQLayout *subTree = (TStoreTypeQLayout*)item->objectTree;
	
			os << "\t" << var << "->addLayout( " << subTree->validName() 
			   << ", " << item->size << ");\n" << endl;
		
			subTree->writePropertiesLayoutTree(os, subTree->validName());
		}
		break;
		case TBoxLayoutDataItem::Stretch:
			os << "\t" << var << "->addStretch( " << item->size << " );" << endl;
			break;
		case TBoxLayoutDataItem::Strut:
			os << "\t" << var << "->addStrut( " << item->size << " );" << endl;
			break;
		case TBoxLayoutDataItem::Space:
			os << "\t" << var << "->addSpacing( " << item->size << " );" << endl;
			break;
		};
	};
	//  cout << "Finished writing layout" << endl;
}
  



void TStoreTypeQBoxLayout::writeLayout(QTextStream &os, int tabs) {
	// Dump layout for the layout here.
	TStoreTypeQLayout::writeLayout(os, tabs);

	QString tabstr = TFileIO::setTabs(tabs);
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");
	TLayoutDataItem *li = (TLayoutDataItem*) findAppliedData("QLayout");

	TFileIO::writeLayoutString(os, tabstr, "Direction", 
							   QBoxLayoutGetDirectionFunc() );
	TFileIO::writeLayoutString(os, tabstr, "Border",
							   QBoxLayoutGetBorderFunc() );

	if (li && li->parentLayoutTree)
		TFileIO::writeLayoutString(os, tabstr, "ParentLayout", 
								   li->parentLayoutTree->validName());
  
	QListIterator<TBoxLayoutDataItem> it(data->items);
	for ( ; it.current() ; ++it) {
		TBoxLayoutDataItem *item = it.current();

		switch (item->type) {
		case TBoxLayoutDataItem::Widget:
			os << tabstr << "Item { Widget " << item->objectTree->object()->name() 
			   << " " << item->size << " ";
			switch (item->alignment) {
			case AlignTop:
				os << "AlignTop }" << endl; break;
			case AlignLeft:
				os << "AlignLeft }" << endl; break;
			case AlignRight:
				os << "AlignRight }" << endl; break;
			case AlignBottom:
				os << "AlignBottom }" << endl; break;
			case AlignCenter:
				os << "AlignCenter }" << endl; break;
			}
			break;
		case TBoxLayoutDataItem::Layout:
			os << tabstr << "Item { Layout " << item->objectTree->object()->name() 
			   << " " << item->size << " }" << endl;
			break;
		case TBoxLayoutDataItem::Stretch:
			os << tabstr << "Item { Stretch " << item->size << " }" << endl;
			break;
		case TBoxLayoutDataItem::Strut:
			os << tabstr << "Item { Strut " << item->size << " }" << endl;
			break;
		case TBoxLayoutDataItem::Space:
			os << tabstr << "Item { Spacing " << item->size << " }" << endl;
			break;
		};
	};
}



void TStoreTypeQBoxLayout::readLayoutItem(QTextStream &is, QString propName) {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	if (propName=="Direction") {
		QString str = TFileIO::readLayoutString(is);
		if ("TopToBottom" == str)
			data->direction = QBoxLayout::TopToBottom;
		else if ("BottomToTop" == str)
			data->direction = QBoxLayout::BottomToTop;
		else if ("LeftToRight" == str)
			data->direction = QBoxLayout::LeftToRight;
		else if ("RightToLeft" == str)
			data->direction = QBoxLayout::RightToLeft;
	}
	else if (propName=="Border") {
		QString str = TFileIO::readLayoutString(is);
		data->border = str.toInt();
	}
	else if (propName=="ParentLayout") {
		data->parentName = TFileIO::readLayoutString(is);
	}
	else if (propName=="Item") {
		TBoxLayoutDataItem *newItem = new TBoxLayoutDataItem;
		QString type;
		is >> type;
		if ("Widget" == type) {
			newItem->type = TBoxLayoutDataItem::Widget;
			is >> newItem->name;
			is >> newItem->size;
			QString align;
			is >> align;
			if ("AlignLeft" == align)
				newItem->alignment = AlignLeft;
			else if ("AlignRight" == align)
				newItem->alignment = AlignRight;
			else if ("AlignTop" == align)
				newItem->alignment = AlignTop;
			else if ("AlignBottom" == align)
				newItem->alignment = AlignBottom;
			else
				newItem->alignment = AlignCenter;
		}
		else if ("Layout" == type) {
			newItem->type = TBoxLayoutDataItem::Layout;
			is >> newItem->name;
			is >> newItem->size;
		}
		else if ("Stretch" == type) {
			newItem->type = TBoxLayoutDataItem::Stretch;
			is >> newItem->size;
		}
		else if ("Strut" == type) {
			newItem->type = TBoxLayoutDataItem::Strut;
			is >> newItem->size;
		}
		else if ("Spacing" == type) {
			newItem->type = TBoxLayoutDataItem::Space;
			is >> newItem->size;
		}
		TFileIO::readLayoutUntil(is, '}');
		data->items.append(newItem);
	}
	else
		TStoreTypeQLayout::readLayoutItem(is, propName);
}




// Copy the details from one object to the other
void TStoreTypeQBoxLayout::copyDetails(TStoreType *oldObj) {
	//cout << "QBoxLayout::copyDetails" << endl;
	TStoreTypeQLayout::copyDetails(oldObj);

	//  n->setText(o->text());
	//n->setAutoResize(o->autoResize());
	//if (o->pixmap())
	//  n->setPixmap(*o->pixmap());
	//n->setAutoRepeat(o->autoRepeat());
  
	//cout << "QBoxLayout::copyDetails" << endl;
}




void TStoreTypeQBoxLayout::rebuildLayout() {
	TLayoutDataItem *layoutItem = (TLayoutDataItem*) findAppliedData("QLayout");

	//cout << "\n###\n###rebuildLayout: " << object()->name() << "\n###" << endl;

	if (layoutItem && layoutItem->parentLayoutTree) {
		TStoreTypeQLayout *parTree = (TStoreTypeQLayout*)layoutItem->parentLayoutTree;
		parTree->rebuildLayout();
	}
	else {
		//cout << "No parent, so recreate" << endl;

		//cout << "Set Rebuild Flag" << endl;
		rebuild(TRUE);

		recreateLayout(NULL);
		repopulateLayout();


		TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");
		if (data->items.count() > 0) {
			QLayout *lay = (QLayout*) object();
			//cout << "About to activate... " << lay->name() << endl;
			lay->activate();
			//cout << "Have just activated..." << endl;
		}

		//cout << "Clear Rebuild Flag" << endl;
		rebuild(FALSE);
	}
}



void TStoreTypeQBoxLayout::deleteLayoutTree() {
	//cout << "Deleting layouttree for " << storeTypeName() << ":" 
	//	 << object()->name() << endl;

	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	QListIterator<TBoxLayoutDataItem> it(data->items);
	for ( ; it.current() ; ++it) {
		TBoxLayoutDataItem *item = it.current();
		
		if (item->type == TBoxLayoutDataItem::Layout) {
			if (item->objectTree->parent() == parent())
				delete item->objectTree->object();
			else
				((TStoreTypeQLayout*)item->objectTree)->deleteLayoutTree();
		}
	}
	delete object();
}


void TStoreTypeQBoxLayout::recreateLayout(QLayout *par) {
	QLayout *layout = NULL;
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");


	TLayoutDataItem *layoutItem = (TLayoutDataItem*) findAppliedData("QLayout");
	if (!layoutItem ||  !layoutItem->parentLayoutTree) {
		//cout << "Deleting Layout" << data->layoutName << endl;
		//object()->dumpObjectTree();
		
		deleteLayoutTree();
//delete (object());

		// WARNING: NEED TO DELETE ALL CHILD LAYOUTS FIRST, SO THAT WE KNOW WE
		// HAVE DELETED THEM ALL
		//cout << "Deleted Layout" << data->layoutName << endl;
	}

	if (par) {
		//cout << "recreateLayout: " << data->layoutName << ", " 
		//	 << par->name() << endl;
		layout = new QBoxLayout(data->direction, data->border, data->layoutName);
	}
	else {
		layout = new QBoxLayout((QWidget*)parent()->object(), 
								data->direction, 
								data->border, -1, data->layoutName);
	}
	//cout << "recreatedLayoutOld: " << (void*)object() << endl;
	object(layout);

	TLayoutWindow *lw = (TLayoutWindow*) root();
	object()->installEventFilter(lw);
	lw->connectToChildDestroyed(this);

	//cout << "recreatedLayoutNew: " << (void*)object() << endl;
	//cout << "recreatedLayout: " << object()->name() << endl;
}    



void TStoreTypeQBoxLayout::repopulateLayout() {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");
	QBoxLayout *layout = (QBoxLayout*) object();

	//cout << "repopulateLayout: " << data->layoutName << endl;

	QListIterator<TBoxLayoutDataItem> it(data->items);
	for ( ; it.current() ; ++it) {
		TBoxLayoutDataItem *item = it.current();

		switch (item->type) {
		case TBoxLayoutDataItem::Widget: 
		{
			//cout << "AddingWidget: name " << item->name << endl;
			//cout << "AddingWidget: tree " << (void*)item->objectTree << endl;
			//cout << "AddingWidget: object " << (void*)item->objectTree->object() << endl;
			//cout << "AddingWidget: " << item->objectTree->object()->name() 
			//	 << ", " << item->size
			//	 << ", " << item->alignment << endl;
			if (!item->objectTree) {
				cerr << "Can not add Widget: it is not there!" << endl;
			}
			else {
				layout->addWidget((QWidget*) item->objectTree->object(), 
								  item->size, item->alignment);
				TLayoutDataItem *li = (TLayoutDataItem*)item->objectTree->findAppliedData("QLayout");
				if (!li) {
					li = new TLayoutDataItem;
					item->objectTree->addAppliedData("QLayout", li);
				}
				li->parentLayoutTree = this;
			}
		}
		break;
		case TBoxLayoutDataItem::Layout:
		{
			TStoreTypeQLayout *subTree = (TStoreTypeQLayout*)item->objectTree;
	
			//cout << "Adding Layout" << endl;
			subTree->recreateLayout((QLayout*) object());

			//cout << "Adding Layout ["<< subTree->object()->name() <<"] into parent: " 
			//	 <<  layout->name() << " with stretch: " << item->size << endl;
			layout->addLayout((QLayout*) subTree->object(), item->size);

			TLayoutDataItem *li = (TLayoutDataItem*)subTree->findAppliedData("QLayout");
			if (!li) {
				li = new TLayoutDataItem;
				subTree->addAppliedData("QLayout", li);
			}
			li->parentLayoutTree = this;

			subTree->repopulateLayout();
			//cout << "Finished Adding Layout" << endl;

		}
		break;
		case TBoxLayoutDataItem::Stretch:
			//cout << "Adding Stretch: " << item->size << endl;
			layout->addStretch(item->size);
			break;
		case TBoxLayoutDataItem::Strut:
			//cout << "Adding Strut: " << item->size << endl;
			layout->addStrut(item->size);
			break;
		case TBoxLayoutDataItem::Space:
			//cout << "Adding Spacing: " << item->size << endl;
			layout->addSpacing(item->size);
			break;
		};
	};
}



void TStoreTypeQBoxLayout::propertyChanged(const char *propName, 
										   const char *oldValue,
										   const char *newValue) {
	if (0==strcasecmp(propName, "Name")) {
		//cout << "+++ NAME CHANGE, from: " << oldValue << " to: " << newValue << endl;
		TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");
		data->layoutName = newValue;
	}
	TStoreTypeQLayout::propertyChanged(propName, oldValue, newValue);
}


void TStoreTypeQBoxLayout::postLayoutLoad() {
	TStoreTypeQLayout::postLayoutLoad();

	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");
	if (!data) {
		cerr << "error: data is NULL" << endl;
		return;
	}
	data->layoutName = object()->name();
	QListIterator<TBoxLayoutDataItem> it(data->items);
	for ( ; it.current() ; ++it) {
		TBoxLayoutDataItem *item = it.current();
      
		switch (item->type) {
		case TBoxLayoutDataItem::Widget:
		case TBoxLayoutDataItem::Layout:
		{
			item->objectTree = root()->findByObjName(item->name);
			TLayoutDataItem *li = (TLayoutDataItem*)item->objectTree->findAppliedData("QLayout");
			if (!li) {
				li = new TLayoutDataItem;
				item->objectTree->addAppliedData("QLayout", li);
			}
			li->parentLayoutTree = this;
		}
		break;
		case TBoxLayoutDataItem::Space:
			break;
		case TBoxLayoutDataItem::Strut:
			break;
		case TBoxLayoutDataItem::Stretch:
			break;
		}
	}
  
	rebuildLayout();
}

/********************************************************************************
**
**  Property Functions
**
**  These are used by the addProperties function to perform the underlying 
**  functionality
**
********************************************************************************/


// A ptText property
bool TStoreTypeQBoxLayout::QBoxLayoutSetBorderFunc(QString AValue) {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	if (data) {
		data->border = AValue.toInt();
		rebuildLayout();
	}
	return TRUE;
}

QString TStoreTypeQBoxLayout::QBoxLayoutGetBorderFunc() {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	if (data) {
		QString str;
		str.setNum(data->border);
		return str;
	}
	return "";
}


// A ptList Property

bool TStoreTypeQBoxLayout::QBoxLayoutSetDirectionFunc(QString AValue) {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	if (data) {
		if (AValue=="LeftToRight")
			data->direction = QBoxLayout::LeftToRight;
		else if (AValue=="RightToLeft")
			data->direction = QBoxLayout::RightToLeft;
		else if (AValue=="TopToBottom")
			data->direction = QBoxLayout::TopToBottom;
		else
			data->direction = QBoxLayout::BottomToTop;

		//cout << "\n\n\nSet Direction\n\n\n" << endl;
		rebuildLayout();
	}
	return TRUE;
}

QString TStoreTypeQBoxLayout::QBoxLayoutGetDirectionFunc() {
	TBoxLayoutData *data = (TBoxLayoutData *) findData("QBoxLayout");

	switch (data->direction) {
	case QBoxLayout::TopToBottom:
		return "TopToBottom"; break;
	case QBoxLayout::BottomToTop:
		return "BottomToTop"; break;
	case QBoxLayout::LeftToRight:
		return "LeftToRight"; break;
	case QBoxLayout::RightToLeft:
		return "RightToLeft"; break;
	default:
		return "";
	}
}

void TStoreTypeQBoxLayout::QBoxLayoutInitDirectionFunc(QObject *w) {
	// set up the list with the required strings
	if (w->inherits("QComboBox")) {
		((QComboBox *)w)->clear();
		((QComboBox *)w)->insertItem("BottomToTop");
		((QComboBox *)w)->insertItem("LeftToRight");
		((QComboBox *)w)->insertItem("RightToLeft");
		((QComboBox *)w)->insertItem("TopToBottom");
	}
	else {
		cerr << "Tried to initialise " << w->name()
			 << " as an AutoResize property editor" << endl;
		cerr << "however it is not a QComboBox" << endl;
	}
}





// ptButton Property
bool TStoreTypeQBoxLayout::QBoxLayoutSetLayoutFunc(QString) {
	return TRUE;
}

QString TStoreTypeQBoxLayout::QBoxLayoutGetLayoutFunc() {
	return ("Layout");
}

void TStoreTypeQBoxLayout::QBoxLayoutActivateLayoutFunc() {
	TBoxLayoutWindow *win = new TBoxLayoutWindow(this);

	if (win->exec()) {
		TBoxLayoutData *data = (TBoxLayoutData*)findData("QBoxLayout");

		// For every item in the data list, remove the layout applied data
		// (We do not know if the item in the list will still be in the layout)
		QListIterator<TBoxLayoutDataItem> oldit(data->items);
		for ( ; oldit.current() ; ++oldit) {
			if (oldit.current()->objectTree) {
				TStoreType *node = oldit.current()->objectTree;
				node->deleteAppliedData("QLayout");
				disconnect(node->object(), SIGNAL(destroyed()), this, SLOT(deleted())); 
			}
		}
		data->items.clear();
    
		QList<TBoxLayoutDataItem> *newItems = win->items();
		QListIterator<TBoxLayoutDataItem> it(*newItems);

		for ( ; it.current() ; ++it) {
			TBoxLayoutDataItem *item = it.current();
      
			if (item->type == TBoxLayoutDataItem::Widget || 
				item->type == TBoxLayoutDataItem::Layout) {
				// We have an external object associated with this layout item,
				// So we need to set the applied data to indicate that it is a member
				// of this list.
				TStoreType *node = item->objectTree;
				assert (node != NULL); // if it is part of a layout it must exist
	
				TLayoutDataItem *data = (TLayoutDataItem *) node->findAppliedData("QLayout");

				if (data) {
					data->parentLayoutTree = this;
				}
				else {
					data = new TLayoutDataItem;
					data->parentLayoutTree = this;
	
					node->addAppliedData("QLayout", data);
				}

				connect(node->object(), SIGNAL(destroyed()), this, SLOT(deleted())); 
			}
			data->items.append(new TBoxLayoutDataItem(*item));
		}


		rebuildLayout();
	}
	delete (win);
}



void TStoreTypeQBoxLayout::layoutMemberDeleted(TStoreType *deletedTree) {
	TBoxLayoutData *data = (TBoxLayoutData*)findData("QBoxLayout");
	QListIterator<TBoxLayoutDataItem> oldit(data->items);

	for ( ; oldit.current() ; ++oldit) {
		if (oldit.current()->objectTree == deletedTree) {
			TStoreType *node = oldit.current()->objectTree;
			node->deleteAppliedData("QLayout");
			disconnect(node->object(), SIGNAL(destroyed()), this, SLOT(deleted()));
			data->items.remove(oldit.current());
			break;
		}
	}

	TStoreTypeQLayout::layoutMemberDeleted(deletedTree);
}






/******************************************************
**
**  TBoxLayoutDataItem
**
******************************************************/

TBoxLayoutDataItem::TBoxLayoutDataItem() {
	type = TBoxLayoutDataItem::Space;
	objectTree = NULL;
	size = 0;
	alignment = AlignCenter;
}

TBoxLayoutDataItem::TBoxLayoutDataItem(TBoxLayoutDataItem &item) {
	type = item.type;
	objectTree = item.objectTree;
	size = item.size;
	alignment = item.alignment;
}

TBoxLayoutDataItem &TBoxLayoutDataItem::operator=(TBoxLayoutDataItem &item) {
	type = item.type;
	objectTree = item.objectTree;
	size = item.size;
	alignment = item.alignment;

	return *this;
}





/******************************************************
**
**  TBoxLayoutData
**
******************************************************/

TBoxLayoutData::TBoxLayoutData() {
	direction = QBoxLayout::LeftToRight;
	border = 0;
};


TBoxLayoutData::TBoxLayoutData(TBoxLayoutData &data) {
	direction = data.direction;
	border = data.border;

	for (uint i=0 ; i<data.items.count() ; i++) {
		TBoxLayoutDataItem *ni = new TBoxLayoutDataItem(*data.items.at(i));
		items.append(ni);
	}
};



TBoxLayoutData &TBoxLayoutData::operator=(TBoxLayoutData &data) {
	direction = data.direction;
	border = data.border;

	for (uint i=0 ; i<data.items.count() ; i++) {
		TBoxLayoutDataItem *ni = new TBoxLayoutDataItem(*data.items.at(i));
		items.append(ni);
	}
	return *this;
}








void TStoreTypeQBoxLayout::rebuild(bool setRebuild) {
	TBoxLayoutData *data = (TBoxLayoutData*) findData("QBoxLayout");

	QListIterator<TBoxLayoutDataItem> it(data->items);
	for ( ; it.current() ; ++it) {
		TBoxLayoutDataItem *item = it.current();

		if (item->type == TBoxLayoutDataItem::Layout)
			((TStoreTypeQBoxLayout*)item->objectTree)->rebuild(setRebuild);
	}
	TStoreTypeQLayout::rebuild(setRebuild);
}
