// Copyright (C) 1999 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifndef	__APE_SMDI_H__
#define	__APE_SMDI_H__

#ifndef	__APE_THREAD_H__
#include <APE/thread.h>
#endif

#ifndef	__APE_SOCKET_H__
#include <APE/socket.h>
#endif

#ifndef	__APE_SERIAL_H__
#include <APE/serial.h>
#endif

enum
{
	SMDI_DIRECT = 1,
	SMDI_FWDALL,
	SMDI_FWDBUSY,
	SMDI_FWDNA
};

#ifdef	WIN32
class __EXPORT SMDI;
class __EXPORT SMDISerial;
class __EXPORT SMDINetwork;
#endif

/**
 * The SMDI service thread responds to SMDI service events that are
 * received and distributes those events through several abstract
 * calls that can be defined in an application specific class.  The
 * SMDI service thread also defines entry points for sending SMDI
 * control messages such as changing message waiting lamp status.
 * 
 * @author David Sugar <dyfet@ostel.com>
 * @short SMDI service thread base class.
 */

class SMDI : public Thread, public Mutex
{
private:
	time_t	lastio;

protected:
	/**
	 * Handler to call in a derived class when contact with the SMDI
	 * data source has been broken.  By default, it simply exits the
	 * SMDI service thread.
	 */
	virtual void OnExit(void)
		{Exit();};
	
	/**
	 * Handler to call in derived class when an incoming ring event
	 * is received.  SMDI ring events include SMDI_DIRECT, SMDI_FWDALL,
	 * SMDI_FWDBUSY, and SMDI_FWDNA.
	 * 
	 * @param desk number of SMDI entity ringing.
	 * @param port number or line of SMDI entity ringing.
	 * @param mtype message type identifier.
	 * @param cid calling station id.
	 * @param fwd forwarding station for integrated greetings.
	 */
	virtual void OnRing(int desk, int port, int mtype, char *cid, char *fwd)
		{return;};

	/**
	 * Handler to call in derived class when SMDI status messages are
	 * received.  These are typically associated with message waiting
	 * lamp control requests.
	 * 
	 * @param msg status string from SMDI source.
	 */
	virtual void OnStat(char *msg)
		{return;};

	/**
	 * Send SMDI message waiting message in either the network or
	 * serial protocol derived class.
	 * 
	 * @return -1 if error.
	 * @param msg waiting control string.
	 */
	virtual int SendMsgWaiting(char *msg) = 0;
	/**
	 * Read input from either the network or serial session protocol
	 * class.
	 * 
	 * @return null if no data is available or null terminated string.
	 */
	virtual char *Readline() = 0;
	/**
	 * SMDI service thread routine.  This calls Readline to receive
	 * input from the session protocol (network or serial) and the
	 * various dispatch routines based on the type of message received.
	 */
	void Run(void);

public:
	/**
	 * Create an SMDI service thread for the derived class at the
	 * specified priority level.  The Start() method should be used
	 * to kick it off once the class is created.
	 * 
	 * @param pri level for the service thread.
	 */
	SMDI(int pri);
	/**
	 * Destroy the SMDI service thread and any resources it uses.
	 */
	virtual ~SMDI();

	/**
	 * Turn the message waiting lamp on at the specified targeted
	 * directory number.
	 * 
	 * @return 0 on success, -1 on error.
	 * @param target directory station number.
	 */
	int setMsgWaiting(char *target);
	/**
	 * Turn the message waiting lamp off at the specified targeted
	 * directory number.
	 * 
	 * @return 0 on success, -1 on error.
	 * @param target directory station number.
	 */
	int clrMsgWaiting(char *target);
	/**
	 * Return number of seconds that have transpired since any SMDI
	 * message has been received and processed by the service thread.
	 * This can be used to determine if the link is broken.
	 * 
	 * @return int number of seconds since last input received.
	 */
	int getActivity(void);
};

/**
 * A class to implement SMDI through a serial device connection using
 * the APE serial class.
 * 
 * @author David Sugar <dyfet@ostel.com>
 * @short SMDI service protocol via serial device.
 */
class SMDISerial : public SMDI, protected Serial
{
protected:
	/**
	 * Serial port derived method for actually sending SMDI
	 * message waiting control messages to the switch.
	 * 
	 * @return -1 if error.
	 * @param msg waiting control string.
	 */
	int SendMsgWaiting(char *msg)
		{return Write(msg);};

	/**
	 * Serial port derived method for reading SMDI input into
	 * the service thread.
	 * 
	 * @return null if no data is available or null terminated string.
	 */
	char *Readline(void);

public:
	/**
	 * Connect SMDI service thread to a specified serial device.
	 * 
	 * @param device name of serial port.
	 * @param pri level of SMDI service thread.
	 * @param serial baud of serial connection (normally 1200bps).
	 */
	SMDISerial(char *device, int pri = 0, long speed = 1200);
};

/**
 * A class to implement SMDI through a TCP network connection using
 * the APE TCP socket class.
 * 
 * @author David Sugar <dyfet@ostel.com>
 * @short SMDI service protocol via TCP network session.
 */

class SMDINetwork : public SMDI, protected TCPSocket
{
protected:
	/**
	 * TCP session derived method for actually sending SMDI
	 * message waiting control messages to the switch.
	 * 
	 * @return -1 if error.
	 * @param msg waiting control string.
	 */	
	int SendMsgWaiting(char *msg)
		{return Write(msg);};
	
	/**
	 * TCP session derived method for reading SMDI input into
	 * the service thread.
	 * 
	 * @return null if no data is available or null terminated string.
	 */
	char *Readline(void);

public:
	/**
	 * Connect SMDI service thread to a TCP network session with a
	 * TCP SMDI protocol server.
	 * 
	 * @param host address for TCP SMDI session server.
	 * @param port address of TCP SMDI session server.
	 * @param pri level for service thread.
	 */
	SMDINetwork(InetHostAddress &host, short port, int pri = 0);
};

#endif
