/*
 * binedit.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include "binedit.h"
#include "button.h"
#include "dialogs.h"
#include "inpline.h"
#include "label.h"
#include "scrbar.h"

/*
 * BinIndicator constructor.
 */

BinIndicator::BinIndicator(Rect bounds): View(bounds)
{
	DEBUG("new BinIndicator at %p\n", this);
	link = NULL;
}

/*
 * Draws the BinIndicator on the screen.
 */

void BinIndicator::draw()
{
	DrawBuf buf;
	int col = getColor(C_BININDICATOR);

	moveChar(buf, ' ', col, size.x);
	if (link != NULL)
	{
		char str[MAX_WIDTH];

		sprintf(str, "%c %7d bytes", link->modified ? '\xf' : ' ',
			link->length);
		moveStr(buf, str, col);
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Sets the link of the BinIndicator.
 */

void BinIndicator::setLink(BinEditor *alink)
{
	link = alink;
	drawView();
}

/*
 * BinEditor constructor.
 */

BinEditor::BinEditor(int x, int y, char *afile, ScrollBar *avs,
	BinIndicator *aind): View(Rect(x, y, x + 73, y + 16))
{
	DEBUG("new BinEditor at %p\n", this);
	options |= OF_SELECTABLE;
	block = NULL;
	current = home = length = modified = 0;
	strncpy2(file, afile, sizeof(file));
	ind = aind;
	if (file[0] != '\0') loadFile();
	if ((vs = avs) != NULL) vs->setParams(0, 0, maxScroll(), 256, 16);
}

/*
 * BinEditor destructor.
 */

BinEditor::~BinEditor()
{
	DEBUG("delete BinEditor at %p\n", this);
	delete block;
}

/*
 * Modifies a byte in the file.
 */

void BinEditor::adjustValue()
{
	while (1)
	{
		Dialog *dlg = new Dialog(Rect(0, 0, 26, 9), "Change data");
		InputLine *il;
		InputLineData ild;

		sprintf(ild.value, "0x%02x", block[current]);
		dlg->insert(new Button(Rect(3, 6, 12, 8), "O~K~", CM_OK,
			BF_DEFAULT));
		dlg->insert(new Button(Rect(14, 6, 23, 8), "~C~ancel",
			CM_CANCEL));
		dlg->insert(il = new InputLine(Rect(3, 3, 23, 4), ""));
		dlg->insert(new Label(Rect(3, 2, 8, 3), "~V~alue", il));
		if (program->execDialog(dlg, &ild) == CM_OK)
		{
			char *tail;
			int val = strtol(ild.value, &tail, 0);

			if (ild.value != tail && val >= 0 && val <= 255)
			{
				if (block[current] != val)
				{
					block[current] = val;
					modified = 1;
					updateScreen();
				}
				break;
			}
			else messageBox("Wrong value", NULL, MF_ERROR |
				MF_OKBUTTON);
		}
		else break;
	}
}

/*
 * Draws the BinEditor on the screen.
 */

void BinEditor::draw()
{
	DrawBuf buf;
	Video *line = buf;
	int col_left = getColor(C_BINEDITOR);

	moveChar(buf, ' ', col_left, size.x * size.y);
	if (block != NULL)
	{
		int col_mid = getColor(C_BINEDITOR + 1);
		int col_midcur = getColor(C_BINEDITOR + 2);
		int col_right = getColor(C_BINEDITOR + 3);
		int col_rightcur = getColor(C_BINEDITOR + 4);
		unsigned char *end = block + MIN(home + 256, length);
		unsigned char *src = block + home;

		while (src < end)
		{
			char str[MAX_WIDTH];
			int i;

			sprintf(str, "%08x", src - block);
			moveStr(line, str, col_left);
			for (i = 0; i < 16 && src < end; i++)
			{
				int col_ascii, col_hex;

				if (src == block + current)
				{
					col_ascii = col_rightcur;
					col_hex = col_midcur;
				}
				else
				{
					col_ascii = col_right;
					col_hex = col_mid;
				}
				sprintf(str, "%02x", *src);
				moveStr(line + i * 3 + 9, str, col_hex);
				moveChar(line + i + 57, *src++, col_ascii, 1);
			}
			line += size.x;
		}
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Enables or disables BinEditor commands.
 */

void BinEditor::enableCommands(int enable)
{
	setCommandState(CM_SAVE, enable);
	setCommandState(CM_SAVEAS, enable);
}

/*
 * Handles BinEditor events.
 */

void BinEditor::handleEvent(Event &event)
{
	View::handleEvent(event);
	if (block != NULL) switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_SCROLLBARCHANGED:
			if (event.sender == vs && vs != NULL)
			{
				home = vs->getValue();
				updateScreen();
			}
		}
		break;
	case EV_COMMAND:
		switch (event.command)
		{
		case CM_SAVE:
			saveFile();
			clearEvent(event);
			break;
		case CM_SAVEAS:
			saveFileAs();
			clearEvent(event);
		}
		break;
	case EV_KEYDOWN:
		trackKey(event.keycode);
		clearEvent(event);
		break;
	case EV_MOUSEDOUBLE:
		if (trackMouse(event)) adjustValue();
		clearEvent(event);
		break;
	case EV_MOUSEDOWN:
		trackMouse(event);
		clearEvent(event);
	}
}

/*
 * Returns nonzero if the given command can be executed.
 */

int BinEditor::isValid(int command)
{
	switch (command)
	{
	case CM_CLOSE:
	case CM_QUIT:
		if (modified)
		{
			char dir[PATH_MAX];
			char name[PATH_MAX];

			expandPath(file, dir, name);
			switch (messageBox("%s has been modified. Save?",
				name, MF_INFORMATION | MF_YESNOCANCEL))
			{
			case CM_YES:
				return saveFile() >= 0;
			case CM_CANCEL:
				return 0;
			}
		}
	}
	return 1;
}

/*
 * Loads the file.
 */

int BinEditor::loadFile()
{
	FILE *fs;

	if ((fs = fopen(file, "r")) == NULL)
	{
		messageBox("Error opening file %s", file,
			MF_ERROR | MF_OKBUTTON);
		return -1;
	}
	fseek(fs, 0, SEEK_END);
	if ((length = ftell(fs)) == 0)
	{
		fclose(fs);
		messageBox("File %s is empty", file, MF_ERROR | MF_OKBUTTON);
		return -1;
	}
	rewind(fs);
	block = new unsigned char [length];
	if (fread(block, length, 1, fs) != 1)
	{
		fclose(fs);
		messageBox("Error reading file %s", file,
			MF_ERROR | MF_OKBUTTON);
		return -1;
	}
	fclose(fs);
	return 0;
}

/*
 * Returns the offset of the last line of the file.
 */

int BinEditor::maxScroll()
{
	return MAX(0, ((length + 15) / 16 - 16) * 16);
}

/*
 * Saves the file.
 */

int BinEditor::saveFile()
{
	FILE *fs;
	char backup[PATH_MAX];

	/* first make a backup file */

	sprintf(backup, "%s~", file);
	rename(file, backup);

	/* now save the file */

	if ((fs = fopen(file, "w")) == NULL)
	{
		messageBox("Error opening file %s", file,
			MF_ERROR | MF_OKBUTTON);
		return -1;
	}
	if (fwrite(block, length, 1, fs) != 1)
	{
		fclose(fs);
		messageBox("Error writing file %s", file,
			MF_ERROR | MF_OKBUTTON);
		return -1;
	}
	fclose(fs);
	modified = 0;
	updateScreen();
	return 0;
}

/*
 * Saves the file with a new name.
 */

int BinEditor::saveFileAs()
{
	FileDialogData fdd;

	strcpy(fdd.file, file);
	if (program->execDialog(new FileDialog("Save file as"),
		&fdd) != CM_OK) return -1;
	strcpy(file, fdd.file);
	sendMessage(owner, EV_BROADCAST, CM_UPDATETITLE);
	return saveFile();
}

/*
 * Handles keyboard events.
 */

void BinEditor::trackKey(int keycode)
{
	switch (keycode)
	{
	case KC_ARROWLEFT:
		if (current > 0)
		{
			current--;
			if (current < home) home -= 16;
		}
		break;
	case KC_ARROWUP:
		if (current >= 16)
		{
			current -= 16;
			if (current < home) home -= 16;
		}
		break;
	case KC_ARROWRIGHT:
		if (current < length - 1)
		{
			current++;
			if (current > home + 255) home += 16;
		}
		break;
	case KC_ARROWDOWN:
		if (current <= length - 17)
		{
			current += 16;
			if (current > home + 255) home += 16;
		}
		break;
	case KC_END:
		current = length - 1;
		home = maxScroll();
		break;
	case KC_HOME:
		current = home = 0;
		break;
	case KC_PAGEDOWN:
		current += 256;
		home += 256;
		break;
	case KC_PAGEUP:
		current -= 256;
		home -= 256;
		break;
	case ' ':
	case KC_ENTER:
		adjustValue();
	}
	updateScreen();
}

/*
 * Selects another byte cell.
 */

int BinEditor::trackMouse(Event &event)
{
	int pos = -1;

	makeLocal(event.where, event.where);
	if (event.where.x >= 9 && event.where.x <= 55)
	{
		pos = home + event.where.y * 16 + (event.where.x - 9) / 3;
	}
	else if (event.where.x >= 57 && event.where.x <= 72)
	{
		pos = home + event.where.y * 16 + event.where.x - 57;
	}
	if (pos >= 0 && pos < length)
	{
		current = pos;
		updateScreen();
		return 1;
	}
	return 0;
}

/*
 * Redraws the screen and the ScrollBar.
 */

void BinEditor::updateScreen()
{
	home = RANGE(home, 0, maxScroll());
	current = RANGE(current, home, MIN(home + 255, length - 1));
	if (ind != NULL) ind->drawView();
	if (vs != NULL) vs->setValue(home);
	drawView();
}

/*
 * BinEditWindow constructor.
 */

BinEditWindow::BinEditWindow(int x, int y, char *afile): Window(Rect(x, y, 75,
	18), "", Window::initFrame)
{
	DEBUG("new BinEditWindow at %p\n", this);
	BinIndicator *ind;
	Rect r;
	ScrollBar *vs;

	flags = WF_CLOSE | WF_MOVE;
	editor = NULL;
	getExtent(r);
	insert(vs = new ScrollBar(Rect(r.b.x - 1, 1, r.b.x, r.b.y - 1)));
	insert(ind = new BinIndicator(Rect(1, r.b.y - 1, 16, r.b.y)));
	insert(editor = new BinEditor(1, 1, afile, vs, ind));
	ind->setLink(editor);
}

/*
 * Updates the EditWindow's commands.
 */

void BinEditWindow::enableCommands(int enable)
{
	if (editor != NULL) editor->enableCommands(enable);
	Window::enableCommands(enable);
}

/*
 * Returns the title of the BinEditWindow.
 */

char *BinEditWindow::getTitle()
{
	return editor != NULL ? editor->file : Window::getTitle();
}
