/*
 * cluster.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stddef.h>
#include "cluster.h"

/*
 * Cluster constructor.
 */

Cluster::Cluster(Rect bounds, ClusterEntry *afirst): View(bounds)
{
	DEBUG("new Cluster at %p\n", this);
	options |= OF_FIRSTCLICK | OF_POSTPROCESS | OF_SELECTABLE;
	current = 0;
	data.bit = 0;
	data.mask = ~0;
	first = afirst;
}

/*
 * Cluster destructor.
 */

Cluster::~Cluster()
{
	while (first != NULL)
	{
		ClusterEntry *next = first->next;

		delete first;
		first = next;
	}
}

/*
 * Returns the size of the Cluster's data.
 */

int Cluster::dataSize()
{
	return sizeof(ClusterData);
}

/*
 * Draws the Cluster on the screen.
 */

void Cluster::draw()
{
	drawState(-1);
}

/*
 * Draws all possible states of the Cluster.
 */

void Cluster::drawBasic(char sign, char *style, int blip)
{
	ClusterEntry *obj = first;
	DrawBuf buf;
	Video *where = buf;
	int col_disabled = getColor(C_CLUSTER), col_normal, col_selected;
	int item = 0;
	int bit = data.bit;
	int mask = data.mask;

	if (state & SF_ACTIVE)
	{
		col_normal = getColor(C_CLUSTER + 1);
		col_selected = state & SF_FOCUSED ? getColor(C_CLUSTER + 2) :
			col_normal;
	}
	else col_normal = col_selected = col_disabled;
	while (obj != NULL)
	{
		int col;

		if (mask & 1)
		{
			col = item == current ? col_selected : col_normal;
		}
		else col = col_disabled;
		style[1] = blip-- == 0 ? '\xf' : bit & 1 ? sign : ' ';
		moveChar(where, ' ', col, size.x);
		moveStr(where, style, col);
		moveCStr(where + 4, obj->name, col); 
		item++;
		obj = obj->next;
		bit >>= 1;
		mask >>= 1;
		where += size.x;
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Finds an item in the specified direction with the mask bit enabled.
 */

void Cluster::findValid(int delta)
{
	if (delta == 0)
	{
		/* select the first available item */

		current = 0;
		while (current < size.y)
		{
			if (data.mask & (1 << current)) break;
			current++;
		}
		if (current >= size.y) current = -1;
	}
	else
	{
		int test = current;

		/* move forward ? */

		if (delta > 0) while (++test < size.y)
		{
			if (data.mask & (1 << test))
			{
				current = test;
				break;
			}
		}
		else if (delta < 0) while (--test >= 0)
		{
			/* move backward */

			if (data.mask & (1 << test))
			{
				current = test;
				break;
			}
		}
	}
	drawView();
}

/*
 * Gets the Cluster's data.
 */

void Cluster::getData(void *buf)
{
	ClusterData *cd = (ClusterData *) buf;

	*cd = data;
}

/*
 * Handles the Cluster's events.
 */

void Cluster::handleEvent(Event &event)
{
	ClusterEntry *obj;
	Point pnt;
	int item = -1, test;

	View::handleEvent(event);
	switch (event.what)
	{
	case EV_KEYDOWN:
		switch (owner->getPhase())
		{
		case PH_FOCUSED:
			switch (event.keycode)
			{
			case KC_ARROWUP:
				findValid(-1);
				clearEvent(event);
				break;
			case KC_ARROWDOWN:
				findValid(1);
				clearEvent(event);
			}
			break;
		case PH_POSTPROCESS:
			obj = first;
			test = 0;
			while (obj != NULL)
			{
				if (event.keycode ==
					MK_META(hotKey(obj->name)) &&
					data.mask & (1 << test))
				{
					current = item = test;
					getFocus();
					clearEvent(event);
					break;					
				}
				obj = obj->next;
				test++;
			}
		}
		break;
	case EV_MOUSEDOWN:
		do
		{
			makeLocal(event.where, pnt);
			switch (event.what)
			{
			case EV_MOUSEDOWN:
				if (data.mask & (1 << pnt.y))
				{
					current = item = pnt.y;
					drawState(item);
				}
				break;
			case EV_MOUSEMOVE:
				if (item >= 0 && (!localPointInside(pnt) ||
					pnt.y != item))
				{
					item = -1;
					drawState(item);
				}
			}
			getEvent(event);
		}
		while (event.what != EV_MOUSEUP);
		drawView();
		clearEvent(event);
	}
	if (item >= 0)
	{
		makePress(item);
		sendMessage(owner, EV_BROADCAST, CM_CLUSTERCHANGED);
	}
}

/*
 * Sets the Cluster's data.
 */

void Cluster::setData(void *buf)
{
	ClusterData *cd = (ClusterData *) buf;

	data = *cd;
	setState(SF_ACTIVE, data.mask != 0);
	findValid(0);
}

/*
 * ClusterEntry constructor.
 */

ClusterEntry::ClusterEntry(char *aname, ClusterEntry *anext)
{
	name = aname;
	next = anext;
}

/*
 * CheckBox constructor.
 */

CheckBox::CheckBox(Rect bounds, ClusterEntry *obj): Cluster(bounds, obj)
{
	findValid(0);
}

/*
 * Draws the CheckBox on the screen.
 */ 

void CheckBox::drawState(int blip)
{
	static char style[] = "[ ]";

	drawBasic('X', style, blip);
}

/*
 * Handles the CheckBox events.
 */

void CheckBox::handleEvent(Event &event)
{
	Cluster::handleEvent(event);
	switch (event.what)
	{
	case EV_KEYDOWN:
		switch (owner->getPhase())
		{
		case PH_FOCUSED:
			switch (event.keycode)
			{
			case KC_ENTER:
			case ' ':
				if (current >= 0)
				{
					makePress(current);
					sendMessage(owner, EV_BROADCAST,
						CM_CLUSTERCHANGED);
					clearEvent(event);
				}
			}
		}
	}
}

/*
 * Makes some effects on the bits.
 */

void CheckBox::makePress(int item)
{
	data.bit ^= 1 << item;
	drawView();
}

/*
 * RadioBox constructor.
 */

RadioBox::RadioBox(Rect bounds, ClusterEntry *obj): Cluster(bounds, obj)
{
	findValid(0);
}

/*
 * Finds an item in the specified direction with the mask bit enabled.
 */

void RadioBox::findValid(int delta)
{
	Cluster::findValid(delta);
	if (current != -1) makePress(current);
}

/*
 * Draws the RadioBox on the screen.
 */

void RadioBox::drawState(int blip)
{
	static char style[] = "( )";

	drawBasic('\x7', style, blip);
}

/*
 * Makes some effects on the bits.
 */

void RadioBox::makePress(int item)
{
	data.bit = 1 << item;
	drawView();
}
