/*
 * config.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include "config.h"

/*
 * Config constructor.
 */

Config::Config()
{
	stream = NULL;
}

/*
 * Config destructor.
 */

Config::~Config()
{
	if (stream != NULL) fclose(stream);
}

/*
 * Finds a field in the file.  Returns -1 if error.
 */

int Config::getField(char *section, char *field, char *value, int &line)
{
	char buf[CONFIG_MAXLEN + 1];
	char sec[CONFIG_MAXLEN + 1];

	/* there isn't a config file */

	if (stream == NULL) return -1;

	/* move to the file start */

	line = 1;
	rewind(stream);

	/* find the right section */

	sprintf(sec, "[%s]", section);
	do
	{
		/* go out if read error */

		if (getToken(buf, line) < 0) return -1;
	}
	while (strcasecmp(buf, sec) != 0);

	/* find the right field */

	do
	{
		/* go out if read error */

		if (getToken(buf, line) < 0) return -1;

		/* go out if it is the next section */

		if (buf[0] == '[' && buf[strlen(buf) - 1] == ']') return -1;
	}
	while (strcasecmp(buf, field) != 0);

	/* go out if read error or if it isn't an equal character */

	if (getToken(buf, line) < 0 || strcmp(buf, "=") != 0) return -1;

	/* read the value */

	if (getToken(buf, line) < 0) return -1;

	strcpy(value, buf);		/* update the string */
	return 0;
}

/*
 * Extracts a flag variable from the file.  Returns -1 if error.
 */
 
int Config::getFlag(char *section, char *field, int *flag)
{
	char buf[CONFIG_MAXLEN + 1];
	int line;

	if (getField(section, field, buf, line) >= 0)
	{
		if (strcmp(buf, "0") == 0 || strcasecmp(buf, "no") == 0)
		{
			*flag = 0;
			return 0;
		}
		if (strcmp(buf, "1") == 0 || strcasecmp(buf, "yes") == 0)
		{
			*flag = 1;
			return 0;
		}
		cerr << __FILE__": flag " << field << " in section " <<
			section << " at line " << line << " is not valid\n";
	}
	return -1;
}

/*
 * Extracts an integer variable from the file.  Returns -1 if error.
 */

int Config::getInt(char *section, char *field, int *num)
{
	char buf[CONFIG_MAXLEN + 1];
	int line;

	if (getField(section, field, buf, line) >= 0)
	{
		char *tail;
		int val = strtol(buf, &tail, 0);

		if (buf != tail)
		{
			*num = val;
			return 0;
		}
		cerr << __FILE__": integer " << field << " in section " <<
			section << " at line " << line << " is not valid\n";
	}
	return -1;
}

/*
 * Extracts a string variable from the file.  Returns -1 if error.
 */

int Config::getStr(char *section, char *field, char *str)
{
	int line;

	return getField(section, field, str, line);
}

/*
 * Gets a token from the file.  Returns -1 if error.
 */

int Config::getToken(char *buf, int &line)
{
	int c, comment, len = 0;

	/* there isn't a config file */

	if (stream == NULL) return -1;

	/* skip space and comment characters */

	do
	{
		while ((c = fgetc(stream)) != EOF && isspace(c))
		{
			if (c == '\n') line++;
		}

		/* is it the comment character ? */

		if ((comment = c == '#') != 0)
		{
			/* skip to the end of the line */
	
			while ((c = fgetc(stream)) != EOF)
			{
				if (c == '\n')
				{
					line++;
					break;
				}
			}
		}
	}
	while (comment);

	/* collects all the non-space characters */

	while (c != EOF && !isspace(c) && len <= CONFIG_MAXLEN)
	{
		buf[len++] = c;
		c = fgetc(stream);
	}
	if (c != EOF) ungetc(c, stream);	/* restore last character */

	if (len > CONFIG_MAXLEN)
	{
		buf[CONFIG_MAXLEN] = '\0';	/* truncate string */
		cerr << __FILE__": string too long at line " << line << "\n";
		return -1;
	}
	buf[len] = '\0';		/* append null terminator */
	return len > 0 ? 0 : -1;
}

/*
 * Opens a file.
 */

void Config::openConfig(char *fname)
{
	if (stream != NULL)
	{
		fclose(stream);
		stream = NULL;
	}
	if (fname != NULL && fname[0] != '\0')
	{
		if ((stream = fopen(fname, "r")) == NULL)
		{
			cerr << __FILE__": unable to open config " << fname <<
				"\n";
		}
	}
}
