/*
 * editor.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "button.h"
#include "config.h"
#include "dialogs.h"
#include "editor.h"
#include "label.h"
#include "scrbar.h"

char Editor::findstr[MAX_WIDTH] = "";
char Editor::replacestr[MAX_WIDTH] = "";

/*
 * Indicator constructor.
 */

Indicator::Indicator(Rect bounds): View(bounds)
{
	DEBUG("new Indicator at %p\n", this);
	growmode = GF_GROWLOY | GF_GROWHIY;
	link = NULL;
}

/*
 * Draws the indicator on the screen.
 */

void Indicator::draw()
{
	DrawBuf buf;
	int col = state & SF_ACTIVE ? getColor(C_INDICATOR + 1) :
		getColor(C_INDICATOR);

	moveChar(buf, ' ', col, size.x);
	if (link != NULL)
	{
		char str[MAX_WIDTH];

		sprintf(str, "%c %3d:%d ",
			link->flags & EF_MODIFIED ? '\xf' : ' ',
			link->curpos.x + 1, link->curpos.y + 1);
		moveStr(buf, str, col);
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Sets the link of the Indicator.
 */

void Indicator::setLink(Editor *alink)
{
	link = alink;
	drawView();
}

/*
 * Editor constructor.
 */

Editor::Editor(Rect bounds, ScrollBar *ahs, ScrollBar *avs, Indicator *aind):
	View(bounds)
{
	DEBUG("new Editor at %p\n", this);
	growmode = GF_GROWHIX | GF_GROWHIY;
	options |= OF_SELECTABLE;

	/* setup buffer */

	bufsize = EDITOR_BLOCKSIZE;
	buffer = (char *) malloc(bufsize);
	buflen = curptr = 0;

	/* setup pointers */

	hs = ahs;
	vs = avs;
	ind = aind;

	/* setup others */

	curpos.x = curpos.y = 0;
	delta.x = delta.y = 0;
	drawline = drawptr = 0;
	flags = 0;
	limit.x = MAX_WIDTH;
	limit.y = 1;
	locked = 0;
	selstart = selend = 0;
	tabwidth = 8;
	showCursor();
	updateScreen();
}

/*
 * Editor destructor.
 */

Editor::~Editor()
{
	DEBUG("delete Editor at %p\n", this);
	if (isClipboard()) clipboard = NULL;
	free(buffer);
}

/*
 * Returns the specified character in the buffer.
 */

int Editor::bufChar(int ptr)
{
	if (ptr >= buflen) return EOF;
	if (ptr >= curptr) ptr += bufsize - buflen;
	return (unsigned char) buffer[ptr];
}

/*
 * Returns the offset of the given character in the buffer.
 */

int Editor::bufPtr(int ptr)
{
	if (ptr >= curptr) ptr += bufsize - buflen;
	return ptr;
}

/*
 * Changes the bounds of the Editor.
 */

void Editor::changeBounds(Rect bounds)
{
	setBounds(bounds);
	delta.x = MAX(0, MIN(delta.x, limit.x - size.x));
	delta.y = MAX(0, MIN(delta.y, limit.y - size.y));
	updateScreen();
}

/*
 * Finds the position of the specified character.
 */

int Editor::charPos(int ptr, int target)
{
	int x = 0;

	while (ptr < target)
	{
		/* expand the tab character */

		if (bufChar(ptr++) == '\t') x = getNextTab(x);
		else x++;
	}
	return x;
}

/*
 * Finds the address of the specified character.
 */

int Editor::charPtr(int ptr, int target)
{
	int c = 0, x = 0;

	while (x < target && ptr < buflen && (c = bufChar(ptr)) != '\n')
	{
		/* expand the tab character */

		if (c == '\t') x = getNextTab(x);
		else x++;
		ptr++;
	}
	if (x > target) ptr--;
	return ptr;
}

/*
 * Copies the selected text to the clipboard.
 */

int Editor::clipCopy()
{
	if (clipboard != NULL && clipboard != this)
	{
		insertTo(clipboard, 1);
		return 1;
	}
	return 0;
}

/*
 * Cuts the selected text to the clipboard.
 */

void Editor::clipCut()
{
	if (clipCopy()) deleteSelect();
}

/*
 * Pastes text from the clipboard.
 */

void Editor::clipPaste()
{
	if (clipboard != NULL && clipboard != this)
	{
		clipboard->insertTo(this, 0);
	}
}

/*
 * Finds the command associated with the key.
 */

void Editor::convertEvent(Event &event)
{
	translatekey_s *table = translateTable();

	if (event.what != EV_KEYDOWN) return;
	while (table->command != CM_NONE || table->keycode != KC_NONE)
	{
		if (event.keycode == table->keycode)
		{
			event.what = EV_COMMAND;
			event.command = table->command;
			return;
		}
		table++;
	}
}

/*
 * Counts the lines in this editor buffer.
 */

int Editor::countBufLines(int from, int to)
{
	int lines = 0;

	if (from > to)
	{
		from ^= to;	/* swap from & to */
		to ^= from;
		from ^= to;
	}
	while (from < to)
	{
		if (bufChar(from++) == '\n') lines++;
	}
	return lines;
}

/*
 * Counts the lines in the specified buffer.
 */

int Editor::countLines(char *buf, int count)
{
	int lines = 0;

	while (count-- > 0)
	{
		if (*buf++ == '\n') lines++;
	}
	return lines;
}

/*
 * Deletes the selected text.
 */

void Editor::deleteSelect()
{
	deleteText(selstart, selend);
}

/*
 * Removes some text from the buffer.
 */

void Editor::deleteText(int start, int end)
{
	int length = end - start;
	int newcurptr = curptr;

	if (length <= 0 || flags & EF_READONLY) return;
#ifdef NOT_FULLY_DEBUGGED

	/* update selection */

	if (selstart > start) selstart = MAX(start, selstart - length);
	if (selend > start) selend = MAX(start, selend - length);
	if (curptr > start) newcurptr = MAX(start, curptr - length);

	if (drawptr > newcurptr)
	{
		int olddrawptr = drawptr;

		drawptr = lineStart(newcurptr);
		drawline -= countBufLines(drawptr, olddrawptr);
	}

	/* remove the text */

	setCurAddr(start);
	limit.y -= countBufLines(newcurptr, curptr);
#else
	setCurAddr(start);

	/* update selection */

	if (newcurptr > curptr) newcurptr = MAX(curptr, newcurptr - length);
	if (selstart > curptr) selstart = MAX(curptr, selstart - length);
	if (selend > curptr) selend = MAX(curptr, selend - length);

	/* remove the text */

	limit.y -= countLines(buffer + curptr + bufsize - buflen, length);
#endif
	buflen -= length;
	if (!isClipboard()) flags |= EF_MODIFIED;
	setCurAddr(newcurptr);
}

/*
 * Finds or replaces a string.
 */

void Editor::doSearchReplace()
{
	int command;

	do
	{
		command = CM_CANCEL;
		if (!searchString(findstr, flags))
		{
			if (!(flags & EF_REPLACE && flags & EF_REPLACEALL))
			{
				makeDialog(ED_SEARCHFAILED, NULL);
			}
		}
		else if (flags & EF_REPLACE)
		{
			command = CM_YES;
			if (flags & EF_PROMPTONREPLACE)
			{
				command = makeDialog(ED_REPLACEPROMPT, NULL);
			}
			if (command == CM_YES)
			{
				deleteSelect();
				insertText(replacestr, strlen(replacestr), 0);
			}
		}
	}
	while (command != CM_CANCEL && flags & EF_REPLACEALL);
}

/*
 * Draws the Editor on the screen.
 */

void Editor::draw()
{
	/* the first line is changed ? */

	if (drawline != delta.y)
	{
		/* move to the first line */

		drawptr = lineMove(drawptr, delta.y - drawline);
		drawline = delta.y;
	}
	drawLines(0, size.y, drawptr);
}

/*
 * Draws some text lines on the screen.
 */

void Editor::drawLines(int y, int count, int ptr)
{
	drawinfo_s info;

	if (state & SF_ACTIVE)
	{
		info.col_codenormal = getColor(C_EDITOR + 1);
		info.col_codeselect = getColor(C_EDITOR + 2);
		info.col_textnormal = getColor(C_EDITOR + 3);
		info.col_textselect = getColor(C_EDITOR + 4);
	}
	else
	{
		info.col_codenormal = info.col_codeselect =
		info.col_textnormal = info.col_textselect =
			getColor(C_EDITOR);
	}
	info.width = size.x + delta.x;
	if (flags & EF_SHOWCODES)
	{
		info.code_nl = 0x14;
		info.code_tab = 0xfa;
	}
	else info.code_nl = info.code_tab = ' ';
	while (count-- > 0)
	{
		formatLine(&info, ptr);
		writeBuf(0, y++, size.x, 1, info.buf + delta.x);

		/* move to the begin of the next line */

		ptr = lineEnd(ptr) + 1;
	}
}

/*
 * Updates the Editor's commands.
 */

void Editor::enableCommands(int enable)
{
	int clip = enable && clipboard != NULL && !isClipboard();

	setCommandState(CM_CUT, clip && hasSelection());
	setCommandState(CM_COPY, clip && hasSelection());
	setCommandState(CM_PASTE, clip && clipboard->hasSelection());
	setCommandState(CM_FIND, enable);
	setCommandState(CM_REPLACE, enable);
	setCommandState(CM_SEARCHAGAIN, enable);
	setCommandState(CM_SETOPTIONS, enable);
}

/*
 * Finds a string.
 */

void Editor::findString()
{
	FindDialogData fdd =
	{
		{""},		/* InputLine */
		{0, ~0}		/* Cluster */
	};

	strcpy(fdd.find.value, findstr);
	fdd.options.bit = flags;
	if (makeDialog(ED_FIND, &fdd) == CM_OK)
	{
		strcpy(findstr, fdd.find.value);
		flags = fdd.options.bit & ~EF_REPLACE;
		doSearchReplace();
	}
}

/*
 * Draws the specified line in the video buffer.
 */

void Editor::formatLine(drawinfo_s *info, int ptr)
{
	Video *buf = info->buf;
	int x = 0;

	while (x < info->width && ptr < buflen)
	{
		int c, col_code, col_text;

		if (selstart <= ptr && ptr < selend)
		{
			col_code = info->col_codeselect;
			col_text = info->col_textselect;
		}
		else
		{
			col_code = info->col_codenormal;
			col_text = info->col_textnormal;
		}

		/* expand the tab character */

		if ((c = bufChar(ptr++)) == '\t')
		{
			int end = getNextTab(x);

			while (x < info->width && x < end)
			{
				buf[x].split.code = info->code_tab;
				buf[x++].split.attr = col_code;
			}
		}
		else if (c == '\n')
		{
			if (info->code_nl != ' ')
			{
				buf[x].split.code = info->code_nl;
				buf[x++].split.attr = col_code;
			}
			break;
		}
		else
		{
			buf[x].split.code = c;
			buf[x++].split.attr = col_text;
		}
	}

	/* clear last characters */

	moveChar(buf + x, ' ', info->col_textnormal, info->width - x);
}

/*
 * Returns the address of the character at the given coordinates.
 */

int Editor::getMousePtr(Point mouse)
{
	mouse.x = MAX(0, MIN(mouse.x, size.x - 1));
	mouse.y = MAX(0, MIN(mouse.y, size.y - 1));
	return charPtr(lineMove(drawptr, mouse.y + delta.y - drawline),
		mouse.x + delta.x);
}

/*
 * Returns the coordinate of the next tab stop.
 */

int Editor::getNextTab(int x)
{
	return x - x % tabwidth + tabwidth; 
}

/*
 * Handles the Editor's events.
 */

void Editor::handleEvent(Event &event)
{
	int centercursor = !isCursorVisible();
	int start;

	View::handleEvent(event);

	/* convert keycodes to commands */

	convertEvent(event);
	switch (event.what)
	{
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_SCROLLBARCHANGED:
			if (event.sender == hs || event.sender == vs)
			{
				if (hs != NULL) delta.x = hs->getValue();
				if (vs != NULL) delta.y = vs->getValue();
				scrollTo(delta.x, delta.y);
			}
		}
		break;
	case EV_COMMAND:
		lockDraw();
		switch (event.command)
		{
		case CM_COPY:
			clipCopy();
			break;
		case CM_CUT:
			clipCut();
			break;
		case CM_PASTE:
			clipPaste();
			break;
		case CM_FIND:
			findString();
			break;
		case CM_REPLACE:
			replaceString();
			break;
		case CM_SEARCHAGAIN:
			doSearchReplace();
			break;
		case CM_SETOPTIONS:
			setOptions();
			break;
		case CM_CHARLEFT:
			setCurAddr(prevChar(curptr));
			break;
		case CM_CHARRIGHT:
			setCurAddr(nextChar(curptr));
			break;
		case CM_LINESTART:
			setCurAddr(lineStart(curptr));
			break;
		case CM_LINEEND:
			setCurAddr(lineEnd(curptr));
			break;
		case CM_LINEUP:
			setCurAddr(lineMove(curptr, -1));
			break;
		case CM_LINEDOWN:
			setCurAddr(lineMove(curptr, 1));
			break;
		case CM_NEWLINE:
			newLine();
			break;
		case CM_PAGEUP:
			setCurAddr(lineMove(curptr, -size.y));
			break;
		case CM_PAGEDOWN:
			setCurAddr(lineMove(curptr, size.y));
			break;
		case CM_SELEND:
			setSelect(selstart, curptr);
			break;
		case CM_SELSTART:
			setSelect(curptr, selend);
			break;
		case CM_TEXTSTART:
			setCurAddr(0);
			break;
		case CM_TEXTEND:
			setCurAddr(buflen);
			break;
		case CM_WORDLEFT:
			setCurAddr(prevWord(curptr));
			break;
		case CM_WORDRIGHT:
			setCurAddr(nextWord(curptr));
			break;
		case CM_BACKSPACE:
			deleteText(prevChar(curptr), curptr);
			break;
		case CM_DELCHAR:
			deleteText(curptr, nextChar(curptr));
			break;
		case CM_DELSTART:
			deleteText(lineStart(curptr), curptr);
			break;
		case CM_DELEND:
			deleteText(curptr, lineEnd(curptr));
			break;
		case CM_DELLINE:
			deleteText(lineStart(curptr), nextLine(curptr));
			break;
		case CM_DELWORD:
			deleteText(curptr, nextWord(curptr));
			break;
		default:
			unlockDraw();
			return;
		}
		trackCursor(centercursor);
		unlockDraw();
		clearEvent(event);
		break;
	case EV_KEYDOWN:
		if (event.keycode == KC_TAB) event.keycode = '\t';
		if ((event.keycode >= ' ' && event.keycode <= 255) ||
			event.keycode == '\t')
		{
			lockDraw();
			insertText((char *) &event.keycode, 1, 0);
			trackCursor(centercursor);
			unlockDraw();
		}
		clearEvent(event);
		break;
	case EV_MOUSEDOWN:
		makeLocal(event.where, event.where);
		setCurAddr(start = getMousePtr(event.where));
		do
		{
			getEvent(event);
			if (event.what == EV_MOUSEMOVE)
			{
				Point mouse;

				makeLocal(event.where, mouse);
				selstart = start;
				selend = getMousePtr(mouse);
				if (mouse.x < 0) delta.x--;
				if (mouse.x >= size.x) delta.x++;
				if (mouse.y < 0) delta.y--;
				if (mouse.y >= size.y) delta.y++;
				scrollTo(delta.x, delta.y);
			}
		}
		while (event.what != EV_MOUSEUP);
		clearEvent(event);
	}
}

/*
 * Returns nonzero if the selection is active.
 */

int Editor::hasSelection()
{
	return selstart < selend;
}

/*
 * Inserts some text in the buffer at the cursor location.
 */

void Editor::insertText(char *buf, int length, int select)
{
	int lines;

	if (length <= 0 || flags & EF_READONLY) return;

	/* resize the buffer ? */

	if (buflen + length >= bufsize)
	{
		int oldbufsize = bufsize;
		int tomove = buflen - curptr;

		bufsize += length + EDITOR_BLOCKSIZE;
		buffer = (char *) realloc(buffer, bufsize);
		memmove(buffer + bufsize - tomove,
			buffer + oldbufsize - tomove, tomove);
	}

	/* update selection */

	if (select) selstart = curptr;
	else
	{
		if (selstart > curptr) selstart += length;
		if (selend > curptr) selend += length;
	}

	/* insert the text */

	memmove(buffer + curptr, buf, length);
	buflen += length;
	curptr += length;
	if (select) selend = curptr;

	/* update cursor position */

	lines = countLines(buf, length);
	curpos.x = charPos(lineStart(curptr), curptr);
	curpos.y += lines;
	limit.y += lines;
	if (!isClipboard()) flags |= EF_MODIFIED;
	updateScreen();
}

/*
 * Inserts the selected text to the specified Editor.
 */

void Editor::insertTo(Editor *target, int select)
{
	if (hasSelection())
	{
		int old = curptr;

		setCurAddr(0);
		target->insertText(buffer + bufPtr(selstart),
			selend - selstart, select);
		setCurAddr(old);
		target->trackCursor(!target->isCursorVisible());
	}
}

/*
 * Returns nonzero if the Editor is the clipboard.
 */

int Editor::isClipboard()
{
	return clipboard == this;
}

/*
 * Returns nonzero if the y coordinate of the cursor is inside the view.
 */

int Editor::isCursorVisible()
{
	return curpos.y >= delta.y && curpos.y < delta.y + size.y;
}

/*
 * Returns nonzero if the specified string is a whole word.
 */

int Editor::isWholeWord(int ptr, int len)
{
	if (ptr > 0 && isWordChar(bufChar(ptr - 1))) return 0;
	if (ptr + len < buflen && isWordChar(bufChar(ptr + len))) return 0;
	return 1;
}

/*
 * Returns nonzero if it is an alphanumeric character.
 */

int Editor::isWordChar(int code)
{
	return isalnum(code) || code == '_';
}

/*
 * Moves the cursor to the end of the line.
 */

int Editor::lineEnd(int ptr)
{
	while (ptr < buflen && bufChar(ptr) != '\n')
	{
		ptr++;
	}
	return MIN(ptr, buflen);
}

/*
 * Moves the cursor to the specified line.
 */

int Editor::lineMove(int ptr, int count)
{
	int cur = lineStart(ptr);

	/* find the x position */

	int x = charPos(cur, ptr);

	/* move forward */

	while (count > 0)
	{
		cur = nextLine(cur);
		count--;
	}

	/* move backward */

	while (count < 0)
	{
		cur = prevLine(cur);
		count++;
	}

	/* go to the x position */

	return charPtr(lineStart(cur), x);
}

/*
 * Moves the cursor to the start of the line.
 */

int Editor::lineStart(int ptr)
{
	while (ptr > 0 && bufChar(ptr - 1) != '\n')
	{
		ptr--;
	}
	return MAX(ptr, 0);
}

/*
 * Locks the Editor.
 */

void Editor::lockDraw()
{
	locked++;
}

/*
 * Creates and executes the specified Dialog.
 */

int Editor::makeDialog(int type, void *buf)
{
	Dialog *dlg = NULL;
	View *link;
	char dir[PATH_MAX];
	char name[PATH_MAX];

	switch (type)
	{
	case ED_FIND:
		dlg = new Dialog(Rect(0, 0, 38, 11), "Find");
		dlg->insert(link = new InputLine(Rect(3, 3, 35, 4), ""));
		dlg->insert(new Label(Rect(3, 2, 15, 3), "~T~ext to find",
			link));
		dlg->insert(new CheckBox(Rect(3, 5, 35, 7),
			new ClusterEntry("Case ~s~ensitive",
			new ClusterEntry("~W~hole words only",
			NULL))
		));
		dlg->insert(new Button(Rect(13, 8, 23, 10), "O~K~", CM_OK,
			BF_DEFAULT));
		dlg->insert(new Button(Rect(25, 8, 35, 10), "Cancel",
			CM_CANCEL));
		dlg->focusNext(1);
		break;
	case ED_REPLACE:
		dlg = new Dialog(Rect(0, 0, 38, 16), "Replace");
		dlg->insert(link = new InputLine(Rect(3, 3, 35, 4), ""));
		dlg->insert(new Label(Rect(3, 2, 15, 3), "~T~ext to find",
			link));
		dlg->insert(link = new InputLine(Rect(3, 6, 35, 7), ""));
		dlg->insert(new Label(Rect(3, 5, 11, 6), "~N~ew text", link));
		dlg->insert(new CheckBox(Rect(3, 8, 35, 12),
			new ClusterEntry("Case ~s~ensitive",
			new ClusterEntry("~W~hole words only",
			new ClusterEntry("~P~rompt on replace",
			new ClusterEntry("~R~eplace all",
			NULL))))
		));
		dlg->insert(new Button(Rect(13, 13, 23, 15), "O~K~", CM_OK,
			BF_DEFAULT));
		dlg->insert(new Button(Rect(25, 13, 35, 15), "Cancel",
			CM_CANCEL));
		dlg->focusNext(1);
		break;
	case ED_SETOPTIONS:
		dlg = new Dialog(Rect(0, 0, 38, 11), "Editor options");
		dlg->insert(link = new InputLine(Rect(3, 3, 35, 4), ""));
		dlg->insert(new Label(Rect(3, 2, 12, 3), "~T~ab width",
			link));
		dlg->insert(new CheckBox(Rect(3, 5, 35, 7),
			new ClusterEntry("~A~uto ident",
			new ClusterEntry("~S~how control codes",
			NULL))
		));
		dlg->insert(new Button(Rect(13, 8, 23, 10), "O~K~", CM_OK,
			BF_DEFAULT));
		dlg->insert(new Button(Rect(25, 8, 35, 10), "Cancel",
			CM_CANCEL));
		dlg->focusNext(1);
		break;
	case ED_SAVEAS:
		dlg = new FileDialog("Save file as");
		break;
	case ED_REPLACEPROMPT:
		return messageBox("Replace this occurrence ?", NULL,
			MF_INFORMATION | MF_YESNOCANCEL);
	case ED_SAVEUNTITLED:
		return messageBox("Save untitled file?", NULL,
			MF_INFORMATION | MF_YESNOCANCEL);
	case ED_SEARCHFAILED:
		return messageBox("Search string not found", NULL,
			MF_ERROR | MF_OKBUTTON);
	case ED_READERROR:
		expandPath((char *) buf, dir, name);
		return messageBox("Error reading file %s", name,
			MF_ERROR | MF_OKBUTTON);
	case ED_SAVEMODIFY:
		expandPath((char *) buf, dir, name);
		return messageBox("%s has been modified. Save?", name,
			MF_INFORMATION | MF_YESNOCANCEL);
	case ED_WRITEERROR:
		expandPath((char *) buf, dir, name);
		return messageBox("Error writing file %s", name,
			MF_ERROR | MF_OKBUTTON);
	}
	return program->execDialog(dlg, buf);
}

/*
 * Adds a newline character.
 */

void Editor::newLine()
{
	char nl = '\n';
	int start, end;

	start = end = lineStart(curptr);
	while (end < curptr)
	{
		int c = bufChar(end);

		if (c != ' ' && c != '\t') break;
		end++;
	}
	insertText(&nl, 1, 0);
	if (flags & EF_AUTOIDENT) insertText(buffer + start, end - start, 0);
}

/*
 * Moves the cursor to the next character.
 */

int Editor::nextChar(int ptr)
{
	ptr++;
	return MIN(ptr, buflen);
}

/*
 * Moves the cursor to the next line.
 */

int Editor::nextLine(int ptr)
{
	return nextChar(lineEnd(ptr));
}

/*
 * Moves the cursor to the next word.
 */

int Editor::nextWord(int ptr)
{
	while (ptr < buflen && isWordChar(bufChar(ptr))) ptr = nextChar(ptr);
	while (ptr < buflen && !isWordChar(bufChar(ptr))) ptr = nextChar(ptr);
	return ptr;
}

/*
 * Moves the cursor to the previous character.
 */

int Editor::prevChar(int ptr)
{
	ptr--;
	return MAX(ptr, 0);
}

/*
 * Moves the cursor to the previous line.
 */

int Editor::prevLine(int ptr)
{
	return lineStart(prevChar(ptr));
}

/*
 * Moves the cursor to the previous word.
 */

int Editor::prevWord(int ptr)
{
	ptr = MIN(ptr, buflen - 1);
	while (ptr > 0 && !isWordChar(bufChar(ptr))) ptr = prevChar(ptr);
	while (ptr > 0 && isWordChar(bufChar(ptr))) ptr = prevChar(ptr);
	return ptr;
}

/*
 * Replaces a string.
 */

void Editor::replaceString()
{
	ReplaceDialogData rdd =
	{
		{""},		/* InputLine */
		{""},		/* InputLine */
		{0, ~0}		/* Cluster */
	};

	strcpy(rdd.find.value, findstr);
	strcpy(rdd.replace.value, replacestr);
	rdd.options.bit = flags;
	if (makeDialog(ED_REPLACE, &rdd) == CM_OK)
	{
		strcpy(findstr, rdd.find.value);
		strcpy(replacestr, rdd.replace.value);
		flags = rdd.options.bit | EF_REPLACE;
		doSearchReplace();
	}		
}

/*
 * Returns nonzero if the two characters are the same.
 */

int Editor::sameChar(char a, char b, int sensitive)
{
	return sensitive ? a == b : toupper2(a) == toupper2(b);
}

/*
 * Moves the cursor to another place.
 */

void Editor::scrollTo(int x, int y)
{
	/* clip to the limits */

	delta.x = MAX(0, MIN(x, limit.x - size.x));
	delta.y = MAX(0, MIN(y, limit.y - size.y));
	updateScreen();
}

/*
 * Finds and selects the specified string.
 */

int Editor::searchString(char *find, int opts)
{
	int len = strlen(find);
	int ptr = curptr;

	while (1)
	{
		int where = strScan(buffer + bufPtr(ptr), buflen - ptr, find,
			(opts & EF_CASESENSITIVE) != 0);

		/* go out if string not found */

		if (where < 0) break;

		ptr += where;
		if (!(opts & EF_WHOLEWORDSONLY) || isWholeWord(ptr, len))
		{
			lockDraw();
			setSelect(ptr, ptr + len);
			setCurAddr(ptr + len);
			trackCursor(!isCursorVisible());
			unlockDraw();
			return 1;
		}
		ptr++;
	}
	return 0;
}

/*
 * Moves the cursor to another position.
 */

void Editor::setCurAddr(int ptr)
{
	int gaplen = bufsize - buflen;
	int len;

	if (ptr == curptr) return;

	/* move forward */

	if (ptr > curptr)
	{
		len = ptr - curptr;
		memmove(buffer + curptr, buffer + curptr + gaplen, len);
		curpos.y += countLines(buffer + curptr, len);
		curptr = ptr;
	}
	else
	{
		/* move backward */

		len = curptr - ptr;
		curptr = ptr;
		curpos.y -= countLines(buffer + curptr, len);
		memmove(buffer + curptr + gaplen, buffer + curptr, len);
	}
	curpos.x = charPos(lineStart(curptr), curptr);
	updateScreen();
}

/*
 * Set Editor options.
 */

void Editor::setOptions()
{
	OptionsDialogData odd =
	{
		{""},		/* InputLine */
		{0, ~0},	/* Cluster */
	};

	sprintf(odd.tabwidth.value, "%d", tabwidth);
	odd.flags.bit = (flags & EF_AUTOIDENT ? 1 : 0) |
		(flags & EF_SHOWCODES ? 2 : 0);
	while (makeDialog(ED_SETOPTIONS, &odd) == CM_OK)
	{
		char *tail;
		int val = strtol(odd.tabwidth.value, &tail, 0);

		if (odd.tabwidth.value != tail && val > 0)
		{
			tabwidth = val;
			odd.flags.bit & 1 ? flags |= EF_AUTOIDENT :
				flags &= ~EF_AUTOIDENT;
			odd.flags.bit & 2 ? flags |= EF_SHOWCODES :
				flags &= ~EF_SHOWCODES;
			break;
		}
		messageBox("Wrong value", NULL, MF_ERROR | MF_OKBUTTON);
	}
}

/*
 * Selects the specified area without moving the cursor.
 */

void Editor::setSelect(int start, int end)
{
	selstart = start;
	selend = end;
	updateScreen();
}

/*
 * Sets the state of the Editor.
 */

void Editor::setState(int flag, int enable)
{
	View::setState(flag, enable);
	switch (flag)
	{
	case SF_ACTIVE:
		if (hs != NULL) hs->setState(flag, enable);
		if (vs != NULL) vs->setState(flag, enable);
		if (ind != NULL) ind->setState(flag, enable);
	}
}

/*
 * Finds the first occurrency of the given string in a buffer.
 */

int Editor::strScan(char *buf, int size, char *str, int sense)
{
	char *src = buf;

	if (*str == '\0') return -1;	/* null string is not valid */
	while (size > 0)
	{
		/* find the first character */

		while (size-- > 0 && !sameChar(*src++, *str, sense)) ;
		if (size > 0)
		{
			char *find = str;
			char *scan = src;
			int len = size;

			/* check other characters */

			do
			{
				if (*++find == '\0') return src - buf - 1;
			}
			while (len-- > 0 && sameChar(*scan++, *find, sense));
		}
	}
	return -1;
}

/*
 * Moves the origin coordinates to the correct position.
 */

void Editor::trackCursor(int center)
{
	if (center) scrollTo(curpos.x - size.x + 1, curpos.y - size.y / 2);
	else scrollTo(
		MAX(curpos.x - size.x + 1, MIN(delta.x, curpos.x)),
		MAX(curpos.y - size.y + 1, MIN(delta.y, curpos.y)));
}

/*
 * Returns the address of the default key-to-command table.
 */

translatekey_s *Editor::translateTable()
{
	static translatekey_s table[] =
	{
		{CM_CHARLEFT,		KC_ARROWLEFT},
		{CM_CHARRIGHT,		KC_ARROWRIGHT},
		{CM_DELCHAR,		KC_DELETE},
		{CM_LINESTART,		KC_HOME},
		{CM_LINEEND,		KC_END},
		{CM_LINEUP,		KC_ARROWUP},
		{CM_LINEDOWN,		KC_ARROWDOWN},
		{CM_NEWLINE,		KC_ENTER},
		{CM_PAGEUP,		KC_PAGEUP},
		{CM_PAGEDOWN,		KC_PAGEDOWN},
		{CM_BACKSPACE,		KC_BACKSPACE},
		{CM_SELSTART,		KC_F1},
		{CM_SELEND,		KC_F2},
		{CM_WORDLEFT,		KC_F3},
		{CM_WORDRIGHT,		KC_F4},
		{CM_DELSTART,		KC_F5},
		{CM_DELEND,		KC_F6},
		{CM_DELLINE,		KC_F7},
		{CM_DELWORD,		KC_F8},
		{CM_NONE,		KC_NONE}	/* last element */
	};
	return table;
}

/*
 * Unlocks the Editor.
 */

void Editor::unlockDraw()
{
	if (locked > 0)
	{
		if (--locked == 0) updateScreen();
	}
}

/*
 * Updates the objects on the screen.
 */

void Editor::updateScreen()
{
	if (locked > 0) return;
	setCursor(curpos.x - delta.x, curpos.y - delta.y);
	drawView();
	if (hs != NULL)
	{
		hs->setParams(delta.x, 0, limit.x - size.x, size.x / 2, 1);
	}
	if (vs != NULL)
	{
		vs->setParams(delta.y, 0, limit.y - size.y, size.y, 1);
	}
	if (ind != NULL) ind->drawView();
	enableCommands(state & SF_FOCUSED && state & SF_VISIBLE);
	updateCommands();
}
