/*
 * fedit.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <string.h>
#include "dialogs.h"
#include "fedit.h"
#include "scrbar.h"

/*
 * FileEditor constructor.
 */

FileEditor::FileEditor(Rect bounds, ScrollBar *hs, ScrollBar *vs,
	Indicator *ind, char *afile): Editor(bounds, hs, vs, ind)
{
	DEBUG("new FileEditor at %p\n", this);
	strncpy2(file, afile, sizeof(file));
	if (file[0] != '\0') loadFile();
}

/*
 * Updates the FileEditor's commands.
 */

void FileEditor::enableCommands(int enable)
{
	setCommandState(CM_SAVE, enable);
	setCommandState(CM_SAVEAS, enable);
	Editor::enableCommands(enable);
}

/*
 * Handles the FileEditor's events.
 */

void FileEditor::handleEvent(Event &event)
{
	Editor::handleEvent(event);
	switch (event.what)
	{
	case EV_COMMAND:
		switch (event.command)
		{
		case CM_SAVE:
			saveFile();
			clearEvent(event);
			break;
		case CM_SAVEAS:
			saveFileAs();
			clearEvent(event);
		}
	}
}

/*
 * Returns nonzero if the command is valid.
 */

int FileEditor::isValid(int command)
{
	switch (command)
	{
	case CM_CLOSE:
	case CM_QUIT:
		if (flags & EF_MODIFIED)
		{
			int type;

			if (file[0] == '\0') type = ED_SAVEUNTITLED;
			else type = ED_SAVEMODIFY;
			switch (makeDialog(type, file))
			{
			case CM_YES:
				return saveFile() >= 0;
			case CM_CANCEL:
				return 0;
			}
		}
	}
	return 1;
}

/*
 * Reads the file from the fs.  Returns -1 if error.
 */

int FileEditor::loadFile()
{
	FILE *fs;
	char buf[EDITOR_BLOCKSIZE];
	int len;

	if ((fs = fopen(file, "r")) == NULL)
	{
		makeDialog(ED_READERROR, file);
		return -1;
	}
	lockDraw();
	do
	{
		if ((len = fread(buf, 1, EDITOR_BLOCKSIZE, fs)) > 0)
		{
			insertText(buf, len, 0);
		}
	}
	while (len == EDITOR_BLOCKSIZE);
	fclose(fs);
	flags &= ~EF_MODIFIED;
	setCurAddr(0);
	trackCursor(0);
	unlockDraw();
	return 0;
}

/*
 * Writes the file to the fs.  Returns -1 if error.
 */

int FileEditor::saveFile()
{
	if (isClipboard()) file[0] = '\0';
	if (file[0] == '\0') return saveFileAs();
	return saveFileNamed();
}

/*
 * Gets the name of the file from the user and saves it.  Returns -1 if
 * error.
 */

int FileEditor::saveFileAs()
{
	FileDialogData fdd;

	if (isClipboard()) file[0] = '\0';
	strcpy(fdd.file, file);
	if (makeDialog(ED_SAVEAS, &fdd) != CM_OK) return -1;
	strcpy(file, fdd.file);
	sendMessage(owner, EV_BROADCAST, CM_UPDATETITLE);
	return saveFileNamed();
}

/*
 * Writes the file to the fs.  Returns -1 if error.
 */

int FileEditor::saveFileNamed()
{
	FILE *fs;
	char backup[PATH_MAX];

	/* first make a backup file */

	sprintf(backup, "%s~", file);
	rename(file, backup);

	/* now save the file */

	if ((fs = fopen(file, "w")) == NULL)
	{
		makeDialog(ED_WRITEERROR, file);
		return -1;
	}

	/* save the first part of the buffer */

	if (curptr > 0)
	{
		if (fwrite(buffer, curptr, 1, fs) != 1)
		{
			makeDialog(ED_WRITEERROR, file);
			return -1;
		}
	}

	/* save the second part of the buffer */

	if (curptr < buflen)
	{
		if (fwrite(buffer + curptr + bufsize - buflen,
			buflen - curptr, 1, fs) != 1)
		{
			makeDialog(ED_WRITEERROR, file);
			return -1;
		}
	}
	fclose(fs);
	flags &= ~EF_MODIFIED;
	updateScreen();
	return 0;
}

/*
 * EditWindow constructor.
 */

EditWindow::EditWindow(Rect bounds, char *file, FileEditor *(*peditor)(Rect,
	ScrollBar *, ScrollBar *, Indicator *, char *)): Window(bounds, "",
	Window::initFrame)
{
	DEBUG("new EditWindow at %p\n", this);
	char *filename = file == NULL ? "" : file;

	editor = NULL;
	if (peditor != NULL)
	{
		Indicator *ind;
		Rect r;
		ScrollBar *hs, *vs;

		getExtent(r);
		insert(hs = new ScrollBar(Rect(13, r.b.y - 1, r.b.x - 1,
			r.b.y)));
		insert(vs = new ScrollBar(Rect(r.b.x - 1, 1, r.b.x,
			r.b.y - 1)));
		insert(ind = new Indicator(Rect(1, r.b.y - 1, 12, r.b.y)));
		editor = peditor(Rect(1, 1, r.b.x - 1, r.b.y - 1), hs, vs,
			ind, filename);
		if (editor != NULL)
		{
			insert(editor);
			ind->setLink(editor);
			if (file == NULL)
			{
				clipboard = editor;
				clipwindow = this;
			}
		}
	}
}

/*
 * Closes the EditWindow.
 */

void EditWindow::doClose()
{
	if (editor != NULL && editor->isClipboard()) setState(SF_VISIBLE, 0);
	else Window::doClose();
}

/*
 * Updates the EditWindow's commands.
 */

void EditWindow::enableCommands(int enable)
{
	if (editor != NULL) editor->enableCommands(enable);
	Window::enableCommands(enable);
}

/*
 * Returns the title of the EditWindow.
 */

char *EditWindow::getTitle()
{
	if (editor != NULL)
	{
		if (editor->isClipboard()) return "clipboard";
		if (editor->file[0] == '\0') return "untitled";
		return editor->file;
	}
	return Window::getTitle();
}

/*
 * Creates a standard Frame object.
 */

FileEditor *EditWindow::initFileEditor(Rect bounds, ScrollBar *hs,
	ScrollBar *vs, Indicator *ind, char *file)
{
	return new FileEditor(bounds, hs, vs, ind, file);
}

/*
 * Returns the size limits of the EditWindow.
 */

void EditWindow::sizeLimits(Point &min, Point &max)
{
	Window::sizeLimits(min, max);
	min.y = 4;
}
