/*
 * group.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdlib.h>
#include "venus.h"

/*
 * Group constructor.
 */

Group::Group(Rect bounds, int buffered): View(bounds)
{
	DEBUG("new Group at %p\n", this);
	options |= OF_FIRSTCLICK | OF_SELECTABLE;
	if (buffered) buffer = new Video [size.x * size.y];
		else buffer = NULL;
	current = first = NULL;
	endcommand = CM_NONE;
	lockflag = 0;
	getExtent(clip);
}

/*
 * Group destructor.
 */

Group::~Group()
{
	DEBUG("delete Group at %p\n", this);
	state |= SF_DELETE;		/* enter in a special state */
	while (first != NULL)
	{
		delete first->prev;
	}
	if (buffer != NULL) delete buffer;
}

/* 
 * Changes the Group's bounds.
 */

void Group::changeBounds(Rect bounds)
{
	Point delta;

	delta.x = bounds.b.x - bounds.a.x - size.x;
	delta.y = bounds.b.y - bounds.a.y - size.y;
	setBounds(bounds);

	/* if no delta the size is not changed, so we do not recalculate */

	if (delta.x == 0 && delta.y == 0) drawView();
	else
	{
		if (buffer != NULL)	/* resize buffer, if exists */
		{
			delete buffer;
			buffer = new Video [size.x * size.y];
		}
		getExtent(clip);
		if (first != NULL)	/* recalculate subview's bounds */
		{
			View *obj = first;

			lockGroup();
			do
			{
				Rect bounds;

				obj->calcBounds(bounds, delta);
				obj->changeBounds(bounds);
				obj = obj->next;
			}
			while (obj != first);
			unlockGroup();
		}
	}
}

/*
 * Returns ths size of the Group's data.
 */

int Group::dataSize()
{
	int size = 0;

	if (first != NULL)
	{
		View *obj = first;

		do
		{
			size += obj->dataSize();
			obj = obj->next;
		}
		while (obj != first);
	}
	return size;
}

/*
 * Sends an event to the specified object.
 */

void Group::doHandleEvent(Event &event, View *target)
{
	if (target == NULL) return;
	if (!(target->state & SF_ACTIVE))
	{
		if (event.what & (ET_FOCUSED | ET_POSITIONAL)) return;
	}
	switch (phase)
	{
	case PH_POSTPROCESS:
		if (!(target->options & OF_POSTPROCESS)) return;
		break;
	case PH_PREPROCESS:
		if (!(target->options & OF_PREPROCESS)) return;
	}
	target->handleEvent(event);
}

/*
 * If a buffer exists, the buffer is written to the screen.  Otherwise, each
 * subview is told to draw itself.
 */

void Group::draw()
{
	if (buffer != NULL) writeBuf(0, 0, size.x, size.y, buffer);
	else redrawAll();
}

/*
 * Shows the cursor in the current view.
 */

void Group::drawCursor()
{
	if (current != NULL) current->drawCursor();
	else View::drawCursor();
}

/*
 * Ends the modal state.
 */

void Group::endModal(int command)
{
	if (state & SF_MODAL) endcommand = command;
}

/*
 * This is the Group's main event loop.  It repeatedly gets events and
 * handles them.
 */

int Group::execModal()
{
	Event event;

	setState(SF_MODAL, 1);
	do
	{
		endcommand = CM_NONE;
		do
		{
			getEvent(event);
			handleEvent(event);
		}
		while (endcommand == CM_NONE);
	}
	while (!isValid(endcommand));
	setState(SF_MODAL, 0);
	return endcommand;
}

/*
 * Executes the View in modal state.
 */

int Group::execView(View *obj)
{
	if (obj != NULL)
	{
		insert(obj);
		return obj->execModal();
	}
	return CM_CANCEL;
}

/*
 * Moves the focus to the next or to the previous object.  Returns nonzero
 * if the new object is not the old.
 */

int Group::focusNext(int forward)
{
	if (first == NULL) return 0;
	if (current == NULL)	/* find the first valid object */
	{
		View *obj = first;

		while (!obj->getFocus())
		{
			obj = obj->next;
			if (obj == first) return 0;
		}
	}
	else
	{
		View *obj = current;

		do
		{
			obj = forward ? obj->next : obj->prev;
			if (obj == current) return 0;
		}
		while (!obj->getFocus());
	}
	return 1;
}

/*
 * Forces the focus to be moved to the next or to the previous object.
 * Returns nonzero if the new object is not the old.
 */

int Group::forceFocusNext(int forward)
{
	if (focusNext(forward)) return 1;
	setCurrent(NULL);	/* it is the same object then fails */
	return 0;
}

/*
 * Gets the Group's data.
 */

void Group::getData(void *buf)
{
	if (first != NULL)
	{
		View *obj = first;
		char *dst = (char *) buf;

		do
		{
			obj->getData(dst);
			dst += obj->dataSize();
			obj = obj->next;
		}
		while (obj != first);
	}
}

/*
 * Returns the Group's help context.
 */

int Group::getHelpCtx()
{
	int ctx = HC_NONE;

	if (current != NULL) ctx = current->getHelpCtx();
	if (ctx == HC_NONE) ctx = View::getHelpCtx();
	return ctx;
}

/*
 * Returns the Group's phase.
 */

int Group::getPhase()
{
	return phase;
}

/*
 * Sends an event to all the objects.
 */

void Group::handleBroadcastEvent(Event &event)
{
	if (first != NULL)
	{
		View *obj = first;

		do
		{
			doHandleEvent(event, obj);

			/* the event is handled by somebody, so go out */

			if (event.what == EV_NONE) return;
			obj = obj->next;
		}
		while (obj != first);
	}
}

/*
 * Handles the Group's events.
 */

void Group::handleEvent(Event &event)
{
	View::handleEvent(event);
	if (event.what & ET_BROADCAST)
	{
		/* send the event to all the objects */

		phase = PH_FOCUSED;
		handleBroadcastEvent(event);
	}
	else if (event.what & ET_FOCUSED)
	{
		/*
		 * send the event only to the current object and to all the
		 * the objects with OF_PREPROCESS or OF_POSTPROCESS flag set
		 */
		phase = PH_PREPROCESS;
		handleBroadcastEvent(event);
		phase = PH_FOCUSED;
		doHandleEvent(event, current);
		phase = PH_POSTPROCESS;
		handleBroadcastEvent(event);
	}
	else if (event.what & ET_POSITIONAL && first != NULL)
	{
		Point pnt;
		View *obj = first;

		/* the object that handles the event must contain the point */

		phase = PH_FOCUSED;
		makeLocal(event.where, pnt);
		do
		{
			obj = obj->prev;
			if (obj->state & SF_VISIBLE &&
				obj->ownerPointInside(pnt))
			{
				doHandleEvent(event, obj);
				return;
			}
		}
		while (obj != first);

		/* mouse button pressed outside any object */

		if (event.what == EV_MOUSEDOWN) makeBeep();
	}
}

/*
 * Inserts the given view in the group's subview list.  The new subview is
 * placed on top of all other subviews.  If the subview has the OF_CENTER_X
 * or OF_CENTER_Y flag set, it is centered accordingly in the group.
 */

void Group::insert(View *obj)
{
	if (obj == NULL) return;
	insertInternal(obj);
	if (obj->options & OF_CENTERX)	/* center along x axis */
	{
		obj->origin.x = (size.x - obj->size.x) / 2;
	}
	if (obj->options & OF_CENTERY)	/* center along y axis */
	{
		obj->origin.y = (size.y - obj->size.y) / 2;
	}
	if (obj->options & OF_SELECTABLE) obj->getFocus();
	redrawAll();
}

/*
 * Inserts an object in the list.  It is used only internally.
 */

void Group::insertInternal(View *obj)
{
	if (first == NULL)	/* the list is empty ? */
	{
		first = obj->next = obj->prev = obj;
	}
	else	/* list is not empty */
	{
		obj->prev = first->prev;
		first->prev->next = obj;
		obj->next = first;
		first->prev = obj;
	}
	obj->owner = this;
}

/*
 * Returns nonzero if the command is valid.
 */

int Group::isValid(int command)
{
	if (first != NULL)
	{
		View *obj = first;

		do
		{
			if (!obj->isValid(command)) return 0;
			obj = obj->next;
		}
		while (obj != first);
	}
	return 1;
}

/*
 * Locks the Group.
 */

void Group::lockGroup()
{
	lockflag++;
}

/*
 * Moves a View to the last position in the list.
 */

void Group::makeFirst(View *obj)
{
	removeInternal(obj);
	insertInternal(obj);
}

/*
 * Moves a View next to the specified View.
 */

void Group::movePast(View *obj, View *target)
{
	if (obj != NULL && target != NULL)
	{
		if (obj == current) forceFocusNext(0);
		removeInternal(obj);
		obj->next = target->next;
		obj->prev = target;
		target->next->prev = obj;
		target->next = obj;
		obj->owner = this;
		if (current == NULL) obj->getFocus();
	}
}

/*
 * Redraws the Group's subviews.
 */

void Group::redrawAll()
{
	if (lockflag > 0) return;
	if (first != NULL)
	{
		View *obj = first;

		do
		{
			obj->drawView();
			obj = obj->next;
		}
		while (obj != first);
	}
	drawCursor();
}

/*
 * Removes the subview from the group and redraws the other subviews as
 * required.
 */

void Group::remove(View *obj)
{
	if (first == NULL || obj == NULL) return;

	/*
	 * SF_DELETE is set when the Group destructor is called.  All the
	 * subviews are deleted without redraw them or move the focus.
	 */
	if (state & SF_DELETE) removeInternal(obj);
	else
	{
		lockGroup();
		if (obj == current) forceFocusNext(0);
		removeInternal(obj);
		unlockGroup();
	}
}

/*
 * Removes an object from the list.  It is used only internally.
 */

void Group::removeInternal(View *obj)
{
	obj->prev->next = obj->next;	/* skip this object */
	obj->next->prev = obj->prev;
	obj->next = obj->prev = NULL;	/* dechaining object */

	/* is it the first object ? */

	if (first == obj) first = first->next;
	obj->owner = NULL;
}

/*
 * Makes the specified view the current in the group.
 */

void Group::setCurrent(View *obj)
{
	if (current != NULL && lockflag == 0)
	{
		current->setState(SF_FOCUSED, 0);
	}
	current = obj;
	if (current != NULL) current->setState(SF_FOCUSED, 1);
	redrawAll();
}

/*
 * Sets the Group's data.
 */

void Group::setData(void *buf)
{
	if (first != NULL)
	{
		View *obj = first;
		char *org = (char *) buf;

		do
		{
			obj->setData(org);
			org += obj->dataSize();
			obj = obj->next;
		}
		while (obj != first);
	}
}

/*
 * Changes the Group's state.
 */

void Group::setState(int flag, int enable)
{
	View::setState(flag, enable);
	switch (flag)
	{
	case SF_FOCUSED:
		if (current != NULL) current->setState(SF_FOCUSED, enable);
		redrawAll();
	}
}

/*
 * Unlocks the Group.
 */

void Group::unlockGroup()
{
	if (lockflag > 0)
	{
		if (--lockflag == 0) redrawAll();
	}
}
