/*
 * inpline.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include "inpline.h"

/*
 * InputLine constructor.
 */

InputLine::InputLine(Rect bounds, char *avalue, int amaxlen): View(bounds)
{
	DEBUG("new InputLine at %p\n", this);
	options |= OF_FIRSTCLICK | OF_SELECTABLE;
	curpos = firstpos = 0;
	strncpy2(data.value, avalue, maxlen = amaxlen);
	showCursor();
}

/*
 * Returns the size of the InputLine's data.
 */

int InputLine::dataSize()
{
	return sizeof(InputLineData);
}

/*
 * Draws the InputLine on the screen.
 */

void InputLine::draw()
{
	DrawBuf buf;
	int arrow, color;

	if (state & SF_ACTIVE)
	{
		arrow = getColor(C_INPUTLINE);
		color = getColor(C_INPUTLINE + 2);
	}
	else arrow = color = getColor(C_INPUTLINE + 1);
	moveChar(buf, ' ', color, size.x);
	moveStr(buf + 1, data.value + firstpos, color);
	if (firstpos > 0)
	{
		moveChar(buf, 0x11, arrow, 1); 
	}
	if ((int) strlen(data.value) - firstpos > size.x - 2)
	{
		moveChar(buf + size.x - 1, 0x10, arrow, 1);
	}
	writeBuf(0, 0, size.x, size.y, buf);
	setCursor(curpos - firstpos + 1, 0);
}

/*
 * Gets the InputLine's data.
 */

void InputLine::getData(void *buf)
{
	InputLineData *ild = (InputLineData *) buf;

	*ild = data;
}

/*
 * Gets the address of the string.
 */

char *InputLine::getString()
{
	return data.value;
}

/*
 * Handles the InputLine's events.
 */

void InputLine::handleEvent(Event &event)
{
	char *addr;
	int len;

	View::handleEvent(event);
	switch (event.what)
	{
	case EV_KEYDOWN:
		addr = data.value + curpos;
		len = strlen(data.value);
		switch (event.keycode)
		{
		case KC_ARROWLEFT:
			if (curpos > 0) curpos--;
			break;
		case KC_ARROWRIGHT:
			if (curpos < len) curpos++;
			break;
		case KC_BACKSPACE:
			if (curpos > 0)
			{
				memmove(addr - 1, addr, len - curpos + 1);
				curpos--;
			}
			break;
		case KC_DELETE:
			if (curpos < len)
			{
				memmove(addr, addr + 1, len - curpos + 1);
			}
			break;
		case KC_END:
			curpos = len;
			break;
		case KC_HOME:
			curpos = 0;
			break;
		default:
			if (event.keycode >= ' ' && event.keycode <= 255 &&
				len < maxlen)
			{
				memmove(addr + 1, addr, len - curpos + 1);
				curpos++;
				*addr = event.keycode;
			}
			else return;
		}
		break;
	case EV_MOUSEDOWN:
		makeLocal(event.where, event.where);
		if (event.where.x <= 0)
		{
			firstpos--;	/* scroll left */
		}
		else if (event.where.x >= size.x - 1)
		{
			firstpos++;	/* scroll right */
		}
		else	/* move the cursor in a middle point */
		{
			curpos = MIN(strlen(data.value),
				firstpos + event.where.x - 1);
		}
		break;
	default:
		return;
	}
	firstpos = MAX(0, RANGE(firstpos, curpos - size.x + 2, curpos));
	drawView();
	clearEvent(event);
}

/*
 * Sets the InputLine's data.
 */

void InputLine::setData(void *buf)
{
	InputLineData *ild = (InputLineData *) buf;

	data = *ild;
	setString(data.value);
}

/*
 * Changes the InputLine's string.
 */

void InputLine::setString(char *avalue)
{
	strncpy2(data.value, avalue, maxlen);
	curpos = strlen(data.value);
	firstpos = MAX(0, curpos - size.x + 2);
	drawView();
}
