/*
 * listbox.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include "listbox.h"

/*
 * StrList constructor.
 */

StrList::StrList()
{
	first = NULL;
	length = 0;
}

/*
 * StrList destructor.
 */

StrList::~StrList()
{
	clearList();
}

/*
 * Adds an item in the list in alphabetical order.
 */

void StrList::addAlphabetical(char *name)
{
	string_s *node = new string_s;
	string_s *prev = NULL, *tag = first;

	strncpy2(node->name, name, sizeof(node->name));
	while (tag != NULL && strcmp(tag->name, node->name) < 0)
	{
		prev = tag;
		tag = tag->next;
	}
	if (prev == NULL) first = node; else prev->next = node;
	node->next = tag;
	length++;
}

/*
 * Adds an item in the list.  For more than 50-100 elements the function can
 * be slow due to the internal loop.
 */

void StrList::addUnsorted(char *name)
{
	string_s *node = new string_s;

	strncpy2(node->name, name, sizeof(node->name));
	if (first == NULL) first = node;
	else
	{
		string_s *last = first;

		while (last->next != NULL) last = last->next;
		last->next = node;
	}
	node->next = NULL;
	length++;
}

/*
 * Clears the list.
 */

void StrList::clearList()
{
	while (first != NULL)
	{
		string_s *next = first->next;

		delete first;
		first = next;
	}
	length = 0;
}

/*
 * Deletes the specified object from the list.
 */

void StrList::deleteAt(int index)
{
	if (index >= 0 && index < length)
	{
		string_s *node = first, *prev = NULL;

		while (index-- > 0)
		{
			prev = node;
			node = node->next;
		}
		if (prev == NULL) first = node->next;
		else prev->next = node->next;
		delete node;
		length--;
	}
}

/*
 * Returns the address of the specified object.
 */

char *StrList::getAt(int index)
{
	if (index >= 0 && index < length)
	{
		string_s *node = first;

		while (index-- > 0) node = node->next;
		return node->name;
	}
	return NULL;
}

/*
 * Gets the number of entries.
 */

int StrList::getLength()
{
	return length;
}

/*
 * Returns the size of the biggest string.
 */

int StrList::getMaxStrLength()
{
	int len = 0;
	string_s *node = first;

	while (node != NULL)
	{
		len = MAX(len, strlen(node->name));
		node = node->next;
	}
	return len;
}

/*
 * ListBox constructor.
 */

ListBox::ListBox(Rect bounds, ScrollBar *hs, ScrollBar *vs, StrList *alist):
	ListViewer(bounds, hs, vs)
{
	DEBUG("new ListBox at %p\n", this);
	list = alist;
}

/*
 * ListBox destructor.
 */

ListBox::~ListBox()
{
	DEBUG("delete ListBox at %p\n", this);
	if (list != NULL) delete list;
}

/*
 * Clears the list.
 */

void ListBox::clearList()
{
	if (list != NULL) list->clearList();
	data.current = -1;
	home = 0;
	drawView();
}

/*
 * Returns the address of the list.
 */

StrList *ListBox::getList()
{
	return list;
}

/*
 * Returns the length of the list.
 */

int ListBox::getLength()
{
	return list != NULL ? list->getLength() : 0;
}

/*
 * Returns the size of the biggest element in the list.
 */

int ListBox::getMaxStrLength()
{
	return list != NULL ? list->getMaxStrLength() : 0;
}

/*
 * Returns the address of the current string.
 */

char *ListBox::getString()
{
	if (list != NULL) return list->getAt(data.current);
	return NULL;
}

/*
 * Returns the address of the specified string.
 */

char *ListBox::getText(int item)
{
	if (list != NULL) return list->getAt(item);
	return NULL;
}
