/*
 * menubar.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include "desktop.h"
#include "menubar.h"

enum
{
	DO_AUTOSELECT,
	DO_NOTHING,
	DO_RETURN,
	DO_SELECT
};

/*
 * MenuItem constructor.
 */

MenuItem::MenuItem(char *aname, char *aparam, int akeycode, int acommand,
	MenuItem *achild, MenuItem *anext)
{
	child = standard = achild;
	command = acommand;
	keycode = akeycode;
	name = aname;
	next = anext;
	param = aparam;
}

/*
 * MenuView constructor.
 */

MenuView::MenuView(Rect bounds, MenuItem *amenu, MenuView *aparentmenu):
	View(bounds)
{
	DEBUG("new MenuView at %p\n", this);
	current = NULL;
	menu = amenu;
	parentmenu = aparentmenu;
}

/*
 * Deletes the menu tree.
 */

void MenuView::destroyMenu(MenuItem *menu)
{
	MenuItem *item = menu->child;

	while (item != NULL)
	{
		MenuItem *next = item->next;

		if (item->name != NULL && item->child != NULL)
		{
			destroyMenu(item);
		}
		delete item;
		item = next;
	}
}

/*
 * Main menu handler.
 */

int MenuView::doSelect(Event &event)
{
	MenuItem *itemshown = NULL;
	int action, result = CM_NONE;

	current = menu->standard;
	do
	{
		action = DO_NOTHING;
		getEvent(event);
		switch (event.what)
		{
		case EV_KEYDOWN:
			handleKey(event, action, result);
			break;
		case EV_MOUSEDOWN:
		case EV_MOUSEMOVE:
		case EV_MOUSEUP:
			handleMouse(event, action);
		}

		/* new item selected ? */

		if (itemshown != current)
		{
			itemshown = current;
			drawView();
		}

		/* select the item ? */

		if ((action == DO_AUTOSELECT || action == DO_SELECT) &&
			current != NULL && current->name != NULL &&
			commandEnabled(current->command))
		{
			/* is it a submenu ? */

			if (current->child != NULL)
			{
				MenuView *target;
				Rect r;

				getItemRect(current, r);
				r.a.x += origin.x;
				r.a.y = r.b.y + origin.y - 1;
				target = newSubView(r, current);
				desktop->insert(target);
				result = target->doSelect(event);
				delete target;
			}
			else if (action == DO_SELECT)
			{
				result = current->command;
			}
		}
		if (result != CM_NONE)
		{
			action = DO_RETURN;
			clearEvent(event);
		}
	}
	while (action != DO_RETURN);
	if (current != NULL)
	{
		menu->standard = current;
		current = NULL;
		drawView();
	}
	if (event.what != EV_NONE) putEvent(event);
	return result;
}

/*
 * Finds the hotkey in the menu tree.
 */

MenuItem *MenuView::findHotKey(MenuItem *menu, int keycode)
{
	MenuItem *item = menu->child;

	while (item != NULL)
	{
		if (item->name != NULL && commandEnabled(item->command))
		{
			if (item->child != NULL)
			{
				MenuItem *find = findHotKey(item, keycode);

				if (find != NULL) return find;
			}
			else if (keycode == item->keycode) break;
		}
		item = item->next;
	}
	return item;
}

/*
 * Finds the hotkey in the current menu.
 */

MenuItem *MenuView::findItem(int keycode)
{
	MenuItem *item = menu->child;

	while (item != NULL)
	{
		if (item->name != NULL && commandEnabled(item->command))
		{
			if (keycode == MK_META(hotKey(item->name))) break;
		}
		item = item->next;
	}
	return item;
}

/*
 * Handles MenuView events.
 */

void MenuView::handleEvent(Event &event)
{
	int openmenu = 0;

	View::handleEvent(event);
	if (menu != NULL) switch (event.what)
	{
	case EV_BROADCAST:
		if (event.command == CM_COMMANDSETCHANGED) drawView();
		break;
	case EV_COMMAND:
		if ((openmenu = event.command == CM_MENU))
		{
			/* open the menu */

			event.what = EV_KEYDOWN;
			event.keycode = KC_ARROWDOWN;
		}
		break;
	case EV_KEYDOWN:
		if (!(openmenu = findItem(event.keycode) != NULL))
		{
			MenuItem *item;

			if ((item = findHotKey(menu, event.keycode)) != NULL)
			{
				event.what = EV_COMMAND;
				event.command = item->command;
				putEvent(event);
				clearEvent(event);
			}
		}
		break;
	case EV_MOUSEDOWN:
		openmenu = 1;
	}
	if (openmenu)
	{
		putEvent(event);
		if ((event.command = doSelect(event)) != CM_NONE)
		{
			event.what = EV_COMMAND;
			putEvent(event);
		}
		clearEvent(event);
	}
}

/*
 * Handles keyboard events.
 */

void MenuView::handleKey(Event &event, int &action, int &result)
{
	MenuItem *item;

	switch (event.keycode)
	{
	case KC_ARROWDOWN:
	case KC_ARROWUP:
		if (size.y != 1) trackKey(event.keycode == KC_ARROWDOWN);
		else if (event.keycode == KC_ARROWDOWN)
		{
			action = DO_AUTOSELECT;
		}
		break;
	case KC_ARROWLEFT:
	case KC_ARROWRIGHT:
		if (size.y == 1)
		{
			trackKey(event.keycode == KC_ARROWRIGHT);
			action = DO_AUTOSELECT;
		}
		else action = DO_RETURN;
		break;
	case KC_HOME:
	case KC_END:
		current = menu->child;
		if (event.keycode == KC_END) trackKey(0);
		break;
	case ' ':
	case KC_ENTER:
		action = DO_SELECT;
		break;
	case KC_ESC:
		if (parentmenu == NULL || parentmenu->size.y != 1)
		{
			clearEvent(event);
		}
		action = DO_RETURN;
		break;
	default:
		/* search in the current menu */

		if ((item = findItem(MK_META(toupper2(event.keycode))))
			!= NULL)
		{
			current = item;
			action = DO_SELECT;
		}
		else if ((item = topMenu()->findItem(event.keycode)) != NULL)
		{
			/* search in the MenuBar */

			if (size.y == 1) 
			{
				current = item;
				action = DO_AUTOSELECT;
			}
			else if (topMenu()->current != item)
			{
				action = DO_RETURN;
			}
		}
		else if ((item = findHotKey(topMenu()->menu, event.keycode))
			!= NULL)
		{
			/* search in the menu tree */

			result = item->command;
			action = DO_RETURN;
		}
	}
}

/*
 * Handles mouse events.
 */

void MenuView::handleMouse(Event &event, int &action)
{
	switch (event.what)
	{
	case EV_MOUSEDOWN:
		if (mouseInView(event.where))
		{
			trackMouse(event);
			if (size.y == 1) action = DO_AUTOSELECT;
		}
		else action = DO_RETURN;
		break;
	case EV_MOUSEMOVE:
		if (event.buttons != 0)
		{
			trackMouse(event);
			if (!mouseInView(event.where) &&
				!mouseInOwner(event) && mouseInMenus(event))
			{
				action = DO_RETURN;
			}
			else if (size.y == 1) action = DO_AUTOSELECT;
		}
		break;
	case EV_MOUSEUP:
		trackMouse(event);
		action = DO_SELECT;
	}
}

/*
 * Returns nonzero if the mouse pointer is in a menu object.
 */

int MenuView::mouseInMenus(Event &event)
{
	MenuView *mv = parentmenu;

	while (mv != NULL && !mv->mouseInView(event.where))
	{
		mv = mv->parentmenu;
	}
	return mv != NULL;
}

/*
 * Returns nonzero if the mouse is inside the current menu in the MenuBar.
 */

int MenuView::mouseInOwner(Event &event)
{
	if (parentmenu != NULL && parentmenu->size.y == 1)
	{
		Point mouse;
		Rect rect;

		parentmenu->makeLocal(event.where, mouse);
		parentmenu->getItemRect(parentmenu->current, rect);
		return rect.containsPoint(mouse);
	}
	return 0;
}

/*
 * Creates a new SubMenu.
 */

MenuView *MenuView::newSubView(Rect bounds, MenuItem *menu)
{
	MenuItem *item = menu->child;

	bounds.b.x = bounds.b.y = 0;
	while (item != NULL)
	{
		if (item->name != NULL)
		{
			int len = cStrLen(item->name);

			if (item->child != NULL) len += 2;
			else if (item->param != NULL)
			{
				len += cStrLen(item->param) + 1;
			}
			bounds.b.x = MAX(bounds.b.x, len);
		}
		bounds.b.y++;
		item = item->next;
	}
	bounds.b.x += bounds.a.x + 4;
	bounds.b.y += bounds.a.y + 2;
	return new SubMenu(bounds, menu, this);
}

/*
 * Returns the MenuBar address.
 */

MenuView *MenuView::topMenu()
{
	MenuView *mv = this;

	while (mv->parentmenu != NULL) mv = mv->parentmenu;
	return mv;
}

/*
 * Selects the next or the previous item.
 */

void MenuView::trackKey(int forward)
{
	if (current == NULL) current = menu->child;
	else do
	{
		if (forward)
		{
			current = current->next;
			if (current == NULL) current = menu->child;
		}
		else
		{
			MenuItem *target = current;
			MenuItem *test = menu->child;

			if (target == test) target = NULL;
			while (test != target)
			{
				current = test;
				test = test->next;
			}
		}
	}
	while (current->name == NULL);
}

/*
 * Finds the item under the mouse pointer.
 */

int MenuView::trackMouse(Event &event)
{
	Point mouse;
	Rect rect;

	makeLocal(event.where, mouse);
	current = menu->child;
	while (current != NULL)
	{
		getItemRect(current, rect);
		if (rect.containsPoint(mouse)) return 1;
		current = current->next;
	}
	return 0;
}

/*
 * MenuBar constructor.
 */

MenuBar::MenuBar(Rect bounds, MenuItem *amenu):
	MenuView(bounds, newSubMenu("", amenu, NULL), NULL)
{
	DEBUG("new MenuBar at %p\n", this);
	options |= OF_PREPROCESS;
}

/*
 * MenuBar destructor.
 */

MenuBar::~MenuBar()
{
	DEBUG("delete MenuBar at %p\n", this);
	destroyMenu(menu);
}

/*
 * Draws the MenuBar on the screen.
 */

void MenuBar::draw()
{
	DrawBuf buf;
	MenuItem *item = menu->child;
	int col_norm = getColor(C_MENUBAR);
	int col_normdisabled = getColor(C_MENUBAR + 1);
	int col_sel = getColor(C_MENUBAR + 2);
	int col_seldisabled = getColor(C_MENUBAR + 3);
	int x = 1;

	moveChar(buf, ' ', col_norm, size.x);
	while (item != NULL)
	{
		if (item->name != NULL)
		{
			char str[MAX_WIDTH];
			int col, len = cStrLen(item->name) + 2;

			if (x + len >= size.x) break;
			if (commandEnabled(item->command))
			{
				if (item == current) col = col_sel;
					else col = col_norm;
			}
			else
			{
				if (item == current) col = col_seldisabled;
					else col = col_normdisabled;
			}
			sprintf(str, " %s ", item->name);
			moveCStr(buf + x, str, col);
			x += len;
		}
		item = item->next;
	}
	writeBuf(0, 0, size.x, 1, buf);
}

/*
 * Returns the bounding rectangle of the current item.
 */

void MenuBar::getItemRect(MenuItem *find, Rect &rect)
{
	MenuItem *test = menu->child;

	rect.makeAssign(1, 0, 1, 1);
	while (1)
	{
		rect.a.x = rect.b.x;
		if (test->name != NULL) rect.b.x += cStrLen(test->name) + 2;
		if (test == find) break;
		test = test->next;
	}
}

/*
 * SubMenu constructor.
 */

SubMenu::SubMenu(Rect bounds, MenuItem *menu, MenuView *parentmenu):
	MenuView(bounds, menu, parentmenu)
{
	DEBUG("new SubMenu at %p\n", this);
	state |= SF_SHADOW;
}

/*
 * Draws the SubMenu on the screen.
 */

void SubMenu::draw()
{
	DrawBuf buf;
	MenuItem *item = menu->child;
	Video *where = buf;
	int col_norm = getColor(C_SUBMENU);
	int col_normdisabled = getColor(C_SUBMENU + 1);
	int col_sel = getColor(C_SUBMENU + 2);
	int col_seldisabled = getColor(C_SUBMENU + 3);

	moveChar(buf, ' ', col_norm, size.x * size.y);
	moveFrame(buf, 0, col_norm);
	while (item != NULL)
	{
		where += size.x;
		if (item->name != NULL)
		{
			int col;

			if (commandEnabled(item->command))
			{
				if (item == current) col = col_sel;
					else col = col_norm;
			}
			else
			{
				if (item == current) col = col_seldisabled;
					else col = col_normdisabled;
			}
			moveChar(where + 1, ' ', col, size.x - 2);
			moveCStr(where + 2, item->name, col);
			if (item->child != NULL) moveChar(where + size.x - 3,
				0x10, col, 1);
			else if (item->param != NULL) moveStr(
				where + size.x - 2 - strlen(item->param),
				item->param, col);
		}
		else
		{
			moveChar(where, 0xc3, col_norm, 1);
			moveChar(where + 1, 0xc4, col_norm, size.x - 2);
			moveChar(where + size.x - 1, 0xb4, col_norm, 1);
		}
		item = item->next;
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Returns the bounding rectangle of the current item.
 */

void SubMenu::getItemRect(MenuItem *find, Rect &rect)
{
	MenuItem *test = menu->child;
	int y = 1;

	while (test != NULL && test != find)
	{
		y++;
		test = test->next;
	}
	rect.makeAssign(1, y, size.x - 1, y + 1);
}
