/*
 * rect.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include "venus.h"

/*
 * Adjusts the coordinates if they are invalid.
 */

void Rect::check()
{
	if (a.x >= b.x || a.y >= b.y) a.x = a.y = b.x = b.y = 0;
}

/*
 * Returns non-zero if the rectangle contains the specified point.
 */

int Rect::containsPoint(Point pnt)
{
	return pnt.x >= a.x && pnt.x < b.x && pnt.y >= a.y && pnt.y < b.y;
}

/*
 * Returns non-zero if the rectangle is empty, meaning the rectangle
 * contains no character spaces.  Essentially, the a and b fields are equal.
 */

int Rect::isEmpty()
{
	return a.x >= b.x || a.y >= b.y;
}

/*
 * Returns non-zero if the specified rectangle is the same as the object
 * rectangle.
 */

int Rect::isEqual(Rect rct)
{
	return a.x == rct.a.x && a.y == rct.a.y && b.x == rct.b.x &&
		b.y == rct.b.y;
}

/*
 * Sets the parameter values to the rectangle's point fields.
 */

void Rect::makeAssign(int ax, int ay, int bx, int by)
{
	a.x = ax;
	a.y = ay;
	b.x = bx;
	b.y = by;
}

/*
 * Sets all fields equal to those in the specified rectangle.
 */

void Rect::makeCopy(Rect rct)
{
	a.x = rct.a.x;
	a.y = rct.a.y;
	b.x = rct.b.x;
	b.y = rct.b.y;
}

/*
 * Changes the size of the rectangle.
 */

void Rect::makeGrow(int deltax, int deltay)
{
	a.x -= deltax;
	a.y -= deltay;
	b.x += deltax;
	b.y += deltay;
	check();
}

/*
 * Changes the location and size of the rectangle to the region defined by
 * the intersection of the current location and that of the specified
 * rectangle.
 */

void Rect::makeIntersect(Rect rct)
{
	if (a.x < rct.a.x) a.x = rct.a.x;
	if (a.y < rct.a.y) a.y = rct.a.y;
	if (b.x > rct.b.x) b.x = rct.b.x;
	if (b.y > rct.b.y) b.y = rct.b.y;
	check();
}

/*
 * Moves the rectangle.
 */

void Rect::makeMove(int deltax, int deltay)
{
	a.x += deltax;
	a.y += deltay;
	b.x += deltax;
	b.y += deltay;
}

/*
 * Changes the rectangle to be the union of itself and the rectangle
 * specified (the smallest rectangle containing both the object and the
 * specified rectangle).
 */

void Rect::makeUnion(Rect rct)
{
	if (a.x > rct.a.x) a.x = rct.a.x;
	if (a.y > rct.a.y) a.y = rct.a.y;
	if (b.x < rct.b.x) b.x = rct.b.x;
	if (b.y < rct.b.y) b.y = rct.b.y;
	check();
}
