/*
 * statline.cc
 *
 * Copyright (C) 1996 Sergio Sigala <ssigala@globalnet.it>
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 *
 * SERGIO SIGALA DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SERGIO SIGALA BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdio.h>
#include "statline.h"

/*
 * StatusDef constructor.
 */

StatusDef::StatusDef(unsigned amin, unsigned amax, StatusKey *aitems,
	StatusDef *anext)
{
	items = aitems;
	max = amax;
	min = amin;
	next = anext;
}

/*
 * StatusDef destructor.
 */

StatusDef::~StatusDef()
{
	while (items != NULL)
	{
		StatusKey *next = items->next;

		delete items;
		items = next;
	}
}

/*
 * StatusKey constructor.
 */

StatusKey::StatusKey(char *atext, int akeycode, int acommand,
	StatusKey *anext)
{
	command = acommand;
	keycode = akeycode;
	next = anext;
	text = atext;
}

/*
 * StatusLine constructor.
 */

StatusLine::StatusLine(Rect bounds, StatusDef *adefs): View(bounds)
{
	DEBUG("new StatusLine at %p\n", this);
	options |= OF_PREPROCESS;
	defs = adefs;
	findItems();
}

/*
 * StatusLine destructor.
 */ 

StatusLine::~StatusLine()
{
	DEBUG("delete StatusLine at %p\n", this);
	while (defs != NULL)
	{
		StatusDef *next = defs->next;

		delete defs;
		defs = next;
	}
}

/*
 * Draws the StatusLine on the screen.
 */

void StatusLine::draw()
{
	drawSelect(NULL);
}

/*
 * Draws the StatusLine on the screen.
 */

void StatusLine::drawSelect(StatusKey *selected)
{
	DrawBuf buf;
	StatusKey *item = items;
	char str[MAX_WIDTH];
	int col_normal = getColor(C_STATUSLINE);
	int col_normaldisabled = getColor(C_STATUSLINE + 1);
	int col_select = getColor(C_STATUSLINE + 2);
	int col_selectdisabled = getColor(C_STATUSLINE + 3);
	int x = 0;

	moveChar(buf, ' ', col_normal, size.x);
	while (item != NULL)
	{
		if (item->text != NULL)
		{
			int len = cStrLen(item->text);

			if (x + len < size.x)
			{
				int col;

				if (commandEnabled(item->command))
				{
					if (item == selected)
						col = col_select;
					else col = col_normal;
				}
				else
				{
					if (item == selected)
						col = col_selectdisabled;
					else col = col_normaldisabled;
				}
				sprintf(str, " %s ", item->text);
				moveCStr(buf + x, str, col);
			}
			x += len + 2;
		}
		item = item->next;
	}

	/* append the hint if exists */

	if (x + 2 < size.x && getHint(helpctx) != NULL)
	{
		moveChar(buf + x, 179, col_normal, 1);
		x += 2;
		strncpy2(str, getHint(helpctx), size.x - x);
		moveStr(buf + x, str, col_normal);
	}
	writeBuf(0, 0, size.x, size.y, buf);
}

/*
 * Finds the group of keys to display on the screen.
 */

void StatusLine::findItems()
{
	StatusDef *obj = defs;

	while (obj != NULL)
	{
		if (helpctx >= obj->min && helpctx <= obj->max)
		{
			items = obj->items;
			return;
		}
		obj = obj->next;
	}
	items = NULL;
}

/*
 * Returns the address of the object under the mouse cursor.
 */

StatusKey *StatusLine::findMouse(Point mouse)
{
	makeLocal(mouse, mouse);
	if (localPointInside(mouse))
	{
		StatusKey *obj = items;
		int start = 0;

		while (obj != NULL)
		{
			if (obj->text != NULL)
			{
				int end = start + cStrLen(obj->text) + 2;

				if (mouse.x >= start && mouse.x < end)
				{
					return obj;
				}
				start = end;
			}
			obj = obj->next;
		}
	}
	return NULL;
}

/*
 * Returns the available hint string address, or NULL if nothing.
 */

char *StatusLine::getHint(unsigned helpctx)
{
	helpctx = 0;			/* prevent warning messages */
	return NULL;
}   

/*
 * Handles the StatusLine's events.
 */

void StatusLine::handleEvent(Event &event)
{
	StatusKey *obj = NULL;

	View::handleEvent(event);
	switch (event.what)
	{
	case EV_MOUSEDOWN:
		do
		{
			StatusKey *find;

			switch (event.what)
			{
			case EV_MOUSEDOWN:
			case EV_MOUSEMOVE:
				if ((find = findMouse(event.where)) != obj)
				{
					obj = find;
					drawSelect(obj);
				}
			}
			getEvent(event);
		}
		while (event.what != EV_MOUSEUP);
		drawView();
		clearEvent(event);
		break;
	case EV_KEYDOWN:
		obj = items;
		while (obj != NULL)
		{
			if (event.keycode == obj->keycode)
			{
				clearEvent(event);
				break;
			}
			obj = obj->next;
		}
		break;
	case EV_BROADCAST:
		switch (event.command)
		{
		case CM_COMMANDSETCHANGED:
			drawView();
		}
	}
	if (obj != NULL && commandEnabled(obj->command))
	{
		Event make;

		make.what = EV_COMMAND;
		make.command = obj->command;
		putEvent(make);
	}
}
