// Copyright (C) 1999-2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include "config.h"
#include "macros.h"
#include "thread.h"
#include "serial.h"
#include <stdlib.h>
#include <limits.h>
#include <termios.h>

#ifndef	MAX_INPUT
#define	MAX_INPUT 255
#endif

#ifndef MAX_CANON
#define MAX_CANON MAX_INPUT
#endif

#ifdef	__FreeBSD__
#undef	_PC_MAX_INPUT
#undef	_PC_MAX_CANON
#endif

#if defined(_THR_UNIXWARE) || defined(__hpux) || defined(_AIX)
#include <sys/termiox.h>
#define	CRTSCTS	(CTSXON | RTSXOFF)
#endif

// IRIX

#ifndef	CRTSCTS
#ifdef	CNEW_RTSCTS
#define	CRTSCTS (CNEW_RTSCTS)
#endif
#endif

Serial::Serial(const char *fname)
{
	initSerial();
	dev = ::open(fname, O_RDWR | O_NDELAY);
	if(dev < 0)
	{
		Error(SERIAL_OPEN_FAILED);
		return;
	}
	if(!isatty(dev))
	{
		::close(dev);
		dev = -1;
		Error(SERIAL_OPEN_NOTTY);
		return;
	}
	initConfig();
}

void Serial::initConfig(void)
{
	struct termios *attr = (struct termios *)current;
	struct termios *orig = (struct termios *)original;
	long ioflags = fcntl(dev, F_GETFL);
	tcgetattr(dev, (struct termios *)original);
	tcgetattr(dev, (struct termios *)current);

	attr->c_oflag = attr->c_lflag = attr->c_iflag = 0;
	attr->c_cflag = CLOCAL | CREAD | HUPCL;
	memset(attr->c_cc, 0, sizeof(attr->c_cc));
	attr->c_cc[VMIN] = 1;
	
	cfsetispeed(attr, cfgetispeed(orig));
	cfsetospeed(attr, cfgetospeed(orig));
	attr->c_cflag = orig->c_cflag & (CRTSCTS | CSIZE | PARENB | PARODD | CSTOPB);
	attr->c_lflag = orig->c_lflag & (IXON | IXANY | IXOFF);
	

	tcsetattr(dev, TCSANOW, attr);
	fcntl(dev, F_SETFL, ioflags & ~O_NDELAY);
}

void Serial::Restore(void)
{
	memcpy(current, original, sizeof(struct termios));
	tcsetattr(dev, TCSANOW, (struct termios *)current);
}

void Serial::initSerial(void)
{
	dev = -1;
	flags.thrown = false;
	flags.linebuf = false;
	errid = SERIAL_SUCCESS;
	errstr = NULL;
	current = new struct termios;
	original = new struct termios;
}

void Serial::endSerial(void)
{
	if(dev > -1)
	{
		tcsetattr(dev, TCSANOW, (struct termios *)original);
		::close(dev);
	}

	if(current)
		delete (struct termios *)current;

	if(original)
		delete (struct termios *)original;

	dev = -1;
	current = NULL;
	original = NULL;
}

sioerror_t Serial::Error(sioerror_t err, char *errs) 
{
        errid = err;
        errstr = errs;
	if(!err)
		return err;

        if(flags.thrown)
                return err;
 
        // prevents recursive throws
 
        flags.thrown = true;
	if(getException() == THROW_OBJECT)
	        throw((Serial *)this);
        return err;                                                             
}

int Serial::setPacketInput(int size, unsigned char btimer)
{
#ifdef	_PC_MAX_INPUT
	int max = fpathconf(dev, _PC_MAX_INPUT);
#else
	int max = MAX_INPUT;
#endif
	struct termios *attr = (struct termios *)current;

	if(size > max)
		size = max;

	attr->c_cc[VEOL] = attr->c_cc[VEOL2] = 0;
	attr->c_cc[VMIN] = (unsigned char)size;
	attr->c_cc[VTIME] = btimer;
	attr->c_lflag &= ~ICANON;
	tcsetattr(dev, TCSANOW, attr);
	bufsize = size;
	return size;
}

int Serial::setLineInput(char newline, char nl1)
{
	struct termios *attr = (struct termios *)current;
	attr->c_cc[VMIN] = attr->c_cc[VTIME] = 0;
	attr->c_cc[VEOL] = newline;
	attr->c_cc[VEOL2] = nl1;
	attr->c_lflag |= ICANON;
	tcsetattr(dev, TCSANOW, attr);
#ifdef _PC_MAX_CANON
	bufsize = fpathconf(dev, _PC_MAX_CANON);
#else
	bufsize = MAX_CANON;
#endif
	return bufsize;
}

void Serial::flushInput(void)
{
	tcflush(dev, TCIFLUSH);
}

void Serial::flushOutput(void)
{
	tcflush(dev, TCOFLUSH);
}

void Serial::waitOutput(void)
{
	tcdrain(dev);
}

Serial &Serial::operator=(const Serial &ser)
{
	::close(dev);
	if(ser.dev < 0)
		return *this;

	dev = dup(ser.dev);
	memcpy(current, ser.current, sizeof(struct termios));
	memcpy(original, ser.original, sizeof(struct termios));
	return *this;
}

sioerror_t Serial::setSpeed(unsigned long speed)
{
	struct termios *attr = (struct termios *)current;
	speed_t rate;

	switch(speed)
        {
#ifdef B115200
        case 115200: 
                rate = B115200;
                break; 
#endif
#ifdef B57600
        case 57600: 
                rate = B57600;
                break; 
#endif
#ifdef B38400
        case 38400: 
                rate = B38400;
                break; 
#endif
        case 19200:
                rate = B19200;
                break;
        case 9600:
                rate = B9600;
                break;
        case 4800:
                rate = B4800;
                break;                                                          
        case 2400:
                rate = B2400;
                break;
        case 1200:
                rate = B1200;
                break;
        case 600:
                rate = B600;
                break;
        case 300:
                rate = B300;
                break;
        case 110:
                rate = B110;
                break;
        case 0:
                rate = B0;
                break;
	default:
		return Error(SERIAL_SPEED_INVALID);
	}
	cfsetispeed(attr, rate);
	cfsetospeed(attr, rate);
	tcsetattr(dev, TCSANOW, attr);
	return SERIAL_SUCCESS;
}
		
sioerror_t Serial::setFlowControl(sioflow_t flow)
{
	struct termios *attr = (struct termios *)current;
	
        attr->c_cflag &= ~CRTSCTS;
        attr->c_iflag &= ~(IXON | IXANY | IXOFF);
 
        switch(flow)
        {
        case SERIAL_FLOW_SOFT:
                attr->c_iflag |= (IXON | IXANY | IXOFF);
                break;
        case SERIAL_FLOW_BOTH:
                attr->c_iflag |= (IXON | IXANY | IXOFF);
        case SERIAL_FLOW_HARD:
                attr->c_cflag |= CRTSCTS;
                break;
	case SERIAL_FLOW_NONE:
		break;
	default:
		return Error(SERIAL_FLOW_INVALID);
        }                                                                       
	
	tcsetattr(dev, TCSANOW, attr);
	return SERIAL_SUCCESS;
}

sioerror_t Serial::setStopBits(int bits)
{
	struct termios *attr = (struct termios *)current;
	attr->c_cflag &= ~CSTOPB;

	switch(bits)
	{
	case 1:
		break;
	case 2:
		attr->c_cflag |= CSTOPB;
		break;
	default:
		return Error(SERIAL_STOPBITS_INVALID);
	}
	tcsetattr(dev, TCSANOW, attr);
	return SERIAL_SUCCESS;
}

sioerror_t Serial::setCharBits(int bits)
{
	struct termios *attr = (struct termios *)current;
	attr->c_cflag &= ~CSIZE;

	switch(bits)
	{
	case 5:
		attr->c_cflag |= CS5;
		break;
	case 6:
		attr->c_cflag |= CS6;
		break;
	case 7:
		attr->c_cflag |= CS7;
		break;
	case 8:
		attr->c_cflag |= CS8;
		break;
	default:
		return Error(SERIAL_CHARSIZE_INVALID);
	}
	tcsetattr(dev, TCSANOW, attr);
	return SERIAL_SUCCESS;
}

sioerror_t Serial::setParity(sioparity_t parity)
{
	struct termios *attr = (struct termios *)current;
	attr->c_cflag &= ~(PARENB | PARODD);

	switch(parity)
	{
	case SERIAL_PARITY_EVEN:
		attr->c_cflag |= PARENB;
		break;
	case SERIAL_PARITY_ODD:
		attr->c_cflag |= (PARENB | PARODD);
		break;
	case SERIAL_PARITY_NONE:
		break;
	default:
		return Error(SERIAL_PARITY_INVALID);
	}
	tcsetattr(dev, TCSANOW, attr);
	return SERIAL_SUCCESS;
}

bool Serial::isPending(siopend_t pending, unsigned long timeout)
{
	int status;
#ifdef HAVE_POLL
	struct pollfd pfd;	

	pfd.fd = dev;
	pfd.revents = 0;
	switch(pending)
	{
	case SERIAL_PENDING_INPUT:
		pfd.events = POLLIN;
		break;
	case SERIAL_PENDING_OUTPUT:
		pfd.events = POLLOUT;
		break;
	case SERIAL_PENDING_ERROR:
		pfd.events = POLLERR | POLLHUP;
		break;
	}

	if(timeout == TIMEOUT_INF)
		status = poll(&pfd, 1, -1);
	else
		status = poll(&pfd, 1, timeout);

	if(status < 1)
		return false;

	if(pfd.revents & pfd.events)
		return true;

	return false;
#else
	struct timeval tv;
	fd_set grp;
	struct timeval *tvp = &tv;

	if(timeout == TIMEOUT_INF)
		tvp = NULL;
	else
	{
		tv.tv_usec = (timeout % 1000) * 1000;
		tv.tv_sec = timeout / 1000;
	}

	FD_ZERO(&grp);
	FD_SET(dev, &grp);
	switch(pending)
	{
	case SERIAL_PENDING_INPUT:
		status = select(dev + 1, &grp, NULL, NULL, tvp);
		break;
	case SERIAL_PENDING_OUTPUT:
		status = select(dev + 1, NULL, &grp, NULL, tvp);
		break;
	case SERIAL_PENDING_ERROR:
		status = select(dev + 1, NULL, NULL, &grp, tvp);
		break;
	}
	if(status < 1)
		return false;

	if(FD_ISSET(dev, &grp))
		return true;

	return false;
#endif
}

TTYStream::TTYStream(const char *filename) :
Serial(filename), streambuf(), iostream()
{
	iostream::init((streambuf *)this);
	gbuf = pbuf = NULL;
	if(dev > -1)
		Allocate();
}

TTYStream::TTYStream() :
Serial(), streambuf(), iostream()
{
	iostream::init((streambuf *)this);
	gbuf = pbuf = NULL;
}

TTYStream::~TTYStream()
{
	endStream();
	endSerial();
}

void TTYStream::endStream(void)
{
	if(bufsize)
		sync();

	if(gbuf)
	{
		delete[] gbuf;
		gbuf = NULL;
	}
	if(pbuf)
	{
		delete[] pbuf;
		pbuf = NULL;
	}
	bufsize = 0;
}

void TTYStream::Allocate(void)
{
	if(dev < 0)
		return;

#ifdef _PC_MAX_INPUT
	bufsize = fpathconf(dev, _PC_MAX_INPUT);
#else
	bufsize = MAX_INPUT;
#endif

	gbuf = new char[bufsize];
	pbuf = new char[bufsize];
	if(!pbuf || !gbuf)
	{
		Error(SERIAL_RESOURCE_FAILURE);
		return;
	}

	clear();
	setb(gbuf, gbuf + bufsize, 0);
	setg(gbuf, gbuf + bufsize, gbuf + bufsize);
	setp(pbuf, pbuf + bufsize);
}

int TTYStream::doallocate()
{
	if(bufsize)
		return 0;
	
	Allocate();
	return 1;
}

void TTYStream::Interactive(bool iflag)
{
	if(dev < 0)
		return;

	if(bufsize >= 1)
		endStream();

	if(iflag)
	{
		// setting to unbuffered mode

		bufsize = 1;
		gbuf = new char[bufsize];
		setb(0,0);
		setg(gbuf, gbuf+bufsize, gbuf+bufsize);
		setp(pbuf, pbuf);
		return;
	}

	if(bufsize < 2)
		Allocate();
}

int TTYStream::uflow(void)
{
	int rlen;
	unsigned char ch;

	if(bufsize < 2)
	{
		rlen = ::read(dev, (char *)&ch, 1);
		if(rlen < 1)
		{
			if(rlen < 0)
				clear(ios::failbit | rdstate());
			return EOF;
		}
		return ch;
	}
	else
	{
		ch = underflow();
		gbump(1);
		return ch;
	}
}

int TTYStream::underflow(void)
{
	int rlen;

	if(!gptr())
		return EOF;

	if(gptr() < egptr())
		return (unsigned char)*gptr();

	rlen = (gbuf + bufsize) - eback();
	rlen = ::read(dev, (char *)eback(), rlen);
	if(rlen < 1)
	{
		if(rlen < 0)
			clear(ios::failbit | rdstate());
		return EOF;
	}

	setg(eback(), eback(), eback() + rlen);
	return (unsigned char) *gptr();
}

int TTYStream::sync(void)
{
	if(bufsize > 1 && pbase() && ((pptr() - pbase()) > 0))
	{
		overflow(0);
		waitOutput();
		setp(pbuf, pbuf + bufsize);
	}
	return 0;
}

int TTYStream::overflow(int c)
{
	unsigned char ch;
	int rlen, req;

	if(bufsize < 2)
	{
		if(c == EOF)
			return 0;

		ch = (unsigned char)(c);
		rlen = ::write(dev, (char *)&ch, 1);
		if(rlen < 1)
		{
			if(rlen < 0)
				clear(ios::failbit | rdstate());
			return EOF;
		}
		else
			return c;
	}

	if(!pbase())
		return EOF;

	req = pptr() - pbase();
	if(req)
	{
		rlen = ::write(dev, (char *)pbase(), req);
		if(rlen < 1)
		{
			if(rlen < 0)
				clear(ios::failbit | rdstate());
			return EOF;
		}
		req -= rlen;
	}
	
	if(req)
		memcpy(pptr(), pptr() + rlen, req);
	setp(pbuf + req, pbuf + bufsize);

	if(c != EOF)
	{
		*pptr() = (unsigned char)c;
		pbump(1);
	}
	return c;
}

bool TTYStream::isPending(siopend_t pending, unsigned long timeout)
{
	if(pending == SERIAL_PENDING_INPUT && in_avail())
		return true;
	else if(pending == SERIAL_PENDING_OUTPUT)
		flush();

	return Serial::isPending(pending, timeout);
}

ttystream::ttystream() : 
TTYStream()
{
	setError(false);
}

ttystream::ttystream(const char *name) :
TTYStream()
{
	setError(false);
	open(name);
}

void ttystream::close(void)
{
	if(dev < 0)
		return;

	endStream();
	Restore();
	::close(dev);
	dev = -1;
}

void ttystream::open(const char *name)
{
	char *cp;
	char pathname[256];
	int namelen;
	long opt;

	if(dev)
	{
		Restore();
		::close(dev);
	}

	cp = strrchr(name, ':');
	if(cp)
		namelen = cp - name;
	else
		namelen = strlen(name);

	cp = pathname;
	if(*name != '/')
	{
		strcpy(pathname, "/dev/");
		cp += 5;
	}

	if((cp - pathname) + namelen > 255)
	{
		Error(SERIAL_RESOURCE_FAILURE);
		return;
	}
	strncpy(cp, name, namelen);
	cp += namelen;
	*cp = 0;
	
	dev = ::open(pathname, O_RDWR | O_NDELAY);

	if(dev < 0)
	{
		Error(SERIAL_OPEN_FAILED);
		return;
	}

	if(!isatty(dev))
	{
		::close(dev);
		dev = -1;
		Error(SERIAL_OPEN_NOTTY);
		return;
	}

	initConfig();
	Allocate();

	strncpy(pathname, name + namelen, 255);
	pathname[255] = 0;
	cp = pathname + 1;

	if(*pathname == ':')
		cp = strtok(cp, ",");
	else
		cp = NULL;

	while(cp)
	{
		switch(*cp)
		{
		case 'h':
		case 'H':
			setFlowControl(SERIAL_FLOW_HARD);
			break;
		case 's':
		case 'S':
			setFlowControl(SERIAL_FLOW_SOFT);
			break;
		case 'b':
		case 'B':
			setFlowControl(SERIAL_FLOW_BOTH);
			break;
		case 'n':
		case 'N':
			setParity(SERIAL_PARITY_NONE);
			break;
		case 'O':
		case 'o':
			setParity(SERIAL_PARITY_ODD);
			break;
		case 'e':
		case 'E':
			setParity(SERIAL_PARITY_EVEN);
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			opt = atol(cp);
			if(opt == 1 || opt == 2)
			{
				setStopBits((int)opt);
				break;
			}
			if(opt > 4 && opt < 9)
			{
				setCharBits((int)opt);
				break;
			}
			setSpeed(opt);
			break;
		default:
			Error(SERIAL_OPTION_INVALID);
		}
		cp = strtok(NULL, ","); 
	}			
}

TTYSession::TTYSession(const char *filename, Semaphore *start, int pri, int stack) :
TTYStream(filename), Thread(start, pri, stack)
{
	setError(false);
}

SerialPort::SerialPort(SerialService *svc, const char *name) :
Serial(name),
detect_pending(true),
detect_output(false),
detect_disconnect(true)
{
	next = prev = NULL;
	service = NULL;

	if(dev > -1)
	{
		setError(false);
		service = svc;
		svc->Attach(this);
	}
}

SerialPort::~SerialPort()
{
	if(service)
		service->Detach(this);

	endSerial();
}

void SerialPort::setTimer(timeout_t ptimer)
{
	TimerPort::setTimer(ptimer);
	service->Update();
}

void SerialPort::incTimer(timeout_t ptimer)
{
	TimerPort::incTimer(ptimer);
	service->Update();
}

void SerialPort::setDetectPending( bool val )
{
	if ( detect_pending != val ) {
		detect_pending = val;
#ifdef __CCXX_USE_POLL
		if ( ufd ) {
			if ( val ) {
				ufd->events |= POLLIN;
			} else {
				ufd->events &= ~POLLIN;
			}
		}
#endif
		service->Update();
	}
}

void SerialPort::setDetectOutput( bool val )
{
	if ( detect_output != val ) {
		detect_output = val;
#ifdef  __CCXX_USE_POLL
		if ( ufd ) {
			if ( val ) {
				ufd->events |= POLLOUT;
			} else {
				ufd->events &= ~POLLOUT;
			}
		}
#endif
		service->Update();
	}
}

SerialService::SerialService(int pri) :
Thread(NULL, pri), Mutex()
{
	long opt;

	first = last = NULL;
	count = 0;
	FD_ZERO(&connect);
	::pipe(iosync);
	hiwater = iosync[0] + 1;
	FD_SET(iosync[0], &connect);

	opt = fcntl(iosync[0], F_GETFL);
	fcntl(iosync[0], F_SETFL, opt | O_NDELAY);	
}

SerialService::~SerialService()
{
	Update(0);
	Terminate();

	while(first)
		delete first;
}

void SerialService::Attach(SerialPort *port)
{
	EnterMutex();
#ifdef	__CCXX_USE_POLL
	port->ufd = 0;
#endif
	if(last)
		last->next = port;

	port->prev = last;
	last = port;
	FD_SET(port->dev, &connect);
	if(port->dev >= hiwater)
		hiwater = port->dev + 1;

	if(!first)
	{
		first = port;
		LeaveMutex();
		Start();
	}
	else
	{
		LeaveMutex();
		Update();
	}			
	++count;
}

void SerialService::Detach(SerialPort *port)
{
	EnterMutex();

#ifndef __CCXX_USE_POLL
	FD_CLR(port->dev, &connect);
#endif

	if(port->prev)
		port->prev->next = port->next;
	else
		first = port->next;
		
	if(port->next)
		port->next->prev = port->prev;
	else
		last = port->prev;

	--count;
	LeaveMutex();
	Update();
}

void SerialService::Update(unsigned char flag)
{
	::write(iosync[1], (char *)&flag, 1);
}

		
void SerialService::Run(void)
{
	timeout_t timer, expires;
	SerialPort *port;
	unsigned char buf;
	int so;

#ifdef	__CCXX_USE_POLL
	
	Poller	mfd;
	pollfd *p_ufd;
	int lastcount = 0;

	// initialize ufd in all attached ports :
	// probably don't need this but it can't hurt.
	EnterMutex();
	port = first;
	while(port)
	{
		port->ufd = 0;
		port = port->next;
	}
	LeaveMutex();
	
#else
	struct timeval timeout, *tvp;
	fd_set inp, out, err;
	int dev;
	FD_ZERO(&inp);
	FD_ZERO(&out);
	FD_ZERO(&err);
#endif

	setCancel(THREAD_CANCEL_DEFERRED);
	for(;;)
	{
		timer = TIMEOUT_INF;
		while(1 == ::read(iosync[0], (char *)&buf, 1))
		{
			if(buf)
			{
				OnUpdate(buf);
				continue;
			}

			setCancel(THREAD_CANCEL_IMMEDIATE);
			Sleep(TIMEOUT_INF);
			Exit();
		}

#ifdef	__CCXX_USE_POLL

		bool	reallocate = false;
		
		EnterMutex();
		OnEvent();
		port = first;
		while(port)
		{
			OnCallback(port);
			if ( ( p_ufd = port->ufd ) ) {

				if ( ( POLLHUP | POLLNVAL ) & p_ufd->revents ) {
					// Avoid infinite loop from disconnected sockets
					port->detect_disconnect = false;
					p_ufd->events &= ~POLLHUP;
					port->Disconnect();
				}
	
				if ( ( POLLIN | POLLPRI ) & p_ufd->revents )
					port->Pending();
	
				if ( POLLOUT & p_ufd->revents )
					port->Output();

			} else {
				reallocate = true;
			}

retry:
			expires = port->getTimer();
			if(expires > 0)
				if(expires < timer)
					timer = expires;

			if(!expires)
			{
				port->endTimer();
				port->Expired();
				goto retry;
			}

			port = port->next;
		}

		//
		// reallocate things if we saw a ServerPort without
		// ufd set !
		if ( reallocate || ( ( count + 1 ) != lastcount ) ) {
			lastcount = count + 1;
			p_ufd = mfd.getList( count + 1 );
	
			// Set up iosync polling
			p_ufd->fd = iosync[0];
			p_ufd->events = POLLIN | POLLHUP;
			p_ufd ++;
			
			port = first;
			while(port)
			{
				p_ufd->fd = port->dev;
				p_ufd->events =
					( port->detect_disconnect ? POLLHUP : 0 )
					| ( port->detect_output ? POLLOUT : 0 )
					| ( port->detect_pending ? POLLIN : 0 )
				;
				port->ufd = p_ufd;
				p_ufd ++;
				port = port->next;
			}
		}
		LeaveMutex();

		poll( mfd.getList(), count + 1, timer );

#else
		EnterMutex();
		OnEvent();
		port = first;

		while(port)
		{
			OnCallback(port);
			dev = port->dev;
			if(FD_ISSET(dev, &err)) {
				port->detect_disconnect = false;
				port->Disconnect();
			}

			if(FD_ISSET(dev, &inp))
				port->Pending();

			if(FD_ISSET(dev, &out))
				port->Output();

retry:
			expires = port->getTimer();
			if(expires > 0)
				if(expires < timer)
					timer = expires;

			if(!expires)
			{
				port->endTimer();
				port->Expired();
				goto retry;
			}

			port = port->next;
		}

		FD_ZERO(&inp);
		FD_ZERO(&out);
		FD_ZERO(&err);
		port = first;
		while(port)
		{

			if(port->detect_pending)
				FD_SET(so, &inp);

			if(port->detect_output)
				FD_SET(so, &out);

			if(port->detect_disconnect)
				FD_SET(so, &err);

			port = port->next;
		}
				
		LeaveMutex();
		if(timer == TIMEOUT_INF)
			tvp = NULL;
		else
		{
			tvp = &timeout;
			timeout.tv_sec = timer / 1000;
			timeout.tv_usec = (timer % 1000) * 1000;
		}
		select(hiwater, &inp, &out, &err, tvp);		
#endif
	}
}						
		

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
