/*
 * This file is part of LibCSS
 * Licensed under the MIT License,
 *                http://www.opensource.org/licenses/mit-license.php
 * Copyright 2017 The NetSurf Project
 */


#define BORDER_SPACING_INDEX 0
#define BORDER_SPACING_SHIFT 21
#define BORDER_SPACING_MASK 0xffe00000
static inline uint8_t get_border_spacing(const css_computed_style *style,
		css_fixed *length_a, css_unit *unit_a, css_fixed *length_b,
		css_unit *unit_b)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[BORDER_SPACING_INDEX];
		bits &= BORDER_SPACING_MASK;
		bits >>= BORDER_SPACING_SHIFT;
		
		/* 11bits: aaaaabbbbbt : unit_a | unit_b | type */
		if ((bits & 0x1) == CSS_BORDER_SPACING_SET) {
			*length_a = style->i.uncommon->i.border_spacing_a;
			*length_b = style->i.uncommon->i.border_spacing_b;
			*unit_a = bits >> 6;
			*unit_b = (bits & 0x3e) >> 1;
		}
		
		return (bits & 0x1);
	}
	
	/* Initial value */
	*length_a = 0;
	*unit_a = CSS_UNIT_PX;
	*length_b = 0;
	*unit_b = CSS_UNIT_PX;
	return CSS_BORDER_SPACING_SET;
}
#undef BORDER_SPACING_INDEX
#undef BORDER_SPACING_SHIFT
#undef BORDER_SPACING_MASK

#define BREAK_AFTER_INDEX 3
#define BREAK_AFTER_SHIFT 28
#define BREAK_AFTER_MASK 0xf0000000
static inline uint8_t get_break_after(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[BREAK_AFTER_INDEX];
		bits &= BREAK_AFTER_MASK;
		bits >>= BREAK_AFTER_SHIFT;
		
		/* 4bits: tttt : type */
		
		return (bits & 0xf);
	}
	
	/* Initial value */
	return CSS_BREAK_AFTER_AUTO;
}
#undef BREAK_AFTER_INDEX
#undef BREAK_AFTER_SHIFT
#undef BREAK_AFTER_MASK

#define BREAK_BEFORE_INDEX 1
#define BREAK_BEFORE_SHIFT 0
#define BREAK_BEFORE_MASK 0xf
static inline uint8_t get_break_before(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[BREAK_BEFORE_INDEX];
		bits &= BREAK_BEFORE_MASK;
		bits >>= BREAK_BEFORE_SHIFT;
		
		/* 4bits: tttt : type */
		
		return (bits & 0xf);
	}
	
	/* Initial value */
	return CSS_BREAK_BEFORE_AUTO;
}
#undef BREAK_BEFORE_INDEX
#undef BREAK_BEFORE_SHIFT
#undef BREAK_BEFORE_MASK

#define BREAK_INSIDE_INDEX 3
#define BREAK_INSIDE_SHIFT 24
#define BREAK_INSIDE_MASK 0xf000000
static inline uint8_t get_break_inside(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[BREAK_INSIDE_INDEX];
		bits &= BREAK_INSIDE_MASK;
		bits >>= BREAK_INSIDE_SHIFT;
		
		/* 4bits: tttt : type */
		
		return (bits & 0xf);
	}
	
	/* Initial value */
	return CSS_BREAK_INSIDE_AUTO;
}
#undef BREAK_INSIDE_INDEX
#undef BREAK_INSIDE_SHIFT
#undef BREAK_INSIDE_MASK

#define CLIP_INDEX 2
#define CLIP_SHIFT 6
#define CLIP_MASK 0xffffffc0
static inline uint8_t get_clip(
		const css_computed_style *style,
		css_computed_clip_rect *rect)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[CLIP_INDEX];
		bits &= CLIP_MASK;
		bits >>= CLIP_SHIFT;

		/*
		26bits: tt tttr rrrr bbbb blll llTR BLyy:
		units: top | right | bottom | left
		opcodes: top | right | bottom | left | type
		*/

		if ((bits & 0x3) == CSS_CLIP_RECT) {
			rect->left_auto = (bits & 0x4);
			rect->bottom_auto = (bits & 0x8);
			rect->right_auto = (bits & 0x10);
			rect->top_auto = (bits & 0x20);

			rect->top = style->i.uncommon->i.clip_a;
			rect->tunit = bits & 0x3e00000 >> 21;

			rect->right = style->i.uncommon->i.clip_b;
			rect->runit = bits & 0x1f0000 >> 16;

			rect->bottom = style->i.uncommon->i.clip_c;
			rect->bunit = (bits & 0xf800) >> 11;

			rect->left = style->i.uncommon->i.clip_d;
			rect->lunit = (bits & 0x7c0) >> 6;
		}

		return (bits & 0x3);
	}

	/* Initial value */
	return CSS_CLIP_AUTO;
}
#undef CLIP_INDEX
#undef CLIP_SHIFT
#undef CLIP_MASK

#define COLUMN_COUNT_INDEX 3
#define COLUMN_COUNT_SHIFT 14
#define COLUMN_COUNT_MASK 0xc000
static inline uint8_t get_column_count(const css_computed_style *style, int32_t
		*integer)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[COLUMN_COUNT_INDEX];
		bits &= COLUMN_COUNT_MASK;
		bits >>= COLUMN_COUNT_SHIFT;
		
		/* 2bits: tt : type */
		*integer = style->i.uncommon->i.column_count;
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*integer = 0;
	return CSS_COLUMN_COUNT_AUTO;
}
#undef COLUMN_COUNT_INDEX
#undef COLUMN_COUNT_SHIFT
#undef COLUMN_COUNT_MASK

#define COLUMN_FILL_INDEX 3
#define COLUMN_FILL_SHIFT 18
#define COLUMN_FILL_MASK 0xc0000
static inline uint8_t get_column_fill(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[COLUMN_FILL_INDEX];
		bits &= COLUMN_FILL_MASK;
		bits >>= COLUMN_FILL_SHIFT;
		
		/* 2bits: tt : type */
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	return CSS_COLUMN_FILL_BALANCE;
}
#undef COLUMN_FILL_INDEX
#undef COLUMN_FILL_SHIFT
#undef COLUMN_FILL_MASK

#define COLUMN_GAP_INDEX 1
#define COLUMN_GAP_SHIFT 18
#define COLUMN_GAP_MASK 0x1fc0000
static inline uint8_t get_column_gap(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[COLUMN_GAP_INDEX];
		bits &= COLUMN_GAP_MASK;
		bits >>= COLUMN_GAP_SHIFT;
		
		/* 7bits: uuuuutt : unit | type */
		if ((bits & 0x3) == CSS_COLUMN_GAP_SET) {
			*length = style->i.uncommon->i.column_gap;
			*unit = bits >> 2;
		}
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*length = 0;
	*unit = CSS_UNIT_PX;
	return CSS_COLUMN_GAP_NORMAL;
}
#undef COLUMN_GAP_INDEX
#undef COLUMN_GAP_SHIFT
#undef COLUMN_GAP_MASK

#define COLUMN_RULE_COLOR_INDEX 2
#define COLUMN_RULE_COLOR_SHIFT 0
#define COLUMN_RULE_COLOR_MASK 0x3
static inline uint8_t get_column_rule_color(const css_computed_style *style,
		css_color *color)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[
				COLUMN_RULE_COLOR_INDEX];
		bits &= COLUMN_RULE_COLOR_MASK;
		bits >>= COLUMN_RULE_COLOR_SHIFT;
		
		/* 2bits: tt : type */
		*color = style->i.uncommon->i.column_rule_color;
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*color = 0;
	return CSS_COLUMN_RULE_COLOR_CURRENT_COLOR;
}
#undef COLUMN_RULE_COLOR_INDEX
#undef COLUMN_RULE_COLOR_SHIFT
#undef COLUMN_RULE_COLOR_MASK

#define COLUMN_RULE_STYLE_INDEX 2
#define COLUMN_RULE_STYLE_SHIFT 2
#define COLUMN_RULE_STYLE_MASK 0x3c
static inline uint8_t get_column_rule_style(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[
				COLUMN_RULE_STYLE_INDEX];
		bits &= COLUMN_RULE_STYLE_MASK;
		bits >>= COLUMN_RULE_STYLE_SHIFT;
		
		/* 4bits: tttt : type */
		
		return (bits & 0xf);
	}
	
	/* Initial value */
	return CSS_COLUMN_RULE_STYLE_NONE;
}
#undef COLUMN_RULE_STYLE_INDEX
#undef COLUMN_RULE_STYLE_SHIFT
#undef COLUMN_RULE_STYLE_MASK

#define COLUMN_RULE_WIDTH_INDEX 0
#define COLUMN_RULE_WIDTH_SHIFT 5
#define COLUMN_RULE_WIDTH_MASK 0x1fe0
static inline uint8_t get_column_rule_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[
				COLUMN_RULE_WIDTH_INDEX];
		bits &= COLUMN_RULE_WIDTH_MASK;
		bits >>= COLUMN_RULE_WIDTH_SHIFT;
		
		/* 8bits: uuuuuttt : unit | type */
		if ((bits & 0x7) == CSS_COLUMN_RULE_WIDTH_WIDTH) {
			*length = style->i.uncommon->i.column_rule_width;
			*unit = bits >> 3;
		}
		
		return (bits & 0x7);
	}
	
	/* Initial value */
	*length = 0;
	*unit = CSS_UNIT_PX;
	return CSS_COLUMN_RULE_WIDTH_MEDIUM;
}
#undef COLUMN_RULE_WIDTH_INDEX
#undef COLUMN_RULE_WIDTH_SHIFT
#undef COLUMN_RULE_WIDTH_MASK

#define COLUMN_SPAN_INDEX 3
#define COLUMN_SPAN_SHIFT 16
#define COLUMN_SPAN_MASK 0x30000
static inline uint8_t get_column_span(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[COLUMN_SPAN_INDEX];
		bits &= COLUMN_SPAN_MASK;
		bits >>= COLUMN_SPAN_SHIFT;
		
		/* 2bits: tt : type */
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	return CSS_COLUMN_SPAN_NONE;
}
#undef COLUMN_SPAN_INDEX
#undef COLUMN_SPAN_SHIFT
#undef COLUMN_SPAN_MASK

#define COLUMN_WIDTH_INDEX 1
#define COLUMN_WIDTH_SHIFT 25
#define COLUMN_WIDTH_MASK 0xfe000000
static inline uint8_t get_column_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[COLUMN_WIDTH_INDEX];
		bits &= COLUMN_WIDTH_MASK;
		bits >>= COLUMN_WIDTH_SHIFT;
		
		/* 7bits: uuuuutt : unit | type */
		if ((bits & 0x3) == CSS_COLUMN_WIDTH_SET) {
			*length = style->i.uncommon->i.column_width;
			*unit = bits >> 2;
		}
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*length = 0;
	*unit = CSS_UNIT_PX;
	return CSS_COLUMN_WIDTH_AUTO;
}
#undef COLUMN_WIDTH_INDEX
#undef COLUMN_WIDTH_SHIFT
#undef COLUMN_WIDTH_MASK

#define CONTENT_INDEX 3
#define CONTENT_SHIFT 22
#define CONTENT_MASK 0xc00000
static inline uint8_t get_content(const css_computed_style *style, const
		css_computed_content_item **content_item)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[CONTENT_INDEX];
		bits &= CONTENT_MASK;
		bits >>= CONTENT_SHIFT;
		
		/* 2bits: tt : type */
		if ((bits & 0x3) == CSS_CONTENT_SET) {
			*content_item = style->i.uncommon->content;
		}
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*content_item = NULL;
	return CSS_CONTENT_NORMAL;
}
#undef CONTENT_INDEX
#undef CONTENT_SHIFT
#undef CONTENT_MASK

#define COUNTER_INCREMENT_INDEX 3
#define COUNTER_INCREMENT_SHIFT 11
#define COUNTER_INCREMENT_MASK 0x800
static inline uint8_t get_counter_increment(const css_computed_style *style,
		const css_computed_counter **counter_arr)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[
				COUNTER_INCREMENT_INDEX];
		bits &= COUNTER_INCREMENT_MASK;
		bits >>= COUNTER_INCREMENT_SHIFT;
		
		/* 1bit: t : type */
		*counter_arr = style->i.uncommon->counter_increment;
		
		return (bits & 0x1);
	}
	
	/* Initial value */
	*counter_arr = NULL;
	return CSS_COUNTER_INCREMENT_NONE;
}
#undef COUNTER_INCREMENT_INDEX
#undef COUNTER_INCREMENT_SHIFT
#undef COUNTER_INCREMENT_MASK

#define COUNTER_RESET_INDEX 3
#define COUNTER_RESET_SHIFT 10
#define COUNTER_RESET_MASK 0x400
static inline uint8_t get_counter_reset(const css_computed_style *style, const
		css_computed_counter **counter_arr)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[COUNTER_RESET_INDEX];
		bits &= COUNTER_RESET_MASK;
		bits >>= COUNTER_RESET_SHIFT;
		
		/* 1bit: t : type */
		*counter_arr = style->i.uncommon->counter_reset;
		
		return (bits & 0x1);
	}
	
	/* Initial value */
	*counter_arr = NULL;
	return CSS_COUNTER_RESET_NONE;
}
#undef COUNTER_RESET_INDEX
#undef COUNTER_RESET_SHIFT
#undef COUNTER_RESET_MASK

#define CURSOR_INDEX 0
#define CURSOR_SHIFT 0
#define CURSOR_MASK 0x1f
static inline uint8_t get_cursor(const css_computed_style *style, lwc_string
		***string_arr)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[CURSOR_INDEX];
		bits &= CURSOR_MASK;
		bits >>= CURSOR_SHIFT;
		
		/* 5bits: ttttt : type */
		*string_arr = style->i.uncommon->cursor;
		
		return (bits & 0x1f);
	}
	
	/* Initial value */
	*string_arr = NULL;
	return CSS_CURSOR_AUTO;
}
#undef CURSOR_INDEX
#undef CURSOR_SHIFT
#undef CURSOR_MASK

#define LETTER_SPACING_INDEX 1
#define LETTER_SPACING_SHIFT 11
#define LETTER_SPACING_MASK 0x3f800
static inline uint8_t get_letter_spacing(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[LETTER_SPACING_INDEX];
		bits &= LETTER_SPACING_MASK;
		bits >>= LETTER_SPACING_SHIFT;
		
		/* 7bits: uuuuutt : unit | type */
		if ((bits & 0x3) == CSS_LETTER_SPACING_SET) {
			*length = style->i.uncommon->i.letter_spacing;
			*unit = bits >> 2;
		}
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*length = 0;
	*unit = CSS_UNIT_PX;
	return CSS_LETTER_SPACING_NORMAL;
}
#undef LETTER_SPACING_INDEX
#undef LETTER_SPACING_SHIFT
#undef LETTER_SPACING_MASK

#define OUTLINE_COLOR_INDEX 3
#define OUTLINE_COLOR_SHIFT 12
#define OUTLINE_COLOR_MASK 0x3000
static inline uint8_t get_outline_color(const css_computed_style *style,
		css_color *color)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[OUTLINE_COLOR_INDEX];
		bits &= OUTLINE_COLOR_MASK;
		bits >>= OUTLINE_COLOR_SHIFT;
		
		/* 2bits: tt : type */
		if ((bits & 0x3) == CSS_OUTLINE_COLOR_COLOR) {
			*color = style->i.uncommon->i.outline_color;
		}
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*color = 0;
	return CSS_OUTLINE_COLOR_INVERT;
}
#undef OUTLINE_COLOR_INDEX
#undef OUTLINE_COLOR_SHIFT
#undef OUTLINE_COLOR_MASK

#define OUTLINE_WIDTH_INDEX 0
#define OUTLINE_WIDTH_SHIFT 13
#define OUTLINE_WIDTH_MASK 0x1fe000
static inline uint8_t get_outline_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[OUTLINE_WIDTH_INDEX];
		bits &= OUTLINE_WIDTH_MASK;
		bits >>= OUTLINE_WIDTH_SHIFT;
		
		/* 8bits: uuuuuttt : unit | type */
		if ((bits & 0x7) == CSS_OUTLINE_WIDTH_WIDTH) {
			*length = style->i.uncommon->i.outline_width;
			*unit = bits >> 3;
		}
		
		return (bits & 0x7);
	}
	
	/* Initial value */
	*length = 0;
	*unit = CSS_UNIT_PX;
	return CSS_OUTLINE_WIDTH_MEDIUM;
}
#undef OUTLINE_WIDTH_INDEX
#undef OUTLINE_WIDTH_SHIFT
#undef OUTLINE_WIDTH_MASK

#define WORD_SPACING_INDEX 1
#define WORD_SPACING_SHIFT 4
#define WORD_SPACING_MASK 0x7f0
static inline uint8_t get_word_spacing(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[WORD_SPACING_INDEX];
		bits &= WORD_SPACING_MASK;
		bits >>= WORD_SPACING_SHIFT;
		
		/* 7bits: uuuuutt : unit | type */
		if ((bits & 0x3) == CSS_WORD_SPACING_SET) {
			*length = style->i.uncommon->i.word_spacing;
			*unit = bits >> 2;
		}
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	*length = 0;
	*unit = CSS_UNIT_PX;
	return CSS_WORD_SPACING_NORMAL;
}
#undef WORD_SPACING_INDEX
#undef WORD_SPACING_SHIFT
#undef WORD_SPACING_MASK

#define WRITING_MODE_INDEX 3
#define WRITING_MODE_SHIFT 20
#define WRITING_MODE_MASK 0x300000
static inline uint8_t get_writing_mode(const css_computed_style *style)
{
	if (style->i.uncommon != NULL) {
		uint32_t bits = style->i.uncommon->i.bits[WRITING_MODE_INDEX];
		bits &= WRITING_MODE_MASK;
		bits >>= WRITING_MODE_SHIFT;
		
		/* 2bits: tt : type */
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	return CSS_WRITING_MODE_HORIZONTAL_TB;
}
#undef WRITING_MODE_INDEX
#undef WRITING_MODE_SHIFT
#undef WRITING_MODE_MASK

#define ORPHANS_INDEX 0
#define ORPHANS_SHIFT 22
#define ORPHANS_MASK 0x400000
static inline uint8_t get_orphans(const css_computed_style *style, int32_t
		*integer)
{
	if (style->page != NULL) {
		uint32_t bits = style->page->bits[ORPHANS_INDEX];
		bits &= ORPHANS_MASK;
		bits >>= ORPHANS_SHIFT;
		
		/* 1bit: t : type */
		*integer = style->page->orphans;
		
		return (bits & 0x1);
	}
	
	/* Initial value */
	*integer = 2;
	return CSS_ORPHANS_SET;
}
#undef ORPHANS_INDEX
#undef ORPHANS_SHIFT
#undef ORPHANS_MASK

#define PAGE_BREAK_AFTER_INDEX 0
#define PAGE_BREAK_AFTER_SHIFT 26
#define PAGE_BREAK_AFTER_MASK 0x1c000000
static inline uint8_t get_page_break_after(const css_computed_style *style)
{
	if (style->page != NULL) {
		uint32_t bits = style->page->bits[PAGE_BREAK_AFTER_INDEX];
		bits &= PAGE_BREAK_AFTER_MASK;
		bits >>= PAGE_BREAK_AFTER_SHIFT;
		
		/* 3bits: ttt : type */
		
		return (bits & 0x7);
	}
	
	/* Initial value */
	return CSS_PAGE_BREAK_AFTER_AUTO;
}
#undef PAGE_BREAK_AFTER_INDEX
#undef PAGE_BREAK_AFTER_SHIFT
#undef PAGE_BREAK_AFTER_MASK

#define PAGE_BREAK_BEFORE_INDEX 0
#define PAGE_BREAK_BEFORE_SHIFT 29
#define PAGE_BREAK_BEFORE_MASK 0xe0000000
static inline uint8_t get_page_break_before(const css_computed_style *style)
{
	if (style->page != NULL) {
		uint32_t bits = style->page->bits[PAGE_BREAK_BEFORE_INDEX];
		bits &= PAGE_BREAK_BEFORE_MASK;
		bits >>= PAGE_BREAK_BEFORE_SHIFT;
		
		/* 3bits: ttt : type */
		
		return (bits & 0x7);
	}
	
	/* Initial value */
	return CSS_PAGE_BREAK_BEFORE_AUTO;
}
#undef PAGE_BREAK_BEFORE_INDEX
#undef PAGE_BREAK_BEFORE_SHIFT
#undef PAGE_BREAK_BEFORE_MASK

#define PAGE_BREAK_INSIDE_INDEX 0
#define PAGE_BREAK_INSIDE_SHIFT 24
#define PAGE_BREAK_INSIDE_MASK 0x3000000
static inline uint8_t get_page_break_inside(const css_computed_style *style)
{
	if (style->page != NULL) {
		uint32_t bits = style->page->bits[PAGE_BREAK_INSIDE_INDEX];
		bits &= PAGE_BREAK_INSIDE_MASK;
		bits >>= PAGE_BREAK_INSIDE_SHIFT;
		
		/* 2bits: tt : type */
		
		return (bits & 0x3);
	}
	
	/* Initial value */
	return CSS_PAGE_BREAK_INSIDE_AUTO;
}
#undef PAGE_BREAK_INSIDE_INDEX
#undef PAGE_BREAK_INSIDE_SHIFT
#undef PAGE_BREAK_INSIDE_MASK

#define WIDOWS_INDEX 0
#define WIDOWS_SHIFT 23
#define WIDOWS_MASK 0x800000
static inline uint8_t get_widows(const css_computed_style *style, int32_t
		*integer)
{
	if (style->page != NULL) {
		uint32_t bits = style->page->bits[WIDOWS_INDEX];
		bits &= WIDOWS_MASK;
		bits >>= WIDOWS_SHIFT;
		
		/* 1bit: t : type */
		*integer = style->page->widows;
		
		return (bits & 0x1);
	}
	
	/* Initial value */
	*integer = 2;
	return CSS_WIDOWS_SET;
}
#undef WIDOWS_INDEX
#undef WIDOWS_SHIFT
#undef WIDOWS_MASK

#define ALIGN_CONTENT_INDEX 8
#define ALIGN_CONTENT_SHIFT 14
#define ALIGN_CONTENT_MASK 0x1c000
static inline uint8_t get_align_content(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[ALIGN_CONTENT_INDEX];
	bits &= ALIGN_CONTENT_MASK;
	bits >>= ALIGN_CONTENT_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef ALIGN_CONTENT_INDEX
#undef ALIGN_CONTENT_SHIFT
#undef ALIGN_CONTENT_MASK

#define ALIGN_ITEMS_INDEX 8
#define ALIGN_ITEMS_SHIFT 23
#define ALIGN_ITEMS_MASK 0x3800000
static inline uint8_t get_align_items(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[ALIGN_ITEMS_INDEX];
	bits &= ALIGN_ITEMS_MASK;
	bits >>= ALIGN_ITEMS_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef ALIGN_ITEMS_INDEX
#undef ALIGN_ITEMS_SHIFT
#undef ALIGN_ITEMS_MASK

#define ALIGN_SELF_INDEX 8
#define ALIGN_SELF_SHIFT 20
#define ALIGN_SELF_MASK 0x700000
static inline uint8_t get_align_self(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[ALIGN_SELF_INDEX];
	bits &= ALIGN_SELF_MASK;
	bits >>= ALIGN_SELF_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef ALIGN_SELF_INDEX
#undef ALIGN_SELF_SHIFT
#undef ALIGN_SELF_MASK

#define BACKGROUND_ATTACHMENT_INDEX 9
#define BACKGROUND_ATTACHMENT_SHIFT 6
#define BACKGROUND_ATTACHMENT_MASK 0xc0
static inline uint8_t get_background_attachment(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BACKGROUND_ATTACHMENT_INDEX];
	bits &= BACKGROUND_ATTACHMENT_MASK;
	bits >>= BACKGROUND_ATTACHMENT_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef BACKGROUND_ATTACHMENT_INDEX
#undef BACKGROUND_ATTACHMENT_SHIFT
#undef BACKGROUND_ATTACHMENT_MASK

#define BACKGROUND_COLOR_INDEX 9
#define BACKGROUND_COLOR_SHIFT 14
#define BACKGROUND_COLOR_MASK 0xc000
static inline uint8_t get_background_color(const css_computed_style *style,
		css_color *color)
{
	uint32_t bits = style->i.bits[BACKGROUND_COLOR_INDEX];
	bits &= BACKGROUND_COLOR_MASK;
	bits >>= BACKGROUND_COLOR_SHIFT;
	
	/* 2bits: tt : type */
	*color = style->i.background_color;
	
	return (bits & 0x3);
}
#undef BACKGROUND_COLOR_INDEX
#undef BACKGROUND_COLOR_SHIFT
#undef BACKGROUND_COLOR_MASK

#define BACKGROUND_IMAGE_INDEX 10
#define BACKGROUND_IMAGE_SHIFT 25
#define BACKGROUND_IMAGE_MASK 0x2000000
static inline uint8_t get_background_image(const css_computed_style *style,
		lwc_string **string)
{
	uint32_t bits = style->i.bits[BACKGROUND_IMAGE_INDEX];
	bits &= BACKGROUND_IMAGE_MASK;
	bits >>= BACKGROUND_IMAGE_SHIFT;
	
	/* 1bit: t : type */
	*string = style->i.background_image;
	
	return (bits & 0x1);
}
#undef BACKGROUND_IMAGE_INDEX
#undef BACKGROUND_IMAGE_SHIFT
#undef BACKGROUND_IMAGE_MASK

#define BACKGROUND_POSITION_INDEX 5
#define BACKGROUND_POSITION_SHIFT 21
#define BACKGROUND_POSITION_MASK 0xffe00000
static inline uint8_t get_background_position(const css_computed_style *style,
		css_fixed *length_a, css_unit *unit_a, css_fixed *length_b,
		css_unit *unit_b)
{
	uint32_t bits = style->i.bits[BACKGROUND_POSITION_INDEX];
	bits &= BACKGROUND_POSITION_MASK;
	bits >>= BACKGROUND_POSITION_SHIFT;
	
	/* 11bits: aaaaabbbbbt : unit_a | unit_b | type */
	if ((bits & 0x1) == CSS_BACKGROUND_POSITION_SET) {
		*length_a = style->i.background_position_a;
		*length_b = style->i.background_position_b;
		*unit_a = bits >> 6;
		*unit_b = (bits & 0x3e) >> 1;
	}
	
	return (bits & 0x1);
}
#undef BACKGROUND_POSITION_INDEX
#undef BACKGROUND_POSITION_SHIFT
#undef BACKGROUND_POSITION_MASK

#define BACKGROUND_REPEAT_INDEX 8
#define BACKGROUND_REPEAT_SHIFT 17
#define BACKGROUND_REPEAT_MASK 0xe0000
static inline uint8_t get_background_repeat(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BACKGROUND_REPEAT_INDEX];
	bits &= BACKGROUND_REPEAT_MASK;
	bits >>= BACKGROUND_REPEAT_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef BACKGROUND_REPEAT_INDEX
#undef BACKGROUND_REPEAT_SHIFT
#undef BACKGROUND_REPEAT_MASK

#define BORDER_BOTTOM_COLOR_INDEX 9
#define BORDER_BOTTOM_COLOR_SHIFT 4
#define BORDER_BOTTOM_COLOR_MASK 0x30
static inline uint8_t get_border_bottom_color(const css_computed_style *style,
		css_color *color)
{
	uint32_t bits = style->i.bits[BORDER_BOTTOM_COLOR_INDEX];
	bits &= BORDER_BOTTOM_COLOR_MASK;
	bits >>= BORDER_BOTTOM_COLOR_SHIFT;
	
	/* 2bits: tt : type */
	*color = style->i.border_bottom_color;
	
	return (bits & 0x3);
}
#undef BORDER_BOTTOM_COLOR_INDEX
#undef BORDER_BOTTOM_COLOR_SHIFT
#undef BORDER_BOTTOM_COLOR_MASK

#define BORDER_BOTTOM_STYLE_INDEX 6
#define BORDER_BOTTOM_STYLE_SHIFT 6
#define BORDER_BOTTOM_STYLE_MASK 0x3c0
static inline uint8_t get_border_bottom_style(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BORDER_BOTTOM_STYLE_INDEX];
	bits &= BORDER_BOTTOM_STYLE_MASK;
	bits >>= BORDER_BOTTOM_STYLE_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef BORDER_BOTTOM_STYLE_INDEX
#undef BORDER_BOTTOM_STYLE_SHIFT
#undef BORDER_BOTTOM_STYLE_MASK

#define BORDER_BOTTOM_WIDTH_INDEX 0
#define BORDER_BOTTOM_WIDTH_SHIFT 16
#define BORDER_BOTTOM_WIDTH_MASK 0xff0000
static inline uint8_t get_border_bottom_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[BORDER_BOTTOM_WIDTH_INDEX];
	bits &= BORDER_BOTTOM_WIDTH_MASK;
	bits >>= BORDER_BOTTOM_WIDTH_SHIFT;
	
	/* 8bits: uuuuuttt : unit | type */
	if ((bits & 0x7) == CSS_BORDER_WIDTH_WIDTH) {
		*length = style->i.border_bottom_width;
		*unit = bits >> 3;
	}
	
	return (bits & 0x7);
}
#undef BORDER_BOTTOM_WIDTH_INDEX
#undef BORDER_BOTTOM_WIDTH_SHIFT
#undef BORDER_BOTTOM_WIDTH_MASK

#define BORDER_COLLAPSE_INDEX 9
#define BORDER_COLLAPSE_SHIFT 26
#define BORDER_COLLAPSE_MASK 0xc000000
static inline uint8_t get_border_collapse(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BORDER_COLLAPSE_INDEX];
	bits &= BORDER_COLLAPSE_MASK;
	bits >>= BORDER_COLLAPSE_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef BORDER_COLLAPSE_INDEX
#undef BORDER_COLLAPSE_SHIFT
#undef BORDER_COLLAPSE_MASK

#define BORDER_LEFT_COLOR_INDEX 9
#define BORDER_LEFT_COLOR_SHIFT 24
#define BORDER_LEFT_COLOR_MASK 0x3000000
static inline uint8_t get_border_left_color(const css_computed_style *style,
		css_color *color)
{
	uint32_t bits = style->i.bits[BORDER_LEFT_COLOR_INDEX];
	bits &= BORDER_LEFT_COLOR_MASK;
	bits >>= BORDER_LEFT_COLOR_SHIFT;
	
	/* 2bits: tt : type */
	*color = style->i.border_left_color;
	
	return (bits & 0x3);
}
#undef BORDER_LEFT_COLOR_INDEX
#undef BORDER_LEFT_COLOR_SHIFT
#undef BORDER_LEFT_COLOR_MASK

#define BORDER_LEFT_STYLE_INDEX 4
#define BORDER_LEFT_STYLE_SHIFT 0
#define BORDER_LEFT_STYLE_MASK 0xf
static inline uint8_t get_border_left_style(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BORDER_LEFT_STYLE_INDEX];
	bits &= BORDER_LEFT_STYLE_MASK;
	bits >>= BORDER_LEFT_STYLE_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef BORDER_LEFT_STYLE_INDEX
#undef BORDER_LEFT_STYLE_SHIFT
#undef BORDER_LEFT_STYLE_MASK

#define BORDER_LEFT_WIDTH_INDEX 0
#define BORDER_LEFT_WIDTH_SHIFT 0
#define BORDER_LEFT_WIDTH_MASK 0xff
static inline uint8_t get_border_left_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[BORDER_LEFT_WIDTH_INDEX];
	bits &= BORDER_LEFT_WIDTH_MASK;
	bits >>= BORDER_LEFT_WIDTH_SHIFT;
	
	/* 8bits: uuuuuttt : unit | type */
	if ((bits & 0x7) == CSS_BORDER_WIDTH_WIDTH) {
		*length = style->i.border_left_width;
		*unit = bits >> 3;
	}
	
	return (bits & 0x7);
}
#undef BORDER_LEFT_WIDTH_INDEX
#undef BORDER_LEFT_WIDTH_SHIFT
#undef BORDER_LEFT_WIDTH_MASK

#define BORDER_RIGHT_COLOR_INDEX 9
#define BORDER_RIGHT_COLOR_SHIFT 30
#define BORDER_RIGHT_COLOR_MASK 0xc0000000
static inline uint8_t get_border_right_color(const css_computed_style *style,
		css_color *color)
{
	uint32_t bits = style->i.bits[BORDER_RIGHT_COLOR_INDEX];
	bits &= BORDER_RIGHT_COLOR_MASK;
	bits >>= BORDER_RIGHT_COLOR_SHIFT;
	
	/* 2bits: tt : type */
	*color = style->i.border_right_color;
	
	return (bits & 0x3);
}
#undef BORDER_RIGHT_COLOR_INDEX
#undef BORDER_RIGHT_COLOR_SHIFT
#undef BORDER_RIGHT_COLOR_MASK

#define BORDER_RIGHT_STYLE_INDEX 6
#define BORDER_RIGHT_STYLE_SHIFT 14
#define BORDER_RIGHT_STYLE_MASK 0x3c000
static inline uint8_t get_border_right_style(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BORDER_RIGHT_STYLE_INDEX];
	bits &= BORDER_RIGHT_STYLE_MASK;
	bits >>= BORDER_RIGHT_STYLE_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef BORDER_RIGHT_STYLE_INDEX
#undef BORDER_RIGHT_STYLE_SHIFT
#undef BORDER_RIGHT_STYLE_MASK

#define BORDER_RIGHT_WIDTH_INDEX 0
#define BORDER_RIGHT_WIDTH_SHIFT 8
#define BORDER_RIGHT_WIDTH_MASK 0xff00
static inline uint8_t get_border_right_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[BORDER_RIGHT_WIDTH_INDEX];
	bits &= BORDER_RIGHT_WIDTH_MASK;
	bits >>= BORDER_RIGHT_WIDTH_SHIFT;
	
	/* 8bits: uuuuuttt : unit | type */
	if ((bits & 0x7) == CSS_BORDER_WIDTH_WIDTH) {
		*length = style->i.border_right_width;
		*unit = bits >> 3;
	}
	
	return (bits & 0x7);
}
#undef BORDER_RIGHT_WIDTH_INDEX
#undef BORDER_RIGHT_WIDTH_SHIFT
#undef BORDER_RIGHT_WIDTH_MASK

#define BORDER_TOP_COLOR_INDEX 8
#define BORDER_TOP_COLOR_SHIFT 0
#define BORDER_TOP_COLOR_MASK 0x3
static inline uint8_t get_border_top_color(const css_computed_style *style,
		css_color *color)
{
	uint32_t bits = style->i.bits[BORDER_TOP_COLOR_INDEX];
	bits &= BORDER_TOP_COLOR_MASK;
	bits >>= BORDER_TOP_COLOR_SHIFT;
	
	/* 2bits: tt : type */
	*color = style->i.border_top_color;
	
	return (bits & 0x3);
}
#undef BORDER_TOP_COLOR_INDEX
#undef BORDER_TOP_COLOR_SHIFT
#undef BORDER_TOP_COLOR_MASK

#define BORDER_TOP_STYLE_INDEX 6
#define BORDER_TOP_STYLE_SHIFT 10
#define BORDER_TOP_STYLE_MASK 0x3c00
static inline uint8_t get_border_top_style(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BORDER_TOP_STYLE_INDEX];
	bits &= BORDER_TOP_STYLE_MASK;
	bits >>= BORDER_TOP_STYLE_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef BORDER_TOP_STYLE_INDEX
#undef BORDER_TOP_STYLE_SHIFT
#undef BORDER_TOP_STYLE_MASK

#define BORDER_TOP_WIDTH_INDEX 0
#define BORDER_TOP_WIDTH_SHIFT 24
#define BORDER_TOP_WIDTH_MASK 0xff000000
static inline uint8_t get_border_top_width(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[BORDER_TOP_WIDTH_INDEX];
	bits &= BORDER_TOP_WIDTH_MASK;
	bits >>= BORDER_TOP_WIDTH_SHIFT;
	
	/* 8bits: uuuuuttt : unit | type */
	if ((bits & 0x7) == CSS_BORDER_WIDTH_WIDTH) {
		*length = style->i.border_top_width;
		*unit = bits >> 3;
	}
	
	return (bits & 0x7);
}
#undef BORDER_TOP_WIDTH_INDEX
#undef BORDER_TOP_WIDTH_SHIFT
#undef BORDER_TOP_WIDTH_MASK

#define BOTTOM_INDEX 4
#define BOTTOM_SHIFT 11
#define BOTTOM_MASK 0x3f800
static inline uint8_t get_bottom(
		const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[BOTTOM_INDEX];
	bits &= BOTTOM_MASK;
	bits >>= BOTTOM_SHIFT;

	/* 7bits: uuuuutt : units | type */
	if ((bits & 0x3) == CSS_BOTTOM_SET) {
		*length = style->i.bottom;
		*unit = bits >> 2;
	}

	return (bits & 0x3);
}
static inline uint8_t get_bottom_bits(
		const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BOTTOM_INDEX];
	bits &= BOTTOM_MASK;
	bits >>= BOTTOM_SHIFT;

	/* 7bits: uuuuutt : units | type */
	return bits;
}
#undef BOTTOM_INDEX
#undef BOTTOM_SHIFT
#undef BOTTOM_MASK

#define BOX_SIZING_INDEX 9
#define BOX_SIZING_SHIFT 12
#define BOX_SIZING_MASK 0x3000
static inline uint8_t get_box_sizing(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[BOX_SIZING_INDEX];
	bits &= BOX_SIZING_MASK;
	bits >>= BOX_SIZING_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef BOX_SIZING_INDEX
#undef BOX_SIZING_SHIFT
#undef BOX_SIZING_MASK

#define CAPTION_SIDE_INDEX 9
#define CAPTION_SIDE_SHIFT 20
#define CAPTION_SIDE_MASK 0x300000
static inline uint8_t get_caption_side(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[CAPTION_SIDE_INDEX];
	bits &= CAPTION_SIDE_MASK;
	bits >>= CAPTION_SIDE_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef CAPTION_SIDE_INDEX
#undef CAPTION_SIDE_SHIFT
#undef CAPTION_SIDE_MASK

#define CLEAR_INDEX 8
#define CLEAR_SHIFT 2
#define CLEAR_MASK 0x1c
static inline uint8_t get_clear(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[CLEAR_INDEX];
	bits &= CLEAR_MASK;
	bits >>= CLEAR_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef CLEAR_INDEX
#undef CLEAR_SHIFT
#undef CLEAR_MASK

#define COLOR_INDEX 10
#define COLOR_SHIFT 24
#define COLOR_MASK 0x1000000
static inline uint8_t get_color(const css_computed_style *style, css_color
		*color)
{
	uint32_t bits = style->i.bits[COLOR_INDEX];
	bits &= COLOR_MASK;
	bits >>= COLOR_SHIFT;
	
	/* 1bit: t : type */
	*color = style->i.color;
	
	return (bits & 0x1);
}
#undef COLOR_INDEX
#undef COLOR_SHIFT
#undef COLOR_MASK

#define DIRECTION_INDEX 9
#define DIRECTION_SHIFT 22
#define DIRECTION_MASK 0xc00000
static inline uint8_t get_direction(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[DIRECTION_INDEX];
	bits &= DIRECTION_MASK;
	bits >>= DIRECTION_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef DIRECTION_INDEX
#undef DIRECTION_SHIFT
#undef DIRECTION_MASK

#define DISPLAY_INDEX 6
#define DISPLAY_SHIFT 27
#define DISPLAY_MASK 0xf8000000
static inline uint8_t get_display(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[DISPLAY_INDEX];
	bits &= DISPLAY_MASK;
	bits >>= DISPLAY_SHIFT;
	
	/* 5bits: ttttt : type */
	
	return (bits & 0x1f);
}
#undef DISPLAY_INDEX
#undef DISPLAY_SHIFT
#undef DISPLAY_MASK

#define EMPTY_CELLS_INDEX 9
#define EMPTY_CELLS_SHIFT 18
#define EMPTY_CELLS_MASK 0xc0000
static inline uint8_t get_empty_cells(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[EMPTY_CELLS_INDEX];
	bits &= EMPTY_CELLS_MASK;
	bits >>= EMPTY_CELLS_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef EMPTY_CELLS_INDEX
#undef EMPTY_CELLS_SHIFT
#undef EMPTY_CELLS_MASK

#define FLEX_BASIS_INDEX 2
#define FLEX_BASIS_SHIFT 11
#define FLEX_BASIS_MASK 0x3f800
static inline uint8_t get_flex_basis(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[FLEX_BASIS_INDEX];
	bits &= FLEX_BASIS_MASK;
	bits >>= FLEX_BASIS_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_FLEX_BASIS_SET) {
		*length = style->i.flex_basis;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef FLEX_BASIS_INDEX
#undef FLEX_BASIS_SHIFT
#undef FLEX_BASIS_MASK

#define FLEX_DIRECTION_INDEX 8
#define FLEX_DIRECTION_SHIFT 11
#define FLEX_DIRECTION_MASK 0x3800
static inline uint8_t get_flex_direction(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[FLEX_DIRECTION_INDEX];
	bits &= FLEX_DIRECTION_MASK;
	bits >>= FLEX_DIRECTION_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef FLEX_DIRECTION_INDEX
#undef FLEX_DIRECTION_SHIFT
#undef FLEX_DIRECTION_MASK

#define FLEX_GROW_INDEX 10
#define FLEX_GROW_SHIFT 21
#define FLEX_GROW_MASK 0x200000
static inline uint8_t get_flex_grow(const css_computed_style *style, css_fixed
		*fixed)
{
	uint32_t bits = style->i.bits[FLEX_GROW_INDEX];
	bits &= FLEX_GROW_MASK;
	bits >>= FLEX_GROW_SHIFT;
	
	/* 1bit: t : type */
	if ((bits & 0x1) == CSS_FLEX_GROW_SET) {
		*fixed = style->i.flex_grow;
	}
	
	return (bits & 0x1);
}
#undef FLEX_GROW_INDEX
#undef FLEX_GROW_SHIFT
#undef FLEX_GROW_MASK

#define FLEX_SHRINK_INDEX 10
#define FLEX_SHRINK_SHIFT 23
#define FLEX_SHRINK_MASK 0x800000
static inline uint8_t get_flex_shrink(const css_computed_style *style,
		css_fixed *fixed)
{
	uint32_t bits = style->i.bits[FLEX_SHRINK_INDEX];
	bits &= FLEX_SHRINK_MASK;
	bits >>= FLEX_SHRINK_SHIFT;
	
	/* 1bit: t : type */
	if ((bits & 0x1) == CSS_FLEX_SHRINK_SET) {
		*fixed = style->i.flex_shrink;
	}
	
	return (bits & 0x1);
}
#undef FLEX_SHRINK_INDEX
#undef FLEX_SHRINK_SHIFT
#undef FLEX_SHRINK_MASK

#define FLEX_WRAP_INDEX 9
#define FLEX_WRAP_SHIFT 28
#define FLEX_WRAP_MASK 0x30000000
static inline uint8_t get_flex_wrap(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[FLEX_WRAP_INDEX];
	bits &= FLEX_WRAP_MASK;
	bits >>= FLEX_WRAP_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef FLEX_WRAP_INDEX
#undef FLEX_WRAP_SHIFT
#undef FLEX_WRAP_MASK

#define FLOAT_INDEX 9
#define FLOAT_SHIFT 8
#define FLOAT_MASK 0x300
static inline uint8_t get_float(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[FLOAT_INDEX];
	bits &= FLOAT_MASK;
	bits >>= FLOAT_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef FLOAT_INDEX
#undef FLOAT_SHIFT
#undef FLOAT_MASK

#define FONT_FAMILY_INDEX 8
#define FONT_FAMILY_SHIFT 26
#define FONT_FAMILY_MASK 0x1c000000
static inline uint8_t get_font_family(const css_computed_style *style,
		lwc_string ***string_arr)
{
	uint32_t bits = style->i.bits[FONT_FAMILY_INDEX];
	bits &= FONT_FAMILY_MASK;
	bits >>= FONT_FAMILY_SHIFT;
	
	/* 3bits: ttt : type */
	*string_arr = style->font_family;
	
	return (bits & 0x7);
}
#undef FONT_FAMILY_INDEX
#undef FONT_FAMILY_SHIFT
#undef FONT_FAMILY_MASK

#define FONT_SIZE_INDEX 5
#define FONT_SIZE_SHIFT 3
#define FONT_SIZE_MASK 0xff8
static inline uint8_t get_font_size(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[FONT_SIZE_INDEX];
	bits &= FONT_SIZE_MASK;
	bits >>= FONT_SIZE_SHIFT;
	
	/* 9bits: uuuuutttt : unit | type */
	if ((bits & 0xf) == CSS_FONT_SIZE_DIMENSION) {
		*length = style->i.font_size;
		*unit = bits >> 4;
	}
	
	return (bits & 0xf);
}
#undef FONT_SIZE_INDEX
#undef FONT_SIZE_SHIFT
#undef FONT_SIZE_MASK

#define FONT_STYLE_INDEX 7
#define FONT_STYLE_SHIFT 0
#define FONT_STYLE_MASK 0x3
static inline uint8_t get_font_style(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[FONT_STYLE_INDEX];
	bits &= FONT_STYLE_MASK;
	bits >>= FONT_STYLE_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef FONT_STYLE_INDEX
#undef FONT_STYLE_SHIFT
#undef FONT_STYLE_MASK

#define FONT_VARIANT_INDEX 9
#define FONT_VARIANT_SHIFT 10
#define FONT_VARIANT_MASK 0xc00
static inline uint8_t get_font_variant(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[FONT_VARIANT_INDEX];
	bits &= FONT_VARIANT_MASK;
	bits >>= FONT_VARIANT_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef FONT_VARIANT_INDEX
#undef FONT_VARIANT_SHIFT
#undef FONT_VARIANT_MASK

#define FONT_WEIGHT_INDEX 3
#define FONT_WEIGHT_SHIFT 0
#define FONT_WEIGHT_MASK 0xf
static inline uint8_t get_font_weight(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[FONT_WEIGHT_INDEX];
	bits &= FONT_WEIGHT_MASK;
	bits >>= FONT_WEIGHT_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef FONT_WEIGHT_INDEX
#undef FONT_WEIGHT_SHIFT
#undef FONT_WEIGHT_MASK

#define HEIGHT_INDEX 4
#define HEIGHT_SHIFT 4
#define HEIGHT_MASK 0x7f0
static inline uint8_t get_height(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[HEIGHT_INDEX];
	bits &= HEIGHT_MASK;
	bits >>= HEIGHT_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_HEIGHT_SET) {
		*length = style->i.height;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef HEIGHT_INDEX
#undef HEIGHT_SHIFT
#undef HEIGHT_MASK

#define JUSTIFY_CONTENT_INDEX 8
#define JUSTIFY_CONTENT_SHIFT 8
#define JUSTIFY_CONTENT_MASK 0x700
static inline uint8_t get_justify_content(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[JUSTIFY_CONTENT_INDEX];
	bits &= JUSTIFY_CONTENT_MASK;
	bits >>= JUSTIFY_CONTENT_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef JUSTIFY_CONTENT_INDEX
#undef JUSTIFY_CONTENT_SHIFT
#undef JUSTIFY_CONTENT_MASK

#define LEFT_INDEX 4
#define LEFT_SHIFT 25
#define LEFT_MASK 0xfe000000
static inline uint8_t get_left(
		const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[LEFT_INDEX];
	bits &= LEFT_MASK;
	bits >>= LEFT_SHIFT;

	/* 7bits: uuuuutt : units | type */
	if ((bits & 0x3) == CSS_LEFT_SET) {
		*length = style->i.left;
		*unit = bits >> 2;
	}

	return (bits & 0x3);
}
static inline uint8_t get_left_bits(
		const css_computed_style *style)
{
	uint32_t bits = style->i.bits[LEFT_INDEX];
	bits &= LEFT_MASK;
	bits >>= LEFT_SHIFT;

	/* 7bits: uuuuutt : units | type */
	return bits;
}
#undef LEFT_INDEX
#undef LEFT_SHIFT
#undef LEFT_MASK

#define LINE_HEIGHT_INDEX 1
#define LINE_HEIGHT_SHIFT 25
#define LINE_HEIGHT_MASK 0xfe000000
static inline uint8_t get_line_height(
		const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[LINE_HEIGHT_INDEX];
	bits &= LINE_HEIGHT_MASK;
	bits >>= LINE_HEIGHT_SHIFT;

	/* 7bits: uuuuutt : units | type */
	if ((bits & 0x3) == CSS_LINE_HEIGHT_NUMBER ||
			(bits & 0x3) == CSS_LINE_HEIGHT_DIMENSION) {
		*length = style->i.line_height;
	}

	if ((bits & 0x3) == CSS_LINE_HEIGHT_DIMENSION) {
		*unit = bits >> 2;
	}

	return (bits & 0x3);
}
#undef LINE_HEIGHT_INDEX
#undef LINE_HEIGHT_SHIFT
#undef LINE_HEIGHT_MASK

#define LIST_STYLE_IMAGE_INDEX 10
#define LIST_STYLE_IMAGE_SHIFT 26
#define LIST_STYLE_IMAGE_MASK 0x4000000
static inline uint8_t get_list_style_image(const css_computed_style *style,
		lwc_string **string)
{
	uint32_t bits = style->i.bits[LIST_STYLE_IMAGE_INDEX];
	bits &= LIST_STYLE_IMAGE_MASK;
	bits >>= LIST_STYLE_IMAGE_SHIFT;
	
	/* 1bit: t : type */
	*string = style->i.list_style_image;
	
	return (bits & 0x1);
}
#undef LIST_STYLE_IMAGE_INDEX
#undef LIST_STYLE_IMAGE_SHIFT
#undef LIST_STYLE_IMAGE_MASK

#define LIST_STYLE_POSITION_INDEX 9
#define LIST_STYLE_POSITION_SHIFT 2
#define LIST_STYLE_POSITION_MASK 0xc
static inline uint8_t get_list_style_position(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[LIST_STYLE_POSITION_INDEX];
	bits &= LIST_STYLE_POSITION_MASK;
	bits >>= LIST_STYLE_POSITION_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef LIST_STYLE_POSITION_INDEX
#undef LIST_STYLE_POSITION_SHIFT
#undef LIST_STYLE_POSITION_MASK

#define LIST_STYLE_TYPE_INDEX 6
#define LIST_STYLE_TYPE_SHIFT 18
#define LIST_STYLE_TYPE_MASK 0x3c0000
static inline uint8_t get_list_style_type(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[LIST_STYLE_TYPE_INDEX];
	bits &= LIST_STYLE_TYPE_MASK;
	bits >>= LIST_STYLE_TYPE_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef LIST_STYLE_TYPE_INDEX
#undef LIST_STYLE_TYPE_SHIFT
#undef LIST_STYLE_TYPE_MASK

#define MARGIN_BOTTOM_INDEX 3
#define MARGIN_BOTTOM_SHIFT 4
#define MARGIN_BOTTOM_MASK 0x7f0
static inline uint8_t get_margin_bottom(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MARGIN_BOTTOM_INDEX];
	bits &= MARGIN_BOTTOM_MASK;
	bits >>= MARGIN_BOTTOM_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MARGIN_SET) {
		*length = style->i.margin_bottom;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MARGIN_BOTTOM_INDEX
#undef MARGIN_BOTTOM_SHIFT
#undef MARGIN_BOTTOM_MASK

#define MARGIN_LEFT_INDEX 1
#define MARGIN_LEFT_SHIFT 4
#define MARGIN_LEFT_MASK 0x7f0
static inline uint8_t get_margin_left(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MARGIN_LEFT_INDEX];
	bits &= MARGIN_LEFT_MASK;
	bits >>= MARGIN_LEFT_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MARGIN_SET) {
		*length = style->i.margin_left;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MARGIN_LEFT_INDEX
#undef MARGIN_LEFT_SHIFT
#undef MARGIN_LEFT_MASK

#define MARGIN_RIGHT_INDEX 2
#define MARGIN_RIGHT_SHIFT 18
#define MARGIN_RIGHT_MASK 0x1fc0000
static inline uint8_t get_margin_right(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MARGIN_RIGHT_INDEX];
	bits &= MARGIN_RIGHT_MASK;
	bits >>= MARGIN_RIGHT_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MARGIN_SET) {
		*length = style->i.margin_right;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MARGIN_RIGHT_INDEX
#undef MARGIN_RIGHT_SHIFT
#undef MARGIN_RIGHT_MASK

#define MARGIN_TOP_INDEX 3
#define MARGIN_TOP_SHIFT 18
#define MARGIN_TOP_MASK 0x1fc0000
static inline uint8_t get_margin_top(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MARGIN_TOP_INDEX];
	bits &= MARGIN_TOP_MASK;
	bits >>= MARGIN_TOP_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MARGIN_SET) {
		*length = style->i.margin_top;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MARGIN_TOP_INDEX
#undef MARGIN_TOP_SHIFT
#undef MARGIN_TOP_MASK

#define MAX_HEIGHT_INDEX 2
#define MAX_HEIGHT_SHIFT 4
#define MAX_HEIGHT_MASK 0x7f0
static inline uint8_t get_max_height(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MAX_HEIGHT_INDEX];
	bits &= MAX_HEIGHT_MASK;
	bits >>= MAX_HEIGHT_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MAX_HEIGHT_SET) {
		*length = style->i.max_height;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MAX_HEIGHT_INDEX
#undef MAX_HEIGHT_SHIFT
#undef MAX_HEIGHT_MASK

#define MAX_WIDTH_INDEX 2
#define MAX_WIDTH_SHIFT 25
#define MAX_WIDTH_MASK 0xfe000000
static inline uint8_t get_max_width(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MAX_WIDTH_INDEX];
	bits &= MAX_WIDTH_MASK;
	bits >>= MAX_WIDTH_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MAX_WIDTH_SET) {
		*length = style->i.max_width;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MAX_WIDTH_INDEX
#undef MAX_WIDTH_SHIFT
#undef MAX_WIDTH_MASK

#define MIN_HEIGHT_INDEX 3
#define MIN_HEIGHT_SHIFT 11
#define MIN_HEIGHT_MASK 0x3f800
static inline uint8_t get_min_height(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MIN_HEIGHT_INDEX];
	bits &= MIN_HEIGHT_MASK;
	bits >>= MIN_HEIGHT_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MIN_HEIGHT_SET) {
		*length = style->i.min_height;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MIN_HEIGHT_INDEX
#undef MIN_HEIGHT_SHIFT
#undef MIN_HEIGHT_MASK

#define MIN_WIDTH_INDEX 1
#define MIN_WIDTH_SHIFT 11
#define MIN_WIDTH_MASK 0x3f800
static inline uint8_t get_min_width(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[MIN_WIDTH_INDEX];
	bits &= MIN_WIDTH_MASK;
	bits >>= MIN_WIDTH_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_MIN_WIDTH_SET) {
		*length = style->i.min_width;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef MIN_WIDTH_INDEX
#undef MIN_WIDTH_SHIFT
#undef MIN_WIDTH_MASK

#define OPACITY_INDEX 10
#define OPACITY_SHIFT 22
#define OPACITY_MASK 0x400000
static inline uint8_t get_opacity(const css_computed_style *style, css_fixed
		*fixed)
{
	uint32_t bits = style->i.bits[OPACITY_INDEX];
	bits &= OPACITY_MASK;
	bits >>= OPACITY_SHIFT;
	
	/* 1bit: t : type */
	if ((bits & 0x1) == CSS_OPACITY_SET) {
		*fixed = style->i.opacity;
	}
	
	return (bits & 0x1);
}
#undef OPACITY_INDEX
#undef OPACITY_SHIFT
#undef OPACITY_MASK

#define ORDER_INDEX 10
#define ORDER_SHIFT 27
#define ORDER_MASK 0x8000000
static inline uint8_t get_order(const css_computed_style *style, int32_t
		*integer)
{
	uint32_t bits = style->i.bits[ORDER_INDEX];
	bits &= ORDER_MASK;
	bits >>= ORDER_SHIFT;
	
	/* 1bit: t : type */
	if ((bits & 0x1) == CSS_ORDER_SET) {
		*integer = style->i.order;
	}
	
	return (bits & 0x1);
}
#undef ORDER_INDEX
#undef ORDER_SHIFT
#undef ORDER_MASK

#define OUTLINE_STYLE_INDEX 1
#define OUTLINE_STYLE_SHIFT 0
#define OUTLINE_STYLE_MASK 0xf
static inline uint8_t get_outline_style(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[OUTLINE_STYLE_INDEX];
	bits &= OUTLINE_STYLE_MASK;
	bits >>= OUTLINE_STYLE_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef OUTLINE_STYLE_INDEX
#undef OUTLINE_STYLE_SHIFT
#undef OUTLINE_STYLE_MASK

#define OVERFLOW_X_INDEX 8
#define OVERFLOW_X_SHIFT 5
#define OVERFLOW_X_MASK 0xe0
static inline uint8_t get_overflow_x(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[OVERFLOW_X_INDEX];
	bits &= OVERFLOW_X_MASK;
	bits >>= OVERFLOW_X_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef OVERFLOW_X_INDEX
#undef OVERFLOW_X_SHIFT
#undef OVERFLOW_X_MASK

#define OVERFLOW_Y_INDEX 8
#define OVERFLOW_Y_SHIFT 29
#define OVERFLOW_Y_MASK 0xe0000000
static inline uint8_t get_overflow_y(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[OVERFLOW_Y_INDEX];
	bits &= OVERFLOW_Y_MASK;
	bits >>= OVERFLOW_Y_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef OVERFLOW_Y_INDEX
#undef OVERFLOW_Y_SHIFT
#undef OVERFLOW_Y_MASK

#define PADDING_BOTTOM_INDEX 7
#define PADDING_BOTTOM_SHIFT 20
#define PADDING_BOTTOM_MASK 0x3f00000
static inline uint8_t get_padding_bottom(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[PADDING_BOTTOM_INDEX];
	bits &= PADDING_BOTTOM_MASK;
	bits >>= PADDING_BOTTOM_SHIFT;
	
	/* 6bits: uuuuut : unit | type */
	if ((bits & 0x1) == CSS_PADDING_SET) {
		*length = style->i.padding_bottom;
		*unit = bits >> 1;
	}
	
	return (bits & 0x1);
}
#undef PADDING_BOTTOM_INDEX
#undef PADDING_BOTTOM_SHIFT
#undef PADDING_BOTTOM_MASK

#define PADDING_LEFT_INDEX 7
#define PADDING_LEFT_SHIFT 26
#define PADDING_LEFT_MASK 0xfc000000
static inline uint8_t get_padding_left(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[PADDING_LEFT_INDEX];
	bits &= PADDING_LEFT_MASK;
	bits >>= PADDING_LEFT_SHIFT;
	
	/* 6bits: uuuuut : unit | type */
	if ((bits & 0x1) == CSS_PADDING_SET) {
		*length = style->i.padding_left;
		*unit = bits >> 1;
	}
	
	return (bits & 0x1);
}
#undef PADDING_LEFT_INDEX
#undef PADDING_LEFT_SHIFT
#undef PADDING_LEFT_MASK

#define PADDING_RIGHT_INDEX 7
#define PADDING_RIGHT_SHIFT 8
#define PADDING_RIGHT_MASK 0x3f00
static inline uint8_t get_padding_right(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[PADDING_RIGHT_INDEX];
	bits &= PADDING_RIGHT_MASK;
	bits >>= PADDING_RIGHT_SHIFT;
	
	/* 6bits: uuuuut : unit | type */
	if ((bits & 0x1) == CSS_PADDING_SET) {
		*length = style->i.padding_right;
		*unit = bits >> 1;
	}
	
	return (bits & 0x1);
}
#undef PADDING_RIGHT_INDEX
#undef PADDING_RIGHT_SHIFT
#undef PADDING_RIGHT_MASK

#define PADDING_TOP_INDEX 7
#define PADDING_TOP_SHIFT 14
#define PADDING_TOP_MASK 0xfc000
static inline uint8_t get_padding_top(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[PADDING_TOP_INDEX];
	bits &= PADDING_TOP_MASK;
	bits >>= PADDING_TOP_SHIFT;
	
	/* 6bits: uuuuut : unit | type */
	if ((bits & 0x1) == CSS_PADDING_SET) {
		*length = style->i.padding_top;
		*unit = bits >> 1;
	}
	
	return (bits & 0x1);
}
#undef PADDING_TOP_INDEX
#undef PADDING_TOP_SHIFT
#undef PADDING_TOP_MASK

#define POSITION_INDEX 5
#define POSITION_SHIFT 0
#define POSITION_MASK 0x7
static inline uint8_t get_position(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[POSITION_INDEX];
	bits &= POSITION_MASK;
	bits >>= POSITION_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef POSITION_INDEX
#undef POSITION_SHIFT
#undef POSITION_MASK

#define QUOTES_INDEX 10
#define QUOTES_SHIFT 20
#define QUOTES_MASK 0x100000
static inline uint8_t get_quotes(const css_computed_style *style, lwc_string
		***string_arr)
{
	uint32_t bits = style->i.bits[QUOTES_INDEX];
	bits &= QUOTES_MASK;
	bits >>= QUOTES_SHIFT;
	
	/* 1bit: t : type */
	*string_arr = style->quotes;
	
	return (bits & 0x1);
}
#undef QUOTES_INDEX
#undef QUOTES_SHIFT
#undef QUOTES_MASK

#define RIGHT_INDEX 1
#define RIGHT_SHIFT 18
#define RIGHT_MASK 0x1fc0000
static inline uint8_t get_right(
		const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[RIGHT_INDEX];
	bits &= RIGHT_MASK;
	bits >>= RIGHT_SHIFT;

	/* 7bits: uuuuutt : units | type */
	if ((bits & 0x3) == CSS_RIGHT_SET) {
		*length = style->i.right;
		*unit = bits >> 2;
	}

	return (bits & 0x3);
}
static inline uint8_t get_right_bits(
		const css_computed_style *style)
{
	uint32_t bits = style->i.bits[RIGHT_INDEX];
	bits &= RIGHT_MASK;
	bits >>= RIGHT_SHIFT;

	/* 7bits: uuuuutt : units | type */
	return bits;
}
#undef RIGHT_INDEX
#undef RIGHT_SHIFT
#undef RIGHT_MASK

#define TABLE_LAYOUT_INDEX 9
#define TABLE_LAYOUT_SHIFT 0
#define TABLE_LAYOUT_MASK 0x3
static inline uint8_t get_table_layout(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[TABLE_LAYOUT_INDEX];
	bits &= TABLE_LAYOUT_MASK;
	bits >>= TABLE_LAYOUT_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef TABLE_LAYOUT_INDEX
#undef TABLE_LAYOUT_SHIFT
#undef TABLE_LAYOUT_MASK

#define TEXT_ALIGN_INDEX 2
#define TEXT_ALIGN_SHIFT 0
#define TEXT_ALIGN_MASK 0xf
static inline uint8_t get_text_align(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[TEXT_ALIGN_INDEX];
	bits &= TEXT_ALIGN_MASK;
	bits >>= TEXT_ALIGN_SHIFT;
	
	/* 4bits: tttt : type */
	
	return (bits & 0xf);
}
#undef TEXT_ALIGN_INDEX
#undef TEXT_ALIGN_SHIFT
#undef TEXT_ALIGN_MASK

#define TEXT_DECORATION_INDEX 6
#define TEXT_DECORATION_SHIFT 22
#define TEXT_DECORATION_MASK 0x7c00000
static inline uint8_t get_text_decoration(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[TEXT_DECORATION_INDEX];
	bits &= TEXT_DECORATION_MASK;
	bits >>= TEXT_DECORATION_SHIFT;
	
	/* 5bits: ttttt : type */
	
	return (bits & 0x1f);
}
#undef TEXT_DECORATION_INDEX
#undef TEXT_DECORATION_SHIFT
#undef TEXT_DECORATION_MASK

#define TEXT_INDENT_INDEX 7
#define TEXT_INDENT_SHIFT 2
#define TEXT_INDENT_MASK 0xfc
static inline uint8_t get_text_indent(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[TEXT_INDENT_INDEX];
	bits &= TEXT_INDENT_MASK;
	bits >>= TEXT_INDENT_SHIFT;
	
	/* 6bits: uuuuut : unit | type */
	if ((bits & 0x1) == CSS_TEXT_INDENT_SET) {
		*length = style->i.text_indent;
		*unit = bits >> 1;
	}
	
	return (bits & 0x1);
}
#undef TEXT_INDENT_INDEX
#undef TEXT_INDENT_SHIFT
#undef TEXT_INDENT_MASK

#define TEXT_TRANSFORM_INDEX 6
#define TEXT_TRANSFORM_SHIFT 3
#define TEXT_TRANSFORM_MASK 0x38
static inline uint8_t get_text_transform(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[TEXT_TRANSFORM_INDEX];
	bits &= TEXT_TRANSFORM_MASK;
	bits >>= TEXT_TRANSFORM_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef TEXT_TRANSFORM_INDEX
#undef TEXT_TRANSFORM_SHIFT
#undef TEXT_TRANSFORM_MASK

#define TOP_INDEX 3
#define TOP_SHIFT 25
#define TOP_MASK 0xfe000000
static inline uint8_t get_top(
		const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[TOP_INDEX];
	bits &= TOP_MASK;
	bits >>= TOP_SHIFT;

	/* 7bits: uuuuutt : units | type */
	if ((bits & 0x3) == CSS_TOP_SET) {
		*length = style->i.top;
		*unit = bits >> 2;
	}

	return (bits & 0x3);
}
static inline uint8_t get_top_bits(
		const css_computed_style *style)
{
	uint32_t bits = style->i.bits[TOP_INDEX];
	bits &= TOP_MASK;
	bits >>= TOP_SHIFT;

	/* 7bits: uuuuutt : units | type */
	return bits;
}
#undef TOP_INDEX
#undef TOP_SHIFT
#undef TOP_MASK

#define UNICODE_BIDI_INDEX 9
#define UNICODE_BIDI_SHIFT 16
#define UNICODE_BIDI_MASK 0x30000
static inline uint8_t get_unicode_bidi(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[UNICODE_BIDI_INDEX];
	bits &= UNICODE_BIDI_MASK;
	bits >>= UNICODE_BIDI_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef UNICODE_BIDI_INDEX
#undef UNICODE_BIDI_SHIFT
#undef UNICODE_BIDI_MASK

#define VERTICAL_ALIGN_INDEX 5
#define VERTICAL_ALIGN_SHIFT 12
#define VERTICAL_ALIGN_MASK 0x1ff000
static inline uint8_t get_vertical_align(const css_computed_style *style,
		css_fixed *length, css_unit *unit)
{
	uint32_t bits = style->i.bits[VERTICAL_ALIGN_INDEX];
	bits &= VERTICAL_ALIGN_MASK;
	bits >>= VERTICAL_ALIGN_SHIFT;
	
	/* 9bits: uuuuutttt : unit | type */
	if ((bits & 0xf) == CSS_VERTICAL_ALIGN_SET) {
		*length = style->i.vertical_align;
		*unit = bits >> 4;
	}
	
	return (bits & 0xf);
}
#undef VERTICAL_ALIGN_INDEX
#undef VERTICAL_ALIGN_SHIFT
#undef VERTICAL_ALIGN_MASK

#define VISIBILITY_INDEX 10
#define VISIBILITY_SHIFT 28
#define VISIBILITY_MASK 0x30000000
static inline uint8_t get_visibility(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[VISIBILITY_INDEX];
	bits &= VISIBILITY_MASK;
	bits >>= VISIBILITY_SHIFT;
	
	/* 2bits: tt : type */
	
	return (bits & 0x3);
}
#undef VISIBILITY_INDEX
#undef VISIBILITY_SHIFT
#undef VISIBILITY_MASK

#define WHITE_SPACE_INDEX 6
#define WHITE_SPACE_SHIFT 0
#define WHITE_SPACE_MASK 0x7
static inline uint8_t get_white_space(const css_computed_style *style)
{
	uint32_t bits = style->i.bits[WHITE_SPACE_INDEX];
	bits &= WHITE_SPACE_MASK;
	bits >>= WHITE_SPACE_SHIFT;
	
	/* 3bits: ttt : type */
	
	return (bits & 0x7);
}
#undef WHITE_SPACE_INDEX
#undef WHITE_SPACE_SHIFT
#undef WHITE_SPACE_MASK

#define WIDTH_INDEX 4
#define WIDTH_SHIFT 18
#define WIDTH_MASK 0x1fc0000
static inline uint8_t get_width(const css_computed_style *style, css_fixed
		*length, css_unit *unit)
{
	uint32_t bits = style->i.bits[WIDTH_INDEX];
	bits &= WIDTH_MASK;
	bits >>= WIDTH_SHIFT;
	
	/* 7bits: uuuuutt : unit | type */
	if ((bits & 0x3) == CSS_WIDTH_SET) {
		*length = style->i.width;
		*unit = bits >> 2;
	}
	
	return (bits & 0x3);
}
#undef WIDTH_INDEX
#undef WIDTH_SHIFT
#undef WIDTH_MASK

#define Z_INDEX_INDEX 10
#define Z_INDEX_SHIFT 30
#define Z_INDEX_MASK 0xc0000000
static inline uint8_t get_z_index(const css_computed_style *style, int32_t
		*integer)
{
	uint32_t bits = style->i.bits[Z_INDEX_INDEX];
	bits &= Z_INDEX_MASK;
	bits >>= Z_INDEX_SHIFT;
	
	/* 2bits: tt : type */
	*integer = style->i.z_index;
	
	return (bits & 0x3);
}
#undef Z_INDEX_INDEX
#undef Z_INDEX_SHIFT
#undef Z_INDEX_MASK
