#!/bin/sh
# Cloud Disk, minimalist gui for curlftpfs, davfs2, and sshfs
# Copyright (C) James Budiono 2012, 2019
# License: GNU GPL Version 3 or later
#
# Version 1 - May 2012 - for davfs2 and curlftpfs
# Version 2 - May 2012 - add sshfs support
# Version 3 - Oct 2019 - add ssh pubkey file support
#
# $0-command name, $1-cmd
# commands are:
# - (blank) - interactive (create new mountpoint) or mount if $0 is *AppRun
# - detach - unmount storage
# - edit - modify existing config
# - mount - mount storage (internal only)
# - umount - umount storage (internal only)

### configuration
APPTITLE="Cloud Disk"
MOUNT_ROOT="$HOME/CloudDisk"	# where all the rox apps will be created
ICON_IMAGE=/usr/share/pixmaps/midi-icons/drive48.png
INTERNAL_MOUNTPOINT=mnt			# where the filesystem will be mounted
CONFIG_FILE=config				# where the filesystem details (url, etc) are kept

# defaults, overriden by config file below. Settings to be applied for non-root users
ACCESS_MODE=770			# rwx for non-root. Change to 750 for r-x
UMASK_MODE=007			# rwx for non-root. Change to 027 for r-x
GID=					# filled in by get_USER_GID below

EVENTMANAGER_CONFIG=/etc/eventmanager
. $EVENTMANAGER_CONFIG	# ACCESS_MODE, UMASK_MODE

################ helpers #########################
# $1-mountpoint
is_mounted() {
	grep -qm 1 "$1" /proc/mounts
}

# splash message
splash() {
	Xdialog --title "$APPTITLE" --backtitle "$2" --no-buttons --infobox "$1" 0 0 1000000 &
	XPID=$!
}
stop_splash() {
	[ $XPID ] && kill $XPID
	XPID=
}

# get group id from $user
get_USER_GID() {
	GID=$(awk -F: "/^$USER/ {print \$4}" /etc/passwd)
}
get_USER_GID

# recreate .DirIcon
# $1-directory, $2 state
refresh_icon() {
	case $2 in
		unmounted)
		cat > "$1/.DirIcon" << EOF
<svg width="48" height="48" id="svg1" xmlns:xlink="http://www.w3.org/1999/xlink">
<image xlink:href="$ICON_IMAGE" width="48" height="48"/>
</svg>
EOF
;;
		mounted)
		cat > "$1/.DirIcon" << EOF
<svg width="48" height="48" id="svg1" xmlns:xlink="http://www.w3.org/1999/xlink">
<defs id="defs1">
<linearGradient id="LG1">
   <stop style="stop-color:#00ff00;stop-opacity:1;" id="stop1"/>
   <stop style="stop-color:#333333;stop-opacity:1;" offset="1" id="stop2"/>
</linearGradient>
<radialGradient xlink:href="#LG1" id="RG1" cx="0.5" cy="0.5" r="0.5" fx="0.5" fy="0.5"/>
</defs>
<image xlink:href="$ICON_IMAGE" width="48" height="48" id="image1"/>
<circle cx="40" cy="8" r="8" style="fill:url(#RG1);" id="ellipse1"/>
</svg>
EOF
;;		
	esac
}

############## action handlers ###################

### interactive - open a dialog to create the rox apps - as user
# this will be run under free context (not as Rox App)
# $1 == edit - means editing, otherwise it's new
interactive() {
	while true; do	
		# init gtkdialog parameters
		[ "$URL" ] && URL="<default>$URL</default>"
		[ "$MOUNTPOINT" ] && MOUNTPOINT="<default>$MOUNTPOINT</default>"
		[ "$USERID" ] && USERID="<default>$USERID</default>"
		[ "$PASSWORD" ] && PASSWORD="<default>$PASSWORD</default>"
		[ "$CONFIRM_PASSWORD" ] && CONFIRM_PASSWORD="<default>$CONFIRM_PASSWORD</default>"
		[ "$NOCRED" = true ] && NOCRED=" active=\"true\"" || NOCRED=""
		[ "$KEY_FILE" ] && KEY_FILE="<default>$KEY_FILE</default>"
		
		[ "$TYPE_WEBDAV" = true ] && TYPE_WEBDAV=" active=\"true\"" || TYPE_WEBDAV=""
		[ "$TYPE_FTP" = true ] && TYPE_FTP=" active=\"true\"" || TYPE_FTP=""
		[ "$TYPE_SSH" = true ] && TYPE_SSH=" active=\"true\"" || TYPE_SSH=""
		
		CREATE_LABEL="Create" MOUNT_EDITABLE="true"
		[ "$1" = edit ] && CREATE_LABEL="Update" && MOUNT_EDITABLE="false"
		
		# get type, URL, mountpoint name, userid/password
		export GUI=$(cat << EOF
<window title="$APPTITLE" window_position="1"><vbox>
	<frame Storage Type>
		<radiobutton$TYPE_WEBDAV>
			<label>Webdav</label>
			<variable>TYPE_WEBDAV</variable>
		</radiobutton>
		<radiobutton$TYPE_FTP>
			<variable>TYPE_FTP</variable>
			<label>FTP</label>			
		</radiobutton>
		<radiobutton$TYPE_SSH>
			<variable>TYPE_SSH</variable>
			<label>SSH</label>			
		</radiobutton>		
	</frame>
	<frame Access Details>
		<hbox>
			<text>
				<label>$APPTITLE URL</label>
			</text>
			<entry tooltip-text="Enter the URL of the cloud storage.

For Webdav:
https://webdav.mydrive.ch
https://webdav.4shared.com
https://www.box.com/dav

For FTP:
ftp://localhost

For SSH:
ssh://localhost
ssh://localhost:folder">
				<variable>URL</variable>$URL
			</entry>
		</hbox>
		<hbox>
			<text>
				<label>Mountpoint Name</label>
			</text>
			<entry editable="$MOUNT_EDITABLE" tooltip-text="Do not use spaces, < or >">
				<variable>MOUNTPOINT</variable>$MOUNTPOINT
			</entry>
		</hbox>		
	</frame>
	<frame Credentials>
		<hbox>
			<text>
				<label>User id</label>
			</text>
			<entry tooltip-text="Do not use < or >">
				<variable>USERID</variable>$USERID
			</entry>
		</hbox>
		<hbox>
			<text>
				<label>Password 1</label>
			</text>
			<entry invisible_char="120" visibility="false" tooltip-text="Password 1 and Password 2 must match. Do not use < or >. They cannot be blank - if you don't need password, fill it in with random values.">
				<variable>PASSWORD</variable>$PASSWORD
			</entry>
		</hbox>
		<hbox>
			<text>
				<label>Password 2</label>
			</text>			
			<entry invisible_char="120" visibility="false" tooltip-text="Password 1 and Password 2 must match. Do not use < or >">
				<variable>CONFIRM_PASSWORD</variable>$CONFIRM_PASSWORD
			</entry>			
		</hbox>
		<hbox>
			<text>
				<label>Keyfile</label>
			</text>			
			<entry tooltip-text="Keyfile for connection (SSH only)">
				<variable>KEY_FILE</variable>$KEY_FILE
			</entry>			
		</hbox>		
		<checkbox$NOCRED>
			<label>Ask credentials later during mount</label>
			<variable>NOCRED</variable>
			<action>if true disable:USERID</action>
			<action>if true disable:PASSWORD</action>
			<action>if true disable:CONFIRM_PASSWORD</action>
			<action>if false enable:USERID</action>
			<action>if false enable:PASSWORD</action>
			<action>if false enable:CONFIRM_PASSWORD</action>
		</checkbox>
	</frame>	
	<hbox>
		<button><label>$CREATE_LABEL</label><input file stock="gtk-network"></input></button>
		<button><label>Quit</label><input file stock="gtk-quit"></input></button>
	</hbox>
</vbox>	
</window>
EOF
)
		# get input
		eval $(gtkdialog -p GUI)
		
		# see what we're supposed to do
		case $EXIT in
			abort|Quit) break ;;
		esac
		
		# trim leading & trailing spaces
		URL=$(echo $URL | sed 's/^ *//; s/ *$//' | tr '<>' '+')
		MOUNTPOINT=$(echo $MOUNTPOINT | sed 's/^ *//; s/ *$//' | tr '<>\t *?/\\' '+')
		USERID=$(echo $USERID | sed 's/^ *//; s/ *$//' | tr '<>' '+')
		PASSWORD=$(echo $PASSWORD | sed 's/^ *//; s/ *$//' | tr '<>' '+')
		CONFIRM_PASSWORD=$(echo $CONFIRM_PASSWORD | sed 's/^ *//; s/ *$//' | tr '<>' '+')
		KEY_FILE=$(echo "$KEY_FILE" | sed 's/^ *//; s/ *$//' | tr '<>' '+')
		
		# validate
		case $URL in "") continue ;; esac
		case $MOUNTPOINT in "") continue ;; esac
		case $NOCRED in
			false) case "$USERID" in "") continue ;; esac
				   case "$PASSWORD" in "") continue ;; esac
				   if [ "$PASSWORD" != "$CONFIRM_PASSWORD" ]; then
						Xdialog --title "$APPTITLE" --infobox "Error! Passwords don't match" 0 0 10000
						PASSWORD="" CONFIRM_PASSWORD=""
						continue
				   fi ;;
			true) USERID="" PASSWORD="" CONFIRM_PASSWORD="" ;;
		esac
		case $TYPE_FTP in "true") FS_TYPE="ftp" ;; esac
		case $TYPE_WEBDAV in "true") FS_TYPE="webdav" ;; esac
		case $TYPE_SSH in "true") FS_TYPE="ssh" ;; esac
		
		# create the rox app now - folder
		APP_DIR="$MOUNT_ROOT/$MOUNTPOINT"
		mkdir -p "$APP_DIR" 2> /dev/null
		
		# store the config
		{
			echo URL=\'$URL\'
			echo USERID=\'$USERID\'
			echo PASSWORD=\'$PASSWORD\'
			echo FS_TYPE=\'$FS_TYPE\'
			echo KEY_FILE=\'$KEY_FILE\'
		} > "$APP_DIR/$CONFIG_FILE"
		
		# symlink ourself to AppRun - unless we're already called as AppRun
		case "$0" in *AppRun) ;; 
		*) ln -sfT $(readlink -f "$0") "$APP_DIR/AppRun" ;;
		esac
		
		# create AppInfo.xml
		cat > "$APP_DIR/AppInfo.xml" <<EOF
<?xml version="1.0"?>
<AppInfo>
  <Summary>$APPTITLE for $URL</Summary>
  <About>
    <Purpose>Application to access FTP and Webdav storage in the cloud</Purpose>
    <Version>1.0</Version>
    <Authors>James Budiono</Authors>
    <License>GNU General Public License Version 3 or Later</License>
  </About>
  <AppMenu>
    <Item option="detach" icon="gtk-disconnect">
      <Label>Detach from cloud</Label>
    </Item>
    <Item option="edit" icon="gtk-edit">
      <Label>Modify settings</Label>
    </Item>    
  </AppMenu>
</AppInfo>		
EOF
		# create DirIcon
		refresh_icon "$APP_DIR" unmounted
		
		# done, open the root folder
		[ "$1" = edit ] && break  # no need to do the following if we're just editing
		
		rox -D "$MOUNT_ROOT"
		rox -x "$MOUNT_ROOT" -d "$MOUNT_ROOT"
		Xdialog --title "$APPTITLE" --infobox "Done! An entry has been created in $MOUNT_ROOT.
Just click this entry to connect to $URL, and right-click to detach it.
You can now create another entry, or click Quit to exit." 0 0 10000
		URL="" MOUNTPOINT="" USERID="" PASSWORD="" CONFIRM_PASSWORD=""
	done
}


### run under RoxApp context - as user
# $1-app_dir location
show_folder() {
	# load config
	. "$1/$CONFIG_FILE"

	# check if already mounted before mounting
	is_mounted "$1" && rox -x "$1/$INTERNAL_MOUNTPOINT" -d "$1/$INTERNAL_MOUNTPOINT" && return
	
	# mount it
	if [ $(id -u) -ne 0 ]; then gtksu "Attaching $URL to local folder" "$0" mount
	else storage_mount "$1"
	fi
	
	# check again if it is now mounted, if not, error
	if is_mounted "$1"; then
		refresh_icon "$1" mounted
		rox -x "$MOUNT_ROOT"
		rox -x "$1/$INTERNAL_MOUNTPOINT" -d "$1/$INTERNAL_MOUNTPOINT" && return
	else
		Xdialog --title "$APPTITLE" --infobox "Error! Unable to mount $URL." 0 0 10000
	fi
}

### run under RoxApp context - as user
# $1-app_dir location
detach_folder() {
	# load config
	. "$1/$CONFIG_FILE"

	# check if already mounted before unmounting
	! is_mounted "$1" && return
	rox -D "$1/$INTERNAL_MOUNTPOINT"
	
	# un-mount it
	if [ $(id -u) -ne 0 ]; then gtksu "Detaching $URL" "$0" umount
	else storage_umount "$1"
	fi

	# check again if it is now mounted, if yes, error
	if is_mounted "$1"; then
		Xdialog --title "$APPTITLE" --infobox "Error! Unable to detach $URL." 0 0 10000
	else
		refresh_icon "$1" unmounted
		rox -x "$MOUNT_ROOT"
	fi
	
}


### internal mount - run under RoxApp context - as user or root
# $1-app_dir location
storage_mount() {
	# load config & get credentials if needed
	. "$1/$CONFIG_FILE"
	
	# check if already mounted before mounting
	is_mounted "$1" && return

	if [ -z "$USERID" ]; then
		if [ "$KEY_FILE" ] && [ "$SSH_AUTH_SOCK" ] || [ -z "$KEY_FILE" ]; then
			if cred=$(Xdialog --title "$APPTITLE" --stdout --separator "|" --password=2 --2inputsbox "Please provide credentials for $URL" 0 0 "User ID" "" "Password" ""); then
				USERID=${cred%%|*}
				PASSWORD=${cred#*|}
			else return
			fi
		fi
	fi
	
	# create internal mountpoint & make sure it belongs to the user
	mkdir -p "$1/$INTERNAL_MOUNTPOINT"
	chown $USER:$USER "$1/$INTERNAL_MOUNTPOINT"

	# mount it, use the correct executable depending on the type
	[ $(id -un) = $USER ] && ALLOW_OTHER= # clear allow other if requester is root	
	splash "$URL" "Mounting, please wait ..." 
	case $FS_TYPE in
		webdav)
			echo -e "$USERID\n$PASSWORD\ny" | 
			mount.davfs "$URL" "$1/$INTERNAL_MOUNTPOINT" -o uid=0,gid=$GID,file_mode=$ACCESS_MODE,dir_mode=$ACCESS_MODE
			;;
		ftp) 
		    curlftpfs "$URL" "$1/$INTERNAL_MOUNTPOINT" -o user="$USERID:$PASSWORD",fsname="$URL",uid=0,gid=$GID,umask=$UMASK_MODE,allow_other,default_permissions,transform_symlinks
		    ;;			
		ssh)
			tmpURL=${URL#ssh://} # remove the ssh:// prefix, if any
			case $tmpURL in *:*) ;; *) tmpURL=$tmpURL: ;; esac # add colon, if there isn't any
			tmpURL=$USERID@$tmpURL
			if [ "$KEY_FILE" ]; then
				if [ -z "$SSH_AUTH_SOCK" ]; then
					stop_splash
					ssh-agent "$1/AppRun" mount "$1/AppRun"
				else
					SSH_ASKPASS=/usr/bin/ssh-askpass ssh-add "$KEY_FILE"
					echo "$PASSWORD" | sshfs $tmpURL "$1/$INTERNAL_MOUNTPOINT" -o fsname="$URL",uid=0,gid=$GID,umask=$UMASK_MODE,allow_other,password_stdin,follow_symlinks,transform_symlinks,default_permissions
				fi
			else
				echo "$PASSWORD" | sshfs $tmpURL "$1/$INTERNAL_MOUNTPOINT" -o fsname="$URL",uid=0,gid=$GID,umask=$UMASK_MODE,allow_other,password_stdin,follow_symlinks,transform_symlinks,default_permissions
			fi
			;;
	esac
	stop_splash
}



### internal umount - run under RoxApp context - as user or root
# $1-app_dir location
storage_umount() {
	# load config & get credentials if needed
	. "$1/$CONFIG_FILE"
	
	# un-mount it, use the correct executable depending on the type
	splash "$URL" "Detaching, please wait ..."
	case $FS_TYPE in
		webdav) umount.davfs "$1/$INTERNAL_MOUNTPOINT" ;;
		ftp|ssh) umount "$1/$INTERNAL_MOUNTPOINT" ;;
	esac
	stop_splash
}



########## main ###########
case $1 in
	# public
	"") case "$0" in
			*AppRun) show_folder "${0%/AppRun}" ;;
			*) interactive ;;
		esac ;;
	
	detach) detach_folder "${0%/AppRun}" ;;
	
	edit) . "${0%/AppRun}/$CONFIG_FILE"
		  case $FS_TYPE in
			ftp) TYPE_FTP=true ;;
			webdav) TYPE_WEBDAV=true ;;
			ssh) TYPE_SSH=true ;;
		  esac
		  [ -z "$USERID" ] && NOCRED=true
		  MOUNTPOINT="${0%/AppRun}"; MOUNTPOINT="${MOUNTPOINT##*/}"
		  CONFIRM_PASSWORD="$PASSWORD"
		  interactive edit ;;
	
	# internal
	mount) storage_mount "${0%/AppRun}" ;;	
	umount) storage_umount "${0%/AppRun}" ;;
esac

