/*
 * Direct Digital Synthesis
 * (C) James Budiono 2020, 2021
 * Licence: GPLv3
 *
 * Version 11 - Nov 2021
 * 
 */ 
#include <inttypes.h>
#include <math.h>
#include <stdlib.h>
#include <SDL.h>
#include <SDL_ttf.h>

#define CLEAR_EOL "\e[K\n"
#define SAVE_CURSOR_POS "\e7"
#define RESTORE_CURSOR_POS "\e8"
#define CLEAR_SCREEN "\e[2J\e[H"

struct osc;

static void event_loop();
static void update_status();
static void fill_audio(void *udata, Uint8 *stream, int len);
static void gen_audio(struct osc *channel, int16_t *buf, int size);
static void filter(int16_t *buf, int size, char filter_type,
			float alpha, int16_t *prev_left, int16_t *prev_right);

static int getnote(SDLKey key);
static int change_octave(SDLKey key);
static int change_volume(SDLKey key);
static int change_waveform(SDLKey key);
static int change_output_filter(SDLKey key);
static int panic(SDLKey key);
static int change_sustain(SDLKey key);
static int change_chan_filter(SDLKey key);

static struct osc *allocate_channel(int note);
static struct osc *free_channel(int note);
static int count_active_channels();
static void all_stop();

static void start_channel(struct osc *channel, int note);
static void stop_channel(struct osc *channel, int note);

static Uint16 round_down_to_power_of_two(Uint16 num);

static SDL_AudioSpec hwspec;

int sample_rate = 32000; // default sampling rate in Hz
int latency = 50;        // default latency in ms

float A440; // Tuning 440Hz, calculate based on actual sampling rate
#define SEMITONE 1.059463 // this is twelve-root of 2.

#define MAX_VOLUME 256 // we use 8-bit volume
#define MAX_WAVEFORM 3 // triangle, square, sawtooth, sine

struct osc  {
	int waveform; // 0=triangle, 1=square, 2=sawtooth, 3=sine
	int filter_type;   // 0=none, 1=fir, 2=iir
	float filter_alpha;
	
	int volume;   // 0=silent, max=256	
	int note;     // note currently played
	int octave;   // octave currently played
	float freq;   // freq currently played (Hz)

	// dds osc				
	uint32_t accum; // 16.16 fixed point
	uint32_t phase;

	// filter state variable
    int16_t prev_left, prev_right; 
    int32_t release, release_constant;
};

static int16_t *chan_buf;
static int32_t *mix_buf;

#define DEF_WAVEFORM 0
#define DEF_FILTER_TYPE 0
#define DEF_FILTER_ALPHA 0.5

#define NUM_CHANNELS 8
static struct osc channels[NUM_CHANNELS];
#ifndef NUM_CHANNELS
#define NUM_CHANNELS (sizeof(channels)/sizeof(struct osc))
#endif
static int channel_overflow;

int chan_waveform = DEF_WAVEFORM;
int chan_filter_type = DEF_FILTER_TYPE;
float chan_filter_alpha = DEF_FILTER_ALPHA;

#define A440_OCTAVE 4     // so that "n" is A440 by default
int global_octave = 3;    // "q" is middle C, and "n" is A440
float global_sustain = 0.5;
int global_dither = 0; // disable dither by default

float global_volume = 10.0; // because we enable AGC by default
int use_agc = !0;
int clipping;

int global_filter_type = 2; // 0=none, 1=fir, 2=iir
float global_filter_alpha = 0.5;
static int16_t global_prev_left, global_prev_right;

char *waveform_str[] = {
	"Triangle", "Square", "Sawtooth", "Sine",
};
char *filter_str[] = {
	"None", "FIR", "IIR",
};

#define RENDER_FONT "/usr/share/fonts/X11/TTF/DejaVuSans.ttf"
#define RENDER_FONT_SIZE 16
TTF_Font *font;
SDL_Surface *screen;

//#define USE_EMBEDDED_SINE_TABLE
#ifdef USE_EMBEDDED_SINE_TABLE
// 256-point signed 16-bit sinewave - see make-sine-table
#define SINE_DATA_LEN 256
int16_t sinewave[]={
     0,   804,  1608,  2410,  3212,  4011,  4808,  5602,  6393,  7179,  7962,  8739,  9512, 10278, 11039, 11793,
 12539, 13279, 14010, 14732, 15446, 16151, 16846, 17530, 18204, 18868, 19519, 20159, 20787, 21403, 22005, 22594,
 23170, 23731, 24279, 24811, 25329, 25832, 26319, 26790, 27245, 27683, 28105, 28510, 28898, 29268, 29621, 29956,
 30273, 30571, 30852, 31113, 31356, 31580, 31785, 31971, 32137, 32285, 32412, 32521, 32609, 32678, 32728, 32757,
 32767, 32757, 32728, 32678, 32609, 32521, 32412, 32285, 32137, 31971, 31785, 31580, 31356, 31113, 30852, 30571,
 30273, 29956, 29621, 29268, 28898, 28510, 28105, 27683, 27245, 26790, 26319, 25832, 25329, 24811, 24279, 23731,
 23170, 22594, 22005, 21403, 20787, 20159, 19519, 18868, 18204, 17530, 16846, 16151, 15446, 14732, 14010, 13279,
 12539, 11793, 11039, 10278,  9512,  8739,  7962,  7179,  6393,  5602,  4808,  4011,  3212,  2410,  1608,   804,
     0,  -804, -1608, -2410, -3212, -4011, -4808, -5602, -6393, -7179, -7962, -8739, -9512,-10278,-11039,-11793,
-12539,-13279,-14010,-14732,-15446,-16151,-16846,-17530,-18204,-18868,-19519,-20159,-20787,-21403,-22005,-22594,
-23170,-23731,-24279,-24811,-25329,-25832,-26319,-26790,-27245,-27683,-28105,-28510,-28898,-29268,-29621,-29956,
-30273,-30571,-30852,-31113,-31356,-31580,-31785,-31971,-32137,-32285,-32412,-32521,-32609,-32678,-32728,-32757,
-32767,-32757,-32728,-32678,-32609,-32521,-32412,-32285,-32137,-31971,-31785,-31580,-31356,-31113,-30852,-30571,
-30273,-29956,-29621,-29268,-28898,-28510,-28105,-27683,-27245,-26790,-26319,-25832,-25329,-24811,-24279,-23731,
-23170,-22594,-22005,-21403,-20787,-20159,-19519,-18868,-18204,-17530,-16846,-16151,-15446,-14732,-14010,-13279,
-12539,-11793,-11039,-10278, -9512, -8739, -7962, -7179, -6393, -5602, -4808, -4011, -3212, -2410, -1608,  -804,
};
#else
#define SINE_DATA_LEN 1024
int16_t *sinewave;
#endif

int main (int argc, char *argv[]) {
	/* process params */
	if (argc>1) {
		if (argv[1][0] == '-') { // assume help
			fprintf(stdout, "%s [sample rate Hz] [latency ms]\n", argv[0]);
			exit(0);
		} else {
			sample_rate = atoi(argv[1]);			
		}
	}
	if (argc>2) {
		latency = atoi(argv[2]);
	}
	
	/* Initialise SDL */
	if( SDL_Init( SDL_INIT_AUDIO | SDL_INIT_VIDEO | SDL_INIT_TIMER ) < 0){
		fprintf( stderr, "Could not initialise SDL: %s\n", SDL_GetError() );
		exit( -1 );
	}

#ifndef USE_EMBEDDED_SINE_TABLE
	/* Initialise 16-bit N-point sinewave table */
	sinewave = malloc(SINE_DATA_LEN*sizeof(int16_t));
	for (int i=0; i<SINE_DATA_LEN; i++) {		
		sinewave[i] = 32767*sin((M_PI*2*i)/SINE_DATA_LEN);
	}
#endif

	/* Set a video mode */
	screen = SDL_SetVideoMode( 640, 400, 0, 0 );
	if( !screen ){
		fprintf( stderr, "Could not set video mode: %s\n", SDL_GetError() );
		SDL_Quit();
		exit( -1 );
	}
	SDL_WM_SetCaption("DDS", "DDS");

	if (TTF_Init() != 0){
		fprintf( stderr, "Could not init SDL_ttf: %s\n", SDL_GetError() );
		SDL_Quit();
		exit( -1 );
	}

	font = TTF_OpenFont(RENDER_FONT,RENDER_FONT_SIZE);
	if (!font) {
		fprintf( stderr, "Could not load font: %s\n", SDL_GetError() );
		SDL_Quit();
		exit( -1 );
	}

    /* Set the audio format */
    SDL_AudioSpec wanted;
    wanted.freq = sample_rate;
    wanted.format = AUDIO_S16; // S16_LE
    wanted.channels = 2;   /* 1 = mono, 2 = stereo */
    //wanted.samples = 1024; /* Put 1024 for low-latency */
    wanted.samples = (latency*sample_rate)/1000; /* count samples based on latency */
    wanted.samples = round_down_to_power_of_two(wanted.samples); // round down to nearest power of two
    wanted.callback = fill_audio;
    wanted.userdata = NULL;

    /* Open the audio device, forcing the desired format */
    if ( SDL_OpenAudio(&wanted, &hwspec) < 0 ) {
        fprintf(stderr, "Couldn't open audio: %s\n", SDL_GetError());
        return(-1);
    }

    sample_rate = hwspec.freq; // override with actual sampling rate that we can get from the hardware
    latency = hwspec.samples*1000/hwspec.freq; // override with actual latency
    
    fprintf(stderr, CLEAR_SCREEN "Format: %d\n", hwspec.format);
    fprintf(stderr, "Sample rate: %d Hz\n", sample_rate);
    fprintf(stderr, "Latency: %d samples (%d ms)\n", hwspec.samples, latency);
    
    chan_buf = malloc(hwspec.size);
    mix_buf = malloc(hwspec.size*2);
    A440 = ((1.0 / (((1.0 / 440)) * sample_rate))); // relative to sample size

    /* Let the callback function play the audio chunk */
    SDL_PauseAudio(0);

	fprintf(stderr, "[+] Playing, press Esc to quit.\n" SAVE_CURSOR_POS);
    event_loop();

	fprintf(stderr, "\nGoodbye!\n");    
    SDL_CloseAudio();

	TTF_CloseFont(font);
	TTF_Quit();
	SDL_Quit();    
    return(0);	
}

Uint16 round_down_to_power_of_two(Uint16 num) {
	Uint16 out = 2;
	while (out < num) {
		out = out<<1;
	}
	return out>>1;
}

Uint32 add_update_event(Uint32 interval, void *param) {
	SDL_Event event;
	SDL_UserEvent userevent;

	userevent.type = SDL_USEREVENT;
	userevent.code = 0;
	userevent.data1 = NULL;
	userevent.data2 = NULL;

	event.type = SDL_USEREVENT;
	event.user = userevent;

	SDL_PushEvent(&event);
	return(interval);
}

static void event_loop() {
	SDL_Event event;
	int quit = 0;
	
	int note;
	struct osc *channel;
	SDL_TimerID update_timer;
	
	update_timer = SDL_AddTimer(250, add_update_event, NULL);

	/* Wait for events, loop until an SDL_QUIT event is found */
	while( !quit && SDL_WaitEvent( &event ) ){
		
		switch( event.type ){
			/* Keyboard event */
			/* Pass the event data onto PrintKeyInfo() */
			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_ESCAPE) quit=1;
				else {
					note = getnote(event.key.keysym.sym);
					if (note != 0) {
						channel = allocate_channel(note);
						if (channel) {
							channel->waveform = chan_waveform;
							channel->filter_type = chan_filter_type;
							channel->filter_alpha = chan_filter_alpha;
							start_channel(channel, note);
						}
					} else
					if (change_octave(event.key.keysym.sym)) {
					} else
					if (change_volume(event.key.keysym.sym)) {
					} else
					if (change_waveform(event.key.keysym.sym)) {
					} else
					if (change_output_filter(event.key.keysym.sym)) {
					} else
					if (panic(event.key.keysym.sym)) {
					} else
					if (change_sustain(event.key.keysym.sym)) {
					} else
					if (change_chan_filter(event.key.keysym.sym)) {
					}
				}
				break;
				
			case SDL_KEYUP:
				note = getnote(event.key.keysym.sym);
				if (note != 0) {
					channel = free_channel(note);
					if (channel)
						stop_channel(channel, note);
				}
				break;

			/* SDL_QUIT event (window close) */
			case SDL_QUIT:
				quit = 1;
				break;

			case SDL_USEREVENT:
				update_status();
				break;
				
			default:
				break;
		}
	}

	SDL_RemoveTimer(update_timer);
}

void show_legend(int line);
static SDL_Color white = {0,255,0};
static SDL_Color black = {0,0,0};
static int text_max_width = 0;
static int draw_text(const char *text, int line, int col) {
	SDL_Surface *textbuf;
	int height;
	
	//textbuf = TTF_RenderUTF8_Solid(font, text, white);
	textbuf = TTF_RenderUTF8_Shaded(font, text, white, black);
	if (text_max_width < textbuf->w) text_max_width = textbuf->w;

	height = textbuf->h;
	SDL_Rect loc = { col, line, 0, 0 };
	SDL_BlitSurface(textbuf, NULL, screen, &loc);	
	SDL_FreeSurface(textbuf);
	return height;
}

static void update_status() {
	char buf[512];
	int line = 0;
	
	SDL_FillRect(screen, NULL, SDL_MapRGB(screen->format, 0, 0, 0));
	text_max_width=0;
	
	snprintf(buf, sizeof(buf), "Active channels: %d%s / %d",
		count_active_channels(), channel_overflow?"*":"", NUM_CHANNELS);
	line += draw_text(buf, line, 0);

	snprintf(buf, sizeof(buf), "Waveform: %s", waveform_str[chan_waveform]);
	line += draw_text(buf, line, 0);

	snprintf(buf, sizeof(buf), "Octave: %d", global_octave);
	line += draw_text(buf, line, 0);
	
	snprintf(buf, sizeof(buf), "Volume: %.1f %s%s", global_volume,
		use_agc?"(AGC)":"", clipping?"(Clipping)":"");
	line += draw_text(buf, line, 0);

	snprintf(buf, sizeof(buf), "Dither: %s", global_dither?"Enabled":"Disabled");
	line += draw_text(buf, line, 0);
	
	snprintf(buf, sizeof(buf), "Sustain: %.1f", global_sustain);
	line += draw_text(buf, line, 0);
	
	snprintf(buf, sizeof(buf), "Output Filter: %s - %.1f",
		filter_str[global_filter_type], global_filter_alpha);
	line += draw_text(buf, line, 0);
	
	snprintf(buf, sizeof(buf), "Channel Filter: %s - %.1f",
		filter_str[chan_filter_type], chan_filter_alpha);
	line += draw_text(buf, line, 0);

	show_legend(line);

	SDL_Flip(screen);

/*
	fprintf(stderr, RESTORE_CURSOR_POS);
	fprintf(stderr, "Active channels: %d%s" CLEAR_EOL,
		count_active_channels(), channel_overflow?"*":"");	
	fprintf(stderr, "Waveform: %s" CLEAR_EOL, waveform_str[chan_waveform]);
	fprintf(stderr, "Octave: %d" CLEAR_EOL, global_octave);
	fprintf(stderr, "Volume: %.1f" CLEAR_EOL, global_volume);
	fprintf(stderr, "Sustain: %.1f" CLEAR_EOL, global_sustain);
	fprintf(stderr, "Output Filter: %s - %.1f" CLEAR_EOL,
		filter_str[global_filter_type], global_filter_alpha);		
	fprintf(stderr, "Channel Filter: %s - %.1f" CLEAR_EOL,
		filter_str[chan_filter_type], chan_filter_alpha);
*/
}

void show_legend(int line) {
	char buf[512];
	int col = text_max_width+50;
	
	snprintf(buf, sizeof(buf), " ");
	line += draw_text(buf, line, 0);
	snprintf(buf, sizeof(buf), "Del - Stop all voices");
	line += draw_text(buf, line, 0);
	snprintf(buf, sizeof(buf), "Esc - Quit");
	line += draw_text(buf, line, 0);
	snprintf(buf, sizeof(buf), " ");
	line += draw_text(buf, line, 0);
	snprintf(buf, sizeof(buf), "Sample rate: %d Hz", sample_rate);
	line += draw_text(buf, line, 0);	
	snprintf(buf, sizeof(buf), "Latency: %d samples (%d ms)", hwspec.samples, latency);
	line += draw_text(buf, line, 0);

	line = 0;
	snprintf(buf, sizeof(buf), "Alphanum keys - notes");
	line += draw_text(buf, line, col);
	snprintf(buf, sizeof(buf), "Up/Down - cycle waveform");
	line += draw_text(buf, line, col);
	snprintf(buf, sizeof(buf), "Left/Right - adjust octave");
	line += draw_text(buf, line, col);
	snprintf(buf, sizeof(buf), "PUp/PgDn - adjust volume, F6 - toggle AGC");
	line += draw_text(buf, line, col);
	snprintf(buf, sizeof(buf), "F7 - toggle dither");
	line += draw_text(buf, line, col);
	snprintf(buf, sizeof(buf), "F1/F2 - adjust sustain");
	line += draw_text(buf, line, col);
	
	snprintf(buf, sizeof(buf), "F10 - cycle filter type. F8/F9 - adjust");
	line += draw_text(buf, line, col);
	snprintf(buf, sizeof(buf), "F5 - cycle channel filter, F3/F4 - adjust");
	line += draw_text(buf, line, col);
	
}

static int getnote(SDLKey key) {
	int note;
	switch( key ) {
		case SDLK_z: note = 3; break;
		case SDLK_s: note = 4; break;
		case SDLK_x: note = 5; break;
		case SDLK_d: note = 6; break;
		case SDLK_c: note = 7; break;
		case SDLK_v: note = 8; break;
		case SDLK_g: note = 9; break;
		case SDLK_b: note = 10; break;
		case SDLK_h: note = 11; break;
		case SDLK_n: note = 12; break; // this is A440*2
		case SDLK_j: note = 13; break;
		case SDLK_m: note = 14; break;
		case SDLK_COMMA: note = 15; break;
		case SDLK_l: note = 16; break;
		case SDLK_PERIOD: note = 17; break;
		case SDLK_SEMICOLON: note = 18; break;
		case SDLK_SLASH: note = 19; break;

		case SDLK_q: note = 15; break;
		case SDLK_2: note = 16; break;
		case SDLK_w: note = 17; break;
		case SDLK_3: note = 18; break;
		case SDLK_e: note = 19; break;
		case SDLK_r: note = 20; break;
		case SDLK_5: note = 21; break;
		case SDLK_t: note = 22; break;
		case SDLK_6: note = 23; break;
		case SDLK_y: note = 24; break;
		case SDLK_7: note = 25; break; // this is A440*3
		case SDLK_u: note = 26; break;
		case SDLK_i: note = 27; break;
		case SDLK_9: note = 28; break;
		case SDLK_o: note = 29; break;
		case SDLK_0: note = 30; break;
		case SDLK_p: note = 31; break;

		case SDLK_LEFTBRACKET: note = 32; break;
		case SDLK_EQUALS: note = 33; break;
		case SDLK_RIGHTBRACKET: note = 34; break;

		default: note = 0;
	}
	return note;
}

static int change_octave(SDLKey key) {
	switch (key) {
		case SDLK_RIGHT: global_octave++; return 1; 
		case SDLK_LEFT: global_octave--; return 1;
		default: return 0;
	}
}

static int change_volume(SDLKey key) {
	switch (key) {
		case SDLK_PAGEUP: global_volume += 0.1; break;
		case SDLK_PAGEDOWN: global_volume -= 0.1; break;
		case SDLK_F6: use_agc = !use_agc; break;
		case SDLK_F7: global_dither = !global_dither; break;
		default: return 0;
	}
	if (global_volume < 0) global_volume = 0;
	return 1;
}

static int change_waveform(SDLKey key) {
	switch (key) {
		case SDLK_UP: chan_waveform++; break; 
		case SDLK_DOWN: chan_waveform--; break;
		default: return 0;
	}
	if (chan_waveform>MAX_WAVEFORM) chan_waveform=0;
	if (chan_waveform<0) chan_waveform=MAX_WAVEFORM;
	return 1;
}

static int change_output_filter(SDLKey key) {
	switch (key) {
		case SDLK_F10: global_filter_type++; break;
		case SDLK_F9: global_filter_alpha += 0.1; break; 
		case SDLK_F8: global_filter_alpha -= 0.1; break;		
		default: return 0;
	}
	if (global_filter_alpha > 1.1) global_filter_alpha = 1.0;
	if (global_filter_alpha < 0.1) global_filter_alpha = 0;
	if (global_filter_type>2) global_filter_type=0;
	return 1;
}

static int panic(SDLKey key) {
	if (key == SDLK_DELETE) {
		all_stop();
		return 1;
	}
	return 0;
}

static int change_sustain(SDLKey key) {
	switch (key) {
		case SDLK_F1: global_sustain -= 0.1; break; 
		case SDLK_F2: global_sustain += 0.1; break;
		default: return 0;
	}
	if (global_sustain < 0.1) global_sustain = 0;
	return 1;
}

static int change_chan_filter(SDLKey key) {
	switch (key) {
		case SDLK_F5: chan_filter_type++; break;
		case SDLK_F4: chan_filter_alpha += 0.1; break; 
		case SDLK_F3: chan_filter_alpha -= 0.1; break;		
		default: return 0;
	}
	if (chan_filter_alpha > 1.1) chan_filter_alpha = 1.0;
	if (chan_filter_alpha < 0.1) chan_filter_alpha = 0;	
	if (chan_filter_type>2) chan_filter_type=0;
	return 1;
}


static void fill_audio(void *udata, Uint8 *stream, int len)
{
	int16_t *out_buf = (int16_t*)stream;
	int32_t sample, peak=0;
	
	memset(out_buf, 0, len);
	memset(mix_buf, 0, len*2); // double the integer size of stream

	for (int i=0; i < NUM_CHANNELS; i++) {
		if (channels[i].volume) {
			gen_audio(&channels[i], chan_buf, len);
			if (channels[i].filter_type)
				filter(chan_buf, len,
					channels[i].filter_type,
					channels[i].filter_alpha,
					&channels[i].prev_left, &channels[i].prev_right);
			
			//SDL_MixAudio(stream, chan_buf, len, SDL_MIX_MAXVOLUME);
			for (int k=0; k < len/sizeof(int16_t);k++) {
				mix_buf[k] += chan_buf[k];
			}
		}
	}

	// apply global volume
	for (int k=0; k < len/sizeof(int16_t);k++) {
		sample = (global_volume*mix_buf[k]) / NUM_CHANNELS;
		// dither
		if (global_dither && rand() > RAND_MAX/2) sample++;
		out_buf[k] = sample;
		if (abs(sample) > peak) peak = abs(sample);
	}

	// AGC
	if (peak > 0x7fff) {
		if (use_agc) {
			// clipping, reduce global volume
			global_volume = global_volume * 0x7fff / peak;
			
			// re-apply global volume
			for (int k=0; k < len/sizeof(int16_t);k++) {
				out_buf[k] = (global_volume*mix_buf[k]) / NUM_CHANNELS;
				// dither
				if (global_dither && rand() > RAND_MAX/2) out_buf[k]++;
			}			
		} else {
			clipping = 1;
		}
	} else {
		clipping = 0;
	}
	
	// apply global filter
	if (global_filter_type)
		filter(out_buf, len, global_filter_type, global_filter_alpha,
			&global_prev_left, &global_prev_right);	
}

struct osc *allocate_channel(int note) {
	int min_vol=MAX_VOLUME, min_channel=0;
	for (int i=0; i<NUM_CHANNELS; i++) {
		if (!channels[i].volume) {
			channel_overflow = 0;
			return &channels[i];
			
		} else {
			// find the channel with softest volume
			if (channels[i].volume < min_vol) {
				min_vol = channels[i].volume;
				min_channel = i;
			}
		}
	}
	
	channel_overflow = 1;
	return &channels[min_channel];
}

static struct osc *free_channel(int note) {
	for (int i=0; i<NUM_CHANNELS; i++) {
		if (channels[i].volume
			&& channels[i].release == 0
			&& channels[i].note == note)
		{
			return &channels[i];
		}
	}
	return NULL;
}

static int count_active_channels() {
	int count = 0;
	for (int i=0; i<NUM_CHANNELS; i++) {
		if (channels[i].volume) {
			count++;
		}
	}
	if (count<NUM_CHANNELS) channel_overflow = 0;
	return count;
}

static void start_channel(struct osc *channel, int note) {
	int octave;
	float sample_count, fraction_count; // have to be float to avoid integer overflow

	// reset accumulators and compute phase
	switch(channel->waveform) {
		case 0: // triangle
			channel->accum = 0x4000 << 16; // this is the zero-crossing, start here to avoid click
			sample_count = 65536; // 65536 samples per period
			break;
			
		case 3: // sine
			channel->accum = 0;
			sample_count = SINE_DATA_LEN;
			break;

		default: // square, sawtooth
			channel->accum = 0;
			sample_count = 65536; // 65536 samples per period 
	}
	fraction_count = 65536; // 1.0 in 16.16 fixed point
	channel->phase = sample_count * fraction_count * A440 * powf(SEMITONE, note);

	// octave == 0 is where "n" (note 12) is 2*A440
	octave = global_octave - A440_OCTAVE;
	if (octave>0)
		channel->phase <<= octave;
	else if (octave<0)
		channel->phase >>= -octave;

	// store for informative purposes
	channel->note = note;
	channel->octave = octave;
	channel->freq = 440 * powf(SEMITONE, note) * powf(2, octave);
	//fprintf(stderr, "Freq: %.0f Hz\n", channel->freq);
	
	// reset filter variables
	channel->prev_left = channel->prev_right = 0;

	// start playing
	channel->release = 0;
	channel->volume = MAX_VOLUME; 
}

static void stop_channel(struct osc *channel, int note) {
	if (global_sustain) {
		channel->release = 
		channel->release_constant =
		global_sustain * sample_rate;
	}
	else
		channel->volume = 0;	
}

static void all_stop() {
	for (int i=0; i<NUM_CHANNELS; i++) {
		channels[i].volume = 0;;
	}
}

void gen_audio(struct osc *channel, int16_t *out, int size) {
	int16_t sample;
	
	size >>= 1; // divide by 2 (2 bytes per sample)
	for (int i=0; i<size;) {

		// to make sure all voices sound with the same volume,
		// we restrict amplitude to -0x4000 to 0x4000
		switch (channel->waveform) {
			case 0:
				// triangle wave
				sample = channel->accum >> 16;
				sample >>=1;
				sample = (abs(sample) - 0x2000) << 1;
				break;

			case 1:
				// square wave
				sample = (channel->accum >> 31)?0xC000:0x3fff;
				break;

			case 2:
				// sawtooth wave
				sample = channel->accum >> 16;
				sample >>=1;
				break;

			case 3:
				// N-point sinewave
#ifdef USE_EMBEDDED_SINE_TABLE
				sample = sinewave[(channel->accum>>16) & 0xff];
#else
				sample = sinewave[((channel->accum>>16) % SINE_DATA_LEN)];
#endif
				break;
		}

		channel->accum += channel->phase;

		int volume = channel->volume;
		if (channel->release > 0) {
			channel->release--;
			volume = channel->volume * channel->release / channel->release_constant;
			if (!volume)
				channel->volume = 0; // stop
		}
		
		if (volume) {
			sample = (int32_t)sample*volume/MAX_VOLUME;
		} else sample = 0;
		
		out[i++] = sample;
		out[i++] = sample;
		//fprintf(stderr, "%04hx\n", sample);
	}
	//fprintf(stderr, "\n");
	//fprintf(stderr, "%04hx\n",sample);
}

/*
* calculations
* ============
* accum is 16.16, where the top 16 is "sample index"
* How much the accum needs to be increased is called "phase".
* "phase" is also 16.16.
* 
* 1. Direct samples (no samples, "sample index" is output
*    directly as amplitude.
* 
*    Implicit sample size = 65536 (16-bit) since the "sample index"
*    loops every 65536 points.
*
*    At 32kHz sampling rate, to play a note at 440Hz, we want to
*    the whole "samples" to repeat 72.73 times (that is 32000/440)
*    every second.
*    
*    Since we have 65536 samples, to make that happen, the accum
*    must be incremented by 901.12 at every tick. This makes the
*    whole "samples" repeat itself every 72.73 ticks.
*
*    ==========
*    Another way to calculate it.
*    At 32kHz sampling, every "tick" is 1/32000.
*    If we want to play at 440Hz, we need the whole "samples"
*    to repeat every 1/440 second.
* 
*    1/440 seconds corresponds to corresponds to
*    (1/440) / (1/32000) ticks, that is 72.73 ticks.
*
*    The whole "samples" must be traversed once every 72.73 ticks,
*    and since we have 65536 samples, we have to increment the accum
*    901.12 at every tick.
*
*    ---
*
*    To avoid aliasing we have to ensure that the
*    target frequency (440) must be less than 1/2 the sampling rate.
*    For a sine wave this is okay, but for anything else, the output
*    needs to be filtered, either digitally or analogically.
*
* 2. Sampled synthesis - fixed original sampling rate.
* 
*    Assuming we have an audio sample originally sampled at 8000Hz.
*    The number of data points doesn't matter.
*
*    To play the samples at its original sampling rate, we use
*    top 16 bit of accum as "sample index". 
*
*    Hence, each data point needs to be played once every 1/8000 second,
*    and since each "tick" is 1/32000, the "sample index" has to
*    advance once every 4 ticks, or, it has to advance
*    0.25 at every tick.
*
*    ---
* 
*    To play it a different frequency, say we want to play the
*    sample at 12000Hz, we just repeat the above calculation:
*    "sample index" has to be increased very 1/12000 second,
*    which corresponds to 2.67 ticks. Or, at each tick it has to be
*    increased 0.375.
* 
*    Note that 12000 is 1.5 times 8000, and the ration of the phase
*    between them is also 0.375/0.25 = 1.5.
*
*    In other words, "higher frequency" = "higher phase".
*
*    ---
*
*    To avoid aliasing, we have to ensure the final sample output
*    freq must be less than 1/2 the sampling rate.
* 
*    In this example, assuming that the original audio sample is
*    band-limited at 8000Hz, we can happily play it up to twice
*    the original sampling rate (at 16000) without filtering the
*    output. But if we want to play it any higher, we have to
*    filter the output at 16000.
* 
* 3. Sampled synthesis - fixed sample data
*
*    For example, we have a sine-wave table of 256 data points,
*    of unknown sampling rate, but we know this table represents
*    one period of the audio.
*
*    If we want to play this at 440Hz, the whole sample needs to
*    be played once every 1/440 seconds. In other words, the
*    sample index needs to be increased once every (1/440) /256 second.
*
*    This corresponds to 0.2841 tick (since each tick is 1/32000).
*    Hence at every tick, the sample index needs to be increased
*    by 3.52.
*
*    To see if the logic makes sense, see if the this number
*    matches what we've got if we use 65536 samples instead of 256.
*    Following the sample calculation, we will then have to
*    increment the sample index by 901.12 at every tick.
*    But you see, this is exactly the value we've got in #1, above.
*
*    ---
*
*    If we want to play at double the frequence, say, at 880Hz,
*    the following the same calculation we'll get that the sample
*    index needs to be increased by 7.04 at every tick.
*    This is exactly double (7.04 = 2*3.52) the increment we use at
*    440Hz, so we come to the same conclusion that
*    "higher freq" = "higher increment", linearly.
*
*    ---
*
*    We assume that the original 256 data points is band-limited for
*    its period (which is true for sine-wave, but not true for anything else).
*    In that case, the highest frequence we can play without aliasing
*    and without filtering would be 16000, or, sample index increment
*    of 128.
* 
*    The value of 128 is 1/2 of 256 is not an accident, it is the
*    consequence of sampling rate theory.
*
*    Note that if the original data point is not band-limited (e.g
*    it is square wave, or triangular wave, or something else), we
*    have to find the period of highest frequency sine-wave represented by
*    the waveform. Let's say of the 256 data point, the highest frequency
*    sine-wave is only represented by 4 data points. This means we can
*    only play this waveform up to 1/2 of the data points, that is,
*    the max increment of the sample index is only 2.
*
*    Or, otherwise, we'll have to set up a filter at the output.
*    If we don't, we'll get aliasing.
*/

// stereo filtering
static void filter(int16_t *buf, int size, char filter_type,
float alpha, int16_t *prev_left, int16_t *prev_right)
{
	int16_t sample;
	int16_t *out = buf;
	size >>= 1; // divide by 2 (2 bytes per sample)
	
	float neg_alpha = (1-alpha);

	switch (filter_type) {
		case 0: // no filter
			break;
			
		case 1: // FIR Filter
			//fprintf(stderr, "FIR\n");
			for (int i=0; i<size;) {
				// left
				sample = out[i]*alpha;
				sample += neg_alpha * (*prev_left);
				(*prev_left) = out[i];
				out[i++] = sample;

				// Right
				sample = out[i]*alpha;
				sample += neg_alpha * (*prev_right);
				(*prev_right) = out[i];
				out[i++] = sample;
			}		
			break;
		
		case 2: // IIR Filter
			//fprintf(stderr, "IIR\n");
			for (int i=0; i<size;) {
				// Left
				sample = out[i]*alpha;
				sample += neg_alpha * (*prev_left);
				(*prev_left) = out[i++] = sample;

				// Right
				sample = out[i]*alpha;
				sample += neg_alpha* (*prev_right);
				(*prev_right) = out[i++] = sample;
			}
			break;
	}
}

/* 
* The simplest digital low-pass filter.
* =====================================
* 
* There are two, FIR and IIR.
*
* FIR: y[n] = (a.x[n] + b.x[n-1]) / (a+b)
* a=1 and b=0 --> no filter
* a=1 and b=1 --> averaging
* a=0 and b=1 --> delay 1 sample
* low-performance.
* 
* FIR (another form): y[n] = a.x[n] + (1-a).x[n-1]
* a=1 --> no filter
* a=0.5 --> averaging
* a=0 --> delay 1 sample
*
* IIR: y[n]= a.x[n] + (1-a).y[n-1]
* a=1 --> no filter
* a=0.5 --> averaging but not like what you think it is
* a=0 --> mute (zero output)
*
* IIR provides sharper cut-off at the expense of osciallations
*/
