#!/bin/ash
# FatdogArm/Fatdog64 package build script
# Copyright (C) James Budiono, 2013, 2014, 2015, 2018
# License: GNU GPL Version 3 or later
# 
# This will build all packages, in the correct order, as specified
# by the pkglist. This script calls build-pkg.sh so many options in 
# build-pkg.sh is applicable here too.
#
# Note: this assumes that LFS Chapter 5 "tools" is already available 
# (LFS Chapter 5 can be built with bootstrap.sh)
#
# $1-command 
# command is install, build, download, remove (default is install)
#
# Package will be automatically rebuild/re-installed if the recipe is 
# newer than the compiled one.

export LANG=C # make it run a bit faster

### configuration settings (overridable)
#DRY_RUN= # if not empty, don't actually do it
PKGLIST=${PKGLIST:-build-pkglist}
BASE_DIR=""

DEF_PKGARCH=x86_64
PKGARCH=${PKGARCH:-$DEF_PKGARCH}
PKG_TYPE=${PKG_TYPE:-txz}
PKG_DIR=${PKG_DIR:-pkg}
PKG_DEPTH=5 # only look for package up to this depth

SRC_DIR=$(readlink -f ${SRC_DIR:-src})
OUTPUT_DIR=$(readlink -f ${OUTPUT_DIR:-output})
CMD=$1
INSTALLER=installpkg
REMOVEPKG=removepkg

### run-time variables
FINDPKG_CACHE=   # filled in by main later
TEMP_INSTALL=    # hold temporary installed packages during build
TEMP_UNINSTALL=  # hold temporary removed packages during build
REPLACED_BY=     # package that replaces the package we're about to build

########### helpers ###########

# load our helper functions 
. ${0%/*}/build-lib.sh 

# $1-flag $2-option, $* option variables, out: TEMP_INSTALL, TEMP_UNINSTALL
preprocess_options() {
	local p OPTION PKG PKG_FULL_PATH PKGNAME PKGVER PKGARCH PKGBUILD
	local build_only="" install_only=""
	TEMP_INSTALL=""	TEMP_UNINSTALL="" REPLACED_BY=""

	# process flag
	case $1 in
		"build")   build_only="yes" ;;
		"install") install_only="yes" ;; 
	esac
	shift

	# process options
	for p; do
		case $p in
			%*) OPTION=$p; continue ;;
		esac
		
		# process args according to current options
		case $OPTION in
			%depends) # install temporary dependency, clearing other packages if need be
				[ -z $build_only ] && continue
				find_pkg $PKG_DIR $p $FINDPKG_CACHE || continue # ignore packages not found
				if ! [ -e $BASE_DIR/var/log/packages/$PKG ]; then
					TEMP_INSTALL="$TEMP_INSTALL $PKG"
					#uninstall_only &&
					#TEMP_UNINSTALL="$TEMP_UNINSTALL $UNINSTALLED_PKG"
					[ $DRY_RUN ] && echo "temp-install $PKG" ||
					ROOT=$BASE_DIR $INSTALLER $OUTPUT_DIR/$PKG.$PKG_TYPE
				fi
				;;

			%removes) # temporarily removes some packages
				[ -z $build_only ] && continue
				find_pkg $PKG_DIR $p $FINDPKG_CACHE || continue # ignore packages not found
				if [ -e $BASE_DIR/var/log/packages/$PKG ]; then
					uninstall_only &&
					TEMP_UNINSTALL="$TEMP_UNINSTALL $UNINSTALLED_PKG"
				fi
				;;
				
			%replaces) # remove some packages permanently
				[ -z $install_only ] && continue
				find_pkg $PKG_DIR $p $FINDPKG_CACHE || continue
				if [ -e $BASE_DIR/var/log/packages/$PKG ]; then
					#uninstall_only
				fi				
				;;

			%replaced_by) # mark that this package is replaced by another
				[ -z $install_only ] && continue
				find_pkg $PKG_DIR $p $FINDPKG_CACHE || continue
				REPLACED_BY="$REPLACED_BY $PKG"
				;;				
		esac
	done
}

# $1-option, $* option variables, $TEMP_INSTALL. $TEMP_UNINSTALL
postprocess_options() {
	local p
	
	# undo whatever was done in preprocess_options
	for p in $TEMP_INSTALL; do
		echo "temp-remove $p"
		[ $DRY_RUN ] || ROOT=$BASE_DIR $REMOVEPKG $OUTPUT_DIR/$p > /dev/null
	done
	for p in $TEMP_UNINSTALL; do
		[ $DRY_RUN ] && echo "install-back $p" ||
		ROOT=$BASE_DIR $INSTALLER $OUTPUT_DIR/$p.$PKG_TYPE
	done
	TEMP_INSTALL="" TEMP_UNINSTALL="" REPLACED_BY=""
}


########### ACTIONS ###########


# $PKG, $PKG_FULL_PATH, $@-options
build_install() {
	#echo build install $PKG; return
	if ! [ -e $OUTPUT_DIR/$PKG.$PKG_TYPE ] ||
	     [ $PKG_FULL_PATH/recipe -nt $OUTPUT_DIR/$PKG.$PKG_TYPE ];
	then
		#uninstall_only &&
		build_only "$@" &&
		install_only "$@"		
	else
		install_only "$@"
	fi
}

# $PKG $PKG_FULL_PATH, $@-options
install_only() {
	local p skip=""
	if [ -e $OUTPUT_DIR/$PKG.$PKG_TYPE ]; then # can only install if we have it
		if ! [ -e $BASE_DIR/var/log/packages/$PKG ] || 
			 [ $OUTPUT_DIR/$PKG.$PKG_TYPE -nt $BASE_DIR/var/log/packages/$PKG ];
		then
			preprocess_options install "$@"
			[ "$REPLACED_BY" ] &&
			for p in $REPLACED_BY; do
				if [ -e $BASE_DIR/var/log/packages/$p ]; then
					skip=yes
					[ $DRY_RUN ] &&echo $PKG replaced by $p
					break
				fi
			done
			if [ -z $skip ]; then
				[ $DRY_RUN ] && echo install $PKG ||
				ROOT=$BASE_DIR $INSTALLER $OUTPUT_DIR/$PKG.$PKG_TYPE
			fi
		fi	
	fi
}

# $PKG $PKG_FULL_PATH, $@-options
build_only() {
	local ret
	#echo build only $PKG; return
	if ! [ -e $OUTPUT_DIR/$PKG.$PKG_TYPE ] ||
	   [ $PKG_FULL_PATH/recipe -nt $OUTPUT_DIR/$PKG.$PKG_TYPE ];
	then
		preprocess_options build "$@"
		[ $DRY_RUN ] && echo build $PKG ||
		./build-pkg.sh $PKG; ret=$?
		postprocess_options "$@"
	fi
	return $ret
}

# $PKG
download_only() {
	#echo download only $PKG; return
	unset pkg_download
	[ -e ./functions.sh ] && . ./functions.sh
	[ -e ./env.sh ] && . ./env.sh
	. $PKG_FULL_PATH/recipe
	type pkg_download > /dev/null || return 0
	if ! [ -s $SRC_DIR/$TARBALL ]; then
		[ $DRY_RUN ] && echo download $PKG ||
		pkg_download $SRC_DIR $TARBALL 
	fi
}

# $PKG, out: UNINSTALLED_PKG
uninstall_only() {
	UNINSTALLED_PKG=""
	local PKG=$PKG p # find the base pkgname
	PKG=${PKG%-*}; PKG=${PKG%-*}; PKG=${PKG%-*}; # strip build, arch, version
	for p in $BASE_DIR/var/log/packages/${PKG}-[0-9]*; do # find actual installed pkgname
		PKG=${p##*/}
		
		#echo uninstall only $PKG; return
		if [ -e $BASE_DIR/var/log/packages/$PKG ]; then
			UNINSTALLED_PKG=$PKG
			echo uninstall $PKG
			[ $DRY_RUN ] || ROOT=$BASE_DIR $REMOVEPKG $PKG > /dev/null
		fi
	done
	return 0
}

# $PKG
bootstrap() {
	[ -e $BASE_DIR/var/log/packages/$PKG ] && return 0	
	if [ $DRY_RUN ]; then
		echo bootstrapping $PKG
	else
		type pkg_prepare > /dev/null &&
		mkdir -p $BASE_DIR &&
		download_only &&
		( cd $BASE_DIR; pkg_prepare ) &&
		ROOT=$BASE_DIR $INSTALLER $SRC_DIR/$PKG.$PKG_TYPE
	fi
}

# $1-input file
process() {
	local PKGLIST=$1 line lineno=0
	while read -r line; do
		: $((lineno = lineno+1))
		set -- ${line%%#*}
		[ -z $1 ] && continue # empty line, skip

		# process directive
		case $1 in
			%exit|%quit)
				echo "Exiting because of %exit directive (file: $PKGLIST line: $lineno)"
				exit ;;	
			%return)
				return ;;
			%include)
				process $2;
				continue ;;
			%echo)
				shift
				echo "$@"
				continue ;;
		esac
		
		# process standard pkg-build
		PKG=$1; shift; CCMD=${CMD:-$1}; shift
		find_pkg $PKG_DIR $PKG $FINDPKG_CACHE || exit # return $PKG $PKG_FULL_PATH PKGNAME PKGVER PKGARCH PKGBUILD
		case $CCMD in
			""|install) build_install "$@" ;;
			build) build_only "$@" ;; 
			download) download_only ;;			
			uninstall|remove) uninstall_only ;;
			#bootstrap) bootstrap ;;
			help|-h|--help) 
				echo "[PKGLIST=pkglist] ${0##*/} [build|download|install|uninstall]"
				exit ;;
			*) echo "unknown command $CCMD for $PKG, ignored (file: $PKGLIST line: $lineno)" ;;
		esac || { echo "$PKG failed to build, stopping."; exit; }		
	done < $PKGLIST
}

### main
FINDPKG_CACHE=$(create_find_pkg_cache $PKG_DIR $PKG_DEPTH)
trap 'rm -f $FINDPKG_CACHE; exit' 0 INT HUP TERM
export FINDPKG_CACHE # for use by build-pkg.sh
process $PKGLIST

