#!/bin/dash
# Copyright (C) 2013, 2021 James Budiono
# License: GNU GPL Version 3 or later
# analog clock for tray icon using 'sit'
# clock svg image is from wikimedia: http://commons.wikimedia.org/wiki/File:Swiss_railway_clock.svg
#
# 2021 - fix sit clock not showing after a while
# 2021 - show second hands if UPDATE_PERIOD is every second, use config file
# renamed to tray-clock.sh (original is sit-clock.sh) to make it clear what it does
#

### configuration
[ -r $HOME/.fatdog/sitclockrc ] && . $HOME/.fatdog/sitclockrc

UPDATE_PERIOD=${UPDATE_PERIOD:-60} # in seconds
SITCLOCK_PATH=/tmp/sitclock.$USER.$XSESSION_ID
ICON_PATH=$SITCLOCK_PATH/icon.svg
TOOLTIP_PATH=$SITCLOCK_PATH/tooltip

### helpers
# $1 - hour rotation, $2 - minute rotation, $3 - second rotation
# out: stdout
write_clock_svg() {
	cat << EOF
<?xml version="1.0" encoding="utf-8"?>
<svg xmlns="http://www.w3.org/2000/svg"
     xmlns:xlink="http://www.w3.org/1999/xlink"
     viewBox="-1024 -1024 2048 2048" width="600" height="600">
  <title>Swiss Railway Clock</title>
  <style type="text/css">
    .bg {stroke: none; fill: white;}
    .fc {stroke: none; fill: black;}
    .h1 {stroke: none; fill: black;}
    .h2 {stroke: none; fill: #aa0000;}
  </style>
  <defs>
    <path id="mark1" d="M -20,-1000 l 40,0 0,100 -40,0 z" />
    <path id="mark2" d="M -40,-1000 l 80,0 0,240 -80,0 z" />
    <path id="mark3" d="M -40,-1000 l 80,0 0,300 -80,0 z" />
    <path id="handh" d="M -50,-600  l 50,-50 50,50 0,800  -100,0 z" />
    <path id="handm" d="M -40,-900  l 40,-40 40,40 0,1180 -80,0  z" />
    <g    id="hands">
      <path d="M -10,-910 l  10,-10 10,10 2,300 -24,0 z
               M -13,-390 l  26,0         7,690 -40,0 z" />
      <path d="M   0,-620 a 120,120 0 0 1 0,240
                          a 120,120 0 0 1 0,-240 z
               M   0,-560 a  60,60  0 0 0 0,120
                          a  60,60  0 0 0 0,-120 z" />
    </g>
    <g id="face1">
      <use xlink:href="#mark1" transform="rotate(06)" />
      <use xlink:href="#mark1" transform="rotate(12)" />
      <use xlink:href="#mark1" transform="rotate(18)" />
      <use xlink:href="#mark1" transform="rotate(24)" />
    </g>
    <g id="face2">
      <use xlink:href="#face1" />
      <use xlink:href="#face1" transform="rotate(30)" />
      <use xlink:href="#face1" transform="rotate(60)" />
      <use xlink:href="#mark3" />
      <use xlink:href="#mark2" transform="rotate(30)" />
      <use xlink:href="#mark2" transform="rotate(60)" />
    </g>
    <g id="face">
      <use xlink:href="#face2" />
      <use xlink:href="#face2" transform="rotate(90)"  />
      <use xlink:href="#face2" transform="rotate(180)" />
      <use xlink:href="#face2" transform="rotate(270)" />
    </g>
  </defs>
  <circle class="bg" r="1024" />
  <use xlink:href="#face"  class="fc" />
  <use xlink:href="#handh" class="h1" transform="rotate($1)" />
  <use xlink:href="#handm" class="h1" transform="rotate($2)" />
  $([ -z "$3" ] && echo "<!--")
  <use xlink:href="#hands" class="h2" transform="rotate($3)" />
  $([ -z "$3" ] && echo "-->")
</svg>
EOF
}

# update clock to current time.
update_clock() {
	# date
	date +"%I:%M:%S %A %e %B %Y" > $TOOLTIP_PATH
	
	# time
	time=$(date +%I-%M/%S)
		
	hour=${time%-*} 
	minute=${time#*-}
	minute=${minute%/*}
	second=${time#*/}

	hour=${hour#0}	
	minute=${minute#0}
	second=${second#0}

	#echo $hour $minute $second
	hour_rot=$(( $hour*30 + $minute/2 ))
	min_rot=$(( $minute*6 ))
	sec_rot=$(( $second*6 ))

	[ $UPDATE_PERIOD -ne 1 ] && sec_rot=
	#kill -STOP $XPID # SFR - pause before update
	write_clock_svg $hour_rot $min_rot $sec_rot > $ICON_PATH-new
	mv $ICON_PATH-new $ICON_PATH
	#kill -CONT $XPID # SFR - continue
}

######## main #########

# 1. start sit
mkdir -p $SITCLOCK_PATH
touch $ICON_PATH $TOOLTIP_PATH
sit $ICON_PATH $TOOLTIP_PATH &
XPID=$!

# 2. prepare for clean shutdown
trap 'kill $XPID; rm -rf $SITCLOCK_PATH; exit' 0 HUP INT TERM

# 3. initial display, sync to current minute
update_clock
while [ $(date +"%S") -ne 0 ]; do sleep 1; done 

# 4. run clock
while :; do
	update_clock
	sleep $UPDATE_PERIOD
	! kill -0 $XPID && break
done
rm -rf $SITCLOCK_PATH
kill $XPID
