#!/bin/dash

# fatdog-shortcut-viewer.sh - show keyboard and mouse shortcuts
# Copyright (C) step, 2020-2021
# License: GNU GPL Version 2
Version=1.1.0

export TEXTDOMAIN=fatdog
export OUTPUT_CHARSET=UTF-8

################
#  Initialize  #
################

FIRSTRUN=$HOME/.fatdog/fatdog-shortcut-viewer-run-at-startup

# Set traps and create temporary folder
TMPD="${TMPDIR:-/tmp}/$USER-${0##*/}"
mkdir -p "$TMPD"
chmod 700 "$TMPD" || exit 1
trap "rm -rf \"$TMPD\"/* \"$TMPD\"; exit" HUP INT QUIT TERM ABRT 0

. "${0%.sh}"-lib.sh

#############################
#  Load user configuration  #
#############################

: TODO

#############################
#  Get and format bindings  #
#############################

bindingsf="$TMPD"/bindings.bsv
countersf="$TMPD"/counters
ctx_any="*"
sep=`printf "\b"`

## text appended after the main list
postface="$(gettext "Showing the shortcuts of:")"

# Openbox window manager
get_openbox_config "$bindingsf" "$ctx_any" "$sep"
{ printf "<counter>"; gettext "Openbox window manager"; echo; } >> "$bindingsf"

# Dunst notification manager
get_dunst_config "$bindingsf" "$ctx_any" "$sep"
{ printf "<counter>"; gettext "Dunst notification manager"; echo; } >> "$bindingsf"

# Sven multimedia keyboard manager
get_sven_config "$bindingsf" "$ctx_any" "$sep"
{ printf "<counter>"; gettext "Sven multimedia manager"; echo; } >> "$bindingsf"

## extract counters
prev=1; i=1
while read line; do
	case $line in
		"<counter>"* )
			postface="$postface\n- ${line#<counter>} $(($i - $prev))"
			prev=$i
			continue
			;;
		* )
			echo "$line"
			;;
	esac
	i=$(($i +1))
done < "$bindingsf" > "$bindingsf"1

## group bindings by device
sort -s -t "$sep" -k1,1n -o "$bindingsf" "$bindingsf"1

##################
#  Presentation  #
##################{{{

win_width=$(gettext locale_window_width:760) win_height=$(gettext locale_window_height:400)
win_width=${win_width#*:} win_height=${win_height#*:}

echo > "$TMPD"/gtkrc '
style "label" {
	font_name = "Sans 12"
}
style "backtitle" {
	font_name = "Sans Italic 12"
}
style "textbox" {
	base[NORMAL] = "#E5EAEB"
	text[NORMAL] = "#000000"
	font_name = "Monospace 12"
}
class "GtkLabel" style "label"
widget "*GtkButton*GtkLabel*" style "label"
widget_class "*GtkHBox.GtkLabel*" style "backtitle"
class "GtkTextView" style "textbox"
'

############}}}
#  Dialog  #
############{{{

[ -e "$FIRSTRUN" ] && read state < "$FIRSTRUN"; state=${state%%|*} # on | off
column -t -s "$sep" "$bindingsf" |
	awk -F"$sep" \
		-v G0="$(gettext "KEYBOARD")" \
		-v G1="$(gettext "MOUSE")" \
		-v G0t="$(gettext "C-A-Left means pressing Control+Alt+Left_arrow together")" \
		-v G0m="$(gettext "Abbreviations: A-/Alt   C-/Control   S-/Shift   W-/Window")" \
		-v G0b="$(gettext "The last column describes what a shortcut does")" \
		-v G1t="$(gettext "To start an action place your pointer where column 1 indicates")" \
		-v G1m="$(gettext "Left, Middle, Right and digits identify a mouse button")" \
		-v G1b="$(gettext "Apply action Click, Press, Drag, etc. to the button")" \
		-v POSTFACE="$postface\n\n${0##*/} $Version" \
		'#{{{awk
# Inject group headers
/^0/ && group != G0 { group = G0; print box(group, G0t, G0m, G0b) }
/^1/ && group != G1 { group = G1; print "\n"box(group, G1t, G1m, G1b) }
function box(s, st, sm, sb,   l, t, b, r) {
	l = length(s) +2
	r = sprintf("% "l"s", " ")
	t = "╭" r "╮"
	s = "│ "s" │"
	b = "╰" r "╯"
	r = "│" r "│"
	# ┐┘└┌
	gsub(/ /, "─", t)
	gsub(/ /, "─", b)
	return t"\n" r" "st"\n" s" "sm"\n" r" "sb"\n" b
}

# Delete device field
{ sub(/^[[:digit:]]+[[:space:]]+/, "") }

{ print }

END { if(POSTFACE) print "\n────\n"POSTFACE }
#awk}}}' > "$TMPD"/text.in

Xdialog \
	--title "$(gettext "Predefined Shortcuts")" \
	--rc-file "$TMPD"/gtkrc \
	--backtitle "$(gettext "Speed up your work with predefined keyboard and mouse shortcuts.")
$(gettext "To learn how to add your own shortcuts click the Help icon located on the Desktop.")\n" \
	--stdout \
	--no-cancel \
	--check "$(gettext "Load this dialog at startup")" "${state:-on}" \
	--textbox "$TMPD"/text.in ${win_width}x$win_height \
	> "$TMPD"/firstrun \
	&

###################}}}
#  Save and exit  #
###################

wait $!

# Save 'show at next login' setting.
if [ 0 = $? -a -s "$TMPD"/firstrun ] && read status < "$TMPD"/firstrun; then
	[ unchecked = "$status" ] && show=off || show=on
	echo $show > "$FIRSTRUN"
fi

