#!/bin/dash
# Download and package libreoffice as Fatdog SFS
# (C) James Budiono 2013, 2015
# License: GNU GPL Version 3 or later
# $1 - optional tmpdir
# i18n by L18L

export TEXTDOMAIN=get_app

# configuration
APPTITLE="$(gettext 'Libreoffice SFS Maker')"
MASTER_URL=http://download.documentfoundation.org/libreoffice/stable
PKGFORMAT=tar.gz
NEEDED_SPACE=$(( 1500*1024 ))	# in KB - this is 1.5 GB
COMP_MODE="-comp xz -Xbcj x86"
#COMP_MODE="-comp lzo -Xcompression-level 1"

# run-time variables
TMPDIR=/tmp
TMPFILE=$(mktemp -p /tmp getlibre.XXXXXXXX)
LATEST=		# the version to download
VERSION=	# the major version (4.0, 3.6, etc)
LANGPACK=	# language pack to download
ROOT_URL=
MAIN_PKG=
SDK_PKG=
HELP_PKG=
LANG_PKG=

# 1. must be run on tty
if ! [ -t 0 ]; then
	exec rxvt -e "$0" "$@"
fi

# 1.1 check for help
case "$1" in
	--help|-h) echo "$(gettext 'Usage:') ${0##*/} [$(gettext '/path/to/tmpdir')]"
			  echo "$(gettext 'Download Libreoffice and make SFS in tmpdir') ($(gettext 'default:') $TMPDIR)"
			  exit
esac
TMPDIR=${1:-$TMPDIR}
TMPDIR="${TMPDIR}/getlibre"

# 2. make tempdir & check free space
mkdir -p "$TMPDIR" || exit
cd "$TMPDIR" || exit

AVAIL_SPACE=$(df -k . | awk 'NR==2 { print $4 }')
if [ $AVAIL_SPACE -lt $NEEDED_SPACE ]; then
    AVAIL_SPACE=$(( $AVAIL_SPACE / 1024 ))
    NEEDED_SPACE=$(( $NEEDED_SPACE / 1024 ))
	echo $(eval echo $(gettext 'Not enough space in $TMPDIR, available: $AVAIL_SPACE MB, required: $NEEDED_SPACE MB'))
	exit
fi

# 3. find available versions
echo $(gettext 'Checking available versions ...')
AVAIL_VERSIONS=$(wget -q -O - $MASTER_URL | sed -n "
/href=\"[0-9]\\.[0-9]*\\.[0-9]*\\/\"/ {
	s|[^h]*href=\"\\([0-9]\\.[0-9]*\\.[0-9]*\\)/\".*|\\1|
	p
}") 

# 4. choose versions
pp=""
for p in $AVAIL_VERSIONS; do pp="$pp $p ''"; done
if ! eval dialog --title \"$APPTITLE\" --menu \"$(gettext 'Choose version')\" 0 0 0 $pp 2> $TMPFILE; then
	clear
	echo $(gettext 'Cancelled.')
	rm $TMPFILE
	exit
fi
LATEST=$(cat $TMPFILE)
VERSION=${LATEST%.*}
clear
echo $(eval echo $(gettext 'Chosen version: $LATEST'))

# 4.1 define root URL based on version
ROOT_URL=${MASTER_URL}/${LATEST}/deb/x86_64

# 5. find available languages
echo $(gettext 'Checking available languages ...')
AVAIL_LANGS=$(wget -q -O - $ROOT_URL | sed -n "
/LibreOffice_${LATEST}_Linux_x86-64_deb_langpack_.*.${PKGFORMAT}\"/ {
	s|[^h]*href=\"LibreOffice_${LATEST}_Linux_x86-64_deb_langpack_\\(.*\\).${PKGFORMAT}\".*|\\1|
	p
}" | sort | uniq)

# 6. choose languages
pp=""
for p in $AVAIL_LANGS; do pp="$pp $p ''"; done
if ! eval dialog --title \"$APPTITLE\" --menu \"$(gettext 'Choose languages')\" 0 0 0 $pp 2> $TMPFILE; then
	clear
	echo $(gettext 'Cancelled.')
	rm $TMPFILE
	exit
fi
LANGPACK=$(cat $TMPFILE)
clear
echo $(eval echo $(gettext 'Chosen version: $LATEST language: $LANGPACK'))
rm $TMPFILE

# 6.1 define packages
MAIN_PKG=LibreOffice_${LATEST}_Linux_x86-64_deb
SDK_PKG=LibreOffice_${LATEST}_Linux_x86-64_deb_sdk
HELP_PKG=LibreOffice_${LATEST}_Linux_x86-64_deb_helppack_${LANGPACK}
LANG_PKG=LibreOffice_${LATEST}_Linux_x86-64_deb_langpack_${LANGPACK}

# 7. get all the stuff
for p in $MAIN_PKG $HELP_PKG $LANG_PKG $SDK_PKG; do
	echo $(eval echo $(gettext 'Downloading $p'))
	if ! wget -nv -c ${ROOT_URL}/${p}.${PKGFORMAT}; then
		echo "$(eval echo $(gettext 'Failed to download $p'))"
		exit
	fi
done

# 8. extract package
for p in $MAIN_PKG $HELP_PKG $LANG_PKG; do
	echo $(eval echo $(gettext 'Extracting $p'))
	tar -xf ${p}.${PKGFORMAT}
done
rm *.${PKGFORMAT}

# 9. extract debs
echo $(gettext 'Extracting DEBs ...')
find -name "*.deb" -exec undeb -x '{}' . \;
rm -rf LibreOffice*

# 10. post-extract steps
echo $(gettext 'Post-extract steps')
if [ -L usr/local/bin/libreoffice${VERSION} ]; then
	# 4.4.x, must do more work
	mv usr/local/bin usr
	for p in usr/local/share/*; do
		mv $p usr/share
	done
	rmdir usr/local/share usr/local
fi
ln -sfv libreoffice${VERSION} usr/bin/libreoffice
ln -sfv libreoffice${VERSION} usr/bin/soffice
for app in base calc draw impress math printeradmin startcenter writer; do
	case $app in
		base|calc) cat="Database;Spreadsheet;" ;;
		draw|impress|math) cat="RasterGraphics;" ;;
		printeradmin|startcenter|writer) cat="WordProcessor;" ;;
	esac
	sed -i -e "s/^Categories=/Categories=${cat}/" opt/libreoffice${VERSION}/share/xdg/${app}.desktop
done

# 11. build SFS
echo $(gettext 'Building SFS')
mkdir squashfs-root
mv etc opt usr squashfs-root
mksquashfs squashfs-root libreoffice-$LATEST-$LANGPACK.sfs -noappend $COMP_MODE

# 12. cleanup
find -maxdepth 1 -type d | sed '1 d' | xargs rm -rf
rox .
echo $(gettext 'Done.')
