#!/bin/sh
# James Budiono 2011, 2013, 2015, 2017, 2018, 2019, 2020
# fatdog interactive secure sandbox
# this version uses rw image.
# you can also choose which SFS to use.
# run this from terminal.
#
# version 1 - forked from version 9 of rw-sandbox.sh
# version 2 - support user namespaces when available
# version 3 - updated support for lxc 1.0.4 (unprivileged container)
# version 4 - proper termination if accidentally closed
# version 5 - support command in arguments
# version 6 - support network and replace insecure desktop emulation with remote Xephyr
# version 7 - added NAT support
# version 8 - (2020) add support for mounting directory, empty SFS list, and OS_IMAGE
# version 9 - (2022) add support for sharing sound and IPC
#
# $1 - path to disk image.
# env: IDMAP=yes - enable use of user namespaces
# env: HOST_IP & LXC_IP & WINDOW_SIZE (if blank auto-calculate)

# 0. directory locations
. $BOOTSTATE_PATH # AUFS_ROOT_ID, BASELINE_MOUNT
[ "$NOT_USING_AUFS" ] && exit 1
XTERM="defaultterm"
SANDBOX_ROOT=/mnt/sb
FAKEROOT=$SANDBOX_ROOT/fakeroot   # mounted chroot location of sandbox - ie, the fake root
SANDBOX_IMG=$SANDBOX_ROOT/sandbox # mounted rw image of the sandbox disk image
SANDBOX_ID=
LXC_ID=sb-lxc	# LXC container ID
USE_DESKTOP=${USE_DESKTOP-ask} # blank (no), ask, yes
USE_NETWORK=${USE_NETWORK-ask} # blank (none), ask, proxyarp or nat
SHARE_SOUND=${SHARE_SOUND-ask} # blank (no), ask, yes
SHARE_IPC=${SHARE_IPC-no} # blank (no), yes
WINDOW_SIZE=${WINDOW_SIZE-1024x600}
#HOST_IP= # auto-calculate if blank
#LXC_IP=  # auto-calculate if blank
#OS_IMAGE=1  # image file is a OS image
TMPFILE=$(mktemp -p /tmp)

# user namespaces settings
#IDMAP=			# enable IDMAP=yes on env to use user namespaces
LXC_UID=${LXC_UID:-100000}          # UID and GID when running under user namespaces
LXC_GID=${LXC_GID:-100000}
LXC_IDRANGE=${LXC_IDRANGE:-65536}   # no of ids to map under user name spaces
#LXC_EXTRA= # whatever you want to append


### trap handler
# umount all and stop lxc if we are accidentally killed
trap '[ $CONSOLE_PID] && kill $CONSOLE_PID; 
[ ${LXC_ID} ] && lxc-stop -n ${LXC_ID}; umountall' HUP
umountall() {
	{
	[ $XSERVER_PID ] && kill $XSERVER_PID
	lxc-wait -n $LXC_ID -s STOPPED
	umount -l $FAKEROOT/$SANDBOX_IMG
	if [ "$USE_DESKTOP" ]; then
		umount -l $FAKEROOT/tmp
		xauth remove ${LXC_ID}/unix:0
	fi

	stop_network
	umount -l $FAKEROOT
	umount -l $SANDBOX_IMG
	rmdir $FAKEROOT
	rmdir $SANDBOX_IMG
	rm ${FAKEROOT}.hold
	} 2> /dev/null
}

### general helpers

die() {
	echo "$1"
	exit 1
}

# get random from $1 to $2 (inclusive)
get_random() {
	awk -v min=$1 -v max=$2 'BEGIN{srand(); print int(min+rand()*(max-min+1))}'
}

# $1-ip, $2-which octet, $3-by how much
adjust_ipv4() {
	awk -F. -v ip=$1 -v octet=$2 -v howmuch=$3 \
'BEGIN {
	$0=ip
	$octet=($octet + howmuch) % 256
	printf("%d.%d.%d.%d", $1,$2,$3,$4);
}'
}

### network setup handlers

prepare_network() {
	local guest_start_file
	
	# compute common parameters
	if [ "$HOST_IP" ]; then
		ipcidr=${HOST_IP}
	else
		ipcidr=$(busybox ipaddr | awk '$1=="inet" && $2 !~ /127.0.0.1/ {print $2}' | head -n1)
		HOST_IP=${ipcidr%/*}
	fi
	HOST_NET=$(busybox ipcalc -n $ipcidr); HOST_NET=${HOST_NET#*=}
	HOST_PREFIX=$(busybox ipcalc -p $ipcidr); HOST_PREFIX=${HOST_PREFIX#*=}
	HOST_GW=$(busybox iproute | awk '$1=="default" { print $3}' | head -n1)
	HOST_GW_DEV=$(busybox iproute | awk '$1=="default" { print $5}' | head -n1)

	case "$USE_NETWORK" in
		proxyarp)
			if [ -z "$LXC_IP" ]; then
				# replace 4th octet
				LXC_IP=$(adjust_ipv4 $HOST_IP 4 $(get_random 1 255))
			fi
			#    type         guest-ip guest-net guest-prefix gw       host-ip
			echo $USE_NETWORK $LXC_IP  $HOST_NET $HOST_PREFIX $HOST_GW $HOST_IP > $FAKEROOT/etc/netparams
			;;

		nat)
			if [ -z "$LXC_IP" ]; then
				# replace 3rd octet
				LXC_IP=$(adjust_ipv4 $HOST_IP 3 $(get_random 1 255))
			fi
			GUEST_NET=$(busybox ipcalc -n $LXC_IP); GUEST_NET=${GUEST_NET#*=}
			GUEST_PREFIX=$(busybox ipcalc -p $LXC_IP); GUEST_PREFIX=${GUEST_PREFIX#*=}
			GUEST_GW=$(adjust_ipv4 $LXC_IP 4 $(get_random 1 255))
			#    type         guest-ip guest-net  guest-prefix  gw        host-ip
			echo $USE_NETWORK $LXC_IP  $GUEST_NET $GUEST_PREFIX $GUEST_GW $HOST_IP > $FAKEROOT/etc/netparams
			;;
		
		"") rm -f $FAKEROOT/etc/netparams ;;
	esac
	
	# create guest network init script
	[ -z "$OS_IMAGE" ] && guest_start_file=$FAKEROOT/etc/rc.d/rc.network.lxc ||
	guest_start_file=$FAKEROOT/sbin/netstart-sb-lxc
	
	! [ -e $guest_start_file ] && cat > $guest_start_file << "EOF" &&
#!/bin/sh
[ -e /etc/netparams ] || exit

read type ip net prefix gw host < /etc/netparams
#echo $type $ip $net/$prefix $gw $host

[ $type = proxyarp ] &&
echo 1 > /proc/sys/net/ipv4/conf/all/proxy_arp ||
echo 0 > /proc/sys/net/ipv4/conf/all/proxy_arp

if type busybox > /dev/null && busybox ipaddr > /dev/null 2>&1; then
busybox iplink set lo up; iplink set eth0 up # bring interfaces up
busybox ipaddr add $ip dev eth0              # add own ip address
busybox iproute add $net/$prefix dev eth0    # add own route
busybox iproute add default via $gw dev eth0 # add default gw
else # if busybox
ip link set lo up; ip link set eth0 up # bring interfaces up
ip addr add $ip dev eth0               # add own ip address
ip route add $net/$prefix dev eth0     # add own route
ip route add default via $gw dev eth0  # add default gw
fi  # if busybox
EOF
	chmod +x $guest_start_file
}

start_network() {
	case "$USE_NETWORK" in
		proxyarp)
			echo 1 > /proc/sys/net/ipv4/ip_forward         # enable ip forwarding
			echo 1 > /proc/sys/net/ipv4/conf/all/proxy_arp # enable proxy arp
			busybox iplink set $LXC_ID up           # enable interface
			busybox ipaddr add $HOST_IP dev $LXC_ID # assign ip address, same as host
			busybox iproute add $LXC_IP dev $LXC_ID # add routing to sandbox
			;;
		
		nat)
			echo 1 > /proc/sys/net/ipv4/ip_forward # enable ip forwarding
			busybox iplink set $LXC_ID up             # enable interface
			busybox ipaddr add $GUEST_GW dev $LXC_ID  # assign guest gw ip address
			busybox iproute add $GUEST_NET/$GUEST_PREFIX dev $LXC_ID # add guest net routing
			iptables -t nat -A POSTROUTING -o $HOST_GW_DEV -j MASQUERADE
			;;
	esac
}

stop_network() {
	case "$USE_NETWORK" in
		proxyarp) ;; # nothing to do
		nat) iptables -t nat -D POSTROUTING -o $HOST_GW_DEV -j MASQUERADE ;;
	esac
}




############# main ###############

# 0.1 must be root
[ $(id -u) -ne 0 ] && die "You must be root to use sandbox."

# 0.2 cannot launch sandbox within sandbox
grep -q LXC_ID $BOOTSTATE_PATH &&
die "Cannot launch sandbox within sandbox."

# 0.3 help
case "$1" in
	--help|-h)
	echo "Usage: ${0##*/} [/path/to/persistent.disk-image] [cmd [args]]"
	echo "Starts a persistent sandbox using lxc. Type 'exit' to leave."
	echo "Path is optional, you will be asked to select or create one if not specified."
	exit
esac

# 0.4 - if not running from terminal but in Xorg, then launch via terminal
! [ -t 0 ] && [ -n "$DISPLAY" ] && exec $XTERM -e "$0" "$@"
! [ -t 0 ] && exit

# 0.5 is this the first sandbox? If not, then create another name for mountpoints
if grep -q $FAKEROOT /proc/mounts; then
	FAKEROOT=$(mktemp -d -p $SANDBOX_ROOT ${FAKEROOT##*/}-XXXXXXX)
	SANDBOX_ID="-${FAKEROOT##*-}"
	SANDBOX_IMG=$SANDBOX_ROOT/${SANDBOX_IMG##*/}${SANDBOX_ID}
	LXC_ID=${LXC_ID}${SANDBOX_ID}	
	rmdir $FAKEROOT
fi

# 0.6 check for lxc
if ! type lxc-start > /dev/null 2>&1; then
	die "LXC not found, please use 'rw-sandbox.sh' instead."
fi

# 1. get aufs system-id for the root filesystem
[ -z "$AUFS_ROOT_ID" ] && AUFS_ROOT_ID=$(
awk '{ if ($2 == "/" && $3 == "aufs") { match($4,/si=[0-9a-f]*/); print "si_" substr($4,RSTART+3,RLENGTH-3) } }' /proc/mounts
)

# 2. get branches, then map branches to mount types or loop devices
if [ -z "$OS_IMAGE" ]; then
items=$(
{ echo ==mount==; cat /proc/mounts; 
  echo ==losetup==; losetup-FULL -a; 
  echo ==branches==; ls -v /sys/fs/aufs/$AUFS_ROOT_ID/br[0-9]* | xargs sed 's/=.*//'; } | awk '
  /==mount==/ { mode=1 }
  /==losetup==/ { mode=2 }
  /==branches==/ { mode=3 }
  {
	if (mode == 1) {
		# get list of mount points, types, and devices - index is $3 (mount points)
		mountdev[$2]=$1
		mounttypes[$2]=$3
	} else if (mode == 2) {
		# get list of loop devices and files - index is $1 (loop devs)
		sub(/:/,"",$1)
		sub(/.*\//,"",$3); sub(/)/,"",$3)
		loopdev[$1]=$3
	} else if (mode == 3) {
		# map mount types to loop files if mount devices is a loop
		for (m in mountdev) {
			if ( loopdev[mountdev[m]] != "" ) mounttypes[m]=loopdev[mountdev[m]]
		}
		# for (m in mountdev) print m " on " mountdev[m] " type " mounttypes[m]
		mode=4
	} else if (mode == 4) {
		# print the branches and its mappings
		if (mounttypes[$0])
			print $0, mounttypes[$0], "on"
		else
			print $0, "unknown", "on"
	}
  }  
'
)
# '
# 3. Ask user to choose the SFS
[ -z "$items" ] && die "No layered filesystem found."
dialog --separate-output --backtitle "rw image sandbox" --title "sandbox config" \
	--checklist "Choose which SFS you want to use" 0 60 10 $items 2> $TMPFILE ||
	{ clear && die "Cancelled, exiting."; }
	
chosen="$(cat $TMPFILE)"
clear
[ -z "$chosen" ] && OS_IMAGE=1 # if no SFS is chosen, we're running an OS IMAGE

else ### if OS_IMAGE
chosen=""
fi   ### if OS_IMAGE

# 4. convert chosen SFS to robranches
robranches=""
for a in $(cat $TMPFILE) ; do
	robranches=$robranches:$a=ro
done
rm $TMPFILE

# 5. get location of rw image
if [ -z "$1" ]; then
	# location not specified - then ask
	dialog --backtitle "rw image sandbox" --title "choose rw image" \
	--extra-button --extra-label "Create" --ok-label "Locate" \
	--yesno "You didn't specify the location of the rw image file. Do you want to locate existing file, or do you want to create a new one?" 0 0
	case $? in
		0) # ok - locate
			dialog --backtitle "rw image sandbox" --title "Specify location of existing rw image" --fselect $(pwd) 8 60 2> $TMPFILE
			rwbranch=$(cat $TMPFILE)
			rm $TMPFILE
			if [ -n "$rwbranch" ]; then
				[ ! -f "$rwbranch" ] && die "$rwbranch doesn't exist, exiting."
			else
				die "You didn't specify any file or you pressed cancel, exiting."
			fi
			;;
		3) # create
			#echo "create"
			dialog --backtitle "rw image sandbox" --title "Specify name and path of new the file" --fselect $(pwd) 8 60 2> $TMPFILE
			rwbranch=$(cat $TMPFILE)
			rm $TMPFILE
			if [ -n "$rwbranch" ]; then
				if [ -f "$rwbranch" ]; then
					die "$rwbranch already exist, exiting."
				else
					# get the size
					dialog --title "Create new rw image" --inputbox "Specify size (in megabytes)" 0 40 100 2> $TMPFILE
					size=$(cat $TMPFILE)
					rm $TMPFILE
					
					if [ -n "$size" ]; then
						if dd if=/dev/zero of="$rwbranch" bs=1 count=0 seek="$size"M; then
							if ! mke2fs -F "$rwbranch"; then
								die "I fail to make an ext2 filesystem at $rwbranch, exiting."
							fi
						else
							die "I fail to create a ${size}M file at $rwbranch, exiting."
						fi
					else
						die "You didn't specify the size or your press cancel, exiting."
					fi					
				fi
			else
				die "You didn't specify any file or you pressed cancel, exiting."
			fi
			;;
		1 | 255) # 1 is cancel, 255 is Escape
			;&
		*) # invalid input - treat as cancel
			die "Cancelled, exiting."
			;;
	esac
else
	# use what we've been given
	rwbranch="$1"
fi
[ $# -gt 0 ] && shift

# 5.1 lxc - enable network?
[ "$USE_NETWORK" = ask ] && dialog --backtitle "tmpfs sandbox" \
	--title "Network Configuration" --no-cancel --no-tags \
	--radiolist "Choose how to configure network (required for desktop emulation)" 0 60 0 \
	"" "None" on proxyarp "Proxy ARP" off nat "NAT" off 2> $TMPFILE-net
read USE_NETWORK < $TMPFILE-net
rm -f $TMPFILE-net
clear
#echo "-->$USE_NETWORK<--"

# 5.2 lxc - enable desktop emulation? (only if we use network)
if [ "$USE_NETWORK" ]; then
	case "$USE_DESKTOP" in
		ask) USE_DESKTOP=
			 dialog --backtitle "tmpfs sandbox" --title "use xorg" \
				--defaultno --yesno "Enable desktop emulation?" 0 0 && USE_DESKTOP=yes
			 ;;
		yes) ;; # keep the yes
		*) USE_DESKTOP= # invalid option, ignore
		;;
	esac
else
	USE_DESKTOP=
fi
clear

# 5.3 lxc - share sounds?
[ "$SHARE_SOUND" = "ask" ] && dialog --backtitle "tmpfs sandbox" --title "share sound" \
       --defaultno --yesno "Enable sound output?" 0 0 && SHARE_SOUND=yes
[ "$SHARE_SOUND" = "yes" ] && # bind-mount /dev/snd to guest
LXC_SHARE_SOUND="-s lxc.mount.entry=\"/dev/snd dev/snd none bind,optional,create=dir\""
clear

# 5.4 lxc - share IPC?
[ "$SHARE_IPC" = "yes" ] && LXC_SHARE_IPC="--share-ipc=1" # share ipc from root container

# 6. make the mountpoints if not exist  yet
mkdir -p $FAKEROOT $SANDBOX_IMG

# 7. do the magic - mount the rw image first, and then the rest with aufs
if [ -d "$rwbranch" ]; then
	mount -o bind "$rwbranch" $SANDBOX_IMG
else
	mount -o loop "$rwbranch" $SANDBOX_IMG
fi || die "unable to mount rw image: $rwbranch"

chmod 0777 $SANDBOX_IMG
if mount -t aufs -o "br:$SANDBOX_IMG=rw$robranches" aufs $FAKEROOT; then
	# 8. record our new aufs-root-id so tools don't hack real filesystem
	SANDBOX_AUFS_ID=$(grep $FAKEROOT /proc/mounts | sed 's/.*si=/si_/; s/ .*//') #'
	sed -i -e '/AUFS_ROOT_ID/ d; /LXC_ID/ d' $FAKEROOT/$BOOTSTATE_PATH 2> /dev/null
	echo AUFS_ROOT_ID=$SANDBOX_AUFS_ID >> $FAKEROOT/$BOOTSTATE_PATH
	echo LXC_ID=$LXC_ID >> $FAKEROOT/$BOOTSTATE_PATH

	# 9. sandbox is ready, now just need to mount other supports - pts, proc, sysfs, usb and tmp
	mkdir -p $FAKEROOT/$SANDBOX_IMG
	mount -o bind $SANDBOX_IMG $FAKEROOT/$SANDBOX_IMG	# so we can access it within sandbox

	# old, insecure desktop emulation, no longer use this
	# 10. desktop emulation preparation. We copy files regardless because /etc/profile always tries to start xwin
	#cp /usr/share/sandbox/lxc/xwin $FAKEROOT/usr/bin
	#cp /usr/share/sandbox/lxc/wmexit $FAKEROOT/usr/bin
	#LXC_SHARE_IPC=""
	#if [ "$USE_DESKTOP" ]; then
	#	LXC_SHARE_IPC="--share-ipc=1" # share ipc from root container
	#	mount -o bind /tmp $FAKEROOT/tmp		
	#	xauth add ${LXC_ID}/unix:0 $(xauth list 2> /dev/null | awk "/^$(hostname)\/unix/"' {print $2, $3; exit}') 2> /dev/null
	#	echo "[ -z \"\$DISPLAY\" ] && export DISPLAY=$DISPLAY" >> $FAKEROOT/etc/shinit
	#	rm $FAKEROOT/etc/.XLOADED 2> /dev/null	# hack because of Xephyr always crash the first time			
	#fi
	
	# 11. make sure we identify ourself as in sandbox
	sed -i -e '/^PS1/ s/^.*$/PS1="sandbox-lxc'${SANDBOX_ID}'# "/' $FAKEROOT/etc/profile # other puppies	
	grep -q "export PS1=" $FAKEROOT/etc/shinit &&
	sed -i -e "s/PS1=.*/PS1='sandbox-lxc${SANDBOX_ID}# '/" $FAKEROOT/etc/shinit ||
	echo -e '\nexport PS1="sandbox-lxc'${SANDBOX_ID}'# "' >> $FAKEROOT/etc/shinit #fatdog 600
	grep -q "export TERM=" $FAKEROOT/etc/shinit &&
	sed -i -e "s/TERM=.*/TERM=$TERM/" $FAKEROOT/etc/shinit ||
	echo "export TERM=$TERM" >> $FAKEROOT/etc/shinit  # default TERM is screwed, so use host's
	
	# hostname
	sed -i -e "/${LXC_ID}/ d" $FAKEROOT/etc/hosts
	echo "127.0.0.1 ${LXC_ID}" >> $FAKEROOT/etc/hosts
	echo "${LXC_ID}" > $FAKEROOT/etc/hostname
	# make sure we can shutdown, remove the acpi_poweroff tests
	sed -i -e '/acpi_poweroff/d' $FAKEROOT/sbin/poweroff $FAKEROOT/sbin/reboot

	# 12. prepare init config
if [ -z "$OS_IMAGE" ]; then
	rm $FAKEROOT/sbin/init; cp $BASELINE_MOUNT/sbin/init $FAKEROOT/sbin/init
	cp /usr/share/sandbox/lxc/inittab $FAKEROOT/etc/inittab

	# 13. make minimal /dev
	rm -rf $FAKEROOT/dev; mkdir $FAKEROOT/dev
	mknod $FAKEROOT/dev/tty c 5 0; mknod $FAKEROOT/dev/console c 5 1; mkdir $FAKEROOT/dev/pts
	for dev in "null-3" "zero-5" "full-7" "random-8" "urandom-9"; do
		mknod $FAKEROOT/dev/${dev%-*} c 1 ${dev#*-}
	done
fi ### if OS_IMAGE
	
	# 14. setup when running with user namespaces 
	if [ -z "$OS_IMAGE" ] && [ "$IDMAP" ] && zcat /proc/config.gz | grep -q CONFIG_USER_NS=y; then
		echo User namespaces enabled, running as uid/gid $LXC_UID/$LXC_GID
		LXC_AUTO_MOUNT="-s lxc.mount.auto='proc cgroup sys'"
		LXC_UID_MAP="-s lxc.id_map='u 0 $LXC_UID $LXC_IDRANGE'"
		LXC_GID_MAP="-s lxc.id_map='g 0 $LXC_GID $LXC_IDRANGE'"
		LXC_DROP_CAP=
		chown -Rh $LXC_UID:$LXC_GID $FAKEROOT/dev
		if ! [ -e $FAKEROOT/etc/rc.d/rc.sysinit.lxc ]; then
			#rm -rf $FAKEROOT/root; cp -a /root $FAKEROOT
			#rm -rf $FAKEROOT/home; cp -a /home $FAKEROOT
			chown     $LXC_UID:$LXC_GID $FAKEROOT/ $FAKEROOT/home
			chown -Rh $LXC_UID:$LXC_GID $FAKEROOT/etc $FAKEROOT/root
		fi
	else
		echo *Not* using user namespaces, dropping capabilities
		LXC_AUTO_MOUNT="-s lxc.mount.auto='proc cgroup sys'"
		LXC_UID_MAP= 
		LXC_GID_MAP=
		LXC_DROP_CAP="-s lxc.cap.drop='sys_module sys_time sys_rawio syslog'"
	fi

	# 14.1 network connection
	prepare_network

	# 14.2 post-process if we use desktop
	if [ $USE_DESKTOP ]; then
		XSERVER_PID=
		# get free display number
		for disp in $(seq 1 10); do
			[ ! -e /tmp/.X${disp}-lock ] && break
		done
		Xephyr :$disp -listen tcp -ac -host-cursor -screen $WINDOW_SIZE -keybd ephyr,,xkbmodel=evdev,xkbrules=evdev > /dev/null 2>&1 &
		XSERVER_PID=$!
		cp /usr/bin/xwin $FAKEROOT/usr/bin
		echo "DISPLAY='$HOST_IP:$disp'" > $FAKEROOT/etc/X11/host-x11 # save DISPLAY from host
		sed -i -e '\_xinit $HOME/.xinitrc_ s|xinit|. /etc/X11/host-x11; export DISPLAY;|' $FAKEROOT/usr/bin/xwin
		cp /usr/lib64/uml/wmexit $FAKEROOT/usr/bin # we use UML's wmexit
	else
		> $FAKEROOT/usr/bin/xwin
		> $FAKEROOT/usr/bin/wmexit
	fi

	# 15. create blank sysinit/shutdown scripts for later customisation
if [ -z "$OS_IMAGE" ]; then	
	if ! [ -e $FAKEROOT/etc/rc.d/rc.sysinit.lxc ]; then
		touch $FAKEROOT/etc/rc.d/rc.sysinit.lxc; chmod +x $FAKEROOT/etc/rc.d/rc.sysinit.lxc		
	fi
	if ! [ -e $FAKEROOT/etc/rc.d/rc.shutdown.lxc ]; then
		touch $FAKEROOT/etc/rc.d/rc.shutdown.lxc; chmod +x $FAKEROOT/etc/rc.d/rc.shutdown.lxc		
	fi

	# 15.1 append oneshot commands
	# TODO: Would it be better to use lxc.init.cmd?
	if [ $# -gt 0 ]; then
		{	echo "#!/bin/sh" 
			for p in "$@"; do printf "'%s' " "$p"; done
			echo
		} > $FAKEROOT/etc/rc.d/rc.oneshot
		chmod +x $FAKEROOT/etc/rc.d/rc.oneshot
	else
		rm -f $FAKEROOT/etc/rc.d/rc.oneshot
		sed -i -e '/rc.oneshot/d' $FAKEROOT/etc/inittab 
	fi
fi ### if OS_IMAGE

	#  We're good to go!
	echo "Starting sandbox (id: ${LXC_ID}) now."
	# TODO: consider to use lxc.environment to pass "boot parameter" to init
	eval lxc-start -d -f /etc/lxc/default.conf -n $LXC_ID  \
				-s lxc.devttydir=lxc -s lxc.pts=10 -s lxc.tty=4 \
				-s lxc.rootfs=$FAKEROOT \
				-s lxc.utsname=${LXC_ID} -s lxc.network.type=veth -s lxc.network.veth.pair=${LXC_ID} \
				-s lxc.cgroup.devices.deny=a -s lxc.cgroup.devices.allow=\"c *:* rwm\" \
				-s lxc.autodev=0 $LXC_AUTO_MOUNT \
				$LXC_SHARE_IPC $LXC_SHARE_SOUND $LXC_DROP_CAP $LXC_UID_MAP $LXC_GID_MAP \
				$LXC_EXTRA # -o /tmp/lxc-err.log -l TRACE
				#-- busybox getty -n -l /bin/autologin 38400 tty1
	sleep 1		# allow some time for container to start

	# Post-process if we use network
	if [ "$USE_NETWORK" ]; then
		# wait for the thing to show up
		while ! iplink show $LXC_ID > /dev/null 2>&1; do
			sleep 0.5
		done
		start_network
	fi

	# start console and restart until container is terminated
	set -m # we need console's pid so we can kill it later		
	while true; do
		lxc-console -q -e g -n "$LXC_ID" -t 1 &
		CONSOLE_PID=$!
		fg %%
		lxc-info -n $LXC_ID >/dev/null 2>&1 || break
	done

	# 16. done - clean up everything 
	umountall
	echo "Leaving sandbox."
else
	echo "Unable to mount aufs br:$SANDBOX_IMG=rw$robranches"
	umount -l $SANDBOX_IMG	
fi
