#!/bin/dash
# Tether to a bluetooth phone (specifically, Android phones)
# This is CLI so that it can be used on non-graphical systems
#
# Copyright (C) James Budiono 2015, 2018
# License: GNU GPL Version 3 or later.
#
# Version 2: Support bluez 5.x
#

TEXTDOMAIN=fatdog
. gettext.sh
Back="$(gettext 'Back')"
OK="$(gettext 'OK')"
CANCEL="$(gettext 'Cancel')"

### configuration
APPTITLE="$(gettext 'Bluetooth Tethering')"
XTERM="defaultterm"
BT_CONFIG=/var/lib/bluetooth # /etc/bluetooth/var in older fatdogs
ANS_FILE=/tmp/ans.$$          # output from dialog

############ helpers ##############
trap 'cleanup; exit;' 0 HUP INT TERM

cleanup() {
	[ -f /etc/resolv.conf.blue-backup ] &&
	mv /etc/resolv.conf.blue-backup /etc/resolv.conf
	[ $XPID ] && kill $XPID
	[ $XPID2 ] && kill $XPID2
	rm -f $ANS_FILE 
}

### make our life with dialog a bit easier
# input: $@, output: $ans, $?
dlg() {
	local ret
	dialog --ok-button "$OK" --cancel-button "$CANCEL" --backtitle "$APPTITLE" "$@" 2> $ANS_FILE; ret=$?
	ans=$(cat $ANS_FILE) # because ans can be multiple lines
	return $ret
}

### $*-msg to display
msgbox() {
	dlg --msgbox "$*" 0 0
}

### $*-msg to display
infobox() {
	dlg --infobox "$*" 0 0
}

### make sure we run as root - input $@
run_as_root() {
	if [ $(id -u) -ne 0 ]; then
		if [ $DISPLAY ]; then
			exec gtksu "$APPTITLE" "$0" "$@"
		else
			exec su -c "$0 $*"
		fi
	fi
}

### make sure we run in terminal - input: $@
run_in_terminal() {
	# check we're running in terminal, if not, re-exec
	if ! test -t 0; then
		# not on terminal, re-exec on terminal
		if [ $DISPLAY ]; then
			exec $XTERM -e "$0" "$@"
		else
			exit
		fi
	fi
}

# $1-adapter
get_bt_devices() {
	ls -d /var/lib/bluetooth/$1/* | sed "s|/var/lib/bluetooth/$1/||; /:/!d" | tee /dev/pts/12
}

#################### 


# returns adapter
ui_choose_adapter() {
	local adapters="" count=0 ans=""
	
	for p in $(hcitool dev | awk '{print $2}'); do
		adapters="$adapters $p $p"
		: $((count=count+1))
		ans=$p
	done

	if [ $count -lt 1 ]; then
		msgbox "No bluetooth adapter is detected."
		exit 1
	fi

	while true; do
		eval dlg --no-tags --menu "'"$(gettext 'Choose adapters to use ')"'" 0 0 0 \
		$adapters \
		exit "'$(gettext 'Exit')'" || exit 0
		case $ans in
			exit) exit 0 ;;
			*) break;
		esac
	done
	adapter=$ans
}

#$1-adapter, return device
ui_choose_devices() {
	local devices

	devices=$(ls -d /var/lib/bluetooth/*/* | sed 's|/var/lib/bluetooth/.*/||; /:/!d')
	devices=$(for p in $devices; do
	awk -v mac=$p -F= '/^Name=/ { print "'\''"mac"'\''", "'\''"$2, "(" mac ")'\''" }' /var/lib/bluetooth/*/$p/info
	done)

	while true; do
		device=""
		eval dlg --no-tags --menu "'Choose devices to connect to. If your device \
is not listed here, add it via bluetooth applet first.'" 0 0 0 \
		$devices \
		back "$Back" || return 1
		case $ans in
			back) return 1 ;;
			*) device=$ans; break ;;
		esac
	done
}

#$1-iface
get_ip() {
	while ! ip link show $1 > /dev/null 2>&1; do
		sleep 1
	done
	udhcpc -qi $1 >> $ANS_FILE 2>&1 $ANS_FILE
	{
	echo "===============" 
	echo "Connection is now ready."
	echo "Keep this window open to maintain connection."
	echo "Press EXIT to shutdown." 
	} >> $ANS_FILE
}

#$1-adapter, $2-device
connect_now() {	
	cp /etc/resolv.conf /etc/resolv.conf.blue-backup
	msgbox "Ready?\nPress OK to continue. Remember to turn on tethering on your phone first."
	echo "Connecting ... please accept request on your phone." > $ANS_FILE
	bt-network -a $1 -c $2 nap >> $ANS_FILE 2>&1 &	
	XPID=$!
	get_ip bnep0 &
	XPID2=$!
	dialog --tailbox $ANS_FILE 15 60
	kill $XPID $XPID2
	mv /etc/resolv.conf.blue-backup /etc/resolv.conf
	XPID="" XPID2=""
	return 0
}


### main
run_as_root
run_in_terminal
while true; do
	ui_choose_adapter &&
	ui_choose_devices $adapter &&
	connect_now $adapter $device
done


