#!/bin/dash
# Login Manager - sets login mode for Fatdog
#
# Copyright (C) James Budiono 2012, 2013, 2016, 2019
# License: GNU GPL Version 3 or later
#
# Note: must run as root
#
# i18n Feb 2015 by L18L

export TEXTDOMAIN=fatdog

### configuration
APPTITLE="$(gettext 'Fatdog64 Login Manager')"
LOGIN_MANAGER=/etc/X11/loginmanager

### L18L translations
# we change labels of all languages for jwm_menu_create
trans=''; 
for L in $(ls /usr/share/locale/*/LC_MESSAGES/jwm 2> /dev/null | cut -d'/' -f5); do 
	trans="$trans $L"; 
done
# labels
Logout="$(gettext 'Logout')"                     # graphical login  
Exit_to_prompt="$(gettext 'Exit to prompt')"     # jwm_menu_create
Quit_X_server="$(gettext 'Quit X server')"       # lx_panel
Restart_X="$(gettext 'Restart X')"               # razor-panel  
Restart_X_server="$(gettext 'Restart X server')" # openbox

######### helpers ########

add_logout_text() {   
	# update xdg menu
	sed -i -e '/NoDisplay/d' /usr/share/applications/system/logout.desktop
	echo "NoDisplay=true" >> /usr/share/applications/system/restartx.desktop
	
	# change "Quit X" in lxpanel
	for p in /usr/share/lxpanel/profile/default/panels/panel $USERHOME/.config/lxpanel/default/panels/panel; do
        if [ -f "$p" ]; then
	 		sed -i -e 's/name=Quit X server/name=Logout/; s/wmexit terminal/wmexit logout/' $p
	        sed -i -e "s/name=${Quit_X_server}/name=${Logout}/; s/wmexit terminal/wmexit logout/" $p
        fi
	done	

	# change "Exit to prompt" in jwm
	sed -i -e 's/-Exit to prompt/-Logout/; s/wmexit terminal/wmexit logout/' /usr/bin/jwm_menu_create
    sed -i -e "s/-${Exit_to_prompt}/-${Logout}/" /usr/bin/jwm_menu_create
	LANGUAGE_orig="$LANGUAGE" # just in case...
    for L in $(echo $trans); do 
		f=/usr/share/locale/${L}/LC_MESSAGES/jwm; 
		sed -i -e "/jwm_Exit=/d" $f; 
		export LANGUAGE=$L; 
		echo "jwm_Exit=\"$(gettext 'Logout')\"" >> $f; 
	done
	export LANGUAGE="$LANGUAGE_orig"	
}

remove_logout_text() {
	# update xdg menu
	sed -i -e '/NoDisplay/d' /usr/share/applications/system/restartx.desktop
	echo "NoDisplay=true" >> /usr/share/applications/system/logout.desktop
		
	# undo logout lxpanel menu - back to Quit X
	for p in /usr/share/lxpanel/profile/default/panels/panel $USERHOME/.config/lxpanel/default/panels/panel; do
        if [ -f "$p" ]; then
			sed -i -e 's/name=Logout/name=Quit X server/; s/wmexit logout/wmexit terminal/' $p
	        sed -i -e "s/name=${Logout}/name=${Quit_X_server}/" $p
        fi
	done

	# undo logout jwm menu - back to Exit to prompt
	sed -i -e 's/-Logout/-Exit to prompt/; s/wmexit logout/wmexit terminal/' /usr/bin/jwm_menu_create
	sed -i -e "s/-${Logout}/-${Exit_to_prompt}/" /usr/bin/jwm_menu_create
	LANGUAGE_orig="$LANGUAGE" # just in case...
    for L in $(echo $trans); do
		f=/usr/share/locale/${L}/LC_MESSAGES/jwm;
		sed -i -e "/jwm_Exit=/d" $f;
		export LANGUAGE=$L;
		echo "jwm_Exit=\"$(gettext 'Exit to prompt')\"">>$f;
	done
	export LANGUAGE="$LANGUAGE_orig"	
}

enable_autologin() {
	sed -i -e '/^tty1:/ s|respawn:.*|respawn:/sbin/getty -n -l /bin/autologin 38400 tty1|' /etc/inittab
}

disable_autologin() {
	sed -i -e '/^tty1:/ s|respawn:.*|respawn:/sbin/getty 38400 tty1|' /etc/inittab
}

remove_graphical_login() {
	[ -e $LOGIN_MANAGER ] && chmod -x $LOGIN_MANAGER
	sed -i -e "\|$LOGIN_MANAGER|d" /etc/inittab
}

add_graphical_login() {
	if ! [ -e $LOGIN_MANAGER ]; then
		# default login manager is slim
		> $LOGIN_MANAGER cat << "EOF"
#!/bin/dash
#Note: this script must finish
exec /usr/bin/slim -d
EOF
	fi
	chmod +x $LOGIN_MANAGER
	echo "::sysinit:$LOGIN_MANAGER" >> /etc/inittab
}

get_current_login_mode() {
	MODE="console"
	[ -e $LOGIN_MANAGER ] && [ -x $LOGIN_MANAGER ] && MODE="graphical"
	grep -q autologin /etc/inittab && MODE="autologin"
}

#$1-mode 
set_login_mode() {
	case $1 in
		console)	disable_autologin; add_logout_text; remove_graphical_login ;;
		graphical)	disable_autologin; add_logout_text; remove_graphical_login; add_graphical_login ;;
		autologin)	enable_autologin;  remove_logout_text; remove_graphical_login ;;
	esac
}

########### main #############
export USERHOME=$(awk -F: -v user=$USER '$1==user {print $6}' /etc/passwd) # the user's home
[ $(id -u) -ne 0 ] && exec gtksu "$APPTITLE" "$0" "$@"

get_current_login_mode	# return MODE

status_console=off status_graphical=off status_autologin=off
eval status_$MODE=on

TEXT="$(eval echo $(gettext 'Current login mode is: $MODE.'))
$(gettext 'Please choose the new login mode.') 
$(gettext 'Takes effect at next boot.')"
if NEWMODE=$(Xdialog --stdout --title "$APPTITLE" --no-tags --radiolist "$TEXT" 16 80 0 \
	autologin "$(gettext 'Autologin - no password necessary')" $status_autologin \
	graphical "$(gettext 'Graphical login')" $status_graphical \
	console "$(gettext 'Console login')" $status_console); then
	if [ $NEWMODE != $MODE ]; then
		set_login_mode $NEWMODE
		Xdialog --title "$APPTITLE" --msgbox "$(eval echo $(gettext 'Login mode has been changed to $NEWMODE and will take effect at next boot.'))" 0 0 10000
		[ $MODE = "autologin" ] && echo -e '\a' > /dev/console && 
		Xdialog --title "$(gettext 'Warning')" --msgbox "$(gettext 'You are disabling autologin.')
$(gettext 'At next boot, you are required to provide a password before you can login to the system.')
$(gettext 'Make sure you know the password for the user you are going to use, and the root password too.')
$(gettext "If you don't know, you can set the passwords now using the User Manager from Control Panel.")
$(gettext "If you have not changed anything, the default root password is 'woofwoof'.")" 0 0 90000
		exit
	fi
fi
Xdialog --title "$APPTITLE" --msgbox "$(gettext 'Nothing is changed.') $(eval echo $(gettext 'The login mode stays as $MODE'))". 0 0 10000
