#!/bin/ash
# set system timezone (/etc/localtime)
# Copyright (C) James Budiono 2012, 2015
# License: GNU GPL Version 3 or later
# re-written from scratch for Fatdog64 600 
# 131026 internationalisation by L18L
#
# 2015-03-11: Updated to support per-user timezone
# $1-system or user (default is system)

# std localisation stanza
export TEXTDOMAIN=fatdog
. gettext.sh
# performance tweak - use "C" if there is no localisation
! [ -e $TEXTDOMAINDIR/${LANG%.*}/LC_MESSAGES/$TEXTDOMAIN.mo ] &&
! [ -e $TEXTDOMAINDIR/${LANG%_*}/LC_MESSAGES/$TEXTDOMAIN.mo ] && LANG=C

### configuration
APPTITLE="$(gettext 'Set Timezone')"
HWCLOCKPARM="--localtime" # default
HWCLOCKCONF=/etc/hwclock.conf
[ -e $HWCLOCKCONF ] && . $HWCLOCKCONF
TZ_CONFIG=$FATDOG_STATE_DIR/timezone

### helpers
# get sorted list of available timezones (excluding those at the root of /usr/share/zoneinfo)
available_timezones() {
	for p in $(LANG=C ls -d /usr/share/zoneinfo/*); do
		if [ -d $p ]; then
			find $p -type f | sed 's%/usr/share/zoneinfo/%%' | LANG=C sort
		fi
	done	
}

# convert sorted list of anything to a treeview format suitable for Xdialog
# stdin - the list, $1 - the separator
build_treeview() {
	awk -F"$1" '
{ 
	for (i=1; i <= NF; i++) {
		if (level[i] != $i) {
			print $0, $i, "off", i-1
			level[i]=$i
			for (k=i+1; k <= NF; k++) level[k]=""
		}
	}
}
'
}

# change system timezone: /etc/localtime symlink and kernel timezone
set_system_timezone() {
	[ $(id -u) -ne 0 ] && exec gtksu "$APPTITLE" "$0" system
	CURRENT_TIMEZONE=$(	readlink /etc/localtime 2> /dev/null | sed 's%/usr/share/zoneinfo/%%')

	if choice=$(Xdialog --title "$APPTITLE" --stdout --treeview "$(eval_gettext 'Current system time zone: $CURRENT_TIMEZONE')" 20 50 10 $(available_timezones | build_treeview "/")); then
		ln -sfT /usr/share/zoneinfo/$choice /etc/localtime
		hwclock --hctosys $HWCLOCKPARM
		systz auto # change kernel timezone too
		
		# first-run: don't ask and don't restart
		if [ -z $FIRST_RUN ] && Xdialog --title "$APPTITLE" --yesno "$(eval_gettext 'System time zone set to $choice.')
	$(gettext 'This setting is stored as symlink in /etc/localtime.')
	$(gettext 'You need to restart X server; do you want to restart it now?')" 0 0; then
			restartwm
		fi
	else
		Xdialog --title "$APPTITLE" --infobox "$(gettext 'Cancelled, nothing is changed.')" 0 0 10000
	fi
}

# update entry in $FATDOG_STATE_DIR/timezone
set_user_timezone() {
	# get the current timezone, default to system timezone
	SYSTEM_TIMEZONE="$( readlink /etc/localtime 2> /dev/null | sed 's%/usr/share/zoneinfo/%%')"
	[ -e $TZ_CONFIG ] && read CURRENT_TIMEZONE < $TZ_CONFIG
	CURRENT_TIMEZONE=${CURRENT_TIMEZONE#:}
	CURRENT_TIMEZONE=${CURRENT_TIMEZONE:-$SYSTEM_TIMEZONE "[System]"}

	if choice=$(Xdialog --title "$APPTITLE" --stdout --treeview "$(eval_gettext 'Current user time zone: $CURRENT_TIMEZONE')" 25 70 10 \
		system "$(gettext 'Remove user settings and use system timezone')" off 0 \
	    $(available_timezones | build_treeview "/")); then
	    case $choice in
			system)
				rm -f $TZ_CONFIG
				TEXT="$(gettext 'User timezone settings removed.')
				      $(eval_gettext 'Time zone will revert to system timezone: $SYSTEM_TIMEZONE')"
				;;
			*)
				echo ":$choice" > $TZ_CONFIG
				TEXT="$(eval_gettext 'User time zone set to $choice.')
				      $(eval_gettext 'This setting is stored in $TZ_CONFIG.')"
				;;
	    esac
		
		if Xdialog --title "$APPTITLE" --yesno "$TEXT
	        $(gettext 'You need to restart X server; do you want to restart it now?')" 0 0; then
			restartwm
		fi
	else
		Xdialog --title "$APPTITLE" --infobox "$(gettext 'Cancelled, nothing is changed.')" 0 0 10000
	fi	
}

### main
case "$1" in
	user) set_user_timezone ;;
	system|"") set_system_timezone ;;
	-h|--help)
		printf "Usage: ${0##*/} [system|user]\nDefault is system if not specified.\n"
esac



