/*
 * kmouse.c -- offer vc-independent mouse support as devices 
 *
 * Copyright (C) 1996 Alessandro Rubini (rubini@ipvvis.unipv.it)
 *
 * Part of protocol decoding comes from selection, by Andrew Haylett
 * The idea of the kernel module is by Peter Orbaek
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/


#ifndef __KERNEL__
#define __KERNEL__
#endif

/* By now, it's only a module. Time will tell... */

#ifndef MODULE
#define MODULE
#endif

#include <linux/module.h>

#include <linux/types.h>   /* ulong and friends */
#include <linux/sched.h>   /* current, task_struct, other goodies */
#include <linux/fcntl.h>   /* O_NONBLOCK etc. */
#include <linux/errno.h>   /* return values */
#include <linux/config.h>  /* system name and global items */
#include <linux/malloc.h>  /* kmalloc, kfree */
#include <linux/kdev_t.h>  /* building majors */
#include <linux/major.h>   /* checking who's speaking */
#include <linux/tqueue.h>  /* running task queues */
#include <linux/tty.h>     /* line disciplines */
#include <linux/termios.h> /* N_MOUSE -- it was already there... */

#include <linux/mouse.h>
#include <linux/../../drivers/char/selection.h> /* Hmmm.... not nice */

#include <gpm.h> /* gpm is needed, at least to define the interface */

#include "kmouse.h" /* kmouse interface */




#define inline


/* options are all relative to the current mouse */
#define opt_sequence (mPtr->sequence)

#define opt_three (mPtr->three)
#define DEFAULT_THREE 0

#define opt_delta (mPtr->delta)
#define DEFAULT_DELTA 25

#define opt_accel (mPtr->accel)
#define DEFAULT_ACCEL 2

#define opt_scale (mPtr->scale)
#define opt_scaley (mPtr->scaley)
#define DEFAULT_SCALE 10

#define opt_time (mPtr->time)
#define DEFAULT_TIME 250

#define opt_tap (mPtr->tap) /* unused, yet */
#define DEFAULT_TAP 1

#define opt_ptrdrag (mPtr->ptrdrag)
#define DEFAULT_PTRDRAG 1

/*===================================================== data structures */

/*
 * Data stolen but waiting to be dispatched. It is still device-specific.
 * This structure is missing the various options...
 */

#define KMOUSE_BUFFER 32
struct kmouse_mouse {
    kdev_t dev;
    struct kmouse_mouse *next;
    struct kmouse_protocol *proto;
    int decoded;
    short errors;
    short missing;
    short head;
    short tail;
    char  data[KMOUSE_BUFFER];
    char sequence[8];
    unsigned short three;
    unsigned short delta, accel;
    unsigned short scale, scaley;
    unsigned short time, tap;
    unsigned short ptrdrag;

};

static void kmouse_mouse_init(struct kmouse_mouse *mPtr, kdev_t dev) {
    memset(mPtr,0,sizeof(struct kmouse_mouse));
    mPtr->dev=dev;
    opt_three = DEFAULT_THREE;
    opt_delta = DEFAULT_DELTA;
    opt_accel = DEFAULT_ACCEL;
    opt_scale = opt_scaley = DEFAULT_SCALE;
    opt_time = DEFAULT_TIME;
    opt_tap = DEFAULT_TAP;
    opt_ptrdrag = DEFAULT_PTRDRAG;
}

Static struct kmouse_mouse *kmouse_mlist=NULL;
Static struct kmouse_mouse *kmouse_lastmouse=NULL;

/*
 * Data ready to be retrieved by applications. It is filp-specific
 */

struct event_list {
    Gpm_Event event;
    struct event_list *next;
};

#define KM_FL_RAW  0x0001

struct kmouse_filp {
    struct file *filp;
    struct kmouse_filp *next;   
    struct wait_queue *queue;
    Gpm_Connect conn;
    unsigned short nitems;
    unsigned short flags;
    struct event_list *first;            /* where to dequeue from */
    struct event_list *last;             /* where to enqueue */
};


Static struct kmouse_filp *filp_first=NULL;
Static struct kmouse_filp *filp_defhn=NULL; /* the first default handler */
Static struct kmouse_filp *filp_last=NULL;  /* last used -- chronologically */


/*===================================================== decoding protocols */

Static int M_bare(Gpm_Event *state,  unsigned char *data)
{
    /* a bare ms protocol */
    state->buttons= ((data[0] & 0x20) >> 3) | ((data[0] & 0x10) >> 4);
    state->dx=      (char)(((data[0] & 0x03) << 6) | (data[1] & 0x3F));
    state->dy=      (char)(((data[0] & 0x0C) << 4) | (data[2] & 0x3F));
    return 0;
}

Static int M_msc(Gpm_Event *state,  unsigned char *data)
{
    /* msc protocol (5 bytes) */
    state->buttons= (~data[0]) & 0x07;
    state->dx=      (char)(data[1]) + (char)(data[3]);
    state->dy=     -((char)(data[2]) + (char)(data[4]));
    return 0;
}

Static int M_sun(Gpm_Event *state,  unsigned char *data)
{
    state->buttons= (~data[0]) & 0x07;
    state->dx=      (char)(data[1]);
    state->dy=     -(char)(data[2]);
    return 0;
}

Static int M_logi(Gpm_Event *state,  unsigned char *data)
{
    state->buttons= data[0] & 0x07;
    state->dx=      (data[0] & 0x10) ?   data[1] : - data[1];
    state->dy=      (data[0] & 0x08) ? - data[2] :   data[2];
    return 0;
}
    
Static int M_ps2(Gpm_Event *state,  unsigned char *data)
{
    state->buttons=
        !!(data[0]&1) * GPM_B_LEFT +
        !!(data[0]&2) * GPM_B_RIGHT +
        !!(data[0]&4) * GPM_B_MIDDLE;

    state->dx=   (data[0] & 0x10) ? data[1]-256 : data[1];
    state->dy= -((data[0] & 0x20) ? data[2]-256 : data[2]);
    return 0;
}

/*
 * Busmice return a whole packet at a time, and are all equal.
 * For serial mice, autodetection works for: bare/mman, msc, ps2, logi/mm
 * ms with third button and sun can't be detected.
 */                          

Static struct kmouse_protocol protocols[]={
#if 0
   { PROTO_MMAN,     3, {0x40, 0x40, 0x40, 0x00}, M_mman, 0},
#endif
   { PROTO_BARE,     3, {0x40, 0x40, 0x40, 0x00}, M_bare, 0},
   { PROTO_MSC,      5, {0xf8, 0x80, 0x00, 0x00}, M_msc,  0},
   { PROTO_SUN,      3, {0xf8, 0x80, 0x00, 0x00}, M_sun,  0},
   { PROTO_LOGI,     3, {0xe0, 0x80, 0x80, 0x00}, M_logi, 0},
   { PROTO_PS2,      3, {0xc0, 0x00, 0x00, 0x00}, M_ps2,  1}, /* busmouse */
#if 0
   { PROTO_MS,       3, {0x40, 0x40, 0x40, 0x00}, M_ms,   0},
#endif
   { 0,              0, {0x00, 0x00, 0x00, 0x00}, NULL,   0}
};


Static int statusX,statusY,statusB; /* to return info */
Static int statusC=0; /* clicks */
Static int con_mode, con_fg, con_shift, con_wid, con_hei; /* console info */
Static struct tty_struct *con_tty;
Static kdev_t button_owner;  /* multi-mouse: button-grab - not implem. yet */

Static inline void kmouse_decode(struct kmouse_mouse *mPtr,
                                 Gpm_Event *event)
{
    int n=mPtr->head-mPtr->tail;        /* bytes to process */
    static unsigned long tv1, tv2;      /* jiffies, to check double clicks */
    static int fine_dx, fine_dy;        /* incremental values */
    static int i, m;
    static Gpm_Event nEvent;            /* new one */
    static short newB=0, oldB=0, oldT=0;/* old values to chain events */
    static unsigned long ljiffies;      /* last invocation */
    struct kmouse_protocol *p;

    if (mPtr->errors > 10 && mPtr->errors > mPtr->decoded)
        mPtr->proto=NULL;

    if (mPtr->proto == NULL) { /* new mouse: autodetect */
        if (n<2) {
            mPtr->missing=2-n;
            return;
        }
        ljiffies=jiffies;
        for (p=protocols; p->fun; p++) {
            if ( (mPtr->data[mPtr->tail]&p->proto[0])
                != p->proto[1])
                continue;
            if ( (mPtr->data[mPtr->tail+1]&p->proto[2])
                != p->proto[3])
                continue;
            PDEBUG("found proto: %i\n",p->protoid);
            if (p->fun && p->busmouse && MAJOR(mPtr->dev)!=MOUSE_MAJOR) {
                PDEBUG("Impossible...\n");
                mPtr->proto=NULL; mPtr->tail = mPtr->head = 0;
                mPtr->missing = 2;
                return;
            }
            mPtr->missing=p->howmany - n;
            mPtr->proto=p;
            break;
        }
    }
    
    
    /* decode all the packets in the list */
    
    PDEBUGG("%i bytes to eat\n",n);
    event->dx = event->dy = 0;
    while (n >= mPtr->proto->howmany) {
        
        /* re-align to the byte stream */
        if ( (mPtr->data[mPtr->tail] & mPtr->proto->proto[0])
            != mPtr->proto->proto[1]) {
            if (mPtr->data[mPtr->tail] || mPtr->decoded) {
                PDEBUG("not head %2x? (e=%i,d=%i)\n",
                       mPtr->data[mPtr->tail],mPtr->errors,mPtr->decoded);
                mPtr->errors++;
            }
            n--; mPtr->tail++; /* skip silently a leading 0 */
            continue;
        }
        if ( (mPtr->data[mPtr->tail+1] & mPtr->proto->proto[2])
            != mPtr->proto->proto[3]) {
            PDEBUG("data packet?\n");
            n -= mPtr->proto->howmany-1;
            mPtr->errors++;
            mPtr->tail += mPtr->proto->howmany-1;
            continue;
        }
        
        oldT=event->type;
        (*mPtr->proto->fun)(&nEvent, &(mPtr->data[mPtr->tail]));
        
        mPtr->tail += mPtr->proto->howmany;
        n -= mPtr->proto->howmany;
        mPtr->decoded++;
        
        PDEBUGG("decoded at %li: b %i dx %3i dy %3i\n",jiffies,
                nEvent.buttons,nEvent.dx,nEvent.dy);


        /*---------------------------------- hw decoding is over */

        /* FIXME -- absolute/relative (the pen) */

        /* button-related issues: sequence, 2/3, old/new */

        event->buttons = opt_sequence[event->buttons] & 7; 
        if ((event->buttons&GPM_B_MIDDLE) && !opt_three) opt_three++;
        oldB=newB; event->buttons=newB=nEvent.buttons;
        /* FIXME -- place to put button-lock */
                
        /*
         * If raw data is being collected, return it.
         * Use the last client as a reference, this helps in
         * avoiding a few calls when char-cells are used
         */
        if (filp_last && filp_last->flags & KM_FL_RAW) {
            PDEBUGG("raw\n");
            event->dx = nEvent.dx;
            event->dy = nEvent.dy;
        } else {

            /* motion-related issues: delta, accel */
            
            if (abs(nEvent.dx)+abs(nEvent.dy) > opt_delta)
                nEvent.dx*=opt_accel, nEvent.dy*=opt_accel;

            fine_dx += nEvent.dx;           fine_dy += nEvent.dy;
            event->dx += fine_dx/opt_scale; event->dy += fine_dy/opt_scaley;
            fine_dx %= opt_scale;           fine_dy %= opt_scaley;
        }
        
        if (!event->dx && !event->dy && (event->buttons==oldB)) {
            event->type=0;
            continue; /* nothing new, but FIXME */
        }
        event->x += event->dx; event->y += event->dy;
        statusB = event->buttons;
        
        /* shift_state, current console */
        con_fg=con_get_fg_info(&con_mode, &con_shift,
                                 &con_wid, &con_hei, &con_tty);
        event->vc = con_fg+1;
        event->modifiers = con_shift;
        if (filp_last && (filp_last->flags & KM_FL_RAW)
               && filp_last->conn.vc != event->vc) {
            filp_last=NULL;            /* not valid anymore */
            event->dx = event->dy = 0; /* drop hw deltas */
        }
        
        if (oldB == event->buttons)
            event->type = (nEvent.buttons ? GPM_DRAG : GPM_MOVE);
        else
            event->type = (nEvent.buttons > oldB ? GPM_DOWN : GPM_UP);
        
        
        switch(event->type)          /* now provide the cooked bits */
            {
            case GPM_DOWN:
                tv2=jiffies;
                if (tv1 && (tv2-tv1)*1000/HZ < opt_time)
                    statusC++, statusC%=3; /* 0, 1 or 2 */
                else statusC=0;
                event->type|=(GPM_SINGLE<<statusC);
                break;
                
            case GPM_UP:
                tv1=jiffies;
                event->buttons^=oldB; /* for button-up, tell which one */
                event->type|= (oldT&GPM_MFLAG);
                event->type|=(GPM_SINGLE<<statusC);
                break;
                
            case GPM_DRAG:
                event->type |= GPM_MFLAG;
                event->type|=(GPM_SINGLE<<statusC);
                break;
                
            case GPM_MOVE:
                statusC=0;
            default:
                break;
            }
        event->clicks=statusC;
        
        /* The gpm policy is to force the following behaviour:
         * - At buttons up, must fit inside the screen, though flags are set.
         * - At button down, allow going outside by one single step
         */
        
        /* selection uses 1-based coordinates, so do I */
        
        /*
         * Only one margin is current. Y takes priority over X.
         * The i variable is how much margin is allowed.
         * "m" is which one is there.
         */
        
        m = 0;
        i = ((event->type&(GPM_DRAG|GPM_UP))!=0); /* i is boolean */
        
        if (event->y > con_hei)  {
            event->y = con_hei+1-!i; i=0; m = GPM_BOT;
        }
        else if (event->y<=0) {
            event->y=1-i;             i=0; m = GPM_TOP;
        }
        if (event->x > con_wid) {
            event->x = con_wid+1-!i; if (!m) m = GPM_RGT;
        }
        else if (event->x<=0) {
            event->x=1-i;             if (!m) m = GPM_LFT;
        }
        event->margin=m;
        
        PDEBUGG("m %3i %3i (%3i %3i) - b=%i t=%x cl=%i\n",
                event->dx,event->dy, event->x,event->y,
                event->buttons, event->type, event->clicks);
        
        /* update the global state */
        statusX=event->x; statusY=event->y;
    }
    mPtr->missing=mPtr->proto->howmany-n;
    
    
}


/*============================================== the selection mechanism */

Static inline void selection_copy(int x1, int y1, int x2, int y2, int mode)
{
    unsigned short arg[5];
    
    PDEBUGG("sel %i: %i %i %i %i\n",mode,x1,y1,x2,y2);
    
    arg[0]=(unsigned short)x1;
    arg[1]=(unsigned short)y1;
    arg[2]=(unsigned short)x2;
    arg[3]=(unsigned short)y2;
    arg[4]=(unsigned short)mode;
    
    set_selection((long)arg-1, NULL /* tty unused */, 0 /* kernel */);
}

Static  inline int do_selection(Gpm_Event *event)  /* return 0, always */
{
    static int x1=1, y1=1, x2, y2;
    struct kmouse_mouse *mPtr=kmouse_lastmouse;
    
    x2=event->x; y2=event->y;
    switch(GPM_BARE_EVENTS(event->type))
    {
    case GPM_MOVE:
        if (x2<1) x2++; else if (x2>con_wid) x2--;
        if (y2<1) y2++; else if (y2>con_hei) y2--;
        selection_copy(x2,y2,x2,y2,3); /* just highlight pointer */
        return 0;
        
    case GPM_DRAG:
        if (event->buttons==GPM_B_LEFT) {
            if (event->margin) /* fix margins */
                switch(event->margin)
                {
                case GPM_TOP: x2=1; y2++; break;
                case GPM_BOT: x2=con_wid; y2--; break;
                case GPM_RGT: x2--; break;
                case GPM_LFT: y2<=y1 ? x2++ : (x2=con_wid, y2--); break;
                }
            selection_copy(x1,y1,x2,y2,event->clicks);

            if (event->clicks>=opt_ptrdrag && !event->margin) /* pointer */
                selection_copy(x2,y2,x2,y2,3);
        }
        return 0;
        
    case GPM_DOWN:
        switch (event->buttons) 
        {
        case GPM_B_LEFT:
            x1=x2; y1=y2;
            selection_copy(x1,y1,x2,y2,event->clicks); /* start */
            return 0;
            
        case GPM_B_MIDDLE:
            paste_selection(con_tty);
            return 0;
            
        case GPM_B_RIGHT:
            if (opt_three==1)
                selection_copy(x1,y1,x2,y2,event->clicks);
            else
                paste_selection(con_tty);
            return 0;
        }
    }
    return 0;
}


/*============================================== managing the mouse list */


/*
 * Utility functions
 */

Static inline void kmouse_link(struct kmouse_mouse *mPtr)
{
    mPtr->head=mPtr->tail=0;
    mPtr->missing=1;
    mPtr->next = kmouse_mlist ? : mPtr;
    kmouse_mlist=mPtr;
}

Static inline void kmouse_unlink(struct kmouse_mouse *mPtr)
{
    struct kmouse_mouse *ptr=kmouse_mlist;

    if (!ptr) return;
    while (ptr->next != mPtr) {
        ptr=ptr->next;
        if (ptr==kmouse_mlist)
            return; /* not found */
    }
    if (ptr==mPtr) /* last */
        kmouse_mlist=NULL;
    else
        kmouse_mlist=ptr->next=mPtr->next;
}

Static inline struct kmouse_mouse *kmouse_lookfor(kdev_t dev)
{
    struct kmouse_mouse *mPtr=kmouse_mlist;

    if (!mPtr) return NULL;
    if (kmouse_lastmouse && kmouse_lastmouse->dev == dev)
        return kmouse_lastmouse;
    
    while (mPtr->dev!=dev && mPtr->next!=kmouse_mlist) {
        mPtr=mPtr->next;
    }
    if (mPtr->dev != dev)
        return NULL;
    kmouse_lastmouse=mPtr; /* for the next time, and for steal_packet */
    return mPtr;
}

/*
 * An extra mouse is ready, so that it can be 'allocated' from an irq
 */

Static struct kmouse_mouse *kmouse_new=NULL;

/*
 * Insert an event into the client's queue, and perform clustering
 */

Static inline void kmouse_enqueue(Gpm_Event *event, struct kmouse_filp *client)
{
    
    struct event_list *oldEv;
    
    if (client->nitems==0) {
        client->first = client->last = 
            kmalloc(sizeof(struct event_list),GFP_ATOMIC);
        if (!client->first) {
            PDEBUG("No mem for first\n");
            return; /* ENOMEM */
        }
        PDEBUGG("Waking up client %p\n",client);
        client->nitems++;
        client->first->next=NULL;
        client->first->event=*event;
        wake_up_interruptible(&client->queue);
        return;
    }
    
    oldEv= client->last;
    
    if ( (oldEv->event.modifiers == event->modifiers)
        && (oldEv->event.type == event->type) ) {
        PDEBUG("clustering (b %i->%i) (t %x->%x)\n",
               oldEv->event.buttons,event->buttons,
               oldEv->event.type,event->type);

        oldEv->event.x = event->x;
        oldEv->event.y = event->y;
        oldEv->event.dx += event->dx;
        oldEv->event.dy += event->dy;
        return;
    }
    
    /* new event */
    oldEv->next = kmalloc(sizeof(struct event_list),GFP_ATOMIC);
    if (!oldEv->next) {
        PDEBUG("No mem for next\n");
        return; /* ENOMEM */
    }
    client->nitems++;
    client->last = oldEv = oldEv->next;
    oldEv->next = NULL;
    oldEv->event = *event;
    return;
}


/*
 * The task queue is bound to the timer. It dispatches any pending
 * input to the right client queues, and awakes the client.
 * I used the scheduler, first, but mouse drawing was jumpy...
 * Back to the scheduler, since things are fixed.
 */

#define KMOUSE_TQ tq_scheduler

Static struct tq_struct kmouse_tqueue;
Static int kmouse_tq_counter=0;

Static void kmouse_dispatch (void *unused)
{
    struct kmouse_mouse *mPtr=kmouse_mlist;
    struct kmouse_filp *client;
    static Gpm_Event event;
    int n;

    PDEBUGG("Running tqueue\n");

    /* rebuild the new if needed */
    if (kmouse_new==NULL) {
        kmouse_new=kmalloc(sizeof(struct kmouse_mouse),GFP_ATOMIC);
    }
    
    /* scan the list and process data */
    if (!mPtr) return;
    do {
        if (mPtr->head==mPtr->tail)
            continue;
        kmouse_decode(mPtr,&event);
        n=mPtr->head-mPtr->tail;
        if (n==0) 
            mPtr->tail=mPtr->head=0;
        PDEBUGG("left %i (m %i), (h,t) = (%i,%i)\n",n,mPtr->missing,
                mPtr->head,mPtr->tail);
        if (event.type==0) continue; /* nothing */;
        
    }
    while (mPtr=mPtr->next, mPtr!=kmouse_mlist);
    
    /* scan the client list and awake them, or run selection */
    
    for (client=filp_first; client; client=client->next) {
        if (client->conn.vc  && client->conn.vc != event.vc)
            continue; /* different console */
        if (client->conn.minMod & event.modifiers < client->conn.minMod)
            continue; /* too few modifiers */
        if (client->conn.maxMod & event.modifiers > client->conn.maxMod)
            continue; /* too many of them */
        
        /* now, consider the default mask as well */
        if (client->conn.eventMask & event.type == 0)
            if (client->conn.defaultMask & event.type)
                continue;
            else
                break; /* ignore */
        if (filp_last && (filp_last->flags & KM_FL_RAW)
	    && !(client->flags &KM_FL_RAW))
            event.dx = event.dy = 0;

        filp_last=client;
        
        kmouse_enqueue(&event,client);
        
        if (client->conn.defaultMask & event.type)
            if (client->conn.defaultMask & GPM_HARD)
                continue;
        break; /* eaten */
    }
    
    if (!client)
        do_selection(&event);
    
    /* done */
    kmouse_tq_counter--; /* atomic? */
}

/*===================================================== stealing functions */

/*
 * These are called from within an interrupt during normal operation,
 * but from a write() call if faking mice.
 */

Static void kmouse_steal_char(kdev_t dev, unsigned char datum)
{
    struct kmouse_mouse *mPtr;

    /* save it somewhere */
    mPtr=kmouse_lookfor(dev);
    if (!mPtr) {
        PDEBUG("using new mouse (0x%x)\n",dev);
        mPtr=kmouse_new;
        kmouse_new=NULL; /* can't malloc here */
        if (!mPtr) return; /* shouldn't happen */
        kmouse_mouse_init(mPtr,dev);
        kmouse_link(mPtr);
    }
    if (!mPtr->proto && !datum && MAJOR(dev)!=MOUSE_MAJOR) {
        PDEBUGG("ignoring a zero\n");
        return;
    }

    mPtr->data[mPtr->head++]=datum;
    mPtr->missing--;
    if (mPtr->head==KMOUSE_BUFFER) {
        int i, j;
        PDEBUG("Out of buffer, freeing %i\n",mPtr->tail);
        for (i=mPtr->tail, j=0; i<mPtr->head; i++, j++)
            mPtr->data[j]=mPtr->data[i];
        mPtr->head=j;
        mPtr->tail=0;
        mPtr->missing=0;
        kmouse_tq_counter=0; /* force registering */
    }        
    if (mPtr->missing<=0 && kmouse_tq_counter<=0) {
        queue_task(&kmouse_tqueue, &KMOUSE_TQ);
        kmouse_tq_counter=1;
    }
}

Static void kmouse_steal_packet(kdev_t dev, short buttons, short dx, short dy)
{
    struct kmouse_protocol *p;

    PDEBUG("stolen (%i,%03i,%03i) from %x\n",buttons,dx,dy,dev);

    /* enqueue three bytes, use sun protocol (minimum overhead) */
    kmouse_steal_char(dev,  (buttons & 0x07) | 0x80);

    if (kmouse_lastmouse->proto == NULL) { /* first time, tell it's sun */
        for (p=protocols; p->protoid != PROTO_SUN; p++)
            ;
        kmouse_lastmouse->proto = p;
    }
     
    if (dx>127) dx = 127; else if (dx < -127) dx = -127; /* dirty */
    if (dy>127) dy = 127; else if (dy < -127) dy = -127;
    kmouse_steal_char(dev, dx & 0xff);
    kmouse_steal_char(dev, dy & 0xff);
}

Static struct  mouse_steal kmouse_stealers = {
    kmouse_steal_char,
    kmouse_steal_packet
};

/*===================================================== the line discipline */

Static struct tty_ldisc kmouse_ldisc;

Static int kmouse_ldisc_open(struct tty_struct *unused)
{
    MOD_INC_USE_COUNT;
    PDEBUG("open ldisc, count is %li\n",mod_use_count_);
    return 0;
}

Static void kmouse_ldisc_close(struct tty_struct *unused)
{
    MOD_DEC_USE_COUNT;
    PDEBUG("close ldisc, count is %li\n",mod_use_count_);
}

Static int kmouse_ldisc_room(struct tty_struct *unused)
{
    return 64; /* anything */
}

Static void kmouse_ldisc_receive(struct tty_struct *tty,
                                 const unsigned char *cp, char *fp, int count)
{
  int i=0;
  while (i<count)
      kmouse_steal_char(tty->device,cp[i++]);
}


/*===================================================== file operations */

Static int kmouse_seek(struct inode *inode, struct file *filp,
                       off_t offset, int origin)
{ return -ESPIPE;}


Static  int kmouse_open (struct inode *inode, struct file *filp)
{
    /* the f_op field is already right -- mouse.c did it for us */
    struct kmouse_filp *client;
    client=kmalloc(sizeof(struct kmouse_filp),GFP_KERNEL);
    if (!client) return -ENOMEM;
    memset(client, 0, sizeof(struct kmouse_filp));

    if (MINOR(inode->i_rdev)==KMOUSE_DATA_MINOR) {
        kdev_t device;
        if (current->tty) {
            device = current->tty->device;
        } else {
               int con;
               /* damn: X has no control tty.... */
               PDEBUG("open: no ctl tty\n");
               if (current->comm[0]=='X' && current->comm[1]=='\0') {
                   PDEBUG("X!\n");
                   con = con_get_fg_info(NULL,NULL,NULL,NULL,NULL);
                   device=MKDEV(TTY_MAJOR,con+1);
               } else {
                   kfree(client);
                   return -EPERM;
               }
           }
        if (MAJOR(device) != TTY_MAJOR
            || MINOR(device) > MAX_NR_CONSOLES) {
            kfree(client);
               return -EPERM;
        }
        client->conn.vc = MINOR(device);
        PDEBUG("tty is %i\n",MINOR(device));
        client->flags |= KM_FL_RAW;
        client->conn.minMod = 0;
        client->conn.maxMod = ~0;
        client->conn.eventMask = ~0;
        client->conn.defaultMask = 0;
    }
    else {
        client->conn.vc = ~0; /* can't get data before writing conn info */
    }
    client->filp=filp;
    client->next=filp_first;
    filp_first=client;
    filp->private_data=client;
    MOD_INC_USE_COUNT;
    PDEBUG("Opening at %p (%p) by %i (%s)\n",client,filp,
           current->pid,current->comm);
    PDEBUG("count is %li\n",mod_use_count_);
    return 0;
}


Static  void kmouse_release(struct inode *inode, struct file *filp)
{
    struct kmouse_filp **ptrptr = &filp_first;
    struct kmouse_filp *client;
    struct event_list *ePtr, *nPtr;
    
    while (*ptrptr && (*ptrptr)->filp != filp)
        ptrptr=&(*ptrptr)->next;
    
    if (!*ptrptr) {
        printk("kmouse: closing unopened node\n");
        return;
    }
    client = *ptrptr;
    PDEBUG("Closing at %p (%p) by %i (%s)\n",client,filp,
           current->pid,current->comm);
    *ptrptr = client->next;
    
    /* free it all */
    for (ePtr=client->first; ePtr; ePtr=nPtr) {
        nPtr=ePtr->next;
        kfree(ePtr);
    }
    kfree(client);
    MOD_DEC_USE_COUNT;
    PDEBUG("count is %li\n",mod_use_count_);
}


/********** read/write for the raw device (used by X) */

Static  int kmouse_read_data(struct inode *inode, struct file *filp,
                             char *buf, int count)
{
    struct kmouse_filp *client=filp->private_data;
    char localbuf[5];
    int bytes_read=0;
    
    PDEBUGG("Reading at %p (%p)\n",client,filp);
    
    if (!client->first) {
        if (filp->f_flags & O_NONBLOCK)
            return -EAGAIN;
        interruptible_sleep_on(&client->queue);
        if (current->signal & ~current->blocked)
            return -ERESTARTSYS;
    }
    
    while(count>=5 && client->first) {
        int dx, dy;
        struct event_list *ptr=client->first;
        
        localbuf[0]= (ptr->event.buttons ^ 0x07) | 0x80;
        dx=ptr->event.dx;
        dy=ptr->event.dy;
        PDEBUGG("loop: (%03i,%03i) B %i\n",
                ptr->event.dx,ptr->event.dy,
                ptr->event.buttons);
        if (dx >  250) dx =  250;    /* clustering can be massive... */
        if (dx < -250) dx = -250;
        if (dy >  250) dy =  250;
        if (dy < -250) dy = -250;
        localbuf[1] = localbuf[3] = (dx/2) & 0xff;
        localbuf[2] = localbuf[4] = (-dy/2) & 0xff;
        localbuf[3] += dx & 1;
        localbuf[4] += dy & 1;
        ptr->event.dx -= dx;
        ptr->event.dy -= dy;
        memcpy_tofs(buf,localbuf,5);
        buf += 5; count  -= 5; bytes_read += 5;
        if (ptr->event.dx || ptr->event.dy) {
            PDEBUG("looping: %i,%i left\n",ptr->event.dx,ptr->event.dy);
            continue; /* some dx/dy left */
        }
        client->first = ptr->next;
        if (client->last == ptr)
            client->last = NULL;
        client->nitems--;
        kfree(ptr);
    }
    
    if (bytes_read == 0)
        PDEBUG("Short read (count is %i)\n",count);
    PDEBUGG("read %i bytes\n",bytes_read);
    return bytes_read;
}

Static  int kmouse_write_data (struct inode *inode, struct file *filp,
                               const char *buf, int count)
{
    PDEBUG("data node being written (%i chars)\n",count);
    return count;
}

/*********** read/write for the gpmctl node. Full structures are passed */

#define QUANTUM sizeof(Gpm_Event)

Static  int kmouse_read_ctl (struct inode *inode, struct file *filp,
                             char *buf, int count)
{
    struct kmouse_filp *client=filp->private_data;
    int bytes_read=0;
    
    PDEBUGG("read by %p (%p)\n",client,filp);
    
    if (!client->first) {
        if (filp->f_flags & O_NONBLOCK)
            return -EAGAIN;
        interruptible_sleep_on(&client->queue);
        if (current->signal & ~current->blocked)
            return -ERESTARTSYS;
    }
    while (count>=QUANTUM) {
        struct event_list *ptr=client->first;
        memcpy_tofs(buf,ptr,QUANTUM);
        buf += QUANTUM; count -= QUANTUM; bytes_read += QUANTUM;
        
        client->first = ptr->next;
        if (client->last == ptr)
            client->last = NULL;
        client->nitems--;
        kfree(ptr);
    }
    PDEBUGG("read %i bytes\n",bytes_read);
    return bytes_read;
}


Static  int kmouse_write_ctl (struct inode *inode, struct file *filp,
                              const char *buf, int count)
{
     struct kmouse_filp *client=filp->private_data;
     
     PDEBUG("written by %p (%p)\n",client,filp);
     if (count < sizeof(Gpm_Connect))
         return -EINVAL;
     memcpy_fromfs(&client->conn,buf,sizeof(Gpm_Connect));
     return sizeof(Gpm_Connect);
 }



Static  int kmouse_select (struct inode *inode, struct file *filp,
                           int sel_type, select_table *wait)
{
    struct kmouse_filp *client=filp->private_data;
    
    PDEBUGG("Select\n");
    if (sel_type==SEL_IN) {
        if (client->first)
            return 1;
        select_wait(&client->queue,wait);
        PDEBUGG("select_wait\n");
        return 0;
    }
    if (sel_type==SEL_OUT)
        return 1; /* always writable */
    return 0; /* never exception-able */
}

Static  int kmouse_ioctl(struct inode *inode, struct file * filp,
                         unsigned int cmd, ulong arg)
{
    int err;
    kdev_t dev=0;
    struct kmouse_options options;
    struct kmouse_mouse *mPtr;
    struct kmouse_protocol *p;

    switch(cmd) {

        case KMOUSE_GETVER:
            err=verify_area(VERIFY_WRITE, (void *)arg, sizeof(int));
            if (err) return err;
            put_user_long((long)KMOUSE_VERSION_N,(int *)arg);
            return 0;

        case KMOUSE_SETOPT:
            err=verify_area(VERIFY_READ, (void *)arg,
                            sizeof(struct kmouse_options));
            if (err) return err;
	    memcpy_fromfs(&options,(void *)arg,sizeof(struct kmouse_options));
	    dev=to_kdev_t(options.dev);
	    PDEBUG("ioctl for dev 0x%x\n",dev);
	    mPtr=kmouse_lookfor(dev);
	    if (!mPtr) {
                PDEBUG("using new mouse (0x%x)\n",dev);
                mPtr=kmalloc(sizeof(struct kmouse_mouse),GFP_KERNEL);
                if (!mPtr) return -ENOMEM;
                kmouse_mouse_init(mPtr,dev);
            }

	    if (options.protoid != PROTO_UNKNOWN) {
                for (p=protocols; p->fun; p++)
                    if (p->protoid == options.protoid)
                        break;
                if (p->fun) mPtr->proto=p;
            }

	    for (err=0; err<8; err++)
                opt_sequence[err]=options.sequence[err];
            opt_three = options.three;
            opt_delta = options.delta;
            opt_accel = options.accel;
            opt_scale = options.scale ? : 1;
            opt_scaley = options.scaley ? : opt_scale;
            opt_time = options.time;
            opt_tap = options.tap;
            opt_ptrdrag = options.ptrdrag;
            kmouse_link(mPtr);
            
            PDEBUG("proto is %i\n",options.protoid);
            return 0;
        }
        
    return -ENOIOCTLCMD;
} 

/* What's this? */

Static int kmouse_fasync(struct inode *inode, struct file *filp, int on)
{
    return 0;
}

Static struct file_operations kmouse_data_fops = {
    kmouse_seek,
    kmouse_read_data,
    kmouse_write_data,
    NULL,              /* readdir */
    kmouse_select,
    kmouse_ioctl,
    NULL,              /* mmap */
    kmouse_open,
    kmouse_release,
    NULL,              /* fsync */
    kmouse_fasync,
};

Static struct file_operations kmouse_ctl_fops = {
    kmouse_seek,
    kmouse_read_ctl,
    kmouse_write_ctl,
    NULL,              /* readdir */
    kmouse_select,
    kmouse_ioctl,
    NULL,              /* mmap */
    kmouse_open,
    kmouse_release,
    NULL,              /* fsync */
    kmouse_fasync,
};

Static struct mouse kmouse_data = {
    KMOUSE_DATA_MINOR , "gpmdata", &kmouse_data_fops
    };
Static struct mouse kmouse_ctl = {
    KMOUSE_CTL_MINOR , "gpmctl", &kmouse_ctl_fops
    };

/*
 * Initialize yourself
 */

int init_module(void)
{
    int err;
    
    /* register your stealers */
    if (mouse_stealing) {
#ifdef DEBUG_KMOUSE
        PDEBUG("Overriding stealers\n");
#else
        return -EBUSY;
#endif
    }
    mouse_stealing=&kmouse_stealers;
    
    kmouse_tqueue.routine = kmouse_dispatch;
    kmouse_tqueue.data = NULL; /* unused */
    
    /* register yourself */
    err = mouse_register(&kmouse_data);
    if (!err) err = mouse_register(&kmouse_ctl);

    memset(&kmouse_ldisc, 0, sizeof(kmouse_ldisc));
    kmouse_ldisc.magic = TTY_LDISC_MAGIC;
    kmouse_ldisc.receive_room = kmouse_ldisc_room;
    kmouse_ldisc.receive_buf =  kmouse_ldisc_receive;
    kmouse_ldisc.open =         kmouse_ldisc_open;
    kmouse_ldisc.close =        kmouse_ldisc_close;

    if (!err) err = tty_register_ldisc(N_MOUSE,&kmouse_ldisc);
    
    if (err) {
        printk("kmouse: fatal error %i while registering\n",-err);
        mouse_stealing = NULL;
        return err;
    }
    
    /* create an empty item, so that a new mouse can call us at irq time */
    kmouse_dispatch(NULL);
    
    printk("kmouse " KMOUSE_VERSION " loaded\n");
    return 0;
}

void cleanup_module (void)
{
    mouse_stealing=NULL;
    
    if (kmouse_new) kfree(kmouse_new);
    mouse_deregister(&kmouse_data);
    mouse_deregister(&kmouse_ctl);
    tty_register_ldisc(N_MOUSE, NULL);
    printk("kmouse removed\n");
}
        
        


