/******************************************************************************
 JXDocumentManager.h

	Interface for the JXDocumentManager class

	Copyright  1997 by John Lindal. All rights reserved.

 ******************************************************************************/

#ifndef _H_JXDocumentManager
#define _H_JXDocumentManager

#include <JArray.h>

class JString;
class JXImage;
class JXDocument;
class JXFileDocument;
class JXSafetySaveTask;
class JXDocumentMenu;
class JXUpdateDocMenuTask;

class JXDocumentManager : virtual public JBroadcaster
{
	friend class JXUpdateDocMenuTask;

public:

	enum ShortcutStyle
	{
		kNoShortcuts,
		kCtrlShortcuts,
		kMetaShortcuts
	};

	enum
	{
		kNoShortcutForDoc = -1
	};

	enum SafetySaveReason
	{
		kTimer,

		// following reasons all imply process termination

		kServerDead,
		kAssertFired
	};

public:

	JXDocumentManager(const ShortcutStyle style,
					  const JBoolean useShortcutZero);

	virtual ~JXDocumentManager();

	JBoolean	WillSafetySave() const;
	void		ShouldSafetySave(const JBoolean doIt);

	JSize		GetSafetySaveInterval() const;
	void		SetSafetySaveInterval(const JSize deltaSeconds);

	void		SafetySave(const SafetySaveReason reason);

	void		CloseDocuments();

	// for use by classes derived from JXDocument

	const JString&		GetNewFileName();
	void				DocumentMustStayOpen(JXDocument* doc, const JBoolean stayOpen);
	JBoolean			FileDocumentIsOpen(const JCharacter* fileName,
										   JXFileDocument** doc) const;

	virtual JBoolean	FindFile(const JCharacter* fileName,
								 const JCharacter* currPath,
								 JString* newFileName,
								 const JBoolean askUser = kTrue) const;

	// for use by JXDocument

	virtual void	DocumentCreated(JXDocument* doc);	// must call inherited
	virtual void	DocumentDeleted(JXDocument* doc);	// must call inherited
	JBoolean		OKToCloseDocument(JXDocument* doc) const;

	// for use by JXDocumentMenu

	void	UpdateDocumentMenu(JXDocumentMenu* menu);
	void	ActivateDocument(const JIndex index);

private:

	struct DocInfo
	{
		JXDocument*	doc;
		JXImage*	menuIcon;
		JBoolean	keepOpen;
		JInteger	shortcut;

		DocInfo()
			:
			doc(NULL), menuIcon(NULL),
			keepOpen(kFalse), shortcut(kNoShortcutForDoc)
		{ };

		DocInfo(JXDocument*	d)
			:
			doc(d), menuIcon(NULL),
			keepOpen(kFalse), shortcut(kNoShortcutForDoc)
		{ };
	};

	struct FileMap
	{
		JString*	oldName;
		JString*	newName;

		FileMap()
			:
			oldName(NULL), newName(NULL)
		{ };

		FileMap(JString* s1, JString* s2)
			:
			oldName(s1), newName(s2)
		{ };
	};

private:

	JArray<DocInfo>*		itsDocList;

	JString*				itsLastNewFileName;		// so we can return const JString&
	JIndex					itsNewDocCount;

	JArray<FileMap>*		itsFileMap;

	const ShortcutStyle		itsShortcutStyle;
	const JInteger			itsFirstShortcut;

	JBoolean				itsPerformSafetySaveFlag;
	JXSafetySaveTask*		itsSafetySaveTask;

	JXUpdateDocMenuTask*	itsUpdateDocMenuTask;

private:

	JBoolean	SearchFileMap(const JCharacter* fileName,
							  JString* newFileName) const;

	void	DocumentMenusNeedUpdate();
	void	UpdateAllDocumentMenus();

	// not allowed

	JXDocumentManager(const JXDocumentManager& source);
	const JXDocumentManager& operator=(const JXDocumentManager& source);

public:

	// JBroadcaster messages

	static const JCharacter* kDocMenuNeedsUpdate;

	class DocMenuNeedsUpdate : public JBroadcaster::Message
		{
		public:

			DocMenuNeedsUpdate()
				:
				JBroadcaster::Message(kDocMenuNeedsUpdate)
				{ };
		};
};


/******************************************************************************
 WillSafetySave

 ******************************************************************************/

inline JBoolean
JXDocumentManager::WillSafetySave()
	const
{
	return itsPerformSafetySaveFlag;
}

#endif
