// $Id: 

#include <qpainter.h>
#include <qfontmet.h>
#include <qcursor.h>

#include "ProjectWnd.h"
#include "Project.h"
#include "DialogWnd.h"

#include "DialogList.h"

#include "bmp/bitmaps.h"

QPixmap *DialogListItem::toBeSaved = NULL;

QPixmap *DialogListItem::projectFolded = NULL;
QPixmap *DialogListItem::projectUnfolded = NULL;

DialogListItem::DialogListItem( DialogWnd *dlg )
  : QListBoxItem()
{
  dialog = dlg;
  project = NULL;

  Initialize();
}

DialogListItem::DialogListItem( Project *prj )
  : QListBoxItem()
{
  dialog = NULL;
  project = prj;

  Initialize();
}

void DialogListItem::Initialize( )
{
  saveItem = FALSE;
  dlgAreShown = TRUE;
  dlgClosed = FALSE;

  if( ! toBeSaved )
  {
    toBeSaved = new QPixmap();
    toBeSaved->loadFromData( ToBeSaved_bmp_data, ToBeSaved_bmp_len );
  
    projectFolded = new QPixmap();
    projectFolded->loadFromData( ProjectFolded_bmp_data, 
                                 ProjectFolded_bmp_len );
  
    projectUnfolded = new QPixmap();
    projectUnfolded->loadFromData( ProjectUnfolded_bmp_data, 
                                   ProjectUnfolded_bmp_len );
  }
}


int DialogListItem::height( const QListBox *lb ) const
{
    return QMAX( toBeSaved->height() + 2,
		 lb->fontMetrics().lineSpacing() + 2 );
}


void DialogListItem::paint( QPainter *painter )
{
  QString str;

  if( project && ! dialog )
    str = project->GetName();
  else if( dialog )
    str = dialog->GetDlgName();

  if( str.isEmpty() )
    str = "NoName";

  int i = str.findRev( "/" );
  if( i >= 0 )
     str = str.right( str.length() - i - 1 );
  
  if( saveItem && toBeSaved )
    painter->drawPixmap( 0, 0, *toBeSaved );

  QFontMetrics fontMetrics = painter->fontMetrics();
  int y = fontMetrics.ascent() + fontMetrics.leading()/2;

  int w1 = (toBeSaved) ? toBeSaved->width() : 0;
  int w2 = (projectFolded) ? projectFolded->width() : 0;
  int w = w1 + w2;

  if( project && dialog )
    w += 10;

  if( IsProject() ) // Project files
  {
    if( dlgAreShown )
      painter->drawPixmap( w1 + 4, 0, *projectUnfolded );
    else
      painter->drawPixmap( w1 + 4, 0, *projectFolded );

    painter->drawText( w + 8 , y, str );
  } else if( IsInProject() ) // Dialogs in projects
    painter->drawText( w + 8 , y, str );
  else // Dialogs w/o projects
    painter->drawText( w1 + 4 , y, str );
}  


DialogList::DialogList( ProjectWnd *p, const char *name )
  : QListBox( p, name )
{
  theProjectWnd = p;
  showChange = TRUE;
}

void DialogList::Append( DialogWnd *dlg )
{
  
  DialogListItem *item = new DialogListItem( dlg );

  insertItem( item );
}

void DialogList::Remove( DialogWnd *dlg )
{
  int index = FindItem( dlg );
  
  if( index >= 0 )
  {
      removeItem( index );
  }
    
}

void DialogList::Append( Project *prj )
{
  DialogListItem *item = new DialogListItem( prj );

  insertItem( item );

  DlgList &dialogs = prj->GetDialogs();

  for( DialogWnd *dlg = dialogs.first();
       dlg;
       dlg = dialogs.next() )
  {
    DialogListItem *item = new DialogListItem( dlg );

    item->SetProject( prj );

    insertItem( item );
  }    
  
}

void DialogList::Remove( Project *prj )
{
  int index = FindItem( prj );
  
  if( index >= 0 )
  {
      removeItem( index );
  }

  unsigned int j;
  for( j = index + 1;
       j < count();
       j++ )
  {
    DialogListItem *dlgItem = (DialogListItem*)item( j );    
    if( dlgItem->GetProject() == prj )
      removeItem( j );
    else
      break;
  }
    
}

DialogWnd *DialogList::CurrentItem()
{
  int index = currentItem();

  if( index < 0 )
    return NULL;

  return ((DialogListItem *)item( index ))->GetDialog();
}
  
Project *DialogList::CurrentProject()
{
  int index = currentItem();

  if( index < 0 )
    return NULL;

  return ((DialogListItem *)item( index ))->GetProject();
}
  
void DialogList::Update( DialogWnd *dlg )
{
  int index = FindItem( dlg );

  if( index >= 0 )
  {
    DialogListItem *dlgItem = (DialogListItem *) item( index );
    dlgItem->SetDialog( dlg );
    dlgItem->ToBeSaved( showChange );
    updateItem( index );
  }
}

void DialogList::Update( Project *prj )
{
  int index = FindItem( prj );

  if( index >= 0 )
  {
    DialogListItem *dlgItem = (DialogListItem *) item( index );
    dlgItem->ToBeSaved( showChange );
    updateItem( index );
  }
}

int DialogList::FindItem( DialogWnd *dlg )
{
  for( unsigned int i = 0;
       i < count();
       ++i )
  {
    DialogListItem *dlgItem = (DialogListItem *) item( i );
    if( dlgItem->GetDialog() == dlg )
    {
      return i;
    }
  }

  return -1;
}

int DialogList::FindItem( Project *prj )
{
  for( unsigned int i = 0;
       i < count();
       ++i )
  {
    DialogListItem *dlgItem = (DialogListItem *) item( i );
    if( dlgItem->IsProject() && dlgItem->GetProject() == prj )
    {
      return i;
    }
  }

  return -1;
}

void DialogList::mousePressEvent( QMouseEvent *event )
{
  QListBox::mousePressEvent( event );

  if( event->button() != RightButton )
    return;

  int i = findItem( event->pos().y() );
  
  if( i >= 0 )
  {
      DialogListItem *dlgItem = (DialogListItem *) item( i );

      if( dlgItem->IsProject() )
	  theProjectWnd->itsPrjListMenu->popup( QCursor::pos() );
      else if( dlgItem->IsInProject() )
      {
      if( dlgItem->IsDialogClosed() )
        theProjectWnd->itsClosedDlgPrjListMenu->popup( QCursor::pos() );
      else
	  theProjectWnd->itsDlgPrjListMenu->popup( QCursor::pos() );
      }
      else
	  theProjectWnd->itsDlgListMenu->popup( QCursor::pos() );
  }
}

void DialogList::mouseDoubleClickEvent( QMouseEvent *event )
{
  QListBox::mouseDoubleClickEvent( event );

  int i = findItem( event->pos().y() );

  if( i >= 0 )
  {
      DialogListItem *dlgItem = (DialogListItem *) item( i );

      if( dlgItem->IsProject() )
	  ToggleView( dlgItem->GetProject() );
  }
}

void DialogList::Saved( DialogWnd* dlg )
{
    ShowChange( FALSE );
    Update( dlg );
    ShowChange( TRUE );
}

void DialogList::Saved( Project* prj )
{
    ShowChange( FALSE );
    Update( prj );
    ShowChange( TRUE );
}

int DialogList::Contains( const QString& name )
{
    int found = FALSE;
    QString str;
    unsigned int i;
    int j;
    
    for( i = 0; (i < count()) && !found; i++ )
    {
	DialogListItem *dlgItem = (DialogListItem*)item(i);

    if( dlgItem->IsProject() )
      str = dlgItem->GetProject()->GetName();
    else
	str = dlgItem->GetDialog()->GetDlgName();

	j = str.findRev( "/" );
	if( j >= 0 )
	    str = str.right( str.length() - j - 1 );

    if( str == name )
	{
	    found = TRUE;
	}
    }
    
    return found;
}

void DialogList::AddDialogToProject( Project *prj, DialogWnd *dlg )
{
  int i = FindItem( prj );

  if( i < 0 )
    return;
  
  // Search the last dlg in this project
  unsigned int j;
  for( j = i;
       j < count();
       j++ )
  {
    DialogListItem *dlgItem = (DialogListItem*)item( j );    
    if( dlgItem->GetProject() != prj )
      break;
  }

  DialogListItem *item = new DialogListItem( dlg );

  item->SetProject( prj );

  insertItem( item, j );

}

void DialogList::RemoveDialogFromProject( Project *prj, DialogWnd *dlg )
{
  int i = FindItem( prj );

  if( i < 0 )
    return;
  
  // Search the dlg in this project
  unsigned int j;
  bool found = FALSE;
  for( j = i;
       j < count() && ! found;
       j++ )
  {
    DialogListItem *dlgItem = (DialogListItem*)item( j );    
    if( dlgItem->GetProject() != prj )
      break;
    if( dlgItem->GetDialog() == dlg )
      found = TRUE;
  }

  if( found )
    removeItem( --j );

}

void DialogList::ToggleView( Project *prj )
{
  static bool showDlgs = TRUE;

  int i = FindItem( prj );

  if( i < 0 )
    return;

  // Toggle
  showDlgs = showDlgs ? FALSE : TRUE;
  
  if( showDlgs )
  {
    DlgList &dialogs = prj->GetDialogs();

    for( DialogWnd *dlg = dialogs.first();
         dlg;
         dlg = dialogs.next() )
      AddDialogToProject( prj, dlg );
  } else
  {
    for( unsigned int j = count() - 1;
         j > (unsigned int)i;
         --j )
    {
      DialogListItem *dlgItem = (DialogListItem*)item( j );    
      if( dlgItem->GetProject() != prj )
        continue;
      removeItem( j );
    }
  }

  DialogListItem *dlgItem = (DialogListItem*)item( i );      
  dlgItem->ProjectView( showDlgs );
  updateItem( i );
}

DialogListItem *DialogList::GetCurrentListItem()
{
  return (DialogListItem*)item( currentItem() );
}

DialogListItem *DialogList::GetProjectListItem( Project *prj )
{
  int index = FindItem( prj );
  if( index > -1 )
    return (DialogListItem*)item( index );
  return NULL;
}

DialogListItem *DialogList::GetDialogListItem( DialogWnd *dlg )
{
  int index = FindItem( dlg );
  if( index > -1 )
    return (DialogListItem*)item( index );
  return NULL;
}

DialogListItem *DialogList::GetPrjDlgListItem( Project *prj, DialogWnd *dlg )
{
  int i = FindItem( prj );

  if( i < 0 )
    return NULL;

  bool found = FALSE;
  unsigned int j;
  for( j = i;
       j < count() && ! found;
       j++ )
  {
    DialogListItem *dlgItem = (DialogListItem*)item( j );    
    if( dlgItem->GetProject() != prj )
      break;
    if( dlgItem->GetDialog() == dlg )
      found = TRUE;
  }

  if( found )
    return (DialogListItem*)item( --j );
  return NULL;
}
