#ifndef BoxLayout_included
#define BoxLayout_included

#include <qwidget.h>
#include <qlayout.h>
#include <qlist.h>
#include <qstring.h>

#include "WidgetLayout.h"

#include "layout/LayoutItem.h"

#include "widget/Frame.h"

class BoxSpacing;
class BoxStretch;
class BoxWidget;
class DlgWidget;
class DlgMenuBar;

class BoxLayout : public LayoutItem
{
public: // Methods

    BoxLayout
    (
	QWidget*		parent,
	WidgetLayout*		widgetLayout,
	Frame*			frame = NULL
    );
    // If NULL is givan as the parent, then a 'shell' layout is created with
    // only the properties and not frame widget

    virtual ~BoxLayout();

    bool IsTopLevel( void) const;
    // Is the layout the top level layout

    virtual void Recreate();
    // Recreate the look of the layouts for this item and its children on its
    // layout

    virtual void AddToLayout
    (
	QBoxLayout*	layout
    );
    // Add the item to the layout

    virtual BoxLayout* AddBoxLayout( void );
    // Add a new box layout to the layout's children and return a pointer to it

    BoxSpacing* AddBoxSpacing( void );
    // Add a new spacing to the layout and return a pointer to it

    BoxStretch* AddBoxStretch( void );
    // Add a new stretch to the layout and return a pointer to it

    BoxWidget* AddBoxWidget
    (
	DlgWidget*	widget,
	int		stretch,
	int		align
    );
    // Add a new widget to the layout and return a pointer to it

    void RemoveWidget
    (
	DlgWidget*	widget
    );
    // Remove the widget from its children

    void RemoveItem
    (
	LayoutItem*	item
    );
    // Remove the item from its list of children and delete it

    LayoutItem* NewLayoutItem( const QString&	type,
			       const QString&	id );
    // Create a new layout item specified by type at position id in the 
    // layout tree

    virtual BoxLayout* FindLayout
    (
	QWidget*	widget
    ) const;
    // Given the widget representing a layout, return the layout or NULL if not
    // found

    virtual BoxLayout* FindLayout
    (
	LayoutItem*	item
    ) const;
    // Given a layout item, find the layout that it is on or return NULL

    virtual BoxLayout* FindLayout
    (
	const QRect&	rect
    );
    // Find the topmost layout that contains the given rect in global coords

    virtual LayoutItem* FindLayoutItem
    (
	QWidget*	widget
    ) const;
    // Given the widget representing a layout item, return the item or NULL if
    // not found

    BoxWidget* FindBoxWidget
    (
	DlgWidget*	widget
    ) const;
    // Find the box widget for the given widget and return it, or NULL if none

    virtual void LowerWidget( void );
    // Lower all the widgets representing layout items

    virtual void SaveContents( QTextStream&	stream,
			       QString		id ) const;
    // Save the item's contents to the stream with the given id

    virtual bool RestoreKeyValue( const QString&	key,
				  const QString&	value );
    // Restore a single key/value pair

    virtual void GenerateSource( QTextStream&	stream,
				 const QString&	layoutVarName,
				 const QString&	layoutId ) const;
    // Generate the source for the layout given by layoutVarName.  If the 
    // layoutVarName is empty, the layout is a top level layout

    void GetMenuBarNames( QStrList&	nameList ) const;
    // Return, in nameList, the names of all the menu bars in the window

    virtual QString GetItemName( void ) const;
    // Return a text string for the name of this item

    static QString ItemName( void );
    // Static method to return the name of this item

    virtual QWidget* GetPropertyPage
    (
	QWidget*	parent
    );
    // Return a new property page to be used for this item

    virtual QRect GetGlobalGeometry( void ) const;
    // Get the global geometry for this layout item

    QBoxLayout::Direction GetDirection( void ) const;
    
    void SetDirection
    (
	QBoxLayout::Direction	direction
    );
    // Get/Set methods on the direction    
    
    int GetBorder( void ) const;
    
    void SetBorder
    (
	int	border
    );
    // Get/Set methods on the border
    
    int GetAutoBorder( void ) const;
    
    void SetAutoBorder
    (
	int	autoBorder
    );
    // Get/Set methods on the automatic border

    const QString& GetName( void ) const;
    
    void SetName
    (
	const QString&	name
    );
    // Get/Set methods on the name of the layout

    int GetStrut( void ) const;
    
    void SetStrut
    (
	int	strut
    );
    // Get/Set methods on the amount of strut

    int GetStretch( void ) const;
    
    void SetStretch
    (
	int	stretch
    );
    // Get/Set methods on the amount of stretch

    int GetNumChildren( void ) const;
    // Return the number of children of the layout
    
    void SetMenuBar( const QString&	menuBar );
    // Set the menu bar for this layout

    QString GetMenuBarName( void ) const;
    // Return the name of the menu bar

private: // Methods

    BoxLayout();
    BoxLayout( const BoxLayout& );
    BoxLayout& operator= ( const BoxLayout& );
    
    void GenerateCtor( QTextStream& 	stream,
		       const QString&	layoutName ) const;
    // Generate a constructor for the layout with the variable layoutName

protected: // Attributes

    Frame*			itsFrame;
    // Frame that represents the layout

    QBoxLayout*			itsLayout;
    // Layout connected to the Frame to control the layout of its
    // children

protected: // Attributes

    QBoxLayout::Direction	itsDirection;
    
    int				itsBorder;
    
    int				itsAutoBorder;

    QString			itsName;

    int				itsStrut;

    int 			itsStretch;
    // How much stretch to use when inserting itself into a BoxLayout

    DlgMenuBar*			itsMenuBar;
    // The menu bar used for this layout

    QList< LayoutItem >         itsChildren;
    // child items contained within this layout

private: // Attributes

    bool			isNewFrame;
    // Did the class new a Frame or receive one
};


inline bool BoxLayout::IsTopLevel( void ) const
{
    return !isNewFrame;
}


inline void BoxLayout::RemoveItem
(
    LayoutItem*	item
)
{
    itsChildren.removeRef( item );
}


inline QString BoxLayout::ItemName( void )
{
    return "Box_Layout";
}


inline QBoxLayout::Direction BoxLayout::GetDirection( void ) const
{
    return itsDirection;
}


inline void BoxLayout::SetDirection
(
    QBoxLayout::Direction	direction
)
{
    itsDirection = direction;
}


inline int BoxLayout::GetBorder( void ) const
{
    return itsBorder;
}


inline void BoxLayout::SetBorder
(
    int	border
)
{
    itsBorder = border;
}


inline int BoxLayout::GetAutoBorder( void ) const
{
    return itsAutoBorder;
}


inline void BoxLayout::SetAutoBorder
(
    int	autoBorder
)
{
    itsAutoBorder = autoBorder;
}


inline int BoxLayout::GetStretch( void ) const
{
    return itsStretch;
}


inline void BoxLayout::SetStretch
(
    int	stretch
)
{
    itsStretch = stretch;
}


inline int BoxLayout::GetStrut( void ) const
{
    return itsStrut;
}


inline void BoxLayout::SetStrut
(
    int	strut
)
{
    itsStrut = strut;
}


inline const QString& BoxLayout::GetName( void ) const
{
    return itsName;
}


inline void BoxLayout::SetName
(
    const QString&	name
)
{
    itsName = name;
}


inline int BoxLayout::GetNumChildren( void ) const
{
    return itsChildren.count();
}


#endif // BoxLayout_included
