/* Program tacg - a command line tool for Restriction Enzyme digests of DNA  */
/* Copyright  1996 Harry J Mangalam, University of California, Irvine (mangalam@uci.edu, 714 824 4824) */

/* The use of this software (except that by Harald T. Alvestrand, which is described in 'udping.c')
   is bound by the notice that appears in the file 'tacg.h' which should accompany this file.  In the event 
   that 'tacg.h' is not bundled with this file, please contact the author.
*/

#include <stdio.h>
#include <ctype.h>
#include <string.h> 
#include <stdlib.h>
#include <math.h>
#include "tacg.h" /* contains all the defines, includes, function prototypes for both main() and functions */

/* Start at the beginning and go to the end, also have to check bounds and combinations and 
   incorrect combos to check that errors are caught and processed OK.  Maybe should also have the default be 
   to die at strange errors in flags - otherwise it may generate data that's not wanted and/or 
   use up cpu cycles that users have to actually pay for - also makes it easier to code.... */
/* SetFlags() takes argc/argv and a pointer to the struct that holds all the option values and 
   implements the routines to read them in and do the error checking.  It's related functionally to 
   Interactive(), which it calls, and to CommandLine(), which makes use of these flags to compose a 
   commandline argument which can be used then next time instead of going thru Interactive(). */

char *SetFlags (int argc, char *argv[], long flag_value[2][NFLAGS]) {

   /* flag_val contain the letters and values of the option flags.  Doing it this way 
   allows easy reconstruction of the command line by simply stepping thru the indices */
   /* Here are the delarations from main() for reference */
/*              index         0   1   2   3       4   5   6   7   8   9  10  11  12  13  14  15  16  17  18  19  20  21  22    */
/*char flag_letter[22]  =   {'f','n','o','m',    'M','b','e','g','l','t','T','O','v','h','?','c','r','R','C','F','L','s','w'}; */
/*int flag_value[2][22]  = {{ 1,  4,  1,  1,  32000,  0,  0,  0,  0,  1,  0, -1, -1, -1, -1,  0, -2, -2,  0, -1,  1,  0, 60 }, changed values */
/*                          { 1,  4,  1,  1,  32000,  0,  0,  0,  0,  1,  0, -1, -1, -1, -1,  0, -2, -2,  0, -1,  1,  0, 60 }};   static values */

   /*   0   topo,     -f : for *form* (= topology) - 0 (zero) for circular; 1(d) for linear */
   /*   1   magCuts,  -n : magnitude of RE; upwardly inclusive - 4(d) = all, 5 = 5,6..., 6 = 6,7...  */
   /*   2   overHang, -o : overhang - 5, 3, 0 for blunt, 1(d) for all */
   /*   3   minCuts,  -m : minimum # cuts to be considered; 1(d) for all */
   /*   4   MaxCuts,  -M : Maximum # cuts to be considered; 32,000(d) */
   /*   5   boSubseq, -b : beginning of DNA subsequence; 0(d) for normal beginning  */
   /*   6   eoSubseq, -e : end of DNA subsequence; 0(d) for normal end  */
   /*   7   gel,      -g : include gel map summary - 1 to include; 0(d) to exclude
                           - probably not going to be done for 1st release */
   /*   8   ladder,   -l : ladder map include with graphics above; 1 to include; 0(d) to exclude  */
   /*   9   trans1,   -t : translate with 1 letter codes; 1(d), 3, 6 to incl w/ that many frames of translation; 
                           0 to exclude
   /*  10   trans3    -T : translate with 1 letter codes; 1, 3, 6 to incl w/ that many frames of translation;   
                           0(d) to exclude; -t and -T are of course mutually exclusive.
   /*  11   orfMap,   -O : graphical orf map in 6 frames, not in 1st release */
   /*  12   ver    ,  -v : asks for version of the program (no numeric argument) */
   /*  13,14   help,  -h or -?: brief help page (no numeric argument) */
   /*  15   cite;     -c : cite 1(d) or not cite 0 by udp */
   /*  16   reserved for interactively selecting specific REs; will need a hash function to search matching names */
   /*  17             -R : indicate an alternative rebase name */
   /*  18   CodonUse  -C : Codon Usage table to use for translation: 0(d) to 7 */
   /*  19   Fragments -F : Fragment data to include/exclude; -1(d) if not defined, 0 : no fragment data, 
                           1 : unsorted by size, 2 : sorted by size, 3 : both sorted and unsorted  */
   /*  20   LinearMap -L : whether to print linear map; 0 : don't print the map, 1(d) print the map. */
   /*  21   cutsites  -s : whether to print cutsites; 0(d) : don't print cutsites, 1 : print cutsites */
   /*  22   width,    -w : number of characters wide to print out results - has to be in multiples of 15, from
                           60(d) to 210 bps per line */
   extern char *optarg;  /* externs are for getopt() - makes things SOOOOooooo easy */
   extern int optind;
   long j;   
   char *RebFile;   /* holder for alternative Rebase file string */
   int  c;    /* holder for the 1st char past the '-' (flag indicator) - has to be of type int because of getopt */

      while ((c = getopt(argc, argv, "lLsvqhf:n:o:m:M:b:e:R:g:t:T:C:F:w:")) != EOF) {
      switch(c) { /* switch based on the letter after the '-' */

         default:   fprintf (stderr, "\n\"%c\" - Unrecognized or Unimplemented Flag!!!\n", c); break;

         case 'f':  /* form or topology  - circular or linear */
         switch (optarg[0]) {
            case '0': case '1': flag_value[0][0] = atoi(optarg); break;
            default:  {
               fprintf (stderr, "\'f\' flag requires \"0\" (circular) or \"1\" (linear).  Assuming linear \n"); 
               flag_value[0][0] = 1;
               break;
            }
         } 
         break;

         case 'n':  /* magnitude of RE recognition site - 4 cutter, 5 cutter, etc */
            j= atoi(optarg);
            if (j >3 && j<9) flag_value[0][1] = j;
            else {
               fprintf (stderr, "\'n\' flag requires an integer between 4 and 8.  Assuming \'-n 6\'..\n");
               flag_value[0][1] = 6;
            }
         break;

         case 'o':  /* Type of overhang to consider - omit flag to consider all */
            j = atoi(optarg); 
            if (j==5 || j==3 || j==0) flag_value[0][2] = j;
            else  fprintf (stderr, "\'o\' flag must be 5, 3, or 0; omit to include all overhangs\n.  Assuming all");
         break;  /* above assumption doesn't require changing anything */

         case 'm':  /* Minimum # of cuts to be considered */
            j = atoi(optarg); 
            if (j>=0) flag_value[0][3] = j;
            else fprintf (stderr, "\'m\' flag must be 0 or greater.\n");
            if (flag_value[0][4] != 0 && flag_value[0][4] -j < 0) {
               fprintf (stderr, "\'m\' flag (%ld) must be less than or equal to \'M\' flag (%ld)."  
               " Assuming you want all cuts.\n", j, flag_value[0][4]);  /* Doesn't require changing anything */
            }
         break;

         case 'M':  /* Maximum # of cuts to be considered */
            j = atoi(optarg); 
            if (j>=0) flag_value[0][4] = j;
            else fprintf (stderr, "\'M\' flag must be 0 or greater.\n");
            if (flag_value[0][3] != 0 && j - flag_value[0][3] < 0) {
               fprintf (stderr, "'M' flag (%ld) must be greater than or equal to 'm' flag (%ld). "
               "Assuming you want all cuts.\n", j, flag_value[0][3]);
            } 
         break;

         case 'b':  /* Beginning of subsequence */
            j = atoi(optarg);
					  /* fprintf (stderr, "\'b\' flag is %ld.\n", j); */

            if (j>0) flag_value[0][5] = j;
            else  {
                fprintf (stderr, "\'b\' flag (%ld) must be >0.\n", j);
                exit(1);
            }
            if ((flag_value[0][6] != 0) && (flag_value[0][6] - j < 30)) { /* check if b < e and that sequence is long enough (>30) */
               fprintf (stderr, "\'b\' flag (%ld) must be less than value of \'e\' flag (%ld)\n and subsequence must be >30 bps", j, flag_value[0][6]);
               exit(1);
            }  
         break;

         case 'e':  /* End of subsequence */
            j= atoi(optarg); 
					  /* fprintf (stderr, "\'e\' flag is %ld.\n", j);  */
            if (j>0) flag_value[0][6] = j;
            else  {
                        fprintf (stderr, "\'e\' flag must be > 0.\n");
                exit(1);
            }              
            if ((flag_value[0][5] != 1) && (j - flag_value[0][5] < 30)) { /* check if e > b and that sequence is long enough (>30) */
                        fprintf (stderr, "\'b\' flag (%ld) must be less than value of \'e\' flag (%ld)\n and subsequence must be >30 bps", j, flag_value[0][6]);
                exit(1);
            } 
         break;

         case 'R':   /* uses SearchPaths() to search for an alternative rebase file - SearchPaths() checks if 
                                                the name starts with a '/','~', or '.' and uses the full path and name if entered 
                                                without leading path specifiers, else checks thru a series of paths depending on 
                                                environment variables and returns the pathname of whatever it finds.  If SearchPaths() 
                                                finds nothing, it returns NULL and everything dies */
            flag_value[0][17] = 1; /* mark it so I can reconstruct the command line */
            if ((RebFile = SearchPaths(optarg, "REBASE")) == NULL) {
               fprintf(stderr,"Can't find the alternative REBASE file!! - Check spelling, ownership, existance, etc - Bye!!\n");
               exit (1);
            }  else fprintf (stderr, "In SetFlags, RebFile = %s\n", RebFile);     
         break;

         case 'g':  /* include the gel map; currently a text/graphic, settable with -w flag  */
                    /* flag has 1 argument: '-g minlog', where minlog = log (10) 
                    or log(100) and indicates where the gel should start   */ 	
            j = atoi(optarg);            
            if (j == 10 || j == 100){
               flag_value[0][7] = log10((float)j);
            } else {
              fprintf (stderr, "the '-g' flag has to be of the form '-g10 or -g100'; assuming -g100 \n");
              flag_value[0][7] = 2; /* 2 = log10(100), right? */
              exit (1);
            }
         break;

         case 'l': /* include the ladder map; currently a text/graphic, set with -w, same as everything else */
            flag_value[0][8] = 1;
         break;

         case 't': /* Translate in 1 letter code in 1, 3, or 6 frames */
            j = atoi(optarg);
            if (j==0 || j==1 || j==3 || j==6)  {
               flag_value[0][9] = j;
               flag_value[0][10] = 0; /* Exclude the 3 letter option */
            } else fprintf (stderr, "The \'t\' flag value must be 1, 3, or 6, not %s.\n"
                                     "Will use the default and continue...\n", optarg);
         break;

         case 'T':   /* Translate in 3 letter code in 1, 3, or 6 frames */
            j = atoi(optarg);           
            if (j==0 || j==1 || j==3 || j==6)  {
               flag_value[0][10] = j;
               flag_value[0][9] = 0; /* Exclude the 1 letter option */
            } else {
                fprintf (stderr, "The \'T\' flag value must be 1, 3, or 6, not %s.  "
                                 "Will use the default value (1) and continue...\n", optarg);
                flag_value[0][10] = 1;
                flag_value[0][9] = 0; /* Exclude the 1 letter option */
            }
         break;

         case 'h': 
            Usage(); 
            exit (1);
         break;

         case 'v':
            flag_value[0][12] = 1; /* changes default value in flag_value - useful?  */
            fprintf (stderr, "program tacg, Version %s, Copyright1996 Harry Mangalam (mangalam@uci.edu),"
            "UC Irvine\n", VERSION);
            exit(1);
         break;

         case 'q': /* quiet - don't cite Harry (Swine!) */
               flag_value[0][15] = 0;
               fprintf (stderr, "You've chosen the no-citation option (-q); please reconsider next time!\n");
         break;

         case 'C': /* Codon usage table to use - 0(d) to 7 */
            j = atoi(optarg);
            if (j>=0 && j<8) flag_value[0][18] = j;
            else  {
               fprintf (stderr, "The \'C\' flag value must be between 0 and 7, not %s.\n", optarg);
               Usage(); /* let user see what the values are */
               exit(1); /* and then die */
            }
         break;

         case 'F': /* How to present fragment data - include or exclude 0-3; it's -1(d) (no frag data) if it wasn't called*/
            j = atoi(optarg); 
            if (j>=0 && j<4) flag_value[0][19] = j;
            else fprintf (stderr, "The \'F\' flag value must be: \n"
            "           0 (don't print fragment data), \n"
            "           1 (print fragments sorted by order)\n"
            "           2 (print fragments sorted by size), or \n"
            "           3 (print fragments sorted both ways), \n"
            "not \'%s\'.\n", optarg);
         break;

         case 'L': /* whether to print linear map; -L prints the linear map; omitting it doesn't print it */
            flag_value[0][20] = 1;
         break;

         case 's':  /* whether to print cutsites; if specified, print cutsites; otherwise don't */
            flag_value[0][21] = 1;
         break;

         case 'w': /* change the output width */
            j = atoi(optarg);   
            if (j<60 || j>210)  {
               fprintf (stderr, "\'w\' value must be between 60 and 210; program truncates to a \n"
               "number exactly divisible by 15 (178 would be truncated to 165).\n");
               exit (1); /* and die cleanly */
            }  else flag_value[0][22] = (int) j / 15*15; /* recalc for every value given by flag to cut down on retries */
            /* and so it's passed back to main(); replaces the BASES_PER_LINE #define */
          break;
      }   /* end of switch() statement */
     }  /*    while (--argc > 0 && (*++argv)[0] ... */
   return RebFile;
}    /* end of  SetFlags */


