/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sysml.api.dl

import caffe.Caffe.LayerParameter;
import caffe.Caffe.NetParameter;
import caffe.Caffe.SolverParameter;

import org.apache.sysml.parser.LanguageException;
import org.apache.sysml.runtime.DMLRuntimeException;
import org.apache.sysml.api.ml.ScriptsUtils
import org.apache.sysml.runtime.matrix.MatrixCharacteristics
import org.apache.sysml.runtime.matrix.data.MatrixBlock
import scala.collection.JavaConversions._
import java.util.ArrayList
import caffe.Caffe.Phase
import caffe.Caffe
import java.util.HashSet
import org.apache.sysml.api.DMLScript
import java.io.File
import org.apache.spark.SparkContext
import org.apache.spark.ml.{ Estimator, Model }
import org.apache.spark.sql.DataFrame
import org.apache.spark.sql.types.StructType
import org.apache.spark.ml.param.{ DoubleParam, Param, ParamMap, Params }
import org.apache.sysml.runtime.matrix.MatrixCharacteristics
import org.apache.sysml.runtime.matrix.data.MatrixBlock
import org.apache.sysml.runtime.DMLRuntimeException
import org.apache.sysml.runtime.instructions.spark.utils.{ RDDConverterUtilsExt => RDDConverterUtils }
import org.apache.sysml.api.mlcontext._
import org.apache.sysml.api.mlcontext.ScriptFactory._
import org.apache.sysml.api.ml._
import java.util.Random
import org.apache.commons.logging.Log
import org.apache.commons.logging.LogFactory
import org.apache.sysml.runtime.controlprogram.parfor.stat.InfrastructureAnalyzer
import org.apache.sysml.hops.OptimizerUtils

/***************************************************************************************
DESIGN OF CAFFE2DML:

1. Caffe2DML is designed to fit well into the mllearn framework. Hence, the key methods that were to be implemented are:
- `getTrainingScript` for the Estimator class.
- `getPredictionScript` for the Model class.

These methods should be the starting point of any developer to understand the DML generated for training and prediction respectively.

2. To simplify the DML generation in getTrainingScript and getPredictionScript method, we use DMLGenerator interface.
This interface generates DML string for common operations such as loops (such as if, for, while) as well as built-in functions (read, write), etc.
Also, this interface helps in "code reading" of this class :)

3. Here is an analogy for SystemML developers to think of various moving components of Caffe2DML:
- Like Dml.g4 in the org.apache.sysml.parser.dml package, caffe.proto in the src/main/proto/caffe directory
is used to generate classes to parse the input files.

Dml.g4      ---> antlr  ---> DmlLexer.java, DmlListener.java, DmlParser.java
caffe.proto ---> protoc ---> target/generated-sources/caffe/Caffe.java

- Just like the classes generated by Dml.g4 are used to parse input DML file,
the target/generated-sources/caffe/Caffe.java class is used to parse the input caffe network/deploy prototxt and solver files.

- You can think of .caffemodel file as DML file with matrix values encoded in it (please see below example).
So it is possible to read .caffemodel file with the Caffe.java class. This is done in Utils.scala's readCaffeNet method.

X = matrix("1.2 3.5 0.999 7.123", rows=2, cols=2)
...

- Just like we convert the AST generated by antlr into our DMLProgram representation, we convert
caffe's abstraction into the below given mapping classes for layer, solver and learning rate.
These mapping classes maps the corresponding Caffe abstraction to the SystemML-NN library.
This greatly simplifies adding new layers into Caffe2DML:
trait CaffeLayer {
  // Any layer that wants to reuse SystemML-NN has to override following methods that help in generating the DML for the given layer:
  def sourceFileName:String;
  def init(dmlScript:StringBuilder):Unit;
  def forward(dmlScript:StringBuilder, isPrediction:Boolean):Unit;
  def backward(dmlScript:StringBuilder, outSuffix:String):Unit;
  ...
}
trait CaffeSolver {
  def sourceFileName:String;
  def update(dmlScript:StringBuilder, layer:CaffeLayer):Unit;
  def init(dmlScript:StringBuilder, layer:CaffeLayer):Unit;
}

4. To simplify the traversal of the network, we created a Network interface:
trait Network {
  def getLayers(): List[String]
  def getCaffeLayer(layerName:String):CaffeLayer
  def getBottomLayers(layerName:String): Set[String]
  def getTopLayers(layerName:String): Set[String]
  def getLayerID(layerName:String): Int
}

5. One of the key design restriction of Caffe2DML is that every layer is identified uniquely by its name.
This restriction simplifies the code significantly.
To shield from network files that violates this restriction, Caffe2DML performs rewrites in CaffeNetwork class (search for condition 1-5).

6. Caffe2DML also expects the layers to be in sorted order.

***************************************************************************************/
object Caffe2DML {
  val LOG = LogFactory.getLog(classOf[Caffe2DML].getName())
  // ------------------------------------------------------------------------
  val USE_PLUS_EQ = true
  def layerDir = "nn/layers/"
  def optimDir = "nn/optim/"

  // Naming conventions:
  val X    = "X"; val y        = "y"; val batchSize = "BATCH_SIZE"; val numImages = "num_images"; val numValidationImages = "num_validation"
  val XVal = "X_val"; val yVal = "y_val"

  val USE_NESTEROV_UDF = {
    // Developer environment variable flag 'USE_NESTEROV_UDF' until codegen starts working.
    // Then, we will remove this flag and also the class org.apache.sysml.udf.lib.SGDNesterovUpdate
    val envFlagNesterovUDF = System.getenv("USE_NESTEROV_UDF")
    envFlagNesterovUDF != null && envFlagNesterovUDF.toBoolean
  }

  def main(args: Array[String]): Unit = {
    // Arguments: [train_script | predict_script] $OUTPUT_DML_FILE $SOLVER_FILE $INPUT_CHANNELS $INPUT_HEIGHT $INPUT_WIDTH $NUM_ITER
    if (args.length < 6) throwUsageError
    val outputDMLFile = args(1)
    val solverFile    = args(2)
    val inputChannels = args(3)
    val inputHeight   = args(4)
    val inputWidth    = args(5)
    val caffeObj      = new Caffe2DML(new SparkContext(), solverFile, inputChannels, inputHeight, inputWidth)
    if (args(0).equals("train_script")) {
      Utils.writeToFile(caffeObj.getTrainingScript(true)._1.getScriptString, outputDMLFile)
    } else if (args(0).equals("predict_script")) {
      Utils.writeToFile(new Caffe2DMLModel(caffeObj).getPredictionScript(true)._1.getScriptString, outputDMLFile)
    } else {
      throwUsageError
    }
  }
  def throwUsageError(): Unit =
    throw new RuntimeException("Incorrect usage: train_script OUTPUT_DML_FILE SOLVER_FILE INPUT_CHANNELS INPUT_HEIGHT INPUT_WIDTH");
  
  val rand = new Random
  // Supported Algorithms:
  val MINIBATCH_ALGORITHM = "minibatch"
  val BATCH_ALGORITHM = "batch"
  val ALLREDUCE_ALGORITHM = "allreduce"
  val ALLREDUCE_PARALLEL_BATCHES_ALGORITHM = "allreduce_parallel_batches"
}

class Caffe2DML(val sc: SparkContext,
                val solverParam: Caffe.SolverParameter,
                val solver: CaffeSolver,
                val net: CaffeNetwork,
                val lrPolicy: LearningRatePolicy,
                val numChannels: String,
                val height: String,
                val width: String)
    extends Estimator[Caffe2DMLModel]
    with BaseSystemMLClassifier
    with DMLGenerator {
  // --------------------------------------------------------------
  // Invoked by Python, MLPipeline
  def this(sc: SparkContext, solver1: Caffe.SolverParameter, networkPath: String, numChannels: String, height: String, width: String) {
    this(
      sc,
      solver1,
      Utils.parseSolver(solver1),
      new CaffeNetwork(networkPath, caffe.Caffe.Phase.TRAIN, numChannels, height, width),
      new LearningRatePolicy(solver1),
      numChannels,
      height,
      width
    )
  }
  def this(sc: SparkContext, solver1: Caffe.SolverParameter, numChannels: String, height: String, width: String) {
    this(
      sc,
      solver1,
      Utils.parseSolver(solver1),
      new CaffeNetwork(solver1.getNet, caffe.Caffe.Phase.TRAIN, numChannels, height, width),
      new LearningRatePolicy(solver1),
      numChannels,
      height,
      width
    )
  }
  def this(sc: SparkContext, solverPath: String, numChannels: String, height: String, width: String) {
    this(sc, Utils.readCaffeSolver(solverPath), numChannels, height, width)
  }
  val uid: String = "caffe_classifier_" + (new Random).nextLong
  override def copy(extra: org.apache.spark.ml.param.ParamMap): Estimator[Caffe2DMLModel] = {
    val that = new Caffe2DML(sc, solverParam, solver, net, lrPolicy, numChannels, height, width)
    copyValues(that, extra)
  }
  def fit(X_file: String, y_file: String): Caffe2DMLModel = {
    mloutput = baseFit(X_file, y_file, sc)
    new Caffe2DMLModel(this)
  }
  // Note: will update the y_mb as this will be called by Python mllearn
  def fit(X_mb: MatrixBlock, y_mb: MatrixBlock): Caffe2DMLModel = {
    mloutput = baseFit(X_mb, y_mb, sc)
    new Caffe2DMLModel(this)
  }
  def fit(df: ScriptsUtils.SparkDataType): Caffe2DMLModel = {
    mloutput = baseFit(df, sc)
    new Caffe2DMLModel(this)
  }
  // --------------------------------------------------------------
  // Returns true if last 2 of 4 dimensions are 1.
  // The first dimension refers to number of input datapoints.
  // The second dimension refers to number of classes.
  def isClassification(): Boolean = {
    val outShape = getOutputShapeOfLastLayer
    return outShape._2 == 1 && outShape._3 == 1
  }
  def getOutputShapeOfLastLayer(): (Int, Int, Int) = {
    val out = net.getCaffeLayer(net.getLayers().last).outputShape
    (out._1.toInt, out._2.toInt, out._3.toInt)
  }

  // Used for simplifying transfer learning
  private val layersToIgnore: HashSet[String]                 = new HashSet[String]()
  def setWeightsToIgnore(layerName: String): Unit             = layersToIgnore.add(layerName)
  def setWeightsToIgnore(layerNames: ArrayList[String]): Unit = layersToIgnore.addAll(layerNames)

  // Input parameters to prediction and scoring script
  val inputs: java.util.HashMap[String, String]  = new java.util.HashMap[String, String]()
  def setInput(key: String, value: String): Unit = inputs.put(key, value)
  customAssert(solverParam.getTestIterCount <= 1, "Multiple test_iter variables are not supported")
  customAssert(solverParam.getMaxIter > 0, "Please set max_iter to a positive value")
  customAssert(net.getLayers.filter(net.getCaffeLayer(_).isInstanceOf[IsLossLayer]).length == 1, "Expected exactly one loss layer")

  def trainAlgoContainsParfor():Boolean = getTrainAlgo.toLowerCase.startsWith("allreduce")
  def testAlgoContainsParfor():Boolean = getTestAlgo.toLowerCase.startsWith("allreduce")
  def containsParfor():Boolean = trainAlgoContainsParfor || testAlgoContainsParfor
  def getTrainAlgo(): String = if (inputs.containsKey("$train_algo")) inputs.get("$train_algo") else Caffe2DML.MINIBATCH_ALGORITHM
  def getTestAlgo(): String  = if (inputs.containsKey("$test_algo")) inputs.get("$test_algo") else Caffe2DML.MINIBATCH_ALGORITHM

  // Prints the summary of network
  def summary(sparkSession: org.apache.spark.sql.SparkSession): Unit = {
    val layers = net.getLayers .map(l => (l, net.getCaffeLayer(l)))
    val numDataLayers = layers.filter(l => l._2.isInstanceOf[Data]).length
    val batchSizes = layers.filter(l => l._2.isInstanceOf[Data]).map(l => l._2.param.getDataParam.getBatchSize).distinct
    if(batchSizes.size > 1) {
      Caffe2DML.LOG.warn("Multiple data layers with different batch sizes:" + batchSizes.mkString(",") + ". Using the batch size:" + batchSizes.get(0))
    }
    else if(batchSizes.size == 0) {
      Caffe2DML.LOG.warn("No data layers found and hence ignoring the memory computation.")
    }
    val batchSize = if(batchSizes.size > 0) batchSizes.get(0) else -1 
    val header = Seq("Name", "Type", "Output", "Weight", "Bias", "Top", "Bottom", "Memory* (train/test)")
    val entries = layers
      .map(l => {
        val layer = l._2
        (l._1,
         layer.param.getType,
         "(, " + layer.outputShape._1 + ", " + layer.outputShape._2 + ", " + layer.outputShape._3 + ")",
         if (layer.weightShape != null) {
           val wShapes = "[" + layer.weightShape()(0) + " X " + layer.weightShape()(1) + "]"
           if (layer.extraWeightShape != null) wShapes + ", " +  "[" + layer.extraWeightShape()(0) + " X " + layer.extraWeightShape()(1) + "]"
           else wShapes
         } else "",
         if (layer.biasShape != null) "[" + layer.biasShape()(0) + " X " + layer.biasShape()(1) + "]" else "",
         layer.param.getTopList.mkString(","),
         layer.param.getBottomList.mkString(","), 
         OptimizerUtils.toMB(Utils.getMemInBytes(l._2, batchSize, true)) + "/" + OptimizerUtils.toMB(Utils.getMemInBytes(l._2, batchSize, false))
        )
      })
    import sparkSession.implicits._
    sc.parallelize(entries).toDF(header: _*).show(net.getLayers.size)
    
    val numLayerOutput = layers.map(l => l._2.outputShape._1.toLong * l._2.outputShape._2.toLong * l._2.outputShape._3.toLong).sum * batchSize
    val numLayerError = numLayerOutput
    val numLayerWeights = layers.map(l => if(l._2.weightShape != null) l._2.weightShape()(0).toLong * l._2.weightShape()(1).toLong else 0).sum
    val numLayerBias = layers.map(l => if(l._2.biasShape != null) l._2.biasShape()(0).toLong * l._2.biasShape()(1).toLong else 0).sum
    val numLayerGradients = (numLayerWeights + numLayerBias) * batchSize
    val convLayers = layers.filter(l => l._2.isInstanceOf[Convolution]).map(l => l._2.asInstanceOf[Convolution])
    val crspq = convLayers.map(l => l.numChannels.toLong*l.kernel_h.toLong*l.kernel_w.toLong*l.outputShape._2.toLong*l.outputShape._3.toLong) 
    val kpq = convLayers.map(l => l.outputShape._1.toLong*l.outputShape._2.toLong*l.outputShape._3.toLong)
    
    System.out.println("* => memory in megabytes assuming the parameters (input, output activations, weights and backpropagation errors) are in double precision and in dense format.")
  }
  
  def setDebugFlags(isDebug:Boolean):Unit = {
    net.getLayers.map(layer => {net.getCaffeLayer(layer).debugLayer = isDebug})
    net.getLayers.map(layer => {net.getCaffeLayer(layer).caffe2dmlObj = this})
  }
  
  // Comma is included
  def getParforParameters():String = if (inputs.containsKey("$parfor_parameters")) inputs.get("$parfor_parameters") else ""

  // ================================================================================================
  // The below method parses the provided network and solver file and generates DML script.
  def getTrainingScript(isSingleNode: Boolean): (Script, String, String) = {
    val startTrainingTime = System.nanoTime()

    reset // Reset the state of DML generator for training script.

    // Flags passed by user
    val DEBUG_TRAINING = if (inputs.containsKey("$debug")) inputs.get("$debug").toLowerCase.toBoolean else false
    assign(tabDMLScript, "debug", if (DEBUG_TRAINING) "TRUE" else "FALSE")
    setDebugFlags(DEBUG_TRAINING)

    appendHeaders(net, solver, true) // Appends DML corresponding to source and externalFunction statements.
    val performOneHotEncoding = !inputs.containsKey("$perform_one_hot_encoding") || inputs.get("$perform_one_hot_encoding").toBoolean
    readInputData(net, true, performOneHotEncoding)         // Read X_full and y_full
    // Initialize the layers and solvers. Reads weights and bias if $weights is set.
    initWeights(net, solver, inputs.containsKey("$weights"), layersToIgnore)

    // Split into training and validation set
    // Initializes Caffe2DML.X, Caffe2DML.y, Caffe2DML.XVal, Caffe2DML.yVal and Caffe2DML.numImages
    val shouldValidate = solverParam.getTestInterval > 0 && solverParam.getTestIterCount > 0 && solverParam.getTestIter(0) > 0
    trainTestSplit(if (shouldValidate) solverParam.getTestIter(0) else 0)
    assign(tabDMLScript, "lr", solverParam.getBaseLr.toString)    
    val lossLayers = getLossLayers(net)
    ceilDivide(tabDMLScript, "num_batches_per_epoch", Caffe2DML.numImages, Caffe2DML.batchSize)
    if(solverParam.getDisplay > 0 && !getTrainAlgo.toLowerCase.equals(Caffe2DML.MINIBATCH_ALGORITHM)) {
      tabDMLScript.append(print(dmlConcat(asDMLString("Iterations (for training loss/accuracy) refers to the number of batches processed where batch size="), Caffe2DML.batchSize)))
    }
    if(getTrainAlgo.toLowerCase.equals(Caffe2DML.ALLREDUCE_PARALLEL_BATCHES_ALGORITHM) ||
        getTestAlgo.toLowerCase.equals(Caffe2DML.ALLREDUCE_PARALLEL_BATCHES_ALGORITHM)) {
      assign(tabDMLScript, "parallel_batches", "$parallel_batches")
    }
    // ----------------------------------------------------------------------------
    // Main logic
    getTrainAlgo.toLowerCase match {
      case Caffe2DML.MINIBATCH_ALGORITHM => {
        assign(tabDMLScript, "e", "0")
        assign(tabDMLScript, "max_iter", ifdef("$max_iter", solverParam.getMaxIter.toString))
        forBlock("iter", "1", "max_iter") {
          getTrainingBatch(tabDMLScript)
          // -------------------------------------------------------
          // Perform forward, backward and update on minibatch
          forward; backward; update
          // -------------------------------------------------------
          if(solverParam.getDisplay > 0) {
            ifBlock("iter  %% " + solverParam.getDisplay + " == 0") {
              displayTrainingLoss(lossLayers(0), performOneHotEncoding)
            }
            if(shouldValidate) {
              ifBlock("iter  %% " + solverParam.getTestInterval + " == 0") {
                displayValidationLoss(lossLayers(0), performOneHotEncoding)
              }
            }
          }
          performSnapshot
          ifBlock("iter %% num_batches_per_epoch == 0") {
            // After every epoch, update the learning rate
            assign(tabDMLScript, "e", "e + 1")
            tabDMLScript.append("# Learning rate\n")
            lrPolicy.updateLearningRate(tabDMLScript)
          }
        }
      }
      case Caffe2DML.BATCH_ALGORITHM => {
        assign(tabDMLScript, "max_iter", ifdef("$max_iter", solverParam.getMaxIter.toString))
        assign(tabDMLScript, "max_epochs", ceil("(max_iter*" + Caffe2DML.batchSize + ")/" + Caffe2DML.numImages))
        forBlock("e", "1", "max_epochs") {
          assign(tabDMLScript, "iter", "num_batches_per_epoch*e")
          assign(tabDMLScript, "Xb", Caffe2DML.X)
          assign(tabDMLScript, "yb", Caffe2DML.y)
          // -------------------------------------------------------
          // Perform forward, backward and update on entire dataset
          forward; backward; update
          // -------------------------------------------------------
          if(solverParam.getDisplay > 0) {
            // Show training/validation loss every epoch
            displayTrainingLoss(lossLayers(0), performOneHotEncoding)
            if(shouldValidate)
              displayValidationLoss(lossLayers(0), performOneHotEncoding)
          }
          performSnapshot
          // After every epoch, update the learning rate
          tabDMLScript.append("# Learning rate\n")
          lrPolicy.updateLearningRate(tabDMLScript)
        }
      }
      case Caffe2DML.ALLREDUCE_PARALLEL_BATCHES_ALGORITHM => {
        assign(tabDMLScript, "e", "0")
        assign(tabDMLScript, "max_iter", ifdef("$max_iter", solverParam.getMaxIter.toString))
        forBlock("iter", "1", "max_iter", "parallel_batches") {  
          assign(tabDMLScript, "allreduce_start_index", "((iter-1) * " + Caffe2DML.batchSize + ") %% " + Caffe2DML.numImages + " + 1; ")
          ifBlock("(allreduce_start_index + parallel_batches*" + Caffe2DML.batchSize + " - 1) > " + Caffe2DML.numImages) {
            // TODO: Skip the last batch
            assign(tabDMLScript, "allreduce_start_index", "1")
          }
          initializeGradients("parallel_batches")
          parForBlock("j", "1", "parallel_batches", "1", getParforParameters()) {
            // Get a mini-batch in this group
            assign(tabDMLScript, "beg", "allreduce_start_index + (j-1)*" + Caffe2DML.batchSize)
            assign(tabDMLScript, "end", "allreduce_start_index + j*" + Caffe2DML.batchSize + " - 1")
            rightIndexing(tabDMLScript, "Xb", Caffe2DML.X, "beg", "end")
            rightIndexing(tabDMLScript, "yb", Caffe2DML.y, "beg", "end")
            forward; backward
            flattenGradients
            if(solverParam.getDisplay > 0) {
              ifBlock("(iter + j - 1)  %% " + solverParam.getDisplay + " == 0") {
                displayTrainingLoss(lossLayers(0), performOneHotEncoding)
              }
            }
          }
          aggregateAggGradients
          update
          if(solverParam.getDisplay > 0 && shouldValidate) {
            val iterMatrix = matrix("seq(iter, iter+parallel_batches-1)", "parallel_batches", "1")
            ifBlock(sum(iterMatrix + " %% " + solverParam.getTestInterval + " == 0") + " > 0") {
              displayValidationLoss(lossLayers(0), performOneHotEncoding)
            }
          }
          performSnapshot
        }
      }
      case Caffe2DML.ALLREDUCE_ALGORITHM => {
        forBlock("iter", "1", "max_iter") {
          // This is distributed synchronous gradient descent with batch size of 1
          // -------------------------------------------------------
          // Perform forward, backward and update on minibatch in parallel
          assign(tabDMLScript, "beg", "((iter-1) * " + Caffe2DML.batchSize + ") %% " + Caffe2DML.numImages + " + 1;")
          assign(tabDMLScript, "end", "min(beg + " + Caffe2DML.batchSize + " - 1, " + Caffe2DML.numImages + ")")
          assign(tabDMLScript, "X_group_batch_size", "end - beg + 1")
          initializeGradients("X_group_batch_size")
          parForBlock("j", "beg", "end", "1", getParforParameters()) {
            assign(tabDMLScript, "Xb", Caffe2DML.X + "[j,]")
            assign(tabDMLScript, "yb", Caffe2DML.y + "[j,]")
            forward; backward
            flattenGradients
          }
          aggregateAggGradients
          update
          // -------------------------------------------------------
          if(solverParam.getDisplay > 0) {
            ifBlock("iter  %% " + solverParam.getDisplay + " == 0") {
              assign(tabDMLScript, "Xb", Caffe2DML.X + "[beg:end,]")
              assign(tabDMLScript, "yb", Caffe2DML.y + "[beg:end,]")
              displayTrainingLoss(lossLayers(0), performOneHotEncoding)
            }
            if(shouldValidate) {
              ifBlock("iter  %% " + solverParam.getTestInterval + " == 0") {
                displayValidationLoss(lossLayers(0), performOneHotEncoding)
              }
            }
          }
          performSnapshot
        }
      }
      case _ => throw new DMLRuntimeException("Unsupported train algo:" + getTrainAlgo)
    }
    // ----------------------------------------------------------------------------

    val trainingScript = tabDMLScript.toString()
    // Print script generation time and the DML script on stdout
    System.out.println("Time taken to generate training script from Caffe proto: " + ((System.nanoTime() - startTrainingTime) * 1e-9) + " seconds.")
    if (DEBUG_TRAINING) Utils.prettyPrintDMLScript(trainingScript)

    // Set input/output variables and execute the script
    val script = dml(trainingScript).in(inputs)
    net.getLayers.map(net.getCaffeLayer(_)).filter(_.weight != null).map(l => script.out(l.weight))
    net.getLayers.map(net.getCaffeLayer(_)).filter(_.extraWeight != null).map(l => script.out(l.extraWeight))
    net.getLayers.map(net.getCaffeLayer(_)).filter(_.bias != null).map(l => script.out(l.bias))
    
    setDebugFlags(false)
    
    (script, "X_full", "y_full")
  }
  // ================================================================================================
  // -------------------------------------------------------------------------------------------
  // Helper functions to generate DML
  // Initializes Caffe2DML.X, Caffe2DML.y, Caffe2DML.XVal, Caffe2DML.yVal and Caffe2DML.numImages
  private def trainTestSplit(numValidationBatches: Int): Unit = {
    if (numValidationBatches > 0) {
      if (solverParam.getDisplay <= 0)
        throw new DMLRuntimeException("Since test_iter and test_interval is greater than zero, you should set display to be greater than zero")
      assign(tabDMLScript, Caffe2DML.numValidationImages, numValidationBatches + " * " + Caffe2DML.batchSize)
      tabDMLScript.append("# Sanity check to ensure that validation set is not too large\n")
      val maxValidationSize = ceil("0.3 * " + Caffe2DML.numImages)
      ifBlock(Caffe2DML.numValidationImages + " > " + maxValidationSize) {
        stop(tabDMLScript, dmlConcat(
            asDMLString("Too large validation size. Please reduce test_iter to "), floor(maxValidationSize + " / " + Caffe2DML.batchSize)))
      }
      rightIndexing(tabDMLScript, Caffe2DML.X, "X_full", int_add(Caffe2DML.numValidationImages, "1"), Caffe2DML.numImages)
      rightIndexing(tabDMLScript, Caffe2DML.y, "y_full", int_add(Caffe2DML.numValidationImages, "1"), Caffe2DML.numImages)
      rightIndexing(tabDMLScript, Caffe2DML.XVal, "X_full", "1", Caffe2DML.numValidationImages)
      rightIndexing(tabDMLScript, Caffe2DML.yVal, "y_full", "1", Caffe2DML.numValidationImages)
    } else {
      assign(tabDMLScript, Caffe2DML.X, "X_full")
      assign(tabDMLScript, Caffe2DML.y, "y_full")
    }
    assign(tabDMLScript, Caffe2DML.numImages, nrow(Caffe2DML.y))
  }
  
  private def displayTrainingLoss(lossLayer: IsLossLayer, performOneHotEncoding:Boolean): Unit = {
    val DEBUG_TRAINING = if (inputs.containsKey("$debug")) inputs.get("$debug").toLowerCase.toBoolean else false
    tabDMLScript.append("# Compute training loss & accuracy\n")
    assign(tabDMLScript, "loss", "0"); assign(tabDMLScript, "accuracy", "0")
    lossLayer.computeLoss(dmlScript, numTabs)
    assign(tabDMLScript, "training_loss", "loss"); assign(tabDMLScript, "training_accuracy", "accuracy")
    tabDMLScript.append(
      print(dmlConcat(asDMLString("Iter:"), "iter", asDMLString(", training loss:"), "training_loss", asDMLString(", training accuracy:"), "training_accuracy"))
    )
    if(performOneHotEncoding && DEBUG_TRAINING && !trainAlgoContainsParfor) {
      printClassificationReport
    }
  }
  
  private def displayValidationLoss(lossLayer: IsLossLayer, performOneHotEncoding:Boolean): Unit = {
    if (trainAlgoContainsParfor && testAlgoContainsParfor) {
      Caffe2DML.LOG.warn("The setting: train_algo=" + getTrainAlgo + " and test_algo=" + getTestAlgo + " is not recommended. Consider changing test_algo=minibatch")
    }
    // Append the DML to compute validation loss
    val numValidationBatches = if (solverParam.getTestIterCount > 0) solverParam.getTestIter(0) else 0
    tabDMLScript.append("# Compute validation loss & accuracy\n")
    assign(tabDMLScript, "loss", "0"); assign(tabDMLScript, "accuracy", "0")
    getTestAlgo.toLowerCase match {
      case Caffe2DML.MINIBATCH_ALGORITHM => {
        assign(tabDMLScript, "validation_loss", "0")
        assign(tabDMLScript, "validation_accuracy", "0")
        forBlock("iVal", "1", "num_batches_per_epoch") {
          getValidationBatch(tabDMLScript)
          forward; lossLayer.computeLoss(dmlScript, numTabs)
          tabDMLScript.append("validation_loss = validation_loss + loss\n")
          tabDMLScript.append("validation_accuracy = validation_accuracy + accuracy\n")
        }
        tabDMLScript.append("validation_accuracy = validation_accuracy / num_batches_per_epoch\n")
      }
      case Caffe2DML.BATCH_ALGORITHM => {
        assign(tabDMLScript, "Xb", Caffe2DML.XVal); assign(tabDMLScript, "yb", Caffe2DML.yVal)
        net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, false))
        lossLayer.computeLoss(dmlScript, numTabs)
        assign(tabDMLScript, "validation_loss", "loss"); assign(tabDMLScript, "validation_accuracy", "accuracy")

      }
      case Caffe2DML.ALLREDUCE_PARALLEL_BATCHES_ALGORITHM => {
        // This setting uses the batch size provided by the user
        assign(tabDMLScript, "max_validation_iter", "as.integer(ceil(" + Caffe2DML.numValidationImages + "/" + Caffe2DML.batchSize + "))")
        assign(tabDMLScript, "group_validation_loss", matrix("0", "max_validation_iter", "1"))
        assign(tabDMLScript, "group_validation_accuracy", matrix("0", "max_validation_iter", "1"))
        parForBlock("iVal", "1", "max_validation_iter", "1", getParforParameters()) {
          assign(tabDMLScript, "validation_beg", "(iVal-1) * " + Caffe2DML.batchSize + " + 1")
          assign(tabDMLScript, "validation_end", min(Caffe2DML.numValidationImages, "validation_beg + " + Caffe2DML.batchSize + " - 1"))
          assign(tabDMLScript, "Xb", Caffe2DML.XVal + "[validation_beg:validation_end,]")
          assign(tabDMLScript, "yb", Caffe2DML.yVal + "[validation_beg:validation_end,]")
          net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, false))
          lossLayer.computeLoss(dmlScript, numTabs)
          assign(tabDMLScript, "group_validation_loss[iVal,1]", "loss")
          assign(tabDMLScript, "group_validation_accuracy[iVal,1]", "accuracy")
        }
        assign(tabDMLScript, "validation_loss", "sum(group_validation_loss)")
        assign(tabDMLScript, "validation_accuracy", "mean(group_validation_accuracy)")
      }
      case Caffe2DML.ALLREDUCE_ALGORITHM => {
        // This setting doesnot use the batch size for validation and allows the parfor optimizer to select plan
        // by minimizing the memory requirement (i.e. batch size = 1)
        assign(tabDMLScript, "group_validation_loss", matrix("0", Caffe2DML.numValidationImages, "1"))
        assign(tabDMLScript, "group_validation_accuracy", matrix("0", Caffe2DML.numValidationImages, "1"))
        parForBlock("iVal", "1", Caffe2DML.numValidationImages, "1", getParforParameters()) {
          assign(tabDMLScript, "Xb", Caffe2DML.XVal + "[iVal,]")
          assign(tabDMLScript, "yb", Caffe2DML.yVal + "[iVal,]")
          net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, false))
          lossLayer.computeLoss(dmlScript, numTabs)
          assign(tabDMLScript, "group_validation_loss[iVal,1]", "loss")
          assign(tabDMLScript, "group_validation_accuracy[iVal,1]", "accuracy")
        }
        assign(tabDMLScript, "validation_loss", "sum(group_validation_loss)")
        assign(tabDMLScript, "validation_accuracy", "mean(group_validation_accuracy)")
      }

      case _ => throw new DMLRuntimeException("Unsupported test algo:" + getTestAlgo)
    }
    tabDMLScript.append(
      print(dmlConcat(asDMLString("Iter:"), "iter", asDMLString(", validation loss:"), "validation_loss", asDMLString(", validation accuracy:"), "validation_accuracy"))
    )
  }
  private def appendSnapshotWrite(varName: String, fileName: String): Unit =
    tabDMLScript.append(write(varName, "snapshot_dir + \"" + fileName + "\"", "binary"))
  private def performSnapshot(): Unit =
    if (solverParam.getSnapshot > 0) {
      ifBlock("iter %% " + solverParam.getSnapshot + " == 0") {
        tabDMLScript.append("snapshot_dir= \"" + solverParam.getSnapshotPrefix + "\" + \"/iter_\" + iter + \"/\"\n")
        val allLayers = net.getLayers.map(net.getCaffeLayer(_))
        allLayers.filter(_.weight != null).map(l => appendSnapshotWrite(l.weight, l.param.getName + "_weight.mtx"))
        allLayers.filter(_.extraWeight != null).map(l => appendSnapshotWrite(l.extraWeight, l.param.getName + "_extra_weight.mtx"))
        allLayers.filter(_.bias != null).map(l => appendSnapshotWrite(l.bias, l.param.getName + "_bias.mtx"))
      }
    }

  private def forward(): Unit = {
    tabDMLScript.append("# Perform forward pass\n")
    net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, false))
  }
  private def backward(): Unit = {
    tabDMLScript.append("# Perform backward pass\n")
    net.getLayers.reverse.map(layer => net.getCaffeLayer(layer).backward(tabDMLScript, ""))
  }
  private def update(): Unit = {
    tabDMLScript.append("# Update the parameters\n")
    net.getLayers.map(layer => solver.update(tabDMLScript, net.getCaffeLayer(layer)))
  }
  private def initializeGradients(parallel_batches: String): Unit = {
    tabDMLScript.append("# Data structure to store gradients computed in parallel\n")
    if(Caffe2DML.USE_PLUS_EQ) {
      net.getLayers
        .map(layer => net.getCaffeLayer(layer))
        .map(l => {
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dWeight + "_agg", matrix("0", nrow(l.weight), ncol(l.weight)))
          if (l.shouldUpdateExtraWeight) assign(tabDMLScript, l.dExtraWeight + "_agg", matrix("0", nrow(l.extraWeight), ncol(l.extraWeight)))
          if (l.shouldUpdateBias) assign(tabDMLScript, l.dBias + "_agg", matrix("0", nrow(l.bias), ncol(l.bias)))
        })
    }
    else {
      net.getLayers
        .map(layer => net.getCaffeLayer(layer))
        .map(l => {
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dWeight + "_agg", matrix("0", parallel_batches, multiply(nrow(l.weight), ncol(l.weight))))
          if (l.shouldUpdateExtraWeight) assign(tabDMLScript, l.dExtraWeight + "_agg", matrix("0", parallel_batches, multiply(nrow(l.extraWeight), ncol(l.extraWeight))))
          if (l.shouldUpdateBias) assign(tabDMLScript, l.dBias + "_agg", matrix("0", parallel_batches, multiply(nrow(l.bias), ncol(l.bias))))
        })
    }
  }
  private def flattenGradients(): Unit = {
    if(Caffe2DML.USE_PLUS_EQ) {
      // Note: We multiply by a weighting to allow for proper gradient averaging during the
      // aggregation even with uneven batch sizes.
      assign(tabDMLScript, "weighting", "1/parallel_batches") // "nrow(Xb)/X_group_batch_size")
      net.getLayers
        .map(layer => net.getCaffeLayer(layer))
        .map(l => {
          if (l.shouldUpdateWeight) assignPlusEq(tabDMLScript, l.dWeight + "_agg", l.dWeight + "*weighting")
          if (l.shouldUpdateExtraWeight) assignPlusEq(tabDMLScript, l.dExtraWeight + "_agg", l.dExtraWeight + "*weighting")
          if (l.shouldUpdateWeight) assignPlusEq(tabDMLScript, l.dBias + "_agg", l.dBias + "*weighting")
        })
    }
    else {
      tabDMLScript.append("# Flatten and store gradients for this parallel execution\n")
      // Note: We multiply by a weighting to allow for proper gradient averaging during the
      // aggregation even with uneven batch sizes.
      assign(tabDMLScript, "weighting", "1/parallel_batches") // "nrow(Xb)/X_group_batch_size")
      net.getLayers
        .map(layer => net.getCaffeLayer(layer))
        .map(l => {
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dWeight + "_agg[j,]", matrix(l.dWeight, "1", multiply(nrow(l.weight), ncol(l.weight))) + " * weighting")
          if (l.shouldUpdateExtraWeight) assign(tabDMLScript, l.dExtraWeight + "_agg[j,]", matrix(l.dExtraWeight, "1", multiply(nrow(l.extraWeight), ncol(l.extraWeight))) + " * weighting")
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dBias + "_agg[j,]", matrix(l.dBias, "1", multiply(nrow(l.bias), ncol(l.bias))) + " * weighting")
        })
    }
  }
  
  private def aggregateAggGradients(): Unit = {
    if(Caffe2DML.USE_PLUS_EQ) {
      net.getLayers
        .map(layer => net.getCaffeLayer(layer))
        .map(l => {
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dWeight, l.dWeight + "_agg")
          if (l.shouldUpdateExtraWeight) assign(tabDMLScript, l.dExtraWeight, l.dExtraWeight + "_agg")
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dBias, l.dBias + "_agg")
        })
    }
    else {
      tabDMLScript.append("# Aggregate the gradients\n")
      net.getLayers
        .map(layer => net.getCaffeLayer(layer))
        .map(l => {
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dWeight, matrix(colSums(l.dWeight + "_agg"), nrow(l.weight), ncol(l.weight)))
          if (l.shouldUpdateExtraWeight) assign(tabDMLScript, l.dExtraWeight, matrix(colSums(l.dExtraWeight + "_agg"), nrow(l.extraWeight), ncol(l.extraWeight)))
          if (l.shouldUpdateWeight) assign(tabDMLScript, l.dBias, matrix(colSums(l.dBias + "_agg"), nrow(l.bias), ncol(l.bias)))
        })
    }
  }
  // -------------------------------------------------------------------------------------------
}

class Caffe2DMLModel(val numClasses: String, val sc: SparkContext, val solver: CaffeSolver, val net: CaffeNetwork, val lrPolicy: LearningRatePolicy, val estimator: Caffe2DML)
    extends Model[Caffe2DMLModel]
    with HasMaxOuterIter
    with BaseSystemMLClassifierModel
    with DMLGenerator {
  // --------------------------------------------------------------
  // Invoked by Python, MLPipeline
  val uid: String = "caffe_model_" + (new Random).nextLong
  def this(estimator: Caffe2DML) = {
    this(
      Utils.numClasses(estimator.net),
      estimator.sc,
      estimator.solver,
      estimator.net,
      // new CaffeNetwork(estimator.solverParam.getNet, caffe.Caffe.Phase.TEST, estimator.numChannels, estimator.height, estimator.width),
      estimator.lrPolicy,
      estimator
    )
  }

  override def copy(extra: org.apache.spark.ml.param.ParamMap): Caffe2DMLModel = {
    val that = new Caffe2DMLModel(numClasses, sc, solver, net, lrPolicy, estimator)
    copyValues(that, extra)
  }
  // --------------------------------------------------------------

  def modelVariables(): List[String] = {
    val allLayers = net.getLayers.map(net.getCaffeLayer(_))
    allLayers.filter(_.weight != null).map(_.weight) ++ allLayers.filter(_.extraWeight != null).map(_.extraWeight) ++ allLayers.filter(_.bias != null).map(_.bias)
  }

  // ================================================================================================
  // The below method parses the provided network and solver file and generates DML script.
  def getPredictionScript(isSingleNode: Boolean): (Script, String) = {
    val startPredictionTime = System.nanoTime()

    reset // Reset the state of DML generator for training script.

    val DEBUG_PREDICTION = if (estimator.inputs.containsKey("$debug")) estimator.inputs.get("$debug").toLowerCase.toBoolean else false
    assign(tabDMLScript, "debug", if (DEBUG_PREDICTION) "TRUE" else "FALSE")
    estimator.setDebugFlags(DEBUG_PREDICTION)

    appendHeaders(net, solver, false) // Appends DML corresponding to source and externalFunction statements.
    val performOneHotEncoding = !estimator.inputs.containsKey("$perform_one_hot_encoding") || estimator.inputs.get("$perform_one_hot_encoding").toBoolean
    readInputData(net, false, performOneHotEncoding)         // Read X_full and y_full
    assign(tabDMLScript, "X", "X_full")

    // Initialize the layers and solvers. Reads weights and bias if readWeights is true.
    if (!estimator.inputs.containsKey("$weights") && estimator.mloutput == null)
      throw new DMLRuntimeException("Cannot call predict/score without calling either fit or by providing weights")
    val readWeights = estimator.inputs.containsKey("$weights") || estimator.mloutput != null
    initWeights(net, solver, readWeights)

    // Donot update mean and variance in batchnorm
    updateMeanVarianceForBatchNorm(net, false)

    val lossLayers     = getLossLayers(net)
    val lastLayerShape = estimator.getOutputShapeOfLastLayer
    assign(tabDMLScript, "Prob", matrix("1", Caffe2DML.numImages, (lastLayerShape._1 * lastLayerShape._2 * lastLayerShape._3).toString))
    estimator.getTestAlgo.toLowerCase match {
      case Caffe2DML.MINIBATCH_ALGORITHM => {
        ceilDivide(tabDMLScript(), "num_iters", Caffe2DML.numImages, Caffe2DML.batchSize)
        forBlock("iter", "1", "num_iters") {
          getTestBatch(tabDMLScript)
          net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, true))
          assign(tabDMLScript, "Prob[beg:end,]", lossLayers(0).out)
        }
      }
      case Caffe2DML.BATCH_ALGORITHM => {
        assign(tabDMLScript, "Xb", "X_full")
        net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, true))
        assign(tabDMLScript, "Prob", lossLayers(0).out)
      }
      case Caffe2DML.ALLREDUCE_PARALLEL_BATCHES_ALGORITHM => {
        // This setting uses the batch size provided by the user
        if (!estimator.inputs.containsKey("$parallel_batches")) {
          throw new RuntimeException("The parameter parallel_batches is required for allreduce_parallel_batches")
        }
        // The user specifies the number of parallel_batches
        // This ensures that the user of generated script remembers to provide the commandline parameter $parallel_batches
        assign(tabDMLScript, "parallel_batches", "$parallel_batches")
        assign(tabDMLScript, "group_batch_size", "parallel_batches*" + Caffe2DML.batchSize)
        assign(tabDMLScript, "groups", "as.integer(ceil(" + Caffe2DML.numImages + "/group_batch_size))")
        // Grab groups of mini-batches
        forBlock("g", "1", "groups") {
          assignBatch(tabDMLScript, "X_group_batch", "X_full", null, null, "group_", Caffe2DML.numImages, "g", "group_batch_size")
          assign(tabDMLScript, "X_group_batch_size", nrow("X_group_batch"))
          //  Run graph on each mini-batch in this group in parallel (ideally on multiple GPUs)
          parForBlock("j", "1", "parallel_batches", "1", estimator.getParforParameters()) {
            assignBatch(tabDMLScript, "Xb", "X_group_batch", null, null, "", "X_group_batch_size", "j", Caffe2DML.batchSize)
            net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, true))
            assign(tabDMLScript, "Prob[beg:end,]", lossLayers(0).out)
          }
        }
      }
      case Caffe2DML.ALLREDUCE_ALGORITHM => {
        // This setting doesnot use the batch size for scoring and allows the parfor optimizer to select the best plan
        // by minimizing the memory requirement (i.e. batch size = 1)
        parForBlock("iter", "1", Caffe2DML.numImages, "1", estimator.getParforParameters()) {
          assign(tabDMLScript, "Xb", "X_full[iter,]")
          net.getLayers.map(layer => net.getCaffeLayer(layer).forward(tabDMLScript, true))
          assign(tabDMLScript, "Prob[iter,]", lossLayers(0).out)
        }
      }
      case _ => throw new DMLRuntimeException("Unsupported test algo:" + estimator.getTestAlgo)
    }

    if (estimator.inputs.containsKey("$output_activations")) {
      if (estimator.getTestAlgo.toLowerCase.equals(Caffe2DML.BATCH_ALGORITHM)) {
        net.getLayers.map(
          layer =>
            tabDMLScript.append(
              write(net.getCaffeLayer(layer).out, estimator.inputs.get("$output_activations") + "/" + net.getCaffeLayer(layer).param.getName + "_activations.mtx", "csv") + "\n"
          )
        )
      } else {
        throw new DMLRuntimeException("Incorrect usage of output_activations. It should be only used in batch mode.")
      }
    }

    val predictionScript = dmlScript.toString()
    System.out.println("Time taken to generate prediction script from Caffe proto:" + ((System.nanoTime() - startPredictionTime) * 1e-9) + "secs.")
    if (DEBUG_PREDICTION) Utils.prettyPrintDMLScript(predictionScript)

    // Reset state of BatchNorm layer
    updateMeanVarianceForBatchNorm(net, true)

    val script = dml(predictionScript).out("Prob").in(estimator.inputs)
    if (estimator.mloutput != null) {
      // fit was called
      net.getLayers.map(net.getCaffeLayer(_)).filter(_.weight != null).map(l => script.in(l.weight, estimator.mloutput.getMatrix(l.weight)))
      net.getLayers.map(net.getCaffeLayer(_)).filter(_.extraWeight != null).map(l => script.in(l.extraWeight, estimator.mloutput.getMatrix(l.extraWeight)))
      net.getLayers.map(net.getCaffeLayer(_)).filter(_.bias != null).map(l => script.in(l.bias, estimator.mloutput.getMatrix(l.bias)))
    }
    
    estimator.setDebugFlags(false)
    
    (script, "X_full")
  }
  // ================================================================================================

  def baseEstimator(): BaseSystemMLEstimator = estimator

  // Prediction
  def transform(X_file: String): String =
    if (estimator.isClassification) {
      Caffe2DML.LOG.debug("Prediction assuming classification")
      baseTransform(X_file, sc, "Prob")
    } else {
      Caffe2DML.LOG.debug("Prediction assuming segmentation")
      val outShape = estimator.getOutputShapeOfLastLayer
      baseTransform(X_file, sc, "Prob", outShape._1.toInt, outShape._2.toInt, outShape._3.toInt)
    }
  def transform(X: MatrixBlock): MatrixBlock =
    if (estimator.isClassification) {
      Caffe2DML.LOG.debug("Prediction assuming classification")
      baseTransform(X, sc, "Prob")
    } else {
      Caffe2DML.LOG.debug("Prediction assuming segmentation")
      val outShape = estimator.getOutputShapeOfLastLayer
      baseTransform(X, sc, "Prob", outShape._1.toInt, outShape._2.toInt, outShape._3.toInt)
    }
  def transform_probability(X_file: String): String =
    if (estimator.isClassification) {
      Caffe2DML.LOG.debug("Prediction of probability assuming classification")
      baseTransformProbability(X_file, sc, "Prob")
    } else {
      Caffe2DML.LOG.debug("Prediction of probability assuming segmentation")
      val outShape = estimator.getOutputShapeOfLastLayer
      baseTransformProbability(X_file, sc, "Prob", outShape._1.toInt, outShape._2.toInt, outShape._3.toInt)
    }
  def transform_probability(X: MatrixBlock): MatrixBlock =
    if (estimator.isClassification) {
      Caffe2DML.LOG.debug("Prediction of probability assuming classification")
      baseTransformProbability(X, sc, "Prob")
    } else {
      Caffe2DML.LOG.debug("Prediction of probability assuming segmentation")
      val outShape = estimator.getOutputShapeOfLastLayer
      baseTransformProbability(X, sc, "Prob", outShape._1.toInt, outShape._2.toInt, outShape._3.toInt)
    }

  def transform(df: ScriptsUtils.SparkDataType): DataFrame =
    if (estimator.isClassification) {
      Caffe2DML.LOG.debug("Prediction assuming classification")
      baseTransform(df, sc, "Prob", true)
    } else {
      Caffe2DML.LOG.debug("Prediction assuming segmentation")
      val outShape = estimator.getOutputShapeOfLastLayer
      baseTransform(df, sc, "Prob", true, outShape._1.toInt, outShape._2.toInt, outShape._3.toInt)
    }
}
