/*******************************************************************************
 * Copyright (c) 2010 Engineering Group.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * 			Andrea Zoppello (Engineering Group) - initial API and implementation and/or initial documentation
 * 			Gianfranco Boccalon (Engineering Group) - initial API and implementation and/or initial documentation
 *          Luca Rossato ( Engineering Group ) - initial API and implementation and/or initial documentation
 *          Luca Barozzi ( Engineering Group ) - initial API and implementation and/or initial documentation
 *  		Antonietta Miele ( Engineering Group ) - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.ebpm.connectors.http;

import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.wsdl.Definition;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.wsdl.WSDLException;
import javax.wsdl.extensions.soap.SOAPAddress;
import javax.wsdl.extensions.soap12.SOAP12Address;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;
import javax.xml.namespace.QName;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.eclipse.ebpm.connectors.http.adapters.IHTTPInputProtocolAdapter;
import org.eclipse.ebpm.connectors.http.ssl.SslBean;
import org.eclipse.ebpm.connectors.http.util.WSSecurityConstants;
import org.eclipse.ebpm.constants.SpagicConstants;
import org.eclipse.ebpm.core.AbstractSpagicConnector;
import org.eclipse.ebpm.messaging.api.Exchange;
import org.eclipse.ebpm.messaging.api.Message;
import org.eclipse.ebpm.soap.wsdl.BindingFactory;
import org.eclipse.ebpm.soap.wsdl.WSDLUtils;
import org.eclipse.ebpm.util.resources.IResource;
import org.eclipse.jetty.continuation.Continuation;
import org.eclipse.jetty.continuation.ContinuationSupport;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;



public class HTTPServer extends AbstractSpagicConnector {
	
	private Logger logger = LoggerFactory.getLogger(HTTPServer.class);
	private String locationURI = null;
	private boolean ssl = false; 
	private SslBean sslBean = null;
	private ExchangeContinuationsTracker exchangeContinuationsTracker = new ExchangeContinuationsTracker();
	private String mep = null;
	private long timeout = -1;
	private boolean isSoap = false;
	private IResource wsdl = null;
	private ConcurrentHashMap<String, IHTTPInputProtocolAdapter> marshalers = new ConcurrentHashMap<String, IHTTPInputProtocolAdapter>();
	private String marshalerId;
	private IHTTPInputProtocolAdapter adapter = null;
	private org.eclipse.ebpm.soap.api.model.Binding<?> binding = null;
	
	private QName service = null;
	private String port = null;
	private boolean dynamicTargetPolicy = false;

	public String getPort() {
		return port;
	}

	public void setPort(String port) {
		this.port = port;
	}

	public QName getService() {
		return service;
	}

	public void setService(QName service) {
		this.service = service;
	}

//	private IHTTPInputProtocolAdapter adapter = null; 
	
	public boolean isSoap() {
		return isSoap;
	}

	public void setSoap(boolean isSoap) {
		this.isSoap = isSoap;
	}

	
	
	public void init(){
		logger.info("-- HTTP Server Component Init --");
		this.locationURI = propertyConfigurator.getString("locationURI");
		this.timeout = propertyConfigurator.getLong("timeout", (long)60000);
		this.mep = propertyConfigurator.getString("mep", SpagicConstants.IN_OUT_MEP);
		this.isSoap = propertyConfigurator.getBoolean("isSoap", false);
		this.ssl = propertyConfigurator.getBoolean("isSSL", false);
		this.dynamicTargetPolicy = propertyConfigurator.getBoolean("dynamicTargetPolicy", false);
		
		
		if (ssl)
			this.sslBean = getSslParameters();
		
		if (isSoap){
			this.marshalerId = propertyConfigurator.getString("marshaller","SOAP");
			this.wsdl = propertyConfigurator.getResource("wsdl");
		}
		else{
			this.marshalerId = propertyConfigurator.getString("marshaller","PlainHTTP");
		}
		
		this.adapter = marshalers.get(marshalerId);
		
		Map<String, String> map = new HashMap<String, String>();
	 	String senderAction = propertyConfigurator.getString("senderAction", null);
	 	String receiverAction = propertyConfigurator.getString("receiverAction", null);
	 	String wsTSenderAction = propertyConfigurator.getString("wsTSenderAction", null);
	 	String wsTReceiverAction = propertyConfigurator.getString("wsTReceiverAction", null);
		
	 	if(receiverAction != null){
			if(receiverAction.equals(WSSecurityConstants.SIGNATURE)){
				map.put("receiverAction", receiverAction);
				map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("receiverSignaturePropFile").pathAsString());
		 	}else if(receiverAction.equals("Encryption")){
		 		map.put("receiverAction", WSSecurityConstants.ENCRYPT);
		 		map.put(WSSecurityConstants.DEC_PROP_FILE, propertyConfigurator.getResource("decryptionPropFile").pathAsString());
		 		map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("receiverPWCallbackClass"));
		 	}else if(receiverAction.equals("Encryption + Signature")){
		 		map.put("receiverAction", WSSecurityConstants.ENCRYPT+" "+WSSecurityConstants.SIGNATURE);
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("receiverSignaturePropFile").pathAsString());
		        map.put(WSSecurityConstants.DEC_PROP_FILE, propertyConfigurator.getResource("decryptionPropFile").pathAsString());
		        map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("receiverPWCallbackClass"));
		 	}
	 	}else if(wsTReceiverAction != null){
	 		if(wsTReceiverAction.equals("Signature + Encryption + Timestamp")){
		 		map.put("receiverAction", WSSecurityConstants.SIGNATURE+" "+WSSecurityConstants.ENCRYPT+" "+WSSecurityConstants.TIMESTAMP);
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("wsTreceiverSignaturePropFile").pathAsString());
		        map.put(WSSecurityConstants.DEC_PROP_FILE, propertyConfigurator.getResource("wsTdecryptionPropFile").pathAsString());
		        map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("wsTreceiverPWCallbackClass"));
		 	}
	 	}
		
		if(senderAction != null){
			if(senderAction.equals(WSSecurityConstants.SIGNATURE)){
				map.put("senderAction", senderAction);
		 		map.put(WSSecurityConstants.USER, propertyConfigurator.getString("user"));
		 		map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("senderPWCallbackClass"));
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("senderSignaturePropFile").pathAsString());
		 		map.put(WSSecurityConstants.SIG_KEY_ID, propertyConfigurator.getString("signatureKeyIdentifier"));
		 	}else if(senderAction.equals("Encryption")){
		 		map.put("senderAction", WSSecurityConstants.ENCRYPT);
		 		map.put(WSSecurityConstants.ENCRYPTION_USER, propertyConfigurator.getString("encryptionUser"));
				map.put(WSSecurityConstants.ENC_PROP_FILE, propertyConfigurator.getResource("encryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.ENC_KEY_ID, propertyConfigurator.getString("encryptionKeyIdentifier"));
		 	}else if(senderAction.equals("Encryption + Signature")){
		 		map.put("senderAction", WSSecurityConstants.ENCRYPT+" "+WSSecurityConstants.SIGNATURE);
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("senderSignaturePropFile").pathAsString());
		 		map.put(WSSecurityConstants.SIG_KEY_ID, propertyConfigurator.getString("signatureKeyIdentifier"));
				map.put(WSSecurityConstants.USER, propertyConfigurator.getString("user"));
				map.put(WSSecurityConstants.ENCRYPTION_USER, propertyConfigurator.getString("encryptionUser"));
				map.put(WSSecurityConstants.ENC_PROP_FILE, propertyConfigurator.getResource("encryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.ENC_KEY_ID, propertyConfigurator.getString("encryptionKeyIdentifier"));
				map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("senderPWCallbackClass"));
		 	}
		}
		else if (wsTSenderAction != null){
			if(wsTSenderAction.equals("Signature + Encryption + Timestamp")){
		 		map.put("senderAction", WSSecurityConstants.SIGNATURE+" "+WSSecurityConstants.ENCRYPT+" "+WSSecurityConstants.TIMESTAMP);
		 		map.put(WSSecurityConstants.SIG_PROP_FILE, propertyConfigurator.getResource("wsTsenderSignaturePropFile").pathAsString());
		 		map.put(WSSecurityConstants.SIG_KEY_ID, propertyConfigurator.getString("wsTsignatureKeyIdentifier"));
				map.put(WSSecurityConstants.USER, propertyConfigurator.getString("wsTuser"));
				map.put(WSSecurityConstants.ENCRYPTION_USER, propertyConfigurator.getString("wsTencryptionUser"));
				map.put(WSSecurityConstants.ENC_PROP_FILE, propertyConfigurator.getResource("wsTencryptionPropFile").pathAsString());
				map.put(WSSecurityConstants.ENC_KEY_ID, propertyConfigurator.getString("wsTencryptionKeyIdentifier"));
				map.put(WSSecurityConstants.PW_CALLBACK_CLASS, propertyConfigurator.getString("wsTsenderPWCallbackClass"));
		 	}
		}
		
		boolean isWsTrustSignature = propertyConfigurator.getBoolean("isWsTrustSignature", false);
		boolean isWsTrustEncryption = propertyConfigurator.getBoolean("isWsTrustEncryption", false);
		boolean isWsTrust = propertyConfigurator.getBoolean("isWsTrust", false);
		
		if(isWsTrust)
			map.put("isWsTrust", "true");
		
		if(isWsTrustSignature){
			map.put("wsTrustSignatureActive", "true");
			map.put("wsSignaturePropFile", propertyConfigurator.getString("wsSignaturePropFile"));
		}
		
		if(isWsTrustEncryption){
			map.put("wsTrustEncryptionActive", "true");
			map.put("wsEncryptionPropFile", propertyConfigurator.getString("wsEncryptionPropFile"));
			map.put("callbackClass", propertyConfigurator.getString("callbackClass"));
		}
		
		adapter.setProperty(map);
		
		validate();
	}
	
	
	private void validate() {
		if (isSoap) {
			if (wsdl == null) {
				throw new RuntimeException(
						"If HttpServer service is a SOAP Service wsdl property must be set");
			}
			try {
				SAXReader reader = new SAXReader();
				Document wsdlDocument = reader.read(wsdl.openStream());
				Element rootElement = wsdlDocument.getRootElement();
				if (WSDLUtils.WSDL1_NAMESPACE.equals(rootElement
						.getNamespaceURI())) {
					//
					// It's a WSDL 1 Namespace
					//
					checkWsdl11();
				} else if (WSDLUtils.WSDL2_NAMESPACE.equals(rootElement
						.getNamespaceURI())) {
					//
					// It's a WSDL2 Namespace
					//
					checkWsdl2();
				} else {
					throw new RuntimeException("Unrecognized wsdl namespace: "
							+ rootElement.getNamespaceURI());
				}
			} catch (Exception e) {
				throw new RuntimeException(e);
			}
		}
	}
	
	public void checkWsdl2(){
		// USE WOODEN
	}
	
	public void checkWsdl11(){
        try{
        	WSDLFactory wsdlFactory = WSDLFactory.newInstance();
    		WSDLReader reader = wsdlFactory.newWSDLReader();
    		
    		Definition def = reader.readWSDL(null, new InputSource(wsdl.openStream()));
    		//Definition def = reader.readWSDL(wsdl.asURL().toString());
    		
    		/*
    		WSIBPValidator validator = new WSIBPValidator(def);
            if (!validator.isValid()) {
                throw new RuntimeException("WSDL is not WS-I BP compliant: " + validator.getErrors());
            } 
            */ 
            Service svc = null;
            if (getService() != null) {
                svc = def.getService(getService());
                if (svc == null) {
                    throw new RuntimeException("Could not find service '" + getService() + "' in wsdl");
                }
            } else if (def.getServices().size() == 1) {
                svc = (Service)def.getServices().values().iterator().next();
                setService(svc.getQName());
            } else {
                throw new RuntimeException("If service is not set, the WSDL must contain a single service definition");
            }
            Port port;
            if (getPort() != null) {
                port = svc.getPort(getPort());
                if (port == null) {
                    throw new RuntimeException("Cound not find port '" + getPort()
                                                  + "' in wsdl for service '" + getService() + "'");
                }
            } else if (svc.getPorts().size() == 1) {
                port = (Port)svc.getPorts().values().iterator().next();
                setPort(port.getName());
            } else {
                throw new RuntimeException("If endpoint is not set, the WSDL service '" + getService()
                                              + "' must contain a single port definition");
            }
            SOAPAddress soapAddress = WSDLUtils.getExtension(port, SOAPAddress.class);
            if (soapAddress != null) {
                soapAddress.setLocationURI(this.locationURI);
            } else {
                SOAP12Address soap12Address = WSDLUtils.getExtension(port, SOAP12Address.class);
                if (soap12Address != null) {
                    soap12Address.setLocationURI(this.locationURI);
                }
            }
            this.binding = BindingFactory.createBinding(port);
        }catch (WSDLException e) {
			throw new RuntimeException(e);
		}
	}

	public boolean isSsl() {
		return ssl;
	}

	public void setSsl(boolean ssl) {
		this.ssl = ssl;
	}

	@Override
	public void start() throws Exception {
		super.start();
		boolean isSsl = isSslConfigured();

		Server server = HttpServerManager.configureServer(this.locationURI, this.propertyConfigurator, isSsl, sslBean);
		ClassLoader bundleClassLoader = this.getClass().getClassLoader();
		ClassLoader oldClassLoader = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(bundleClassLoader);
		if (!server.isStarted()){
			server.start();
		}
		ContextHandler context = HttpServerManager.configureContext(server, this.locationURI, this.propertyConfigurator, isSsl, sslBean, this);

		if (!context.isStarted()){
			context.start();
		}

		Thread.currentThread().setContextClassLoader(oldClassLoader);
	}

	@Override
	public void stop() throws Exception {
		super.stop();

		HttpServerManager.unconfigureContext(this.locationURI);


	}
	
	private boolean isSslConfigured() throws Exception {
		URL locationURL = new URL(this.locationURI);
		boolean isSsl = false;
        if (locationURL.getProtocol().equals("https")) {
            if (sslBean == null) {
                throw new IllegalArgumentException("https require SSL Properties");
            }
            isSsl = true;
        } else if (!locationURL.getProtocol().equals("http")) {
            throw new UnsupportedOperationException("Only http and https Protocol Are supported");
        } 
        return isSsl;
	}

	private SslBean getSslParameters(){
		
		IResource keyStoreResource = propertyConfigurator.getResource("keystore", null);
		String keyStorePassword = propertyConfigurator.getString("keystorePassword","");
		String keyStoreType = propertyConfigurator.getString("keystoreType", "JKS");
		
		IResource trustStoreResource = propertyConfigurator.getResource("truststore", null);
		String trustStorePassword = propertyConfigurator.getString("truststorePassword", "");
		String trustStoreType = propertyConfigurator.getString("truststoreType", "JKS");
		
		String clientAuthentication = propertyConfigurator.getString("clientAuthentication", "NoClientAuthentication");
		SslBean sslBean = new SslBean();

		if (clientAuthentication.equalsIgnoreCase("NoClientAuthentication")){
			sslBean.setNeedClientAuth(false);
			sslBean.setWantClientAuth(false);
		}else if (clientAuthentication.equalsIgnoreCase("OptionalClientAuthentication")){
			sslBean.setNeedClientAuth(false);
			sslBean.setWantClientAuth(true);
		}else if (clientAuthentication.equalsIgnoreCase("MandatoryClientAuthentication")){
			sslBean.setNeedClientAuth(true);
			sslBean.setWantClientAuth(true);
			//Trustore needed only if NeedClientAuth=TRUE
			sslBean.setTrustStore(trustStoreResource.asURL().toString());
			sslBean.setTrustStorePassword(trustStorePassword);
			sslBean.setTrustStoreType(trustStoreType);
		}	

		sslBean.setKeyStore(keyStoreResource.asURL().toString());
		sslBean.setKeyStorePassword(keyStorePassword);
		sslBean.setKeyStoreType(keyStoreType);		
		
		return sslBean;
	}
	
	
	public void processHttp(HttpServletRequest request, HttpServletResponse response) throws IOException {
		
		try{
			Continuation continuation = ContinuationSupport.getContinuation(request);
			exchangeContinuationsTracker.handle(request, response, continuation, this.timeout, this);
		} catch (Exception e) {
			e.printStackTrace();
			try{
				responseError(null, e, request, response);
			}catch (Exception ex) {
				ex.printStackTrace();
			}
		}
		
		
	}

	@Override
	public void process(Exchange exchange) {
		exchangeContinuationsTracker.exchangeArrived(exchange);
	}
	
	public String getTarget(HttpServletRequest request){
		if (!isSoap && dynamicTargetPolicy){
			// If the component is not SOAP the spagic-target-service could be specified in a heder called
			// "spagic-target-service"
			return request.getHeader("spagic-target-service");
		}
		return this.target;
	}
	public Exchange createExchange(HttpServletRequest request) throws Exception {
		return adapter.createExchange(request, mep, getSpagicId(), getTarget(request), this.propertyConfigurator, this.binding);
    }

    public void responseAccepted(Exchange exchange, HttpServletRequest request,
                             HttpServletResponse response) throws Exception {
    	 adapter.sendAccepted(exchange, request, response,this.propertyConfigurator, this.binding);
    }

    public void responseError(Exchange exchange, Exception error, HttpServletRequest request,
                          HttpServletResponse response) throws Exception {
    	adapter.sendError(null,error, request, response, this.propertyConfigurator, this.binding);
    		
    }

    public void responseFault(Exchange exchange, Message fault, HttpServletRequest request,
                          HttpServletResponse response) throws Exception {
    	adapter.sendFault(exchange, fault, request, response, this.propertyConfigurator, this.binding);
    }

    public void responseOut(Exchange exchange, Message outMsg, HttpServletRequest request,
                        HttpServletResponse response) throws Exception {
    	adapter.sendOut(exchange, outMsg, request, response,this.propertyConfigurator, this.binding);
    	
    }

    public void setMarshaler(IHTTPInputProtocolAdapter marshaler) {
		this.marshalers.put(marshaler.getAdapterId(), marshaler);
	}

	public void unsetMarshaler(IHTTPInputProtocolAdapter marshaler) {
		this.marshalers.remove(marshaler.getAdapterId());  }
    
}
