(* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is JavaScript Bridge.
 *
 * The Initial Developer of the Original Code is
 * Sterling Bates.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Theo Lustenberger <theo@theo.ch>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** *)

unit jstesting;

interface

uses jsintf, js15decl, jssamples, jsbridge, jstesting_pvt, SysUtils;

const
  SuppressSuccesses: Boolean = false;

function TestAll: String;
function TestBridge: String;
function TestNonBridge: String;
function TestTJSObject: String;
function TestTJSString: String;
function TestTJSDouble: String;
function TestTJSInteger: String;
function TestTJSBoolean: String;
function TestTJSFunction: String;
function TestTJSArray: String;
function TestSamples: String;
function TestSerialization: String;
function TestEnumeration: String;
function TestErrorReporting: String;

implementation

procedure AddOutput(var Str: String; Msg: String);
begin
  Str := Str +Msg +#10;
end;

procedure AddResult(var Str: String; Msg: String; Cond: Boolean;
                    const Value: String = '');
begin
  if (not Cond) then
    AddOutput(Str,'!!! ' +Msg +': failed')
  else if (not SuppressSuccesses) then
    AddOutput(Str,Msg +': succeeded')
end;

function TestAll: String;
begin

end;

function TestBridge: String;
var
  eng: TJSEngine;
  bridge: TTestJSBridge;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    bridge := TTestJSBridge.Create(eng,'bridge');
    try
      eng.Evaluate('bridge.Int = 1668');
      AddResult(Result,'Integer property assignment',bridge.Int = 1668);

      eng.Evaluate('bridge.Value = "lorem ipsum dolor sit amet"');
      AddResult(Result,'String property assignment',
                bridge.Value = 'lorem ipsum dolor sit amet');

      eng.Evaluate('bridge.SetStrFromJS("cogito ergo sum")');
      AddResult(Result,'String method assignment',
                bridge.Value = 'cogito ergo sum');
    finally
      bridge.Free;
    end;
 finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestNonBridge: String;
var
  eng: TJSEngine;
  nonbridge: TTestNonBridge;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    nonbridge := TTestNonBridge.Create;
    try
      eng.Global.AddNativeObject(nonbridge, 'nonbridge');
      eng.Evaluate('nonbridge.Int = 1668');
      AddResult(Result,'Integer property assignment',nonbridge.Int = 1668);

      eng.Evaluate('nonbridge.Value = "lorem ipsum dolor sit amet"');
      AddResult(Result,'String property assignment',
                nonbridge.Value = 'lorem ipsum dolor sit amet');

      eng.Evaluate('nonbridge.SetStrFromJS("cogito ergo sum")');
      AddResult(Result,'String method assignment',
                nonbridge.Value = 'cogito ergo sum');
    finally
      nonbridge.Free;
    end;
 finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
  AddOutput(Result,'Known issue: String property assignment may fail ' +
                   'unexpectedly (SetProperty improperly called)');
end;

function TestTJSObject: String;
var
  eng: TJSEngine;
  obj: TJSObject;
  str: TBridgeString;
  int: Integer;
  dbl: Double;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    obj := TJSObject.Create(eng, 'obj');
    try
      eng.Evaluate('obj.str = "lorem ipsum dolor sit amet";');
      obj.GetProperty('str',str);
      AddResult(Result,'New string property assignment',
                str = 'lorem ipsum dolor sit amet');

      eng.Evaluate('obj.tint = 2876;');
      obj.GetProperty('tint',int);
      AddResult(Result,'New integer property assignment',int = 2876);
      AddResult(Result,'New integer property is an integer',
                obj.IsInteger('tint'));

      eng.Evaluate('obj.dbl = 876.37');
      obj.GetProperty('dbl',dbl);
      AddResult(Result,'New double property assignment',
                FloatToStr(dbl) = '876.37');
      AddResult(Result,'New double property is not an integer',
                not obj.IsInteger('dbl'));

      eng.Evaluate('obj.func = function() { return "lorem ipsum" }');
      AddResult(Result,'New function property assignment is function type',
                obj.IsFunction('func'));
      obj.Call('func',[],str);
      AddResult(Result,'JS function return value from Call',
                str = 'lorem ipsum');

      eng.Evaluate('obj.intfunc = function(p_int) { return 6*p_int }');
      AddResult(Result,
                'New integer function property assignment is function type',
                obj.IsFunction('intfunc'));
      obj.Call('intfunc',[obj.Declare(60)],int);
      AddResult(Result,'JS integer function return value from Call',int = 6*60);

      eng.Evaluate('obj.strfunc = ' +
                   'function(p_str) { return p_str+" ergo spud" }');
      AddResult(Result,
                'New string function property assignment is function type',
                obj.IsFunction('strfunc'));
      obj.Call('strfunc',[obj.Declare('cogito')],str);
      AddResult(Result,'JS string function return value from Call',
                str = 'cogito ergo spud');
    finally
      obj.Free;
    end;
 finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
  AddOutput(Result,'Known issue: Cannot get a property named "int" -- ' +
                   'generates error and returns undefined');
  AddOutput(Result,'Known issue: Have to convert a double to a string to ' +
                   'compare its value correctly');
end;

function TestTJSString: String;
var
  eng: TJSEngine;
  str: TJSString;
  nstr: TBridgeString;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    str := TJSString.Create;
    try
      str.Value := 'Lorem ipsum dolor sit amet';
      AddResult(Result,'Disconnected value set',
                str.Value = 'Lorem ipsum dolor sit amet');
      AddResult(Result,'Object is disconnected',not str.Connected);

      str.Connect(eng, 'str');
      AddResult(Result,'Object is now connected',str.Connected);

      eng.Evaluate('e = str');
      eng.Global.GetProperty('e',nstr);
      AddResult(Result,'New property equivalent to object',nstr = str.Value);

      eng.Evaluate('str = "cogito ergo sum";');
      AddResult(Result,'Object value assignment',str.Value = 'cogito ergo sum');
    finally
      str.Free;
    end;
  finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestTJSDouble: String;
var
  eng: TJSEngine;
  dbl: TJSDouble;
  ndbl: Double;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    dbl := TJSDouble.Create;
    try
      dbl.Value := 9671.17;
      AddResult(Result,'Disconnected value set',
                FloatToStr(dbl.Value) = '9671.17');
      AddResult(Result,'Object is disconnected',not dbl.Connected);

      dbl.Connect(eng, 'dbl');
      AddResult(Result,'Object is now connected',dbl.Connected);

      eng.Evaluate('e = dbl');
      eng.Global.GetProperty('e',ndbl);
      AddResult(Result,'New property equivalent to object',ndbl = dbl.Value);

      eng.Evaluate('dbl = 18567.37;');
      AddResult(Result,'Object value assignment',
                FloatToStr(dbl.Value) = '18567.37');
    finally
      dbl.Free;
    end;
  finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestTJSInteger: String;
var
  eng: TJSEngine;
  int: TJSInteger;
  nint: Integer;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    int := TJSInteger.Create;
    try
      int.Value := 9671;
      AddResult(Result,'Disconnected value set',int.Value = 9671);
      AddResult(Result,'Object is disconnected',not int.Connected);

      int.Connect(eng, 'tint');
      AddResult(Result,'Object is now connected',int.Connected);

      eng.Evaluate('e = tint');
      eng.Global.GetProperty('e',nint);
      AddResult(Result,'New property equivalent to object',nint = int.Value);

      eng.Evaluate('tint = 18567;');
      AddResult(Result,'Object value assignment',int.Value = 18567);
    finally
      int.Free;
    end;
  finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestTJSBoolean: String;
var
  eng: TJSEngine;
  bool: TJSBoolean;
  nbool: Boolean;
begin
  AddOutput(Result,'Starting test');
  eng := TJSEngine.Create(80000);
  try
    bool := TJSBoolean.Create;
    try
      bool.Value := true;
      AddResult(Result,'Disconnected value set',bool.Value = true);
      AddResult(Result,'Object is disconnected',not bool.Connected);

      bool.Connect(eng, 'bool');
      AddResult(Result,'Object is now connected',bool.Connected);

      eng.Evaluate('e = bool');
      eng.Global.GetProperty('e',nbool);
      AddResult(Result,'New property equivalent to object',nbool = bool.Value);

      eng.Evaluate('bool = false;');
      AddResult(Result,'Object value assignment',bool.Value = false);
    finally
      bool.Free;
    end;
  finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestTJSFunction: String;
begin
end;

function TestTJSArray: String;
var
  jsarr: TJSArray;
  jsint: TJSInteger;
  jsstr: TJSString;
  jsobj: TJSObject;
  str: TBridgeString;
  str2: TBridgeString;
  str3: TBridgeString;
  eng: TJSEngine;
begin
  AddOutput(Result,'Starting test TJSArray');
  eng := TJSEngine.Create(80000);
  try
    jsarr := TJSArray.Create(eng,'arr');
    try
      jsint := eng.Declare(10,'myint');
      jsstr := eng.Declare('lorem ipsum');
      jsarr.Value[0] := jsint;
      jsarr.Value[1] := jsstr;
      AddResult(Result,'First entry is integer 10',
                TJSInteger(jsarr.Value[0]).Value = 10);
      AddResult(Result,'Second entry is string "lorem ipsum"',
                TJSString(jsarr.Value[1]).Value = 'lorem ipsum');

      eng.Evaluate('arr[0]=10287; arr[1]="cogito ergo sum";');

      AddResult(Result,'Integer value is now 10287',jsint.Value = 10287);
      AddResult(Result,'First array entry is now 10287',
                TJSInteger(jsarr.Value[0]).Value = 10287);

      AddResult(Result,'String value is now "cogito ergo sum"',
                jsstr.Value = 'cogito ergo sum');
      AddResult(Result,'Second array entry is now "cogito ergo sum"',
                TJSString(jsarr.Value[1]).Value = 'cogito ergo sum');

      eng.Evaluate('e=arr[0]');
      eng.Global.GetProperty('e',str);
      AddResult(Result,'Assignment of first array entry',str = '10287');

      eng.Evaluate('arr[0] = "lorem ipsum";');
      AddResult(Result,'First array entry is now "lorem ipsum"',
                TJSString(jsarr.Value[0]).Value = 'lorem ipsum');

      jsobj := eng.Global.DeclareObject('myobj');
      jsarr.Value[2] := jsobj;
      AddResult(Result,'Assignment of object as third array entry',
                jsarr.Value[2].ClassName = 'TJSObject');

      jsobj.Declare(104819,'newint');
      eng.Evaluate('f=arr[2].newint;');
      eng.Global.GetProperty('f',str);
      AddResult(Result,'Retrieving object property via array',str = '104819');

      eng.Evaluate('arr.length = 2');
      AddResult(Result,'Array length should now be 2',jsarr.Len = 2);

      eng.Evaluate('arr.length = 6');
      AddResult(Result,'Array length should now be 6',jsarr.Len = 6);

      eng.Evaluate('arr[0] = 5; arr[1] = 3; arr[2] = 9;');
      eng.Evaluate('f = new Array(); f.push(36); f.push(96); f.push("test"); ' +
                   'arr.push(f);');
      eng.Evaluate('s = arr.join()');
      eng.Evaluate('arr.reverse();');
      eng.Evaluate('arr.splice(0,1)');
      eng.Evaluate('p = arr.pop()');
      eng.Evaluate('v = ""; for (j in arr) { v += arr[j] +", "; }');
      eng.Global.GetProperty('v',str);
      eng.Global.GetProperty('p',str2);
      eng.Global.GetProperty('s',str3);
      AddOutput(Result,'Value list: ' +str);
      AddOutput(Result,'Pop result: ' +str2);
      AddOutput(Result,'Joined array: ' +str3);
    finally
      jsarr.Free;
    end;
  finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestSamples: String;
begin
  AddOutput(Result,'Starting test');
  (*
  CallString;
  CallInteger;
  CreateObject;
  CreateFunction;
  CreateBridge;
  *)
  AddOutput(Result,'Finished test');
  AddOutput(Result,'Known issue: Suppress alerts during sample execution <g>');
end;

function TestSerialization: String;
var
  eng: TJSEngine;
  jsscript: TJSScript;
  str: TBridgeString;
begin
  AddOutput(Result,'Starting test Serialization');
  eng := TJSEngine.Create(80000);
  try
    jsscript := TJSScript.Create;
    try
      jsscript.Code := 'function mult(p_num) { return 6*p_num }; var a = new ' +
                       'Date(); var s = new String("test"); ' +
                       'function add(p_num) { return 9+p_num; }';
      jsscript.Execute(eng);
      eng.Global.Call('mult',[eng.Global.Declare(10)],str);
      AddResult(Result,'Result of script function is 60',str = '60');

      jsscript.SaveRaw('serialraw.js');
      AddResult(Result,'Saved raw code',true);
      jsscript.SaveCompiled('serial.js',eng);
      AddResult(Result,'Saved compiled bytecode',true);

      // Make sure we recycle all variables
      jsscript.Free;
      eng.Free;

      eng := TJSEngine.Create(80000);
      jsscript := TJSScript.Create;

      jsscript.LoadRaw('serialraw.js');
      AddResult(Result,'Loaded raw code',true);

      jsscript.LoadCompiled('serial.js',eng);
      AddResult(Result,'Loaded compiled bytecode',true);
      jsscript.Execute(eng);
      eng.Global.Call('mult',[eng.Global.Declare(10)],str);
      AddResult(Result,'Result of script function is still 60',str = '60');
    finally
      jsscript.Free;
    end;
  finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestEnumeration: String;
var
  eng: TJSEngine;
  obj: TJSObject;
  subobj: TJSObject;
  list: TStringArray;
begin
  AddOutput(Result,'Starting test Enumeration');
  eng := TJSEngine.Create(80000);
  try
    obj := TJSObject.Create(eng, 'obj');
    try
      eng.Evaluate('obj.str = "lorem ipsum dolor sit amet";');
      eng.Evaluate('obj.tint = 2876;');
      eng.Evaluate('obj.dbl = 876.37');
      eng.Evaluate('obj.func = function() { return "lorem ipsum" }');
      eng.Evaluate('obj.intfunc = function(p_int) { return 6*p_int }');
      eng.Evaluate('obj.strfunc = ' +
                   'function(p_str) { return p_str+" ergo spud" }');
      eng.Evaluate('obj.obj = new Object(); obj.obj.str = "cogito ergo sum";');

      list := obj.Enumerate;
      AddResult(Result,'Property 0 is str',list[0] = 'str');
      AddResult(Result,'Property 1 is tint',list[1] = 'tint');
      AddResult(Result,'Property 2 is dbl',list[2] = 'dbl');
      AddResult(Result,'Property 3 is func',list[3] = 'func');
      AddResult(Result,'Property 4 is intfunc',list[4] = 'intfunc');
      AddResult(Result,'Property 5 is strfunc',list[5] = 'strfunc');
      AddResult(Result,'Property 6 is obj',list[6] = 'obj',list[6]);

      AddResult(Result,'Type of str is string',
                obj.TypeOf('str') = JSTYPE_STRING);
      AddResult(Result,'Type of tint is integer',
                (obj.TypeOf('tint') = JSTYPE_NUMBER)
                  and (obj.IsInteger('tint')));
      AddResult(Result,'Type of dbl is double',
                (obj.TypeOf('dbl') = JSTYPE_NUMBER)
                  and (not obj.IsInteger('dbl')));
      AddResult(Result,'Type of func is function',
                obj.TypeOf('func') = JSTYPE_FUNCTION);
      AddResult(Result,'Type of intfunc is function',
                obj.TypeOf('intfunc') = JSTYPE_FUNCTION);
      AddResult(Result,'Type of strfunc is function',
                obj.TypeOf('strfunc') = JSTYPE_FUNCTION);
      AddResult(Result,'Type of obj is object',
                obj.TypeOf('obj') = JSTYPE_OBJECT);

      obj.GetProperty('obj',subobj);
      list := subobj.Enumerate;
      AddResult(Result,'Subobject property 0 is str',list[0] = 'str');
      AddResult(Result,'Type of subobject.str is string',
                subobj.TypeOf('str') = JSTYPE_STRING);
    finally
      obj.Free;
    end;
 finally
    eng.Free;
  end;
  AddOutput(Result,'Finished test');
end;

function TestErrorReporting: String;
begin
end;

end.

