/*$Id: fkStreamListener.cpp,v 1.14 2006/08/20 16:39:54 jwrobel Exp $*/

/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "fkStreamListener.h"
#include "nsIServiceManager.h"
#include "nsIURI.h"

#include "nsXPCOM.h"
#include "nsMemory.h"
#include "nsStringAPI.h"
#include "nsEmbedString.h"

#include "Firekeeper.h"
#include "fkRule.h"
#include "fkSentence.h"
#include "Error.h"
#include "Common.h"
#include "nsIByteArrayInputStream.h"

//flush points idea comes from Snort stream4 preprocessor
#define FLUSH_POINT_MIN 2000
#define FLUSH_POINT_MAX 4000

#define NO_FLUSH_POINT_MIN 100
#define NO_FLUSH_POINT_MAX 200  



NS_IMPL_ISUPPORTS3(fkStreamListener, nsIStreamListener,
		   nsIRequestObserver, 
		   nsIHttpHeaderVisitor);

fkStreamListener::fkStreamListener(nsIStreamListener *origListener, fkHttpChannel *channel,
				   fkISentence *sentence, ActiveRuleSet *connRules):
	body(nsnull), headers(nsnull), url(nsnull), body_len(0), headers_len(0), 
	body_max(BODY_INIT_SIZE), headers_max(HEADERS_INIT_SIZE), offset(0), debugNoHeaders(true)
{	
	TRACE("constructor %08x", this);
	
	this->origListener = origListener;
	this->sentence = sentence;
	this->connRules = connRules;
	this->channel = channel;
	NS_ADDREF(this->channel);

	
	body = (char *)nsMemory::Alloc(body_max);
	if (body == nsnull){
		body_max = headers_max = 0;
		return;
	}
	
	headers = (char *)nsMemory::Alloc(headers_max);
	if (headers == nsnull){
		body_max = headers_max = 0;
		nsMemory::Free(body);
		body = nsnull;
		return;
	}

	drawFlushPoint();
	drawNoFlushPoint();
}

fkStreamListener::~fkStreamListener()
{
	TRACE("destructor %08x", this);
	NS_RELEASE(this->channel);
		
	if (body != nsnull)
		nsMemory::Free(body);
	if (headers != nsnull)
		nsMemory::Free(headers);
}

void fkStreamListener::drawFlushPoint()
{
	srand(flush_point + time(0) + (int)this);
	flush_point = FLUSH_POINT_MIN + 
		rand() % (FLUSH_POINT_MAX - FLUSH_POINT_MIN);
	TRACE("new flush point %d", flush_point);
}

void fkStreamListener::drawNoFlushPoint()
{
	srand(no_flush_point + time(0) + (int)this);
	no_flush_point = NO_FLUSH_POINT_MIN + 
		rand() % (NO_FLUSH_POINT_MAX - NO_FLUSH_POINT_MIN);
	
	TRACE("new no flush point %d", no_flush_point);
}

const char* 
fkStreamListener::getURL()
{
	if (!url.Length()){
		nsCOMPtr<nsIURI> uri;
		channel->GetURI(getter_AddRefs(uri));
		uri->GetSpec(url);
	}
	return url.get();
	
}

NS_IMETHODIMP
fkStreamListener::headers_append(const char *src, PRUint32 len)
{
	if (headers == nsnull)
		return NS_ERROR_OUT_OF_MEMORY;
	
	if (expandBuf(&headers, len, headers_len, headers_max) != NS_OK)
		return NS_ERROR_OUT_OF_MEMORY;

	memcpy(headers + headers_len, src, len);
	headers_len += len;
	return NS_OK;
}

NS_IMETHODIMP
fkStreamListener::expandBuf(char **buf, PRUint32 len, PRUint32 end, PRUint32 &max_size)
{
	while(max_size  - end < len){	
		max_size *= 2;		
		char *tmp = (char *)nsMemory::Realloc(*buf, max_size);
		if (tmp == nsnull){
			max_size /= 2;
			return NS_ERROR_OUT_OF_MEMORY;
		}
		*buf = tmp;
	}
	return NS_OK;
}

NS_IMETHODIMP
fkStreamListener::checkBody(char *body, int bodylen)
{
	list<const Rule *> match = connRules->checkBody(body, bodylen);
	list<const Rule *>::iterator it;
	for(it = match.begin(); it != match.end(); ++it){
		fkRule *rule_raw = new fkRule(*it);
		if (!rule_raw){
			return NS_ERROR_OUT_OF_MEMORY;
		}
		nsCOMPtr<fkIRule> 
			rule(do_QueryInterface(rule_raw));
		fkJudge->Judge(getURL(), rule, sentence);		
		int action;
		sentence->GetAction(&action);
		TRACE("action = %d", action);
		if (action == sentence->BLOCK){
			TRACE("aborting request");
			channel->Cancel(NS_ERROR_ABORT);
			return NS_ERROR_ABORT;
		}
		else if (action == sentence->DONT_AUDIT){
			TRACE("passing request");
			break;
		}
	}
	return NS_OK;
}

NS_IMETHODIMP
fkStreamListener::do_OnDataAvailable(nsIRequest *request, nsISupports* acontext,
				     PRUint32 aSourceOffset)
{
	nsresult rv;
	FK_ASSERT(body != nsnull);
	TRACE("calling checkBody");
	
	rv = checkBody(body, body_len);
	TRACE("ok");
	if (NS_FAILED(rv))
		return rv;

	nsCOMPtr<nsIByteArrayInputStream> convertedStreamSup;
	
	
	rv = NS_NewByteArrayInputStream(getter_AddRefs(convertedStreamSup), body, body_len);    
	if (NS_FAILED(rv))
		return rv;
	
	body = nsnull;
	nsCOMPtr<nsIInputStream> convertedStream = do_QueryInterface(convertedStreamSup, &rv);
	if (NS_FAILED(rv))
		return rv;
	
	TRACE("OK offset = %d len %d", aSourceOffset, body_len);
	return origListener->OnDataAvailable(request, acontext, convertedStream, 
					     aSourceOffset, body_len);
}

NS_IMETHODIMP
fkStreamListener::do_OnDataAvailable(nsIRequest* request, nsISupports *aContext, 
				     PRUint32 aSourceOffset, char *buffer, PRUint32 aCount)
{
	
	nsresult rv;
	
	TRACE("calling checkBody len =%d", aCount);
	rv = checkBody(buffer, aCount);
	TRACE("ok");
	if (NS_FAILED(rv))
		return rv;
	
	nsCOMPtr<nsIByteArrayInputStream> convertedStreamSup;
	
	char *lBuf = (char *) nsMemory::Alloc(aCount);
	if (!lBuf)
		return NS_ERROR_OUT_OF_MEMORY;
	
	memcpy(lBuf, buffer, aCount);
	
	rv = NS_NewByteArrayInputStream(getter_AddRefs(convertedStreamSup), lBuf, aCount);
	if (NS_FAILED(rv))
		return rv;
	
	nsCOMPtr<nsIInputStream> convertedStream = do_QueryInterface(convertedStreamSup, &rv);
	if (NS_FAILED(rv))
		return rv;
	
	return origListener->OnDataAvailable(request, aContext, convertedStream, 
					 aSourceOffset, aCount);
}

//-----------------------------------------------------------------------------
// fkStreamListener::nsIStreamListener
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkStreamListener::OnDataAvailable(nsIRequest *request, nsISupports *ctxt,
				  nsIInputStream *input,
				  PRUint32 off, PRUint32 count)
{	
	TRACE("got data count = %d offset = %d", count, off);
	FK_ASSERT(debugNoHeaders == false);
	
	int action;
	nsresult rv;
	
	sentence->GetAction(&action);
	if (action == sentence->DONT_AUDIT)
		return origListener->OnDataAvailable(request, ctxt, input, off, count);
	else if (action == sentence->BLOCK){
		FK_ASSERT(false);
		return NS_ERROR_ABORT;
	}

	if (expandBuf(&body, count, body_len, body_max) != NS_OK)
		return NS_ERROR_OUT_OF_MEMORY;

	PRUint32 res;	
	input->Read(body + body_len, count, &res);
	FK_ASSERT(res == count);	
	body_len += count;
	
	if (body_len < flush_point)
		return NS_OK;
	
	TRACE("flushing buffer offset %d body_len = %d", offset, body_len);
	PRUint32 data_len = body_len - no_flush_point;
	TRACE("data_len = %d", data_len);
	rv = do_OnDataAvailable(request, ctxt, offset, body, data_len);
	if (NS_FAILED(rv)){
		TRACE("do_OnDataAvailable FAILED");
		return rv;
	}
	
	offset += data_len;
	
	memmove(body, body + data_len, no_flush_point); 
	body_len = no_flush_point;
	
	drawFlushPoint();
	drawNoFlushPoint();
	return rv;
}


//-----------------------------------------------------------------------------
// fkStreamListener::nsIRequestObserver
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkStreamListener::OnStartRequest(nsIRequest *request, nsISupports* acontext)
{
	TRACE("Request started %08x", this);
	nsresult rv = channel->VisitResponseHeaders(this);
	if (NS_FAILED(rv)){
		TRACE("Visit response headers failed");
		debugNoHeaders = true;
		return rv;
	}
	debugNoHeaders = false;
	rv = headers_append("", 1);
	if (NS_FAILED(rv)){
		return rv;
	}	

	TRACE("headers:\n %s", headers);
	
	list<const Rule *> match = connRules->checkHeaders(headers, strlen(headers));
	list<const Rule *>::iterator it;
	for(it = match.begin(); it != match.end(); ++it){
		fkRule *rule_raw = new fkRule(*it);
		if (!rule_raw){
			return NS_ERROR_OUT_OF_MEMORY;
		}
		nsCOMPtr<fkIRule> 
			rule(do_QueryInterface(rule_raw));
		fkJudge->Judge(getURL(), rule, sentence);		
		int action;
		sentence->GetAction(&action);
		TRACE("action = %d", action);
		if (action == sentence->BLOCK){
			TRACE("aborting request");
			return NS_ERROR_ABORT;
		}
		else if (action == sentence->DONT_AUDIT){
			TRACE("passing request");
			break;
		}
	}

	return origListener->OnStartRequest(request, acontext);
}


NS_IMETHODIMP
fkStreamListener::OnStopRequest(nsIRequest *request, nsISupports* acontext,
				nsresult astatus)
{
	TRACE("Request stopped %08x status = %d", this, astatus);
	nsresult rv;
	int action;
	sentence->GetAction(&action);

	if (body_len && action != sentence->DONT_AUDIT && action != sentence->BLOCK){
		rv = do_OnDataAvailable(request, acontext, offset);	
		if (NS_FAILED(rv)){
			TRACE("do_OnDataAvailable failed %d", rv);
			astatus = rv;
		}
	}
	
	return origListener->OnStopRequest(request, acontext, astatus);

}


//-----------------------------------------------------------------------------
// fkStreamListener::nsIHttpHeaderVisitor
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkStreamListener::VisitHeader(const nsACString &header, const nsACString &value)
{

	nsresult rv;
	const char *h = nsEmbedCString(header).get();
	const char *v = nsEmbedCString(value).get();

	rv = headers_append(h, strlen(h));
	if (NS_FAILED(rv))
		return rv;
	rv = headers_append(": ", 2);
	if (NS_FAILED(rv))
		return rv;

	rv = headers_append(v, strlen(v));
	if (NS_FAILED(rv))
		return rv;
	
	rv = headers_append("\n", 1);
	if (NS_FAILED(rv))
		return rv;
	return NS_OK;
}


