/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.xml;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import org.ovirt.api.metamodel.runtime.util.ArrayListWithHref;
import org.ovirt.api.metamodel.runtime.util.ListWithHref;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.internal.containers.DiskSnapshotContainer;
import org.ovirt.engine.sdk4.types.DiskSnapshot;

public class XmlDiskSnapshotReader {
    
    public static DiskSnapshot readOne(XmlReader reader) {
        // Do nothing if there aren't more tags:
        if (!reader.forward()) {
            return null;
        }
        
        // Create the object:
        DiskSnapshotContainer object = new DiskSnapshotContainer();
        
        // Process the attributes:
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String name = reader.getAttributeLocalName(i);
            String image = reader.getAttributeValue(i);
            switch (name) {
                case "href":
                object.href(image);
                break;
                case "id":
                object.id(image);
                break;
                default:
                break;
            }
        }
        
        // Process the inner elements:
        reader.next();
        while (reader.forward()) {
            String name = reader.getLocalName();
            switch (name) {
                case "active":
                object.active(reader.readBoolean());
                break;
                case "actual_size":
                object.actualSize(reader.readInteger());
                break;
                case "alias":
                object.alias(reader.readString());
                break;
                case "bootable":
                object.bootable(reader.readBoolean());
                break;
                case "comment":
                object.comment(reader.readString());
                break;
                case "description":
                object.description(reader.readString());
                break;
                case "disk":
                object.disk(XmlDiskReader.readOne(reader));
                break;
                case "disk_profile":
                object.diskProfile(XmlDiskProfileReader.readOne(reader));
                break;
                case "format":
                object.format(XmlDiskFormatReader.readOne(reader));
                break;
                case "image_id":
                object.imageId(reader.readString());
                break;
                case "instance_type":
                object.instanceType(XmlInstanceTypeReader.readOne(reader));
                break;
                case "interface":
                object.interface_(XmlDiskInterfaceReader.readOne(reader));
                break;
                case "logical_name":
                object.logicalName(reader.readString());
                break;
                case "lun_storage":
                object.lunStorage(XmlHostStorageReader.readOne(reader));
                break;
                case "name":
                object.name(reader.readString());
                break;
                case "openstack_volume_type":
                object.openstackVolumeType(XmlOpenStackVolumeTypeReader.readOne(reader));
                break;
                case "permissions":
                object.permissions(XmlPermissionReader.readMany(reader));
                break;
                case "progress":
                object.progress(reader.readInteger());
                break;
                case "propagate_errors":
                object.propagateErrors(reader.readBoolean());
                break;
                case "provisioned_size":
                object.provisionedSize(reader.readInteger());
                break;
                case "quota":
                object.quota(XmlQuotaReader.readOne(reader));
                break;
                case "read_only":
                object.readOnly(reader.readBoolean());
                break;
                case "sgio":
                object.sgio(XmlScsiGenericIOReader.readOne(reader));
                break;
                case "shareable":
                object.shareable(reader.readBoolean());
                break;
                case "snapshot":
                object.snapshot(XmlSnapshotReader.readOne(reader));
                break;
                case "sparse":
                object.sparse(reader.readBoolean());
                break;
                case "statistics":
                object.statistics(XmlStatisticReader.readMany(reader));
                break;
                case "status":
                object.status(XmlDiskStatusReader.readOne(reader));
                break;
                case "storage_domain":
                object.storageDomain(XmlStorageDomainReader.readOne(reader));
                break;
                case "storage_domains":
                object.storageDomains(XmlStorageDomainReader.readMany(reader));
                break;
                case "storage_type":
                object.storageType(XmlDiskStorageTypeReader.readOne(reader));
                break;
                case "template":
                object.template(XmlTemplateReader.readOne(reader));
                break;
                case "uses_scsi_reservation":
                object.usesScsiReservation(reader.readBoolean());
                break;
                case "vm":
                object.vm(XmlVmReader.readOne(reader));
                break;
                case "vms":
                object.vms(XmlVmReader.readMany(reader));
                break;
                case "wipe_after_delete":
                object.wipeAfterDelete(reader.readBoolean());
                break;
                case "link":
                readLink(reader, object);
                break;
                default:
                reader.skip();
                break;
            }
        }
        
        // Discard the end tag:
        reader.next();
        
        return object;
    }
    
    public static Iterator<DiskSnapshot> iterateMany(final XmlReader reader) {
        return new Iterator<DiskSnapshot>() {
            private boolean first = true;
            
            @Override
            public void remove() {
                // Empty on purpose
            }
            
            @Override
            public boolean hasNext() {
                if (first) {
                    if (!reader.forward()) {
                        return false;
                    }
                    reader.next();
                    first = false;
                }
                if (!reader.forward()) {
                    reader.next();
                    return false;
                }
                return true;
            }
            
            @Override
            public DiskSnapshot next() {
                DiskSnapshot next = readOne(reader);
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }
    
    public static List<DiskSnapshot> readMany(XmlReader reader) {
        List<DiskSnapshot> list = new ArrayList<>();
        Iterator<DiskSnapshot> iterator = iterateMany(reader);
        while (iterator.hasNext()) {
            list.add(iterator.next());
        }
        return list;
    }
    
    private static void readLink(XmlReader reader, DiskSnapshotContainer object) {
        // Process the attributes:
        String rel = null;
        String href = null;
        for (int i = 0; i < reader.getAttributeCount(); i++) {
            String attrName = reader.getAttributeLocalName(i);
            String attrVal = reader.getAttributeValue(i);
            switch (attrName) {
                case "href":
                href = attrVal;
                break;
                case "rel":
                rel = attrVal;
                break;
                default:
                reader.skip();
                break;
            }
        }
        if (href != null) {
            ListWithHref list = new ArrayListWithHref();
            list.href(href);
            switch (rel) {
                case "permissions":
                object.permissions(list);
                break;
                case "statistics":
                object.statistics(list);
                break;
                case "storagedomains":
                object.storageDomains(list);
                break;
                case "vms":
                object.vms(list);
                break;
                default:
                break;
            }
        }
        reader.skip();
    }
}

