/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.PermissionBuilder;
import org.ovirt.engine.sdk4.types.Permission;

/**
 * Represents a permission sub-collection, scoped by user, group or some entity type.
 */
public interface AssignedPermissionsService extends Service {
    /**
     * Assign a new permission to a user or group for specific entity.
     * 
     * For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
     * send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/vms/123/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>UserVmManager</name>
     *   </role>
     *   <user id="456"/>
     * </permission>
     * ----
     * 
     * To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>SuperUser</name>
     *   </role>
     *   <user id="456"/>
     * </permission>
     * ----
     * 
     * If you want to assign permission to the group instead of the user please replace the `user` element with the
     * `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
     * id `123` to the group with id `789` send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/clusters/123/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>UserRole</name>
     *   </role>
     *   <group id="789"/>
     * </permission>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest permission(Permission permission);
        /**
         * The permission.
         */
        AddRequest permission(PermissionBuilder permission);
    }
    
    /**
     * Assign a new permission to a user or group for specific entity.
     * 
     * For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
     * send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/vms/123/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>UserVmManager</name>
     *   </role>
     *   <user id="456"/>
     * </permission>
     * ----
     * 
     * To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>SuperUser</name>
     *   </role>
     *   <user id="456"/>
     * </permission>
     * ----
     * 
     * If you want to assign permission to the group instead of the user please replace the `user` element with the
     * `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
     * id `123` to the group with id `789` send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/clusters/123/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>UserRole</name>
     *   </role>
     *   <group id="789"/>
     * </permission>
     * ----
     */
    public interface AddResponse extends Response {
        /**
         * The permission.
         */
        Permission permission();
    }
    
    /**
     * Assign a new permission to a user or group for specific entity.
     * 
     * For example, to assign the `UserVmManager` role to the virtual machine with id `123` to the user with id `456`
     * send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/vms/123/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>UserVmManager</name>
     *   </role>
     *   <user id="456"/>
     * </permission>
     * ----
     * 
     * To assign the `SuperUser` role to the system to the user with id `456` send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>SuperUser</name>
     *   </role>
     *   <user id="456"/>
     * </permission>
     * ----
     * 
     * If you want to assign permission to the group instead of the user please replace the `user` element with the
     * `group` element with proper `id` of the group. For example to assign the `UserRole` role to the cluster with
     * id `123` to the group with id `789` send a request like this:
     * 
     * ....
     * POST /ovirt-engine/api/clusters/123/permissions
     * ....
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <permission>
     *   <role>
     *     <name>UserRole</name>
     *   </role>
     *   <group id="789"/>
     * </permission>
     * ----
     */
    AddRequest add();
    
    /**
     * List all the permissions of the specific entity.
     * 
     * For example to list all the permissions of the cluster with id `123` send a request like this:
     * 
     * ....
     * GET /ovirt-engine/api/clusters/123/permissions
     * ....
     * 
     * [source,xml]
     * ----
     * <permissions>
     *   <permission id="456">
     *     <cluster id="123"/>
     *     <role id="789"/>
     *     <user id="451"/>
     *   </permission>
     *   <permission id="654">
     *     <cluster id="123"/>
     *     <role id="789"/>
     *     <group id="127"/>
     *   </permission>
     * </permissions>
     * ----
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
    }
    
    /**
     * List all the permissions of the specific entity.
     * 
     * For example to list all the permissions of the cluster with id `123` send a request like this:
     * 
     * ....
     * GET /ovirt-engine/api/clusters/123/permissions
     * ....
     * 
     * [source,xml]
     * ----
     * <permissions>
     *   <permission id="456">
     *     <cluster id="123"/>
     *     <role id="789"/>
     *     <user id="451"/>
     *   </permission>
     *   <permission id="654">
     *     <cluster id="123"/>
     *     <role id="789"/>
     *     <group id="127"/>
     *   </permission>
     * </permissions>
     * ----
     */
    public interface ListResponse extends Response {
        /**
         * The list of permissions.
         */
        List<Permission> permissions();
    }
    
    /**
     * List all the permissions of the specific entity.
     * 
     * For example to list all the permissions of the cluster with id `123` send a request like this:
     * 
     * ....
     * GET /ovirt-engine/api/clusters/123/permissions
     * ....
     * 
     * [source,xml]
     * ----
     * <permissions>
     *   <permission id="456">
     *     <cluster id="123"/>
     *     <role id="789"/>
     *     <user id="451"/>
     *   </permission>
     *   <permission id="654">
     *     <cluster id="123"/>
     *     <role id="789"/>
     *     <group id="127"/>
     *   </permission>
     * </permissions>
     * ----
     */
    ListRequest list();
    
    /**
     * Sub-resource locator method, returns individual permission resource on which the remainder of the URI is
     * dispatched.
     */
    PermissionService permissionService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

