/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.SnapshotBuilder;
import org.ovirt.engine.sdk4.types.Snapshot;

/**
 * Manages the set of snapshots of a storage domain or virtual machine.
 */
public interface SnapshotsService extends Service {
    /**
     * Creates a virtual machine snapshot.
     * 
     * For example, to create a new snapshot for virtual machine `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/vms/123/snapshots
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <snapshot>
     *   <description>My snapshot</description>
     * </snapshot>
     * ----
     * 
     * [IMPORTANT]
     * ====
     * When a snapshot is created the default value for the <<types/snapshot/attributes/persist_memorystate,
     * persist_memorystate>> attribute is `true`. That means that the content of the memory of the virtual
     * machine will be included in the snapshot, and it also means that the virtual machine will be paused
     * for a longer time. That can negatively affect applications that are very sensitive to timing (NTP
     * servers, for example). In those cases make sure that you set the attribute to `false`:
     * 
     * [source,xml]
     * ----
     * <snapshot>
     *   <description>My snapshot</description>
     *   <persist_memorystate>false</persist_memorystate>
     * </snapshot>
     * ----
     * ====
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest snapshot(Snapshot snapshot);
        AddRequest snapshot(SnapshotBuilder snapshot);
    }
    
    /**
     * Creates a virtual machine snapshot.
     * 
     * For example, to create a new snapshot for virtual machine `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/vms/123/snapshots
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <snapshot>
     *   <description>My snapshot</description>
     * </snapshot>
     * ----
     * 
     * [IMPORTANT]
     * ====
     * When a snapshot is created the default value for the <<types/snapshot/attributes/persist_memorystate,
     * persist_memorystate>> attribute is `true`. That means that the content of the memory of the virtual
     * machine will be included in the snapshot, and it also means that the virtual machine will be paused
     * for a longer time. That can negatively affect applications that are very sensitive to timing (NTP
     * servers, for example). In those cases make sure that you set the attribute to `false`:
     * 
     * [source,xml]
     * ----
     * <snapshot>
     *   <description>My snapshot</description>
     *   <persist_memorystate>false</persist_memorystate>
     * </snapshot>
     * ----
     * ====
     */
    public interface AddResponse extends Response {
        Snapshot snapshot();
    }
    
    /**
     * Creates a virtual machine snapshot.
     * 
     * For example, to create a new snapshot for virtual machine `123` send a request like this:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/vms/123/snapshots
     * ----
     * 
     * With a request body like this:
     * 
     * [source,xml]
     * ----
     * <snapshot>
     *   <description>My snapshot</description>
     * </snapshot>
     * ----
     * 
     * [IMPORTANT]
     * ====
     * When a snapshot is created the default value for the <<types/snapshot/attributes/persist_memorystate,
     * persist_memorystate>> attribute is `true`. That means that the content of the memory of the virtual
     * machine will be included in the snapshot, and it also means that the virtual machine will be paused
     * for a longer time. That can negatively affect applications that are very sensitive to timing (NTP
     * servers, for example). In those cases make sure that you set the attribute to `false`:
     * 
     * [source,xml]
     * ----
     * <snapshot>
     *   <description>My snapshot</description>
     *   <persist_memorystate>false</persist_memorystate>
     * </snapshot>
     * ----
     * ====
     */
    AddRequest add();
    
    /**
     * Returns the list of snapshots of the storage domain or virtual machine.
     * 
     * The order of the returned list of snapshots isn't guaranteed.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of snapshots to return. If not specified all the snapshots are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of snapshots to return. If not specified all the snapshots are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of snapshots to return. If not specified all the snapshots are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    /**
     * Returns the list of snapshots of the storage domain or virtual machine.
     * 
     * The order of the returned list of snapshots isn't guaranteed.
     */
    public interface ListResponse extends Response {
        List<Snapshot> snapshots();
    }
    
    /**
     * Returns the list of snapshots of the storage domain or virtual machine.
     * 
     * The order of the returned list of snapshots isn't guaranteed.
     */
    ListRequest list();
    
    SnapshotService snapshotService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

