#!/usr/bin/env python3
#
# SPDX-License-Identifier: BSD-2-Clause-FreeBSD
#
# Copyright (c) 2019 Kristof Provost <kp@FreeBSD.org>
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

import argparse
import logging
logging.getLogger("scapy").setLevel(logging.CRITICAL)
import scapy.all as sp
import sys
from sniffer import Sniffer

def check_icmp_error(args, packet):
	ip = packet.getlayer(sp.IP)
	if not ip:
		return False
	if ip.dst != args.to[0]:
		return False

	icmp = packet.getlayer(sp.ICMP)
	if not icmp:
		return False
	if icmp.type != 3 or icmp.code != 3:
		return False

	return True

def main():
	parser = argparse.ArgumentParser("CVE-2019-icmp.py",
		description="CVE-2019-icmp test tool")
	parser.add_argument('--sendif', nargs=1,
		required=True,
		help='The interface through which the packet will be sent')
	parser.add_argument('--recvif', nargs=1,
		required=True,
		help='The interface on which to check for the packet')
	parser.add_argument('--src', nargs=1,
		required=True,
		help='The source IP address')
	parser.add_argument('--to', nargs=1,
		required=True,
		help='The destination IP address')

	args = parser.parse_args()

	# Send the allowed packet to establish state
	udp = sp.Ether() / \
	    sp.IP(src=args.src[0], dst=args.to[0]) / \
	    sp.UDP(dport=53, sport=1234)
	sp.sendp(udp, iface=args.sendif[0], verbose=False)

	# Start sniffing on recvif
	sniffer = Sniffer(args, check_icmp_error)

	# Send the bad error packet
	icmp_reachable = sp.Ether() / \
	    sp.IP(src=args.src[0], dst=args.to[0]) / \
	    sp.ICMP(type=3, code=3) / \
	    sp.IP(src="4.3.2.1", dst="1.2.3.4") / \
	    sp.UDP(dport=53, sport=1234)
	sp.sendp(icmp_reachable, iface=args.sendif[0], verbose=False)

	sniffer.join()
	if sniffer.foundCorrectPacket:
		sys.exit(1)

	sys.exit(0)

if __name__ == '__main__':
	main()
