/* Symlink
 * Version:     4.0
 * File:        link_types.c
 * Date:        9/11/93
 * Authors:     CM King with AJ Rixon
 * Comment:     handle different link types
 */

#include "make_links.h"
#include "str_func.h"
#include "linker.h"

int     present_level;		/* current search depth */
int     level_limit;		/* max dir search depth */
int     trial;			/* trial mode */
int     ignore;			/* default: don't make directories */
void    split_target_directories();
void    resolve_link();

/* list_dir:    traverses a directory gathering filenames and link them
 * Comment:     finds the files in the directory concerned
 * Inputs:      directory name
 * Output:      none
 */

void    list_dir(dirname, target)
    char   *dirname;		/* directory with files in */
    char   *target;		/* directory which will have links in */
{
    DIR    *fd;			/* directory handle */
    struct dirent *dp;		/* directory info */
    struct stat buf;		/* directory status info */
    struct stat dummybuf;	/* directory status info (unused) */
    char    full_path[MAXPATHLEN];	/* absolute pathname for directory */
    char    new_target[MAXPATHLEN];	/* subdirectories of the target */
    char    resolved_full_path[MAXPATHLEN]; /* buffer to store resolved path */
    char    err_mesg[MAXPATHLEN];
    int     ans = YES;		/* user response */

    if ((fd = opendir(dirname)) == NULL) {
	perror("error opening dir\n");
	exit(1);
    }
    /* Collect directory entries */
    for (dp = readdir(fd); dp != NULL; dp = readdir(fd)) {
	(void) sprintf(full_path, "%s/%s", dirname, dp->d_name);
	if ((stat(full_path, &buf)) == FSERR) {	/* get status */
	    if ((lstat(full_path, &buf)) == FSERR) {	/* or get link status */
		perror("stat failed");
		exit(2);
	    }
	}
	if (S_ISDIR(buf.st_mode)) {	/* if a directory */
	    /* recursively enter directories if not current or previous dir */
	    if ((strcmp(".", dp->d_name) == 0) || (strcmp("..", dp->d_name) == 0))
		continue;

	    if (present_level == level_limit) /* quit if max search depth */
		make_dir_links(target, full_path);
	    else {
		/* down again if we haven't reached search limit */
		if (present_level < level_limit) {
		    present_level++;
		    (void) sprintf(new_target, "%s/%s", target, dp->d_name);
		    if (stat(new_target, &dummybuf) != 0) {
			/* create it if it doesn't exist */
			if (!ignore) {	/* if -i not specified ask for confirmation */
			    (void) fprintf(stderr, "Directory %s doesn't exist, want me to make it (y/n) ? \n ", new_target);
			    ans = yesno();
			} else
			    ans = YES;

			if (ans == YES) {
			    if (mkdir(new_target, buf.st_mode) != 0) {
				(void) sprintf(err_mesg, "mkdir failed in list_dir for %s", new_target);
				(void) perror(err_mesg);
				exit(2);
			    }
			}
		    }
		    if (ans == YES)	/* only link if dir made */
			list_dir(full_path, new_target);
		    present_level--;
		    ans = YES;	/* reset ans */
		}
	    }
	} else {		/* if a (link to) normal file */
	    if (S_ISREG(buf.st_mode))
		make_dir_links(target, full_path);
	    else {	/* if it is a link resolve the reference */
		if (S_ISLNK(buf.st_mode)) {
		    resolve_link(full_path, resolved_full_path);
		    make_dir_links(target, full_path);
		}
	    }
	}
    }
    (void) closedir(fd);
}

/* make_link_typet:  link files in a directory to a number of directories
 * Inputs:           link source, target dir for links
 * Output:           none
 */

void    make_link_type(link_source, target, multiple_type)
    char   *link_source;	/* source directory */
    char   *target;		/* directories to link to */
    LINK_TYPE multiple_type;	/* single or multiple links */
{
    char  **point;		/* Pointer to array of pointer */
    int     character_count;	/* occurrences of char in line */
    int     x;			/* counter */
    char    source[MAXPATHLEN];	/* absolute pathname to source */
    int     current_limit;	/* store current limit in */

    character_count = countchar_in_line(target, ':');
    if ((point = (char **) malloc(character_count * sizeof(char *))) == NULL) {
	perror("Malloc failed in make_link_type");
	exit(2);
    }
    for (x = 0; x < character_count; x++) {
	if ((point[x] = (char *) malloc(MAXPATHLEN * sizeof(char))) == NULL) {
	    perror("Malloc failed in make_link_type");
	    exit(2);
	}
	memset(point[x], NULL, MAXPATHLEN * sizeof(char));
    }

    split_target_directories(point, target);

    /* concatinate root and the source directory */
    (void) sprintf(source, "%s/%s", root_path, link_source);

    /* create the links from all files in the directory source */
    for (x = 0; x < character_count; x++, point++) {
	switch (multiple_type) {
	case SINGLE:{
		make_link(*point, source);
		break;
	    }
	case MULTIPLE:{
		list_dir(source, *point);
		break;
	    }
	case SUBMULTIPLE:{
		current_limit = level_limit;
		level_limit = 100;	/* arbitary large */
		list_dir(source, *point);
		level_limit = current_limit;
		break;
	    }
	}
    }
}

/* make_man_links:  subdir to subdir multiple links (man dirs)
 * Inputs:          source, target, subdir affixes to link
 * Output:          none
 */

void    make_man_links(source, target, subdirs)
    char   *source;
    char   *target;
    char   *subdirs;
{
    char    submansource[MAXPATHLEN];	/* man sources */
    char    submantarget[MAXPATHLEN];	/* man targets */
    int     x;			/* loop var */

    for (x = 0; x < 10; x++) {
	/* construct dir paths and names */
	(void) sprintf(submansource, "%s/man%c", source, subdirs[x]);
	(void) sprintf(submantarget, "%s/man%c", target, subdirs[x]);
	/* make the links */
	if (test_dir_exist(submansource, root_path))
	    make_link_type(submansource, submantarget, MULTIPLE);
    }
}

/* split_target_directories
 * Comment - Splits elements seperated by DIR_SEPERATOR
 *         - and puts then into a array of char *
 * Inputs  - point array of char *
 * Inputs  - strings containing directory names
 * Outputs - none
*/
void    split_target_directories(point, target)
    char  **point;
    char   *target;
{
    int     row = 0;		/* array index */
    int     column = 0;		/* indexes to 2d array */

    while (*target != '\0') {
	if (*target == ':') {
	    row++;
	    column = 0;
	} else
	    point[row][column++] = *target;

	target++;
    }
}

/* test_dir_exist
 * Comment:     test for link directory target
 * Inputs:      name of directory, path to directory
 * Output:      boolean
 */

int     test_dir_exist(dir_name, root_root)
    char   *dir_name;
    char   *root_root;
{
    struct stat buf;
    char    full_path[MAXPATHLEN];

    /* concatenate for full path */
    (void) sprintf(full_path, "%s/%s", root_root, dir_name);

    if ((stat(full_path, &buf)) == 0)
	return 1;		/* exists */
    else
	return 0;		/* doesn't exist */
}

/* resolve_link
 * Comment:     Resolves a symbolic link until it is a file
 * Inputs:      name of link to resolve
 * Output:      buffer for resolved name
 */

void    resolve_link(path_to_resolve, resolved_path)
    char   *path_to_resolve;
    char   *resolved_path;
{
    struct stat stat_buf;
    char    err_mesg[MAXPATHLEN];

/* do you want to resolve symbolic links */

#ifdef NO_LINK_RESOLVE
    (void) strcpy(resolved_path, path_to_resolve);
    return;
#else
    while (1) {
	if ((stat(path_to_resolve, &stat_buf)) != 0) {
	    (void) sprintf(err_mesg, "stat failed on %s in resolve_link", path_to_resolve);
	    perror(err_mesg);
	    exit(2);
	}
	if (S_ISLNK(stat_buf.st_mode))	/* if it is a link */
	    (void) readlink(path_to_resolve, resolved_path, MAXPATHLEN);
	else
	    return;		/* path has been resolved */
    }
#endif
}

