/* Symlink
 * File:        make_links.c
 * Authors:     CM King with AJ Rixon
 * Comment:     link creation routines
 */

#include "make_links.h"
#include "str_func.h"

int     present_level;		/* current search depth */
int     level_limit;		/* max search depth for directories */
int     backoff = 0;		/* default: remake all links */
int     ignore = 0;		/* default: don't overwrite existing links */
int     trial;			/* for trial runs */
int     ack;			/* acknowledgement before linking each time */

/* make_dir_links
 * Comment:     calls the function to do the linking
 * Inputs:      link target, files to link
 * Output:      none
 */
void    make_dir_links(target_name, source_file)
    char   *target_name;	/* file that exists */
    char   *source_file;	/* dir of new link */
{
    char   *tmp_file_name;	/* ptr to last component of target path */
    char    to_link_name[MAXNAMELEN];	/* link filename */
    char    source[MAXNAMELEN];	/* link target name minus CR */
    char    end[MAXNAMELEN];	/* link filename minus CR */

    /* check if it is a pathname or a filename */
    if ((tmp_file_name = strrchr(source_file, '/')) == NULL)
	tmp_file_name = source_file;
    /* a pathname so append the last element to the target path */
    (void) sprintf(to_link_name, "%s%s", target_name, tmp_file_name);
    /* take \n character out */
    rm_cr(source_file, end);
    rm_cr(to_link_name, source);

    make_link(source, end);	/* create the link */
}

/* make_link
 * Comment:     actually create the link
 * Inputs:      link target, link origin
 * Output:      none
 */
void    make_link(target, source)
    char   *target;
    char   *source;
{
    struct stat bufsource, buftarget;	/* status info for target & source */
    int     ans = NO;		/* answer to unlink query */
    char    dir_stub[MAXPATHLEN];	/* to hold parent of target */
    int     x;			/* counter */

    /* check source exists */
    if (stat(source, &bufsource) == FSERR) {
	if (ignore)
	    return;
	else {
	    (void) fprintf(stderr, "source file %s does not exist", source);
	    (void) fprintf(stderr, "or it can not be acessed\n", source);
	    exit(2);
	}
    }
    /* test if target link already exists */
    if ((lstat(target, &buftarget)) == 0) {
	if (backoff)		/* if -b specified */
	    ans = NO;
	else if (!ignore) {	/* if -i not specified */
	    (void) fprintf(stderr, "Target file %s already exists!\n", target);
	    (void) fprintf(stderr, "Do you wish to relink anyway (y/n)?\n");
	    ans = yesno();
	    (void) printf("Target: %s\n", target);
	} else
	    ans = YES;
	if (ans == NO)
	    return;		/* if no, skip this link */
	else {
	    (void) fprintf(log_file, "Unlinking %s\n", target);
	    if ((unlink(target)) == -1) {	/* remove previous link */
		(void) perror("Unlink failed\n");
		exit(2);
	    }
	}
    }
    /* test if the directory above exists, if not make it */
    (void) strcpy(dir_stub, target);
    /* find the end */
    for (x = strlen(dir_stub); dir_stub[x] != '/'; x--);
    dir_stub[x] = NULL;		/* append terminator */

    if (access(dir_stub, F_OK) == -1) {
	if (ack) {
	    (void) fprintf(stderr, "Directory %s does not exist : make it y/n?", dir_stub);
	    if ((yesno()) == NO)
		return;		/* skip it */
	}
	/* try to make it */
	if (mkdir(dir_stub, DEFAULT_PERMS) == FSERR) {
	    perror("mkdir failed");
	    exit(2);
	} else {
	    (void) fprintf(log_file, "\tmaking dir %s\n", dir_stub);
	    return;
	}
    }
    if (ack) {			/* for -a option */
	(void) fprintf(stderr, "Do you wish to link %s to %s\n", target, source);
	if ((yesno()) == NO)
	    return;		/* skip it */
    }
    /* update log */
    (void) fprintf(log_file, "\tLinking %s to %s\n", target, source);
    if (trial) {		/* for -t option */
	(void) fprintf(log_file, "\tsymlink(%s,%s)\n\n", source, target);
	return;
    } else {
	if (symlink(source, target) == FSERR) {	/* make the link */
	    /* unsuccessful - log it */
	    (void) fprintf(log_file, "Failed to link %s to ", target);
	    (void) fprintf(log_file, "%s \t %s\n", source, sys_errlist[errno]);
	    exit(2);
	}
    }
}

/* yesno
 * Comment:     read one line from stdin and test for Y/N
 * Inputs:      none
 * Output:      boolean, nonzero (YES) or zero (NO)
 */
int     yesno()
{
    int     c;			/* chars from tty */
    int     rv;			/* response boolean */
    FILE   *fdtty;		/* tty handle */

    /* open tty for input, since lex and yacc use stdin */
    if ((fdtty = fopen("/dev/tty", "r")) == NULL) {
	(void) fprintf(stderr, "Failed to open /dev/tty\n");
	exit(2);
    }
    setbuf(fdtty, NULL);	/* turn line buffering off */
    (void) fflush(fdtty);
    c = fgetc(fdtty);		/* get a char from stdin */
    rv = (c == 'y') || (c == 'Y');	/* test for [Yy]es */
    /* get anything else (unused) */
    while (c != EOF && c != '\n')
	c = fgetc(fdtty);
    (void) fclose(fdtty);
    return rv;			/* return boolean response */
}
