/*****************************************************/
/* notify.c                                          */
/* -- DLL tracks open files by hooking int 0x21.     */
/* -- To build:                                      */
/*    cc -wd -DSTRICT notify.c libentry.obj \        */
/*      hook21.obj toolhelp.lib notify.def           */
/*    implib notify.lib notify.dll                   */
/*****************************************************/
#include <windows.h>
#include <windowsx.h>
#include <toolhelp.h>
#include "notify.h"

/* Increment this many file info entries at a time. */
#define dcofi   100

/* Get more space when number of free entries hits */
/* this value. */
#define cofiLow 50

/* Internal prototypes. */
void         CopyLpsz(LPSTR lpszDest, LPSTR lpszSrc);
void _loadds DoNotify21(LPWORD lpwStack);
int CALLBACK LibMain(HINSTANCE hinsThis, WORD wDS,
  WORD cbHeap, LPSTR lpsz);
OFI far *    LpofiFind(int wHandle);
LPVOID       LpvProt(WORD wSeg, WORD wOff);
void         Notify(void);
void         NotifyClose(int wHandle);
void         NotifyDuplicate(int wOrig, int wNew);
void         NotifyOpen(LPSTR lpszFile, int wHandle);
WORD         PdbGetCur(void);
int CALLBACK _export WEP(int wExitCode);

HWND        hwndNotify; /* Notify when list changes. */
OFI huge *  gprgofi;    /* File info list. */
long        cofi;       /* Size of list. */
long        cofiUsed;   /* Number of files in list. */
BOOL        fNotified;  /* Client has been notified. */
WORD        sel;        /* Allocated selector. */
struct
    {
    WORD    wDI, wEDIHi, wSI, wESIHi, wBP, wEBPHi;
    DWORD   lwReserved;
    WORD    wBX, wEBXHi, wDX, wEDXHi, wCX, wECXHi,
            wAX, wEAXHi, wFlags, wES, wDS, wFS, wGS,
            wIP, wCS, wSP, wSS;
    } rmr;              /* Real-Mode Registers. */

/* Store these addresses in code so real-mode code */
/* can access them without requiring a data segment. */
FARPROC _based(_segname("_CODE")) lpfnAlias;
FARPROC _based(_segname("_CODE")) lpfnSav;

int CALLBACK LibMain(HINSTANCE hinsThis, WORD wDS,
  WORD cbHeap, LPSTR lpsz)
/*****************************************************/
/* -- Install the int 21 ISR hook, save the old      */
/*    contents.                                      */
/*****************************************************/
    {
    int     wVal        = 0;    /* Outcome. */
    DWORD   lw, lpfnReal;
    FARPROC lpfnProt    = (FARPROC)Hook21;

    /* Preallocate some space for the file list. */
    cofiUsed = 0;
    if ((gprgofi = GlobalAllocPtr(GMEM_SHARE | GHND,
      (cofi = dcofi) * sizeof(OFI))) == NULL)
        return 0;   /* No memory. */

    lpfnReal = GetSelectorBase(SELECTOROF(lpfnProt));
    if (lpfnReal > 0x00100000 ||    /* Loaded low? */
      lpfnReal & 0x0000000f)    /* Para boundary? */
        return 0;       /* Can't use code segment. */

    _asm push si;   /* Save some registers blown */
    _asm push di;   /* away by ensuing _asm code. */

    lpfnReal = ((lpfnReal & 0x000ffff0) << 12) +
      OFFSETOF(lpfnProt);   /* Get real-mode addr. */

    _asm mov  ax, 0x0200;   /* Save original real- */
    _asm mov  bx, 0x0021;   /* mode int 0x21 ISR. */
    _asm int  0x31;
    _asm mov  WORD PTR [lw], dx;
    _asm mov  WORD PTR [lw + 2], cx;
    MemoryWrite(SELECTOROF(&lpfnSav), OFFSETOF(&lpfnSav),
      &lw, sizeof lw);

    _asm mov  ax, 0x0303;   /* Get a real-mode alias */
    _asm push ds;           /* for the real-mode int */
    _asm pop  es;           /* 0x21 handler to call  */
    _asm mov  di, OFFSET rmr; /* back to. */
    _asm push ds;
    _asm push cs;
    _asm pop  ds;
    _asm mov  si, OFFSET Notify21;
    _asm int  0x31
    _asm pop  ds;
    _asm jc   LCleanup;
    _asm mov  WORD PTR [lw], dx
    _asm mov  WORD PTR [lw + 2], cx
    MemoryWrite(SELECTOROF(&lpfnAlias),
      OFFSETOF(&lpfnAlias), &lw, sizeof lw);

    /* Get a selector/descriptor for obtaining */
    /* protect mode addrs. from real-mode addrs. */
    if ((sel = AllocSelector(wDS)) == 0)
        goto LCleanup;
    SetSelectorLimit(sel, 0x0000ffff);

    _asm mov  ax, 0x0201;   /* Install our real-mode */ 
    _asm mov  bx, 0x0021;   /* int 0x21 handler. */
    _asm mov  dx, WORD PTR [lpfnReal];
    _asm mov  cx, WORD PTR [lpfnReal + 2];
    _asm int  0x31;
    _asm jc   LCleanup;

    wVal = 1;   /* Success! */

LCleanup:
    _asm pop  di;
    _asm pop  si;
    return wVal;
    }

int CALLBACK _export WEP(int wExitCode)
/*****************************************************/
/* -- Remove the int 21 ISR hook.                    */
/*****************************************************/
    {
    _asm mov  ax, 0x0201;   /* Restore previous ISR. */
    _asm mov  bx, 0x0021;
    _asm mov  dx, WORD PTR cs:[lpfnSav];
    _asm mov  cx, WORD PTR cs:[lpfnSav + 2];
    _asm int  0x31;

    /* Free real-mode alias. */
    _asm mov  ax, 0x0304;
    _asm mov  dx, WORD PTR cs:[lpfnAlias];
    _asm mov  cx, WORD PTR cs:[lpfnAlias + 2];
    _asm int  0x31;

    if (gprgofi != NULL)    /* File info list. */
        GlobalFreePtr(gprgofi);
    if (sel != 0)           /* Addr. xlate selector. */
        FreeSelector(sel);

    return 1;
    }

void _loadds DoNotify21(LPWORD lpwStack)
/*****************************************************/
/* -- Protect mode callback from real-mode calls     */
/*    appropriate notification routine to deal with  */
/*    current interrupt 0x21.                        */
/* -- lpwStack : Protect mode address corresponding  */
/*               to real mode stack.                 */
/*****************************************************/
    {
    /* Fix up return address (CS:IP) in real mode */
    /* register data structure. */
    rmr.wIP = lpwStack[0];
    rmr.wCS = lpwStack[1];
    rmr.wSP += 4;   /* Simulate a far return. */

    /* Call the appropriate notification handler. */
    switch (HIBYTE(lpwStack[2]))
        {
    default:    /* Should never happen. */
        break;  /* Should have an assert here. */

    case 0x3c:
    case 0x3d:
    case 0x5a:
    case 0x5b:
        NotifyOpen(LpvProt(rmr.wDS, rmr.wDX), rmr.wAX);
        break;

    case 0x3e:
        NotifyClose(rmr.wBX);
        break;

    case 0x45:
        NotifyDuplicate(rmr.wBX, rmr.wAX);
        break;

    case 0x46:
        NotifyDuplicate(rmr.wCX, rmr.wBX);
        break;

    case 0x6c:
        NotifyOpen(LpvProt(rmr.wDS, rmr.wSI), rmr.wAX);
        break;
        }
    }

LPVOID LpvProt(WORD wSeg, WORD wOff)
/*****************************************************/
/* -- Return the protect mode address corresponding  */
/*    to the given real-mode address.                */
/* -- Uses the pre-allocated selector, so this call  */
/*    cannot be re-entered.                          */
/*****************************************************/
    {
    SetSelectorBase(sel, (DWORD)wSeg * 0x10);
    return MAKELP(sel, wOff);
    }

void NotifyOpen(LPSTR lpszFile, int wHandle)
/*****************************************************/
/* -- Add the file to the list and notify the client */
/*    the list has changed.                          */
/*****************************************************/
    {
    long        iofi;
    OFI huge *  gpofi;
    WORD        pdb     = PdbGetCur();

    if (cofi == cofiUsed)
        return; /* Out of space. */

    /* Check for duplicates. */
    for (iofi = 0, gpofi = gprgofi; iofi < cofiUsed;
      iofi += (gpofi++)->pdb != 0)
        if (gpofi->pdb == pdb &&
          gpofi->wHandle == wHandle)
            return; /* Duplicate. */

    /* Find a free slot in the list. */
    for (iofi = 0, gpofi = gprgofi; iofi < cofi;
      gpofi++, iofi++)
        if (gpofi->pdb == 0)
            {
            /* Add info to list. */
            gpofi->wHandle = wHandle;
            CopyLpsz(gpofi->szPath, lpszFile);
            gpofi->pdb = pdb;
            cofiUsed++;
            Notify();   /* Inform client. */
            break;
            }
    }

WORD PdbGetCur(void)
/*****************************************************/
/* -- Return the current PDB.                        */
/*****************************************************/
    {
    WORD    pdb;

    _asm mov     ax, 0x5100
    _asm int     0x21
    _asm mov     pdb, bx;
    return pdb;
    }

void Notify(void)
/*****************************************************/
/* -- Post notification that file list has changed.  */
/*****************************************************/
    {
    if (IsWindow(hwndNotify))
        {
        if (!fNotified)
            {
            fNotified = TRUE;
            PostMessage(hwndNotify, WM_USER, 0, 0);
            }
        }
    else
        hwndNotify = NULL;
    }

void CopyLpsz(LPSTR lpszDest, LPSTR lpszSrc)
/*****************************************************/
/* -- Our own version of lstrcpy(), since it needs   */
/*    to be called at interrupt time.                */
/*****************************************************/
    {
    while ((*lpszDest++ = *lpszSrc++) != 0)
        ;
    }

void NotifyClose(int wHandle)
/*****************************************************/
/* -- Remove the file from the list and notify the   */
/*    client the list has changed.                   */
/*****************************************************/
    {
    OFI far *   lpofi;

    /* See if the file is on the list. */
    if ((lpofi = LpofiFind(wHandle)) != NULL)
        {
        /* Remove from list. */
        lpofi->pdb = 0;
        cofiUsed--;
        Notify();   /* Inform client. */
        }
    }

OFI far * LpofiFind(int wHandle)
/*****************************************************/
/* -- Find the file handle for the current task.     */
/* -- Return NULL if not found.                      */
/*****************************************************/
    {
    long        iofi;
    OFI huge *  gpofi;
    WORD        pdb     = PdbGetCur();

    for (iofi = 0, gpofi = gprgofi; iofi < cofiUsed;
      iofi += (gpofi++)->pdb != 0)
        if (gpofi->pdb == pdb &&
          gpofi->wHandle == wHandle)
            return gpofi;

    return NULL;
    }

void NotifyDuplicate(int wOrig, int wNew)
/*****************************************************/
/* -- Add the duplicate handle to the list.          */
/*****************************************************/
    {
    OFI far *   lpofi;

    /* See if the file is on the list. */
    if ((lpofi = LpofiFind(wOrig)) != NULL)
        NotifyOpen(lpofi->szPath, wNew);
    }

void WINAPI _export SetPostWnd(HWND hwnd)
/*****************************************************/
/* -- Set the window to receive notifications.       */
/*****************************************************/
    {
    hwndNotify = hwnd;
    }

BOOL WINAPI _export FNextOfi(OFI far * lpofi,
  long far * lpiofi)
/*****************************************************/
/* -- Client service.                                */
/* -- Return the next file in the list to the        */
/*    client.                                        */
/* -- Call with *lpiofi set to -1 to begin           */
/*    enumeration.                                   */
/* -- Updates lpiofi to next entry each call,        */
/*    returns FALSE once all entries have been       */
/*    enumerated (in which case *lpofi is invalid).  */
/*****************************************************/
    {
    if (fNotified)
        {
        fNotified = FALSE;  /* Ok to notify again. */

        /* Grow list? */
        if (cofi - cofiUsed < cofiLow)
            {
            OFI huge *  gprgofiNew;

            if ((gprgofiNew = GlobalReAllocPtr(gprgofi,
              (cofi + dcofi) * sizeof(OFI),
              GMEM_SHARE | GHND)) != NULL)
                {
                gprgofi = gprgofiNew;
                cofi += dcofi;
                }
            }
        }

    /* Is the list empty? */
    if (cofiUsed == 0)
        return FALSE;

    /* Find the next file. */
    for (++*lpiofi; *lpiofi < cofi; ++*lpiofi)
        if (gprgofi[*lpiofi].pdb != 0)
            {
            *lpofi = gprgofi[*lpiofi];
            return TRUE;
            }

    return FALSE;
    }
