// wuiargs.c - implement argument parsing for WUIMAN set attribute
// Copyright (c) 1993 by Ron Burk

#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <windows.h>

#include "wuiargs.h"

// GetIdentifier - extract token matching "[A-Za-z_][A-Za-z0-9_]*".
static
const char *GetIdentifier(const char *Input, char *Output)
    {
    if(isalpha(*Input) || *Input == '_')
        {
        while(isalnum(*Input) || *Input == '_')
            *Output++   = *Input++;
        *Output = '\0';
        return Input;
        }
    else
        return 0;
    }

// EatWhite - skip over white space in the input stream.
static
const char *EatWhite(const char *Input)
    {
    while(*Input && isspace(*Input))
        ++Input;
    return Input;
    }

// Constructor - parse name and arg list
TWuiArgs::TWuiArgs(const char *Input)
    :   NArgs_(0), AttributeName_(0)
    {   MEMBERASSERT();
    char    Name[TWuiName::MAXNAME];
    char    Token[256];

    Input   = EatWhite(Input);
    Input   = GetIdentifier(Input, Token);
    if(!Input)  return;
    Token[TWuiName::MAXNAME-1]    = '\0';
    strcpy(Name, Token);
    Input   = EatWhite(Input);
    if(*Input)  // something after attribute name
        {
        // it better be left paren
        if(*Input++ != '(') return;
        Input   = EatWhite(Input);
        while(*Input && *Input != ')')
            {
            Input   = GetArg(Input, Token);
            if(!Input) return;
            Input   = EatWhite(Input);
            if(*Input == ',')
                Input   = EatWhite(++Input);
            ++NArgs_;
            }
        if(*Input != ')')
            return;
        }
    AttributeName_  = Name;
    }

// GetArg - helper function to extract string or long
const char *TWuiArgs::GetArg(const char *Input, char *Buffer)
    {   MEMBERASSERT();
    char    *Output = Buffer;
    char    *End;       // can't use const because of strtol()
    *Output = '\0';
    if(*Input == '\'' || *Input == '\"')
        {
        char    Quote = *Input++;
        char    Next;
        while((Next=*Input++) != '\0')
            {
            if(Next == Quote)
                if(*Input == Quote)
                    ++Input;    // quoted quote char
                else    // else, end of quote
                    {
                    *Output           = '\0';
                    char *Result      = StringClone(Buffer);
                    ArgTypes_[NArgs_] = STRING_ARG;
                    Args_[NArgs_]     = Result;
                    return Input;   // success!
                    }
            *Output++   = Next;
            }
        // must not have found end quote
        return 0;
        }
    else    // else, try for integral arg
        {
        long    LongArg;
        errno   = 0;    // I hate errno!
        LongArg = strtol(Input, &End, 0);
        if(errno || (End == Input))
            return 0;
        else
            {
            Args_[NArgs_]       = (void *)new long(LongArg);
            ArgTypes_[NArgs_]   = LONG_ARG;
            return End;
            }
        }
    }

// Destructor - free up memory
TWuiArgs::~TWuiArgs()
    {   MEMBERASSERT();
    if(NArgs_ > 0)
        {
        for(int i = 0; i < NArgs_; ++i)
            if(ArgTypes_[i] == STRING_ARG)
                delete (char *)Args_[i];
            else
                delete (long *)Args_[i];
        }
    }

// GetType - return type of a particular argument
TWuiArgs::ArgType TWuiArgs::GetType(int Which)
    {   MEMBERASSERT();
    if(Which >= 0 && Which < NArgs())
        return ArgTypes_[Which];
    else
        return ERROR_ARG;
    }

// GetLong - return Nth argument as a long
long    TWuiArgs::GetLong(int Which)
    {   MEMBERASSERT();
    if(Which >= 0 && Which < NArgs())
        return *(long *)Args_[Which];
    else
        return 0;
    }

// GetString - return Nth argument as a string pointer
char   *TWuiArgs::GetString(int Which)
    {   MEMBERASSERT();
    if(Which >= 0 && Which < NArgs())
        return (char *)Args_[Which];
    else
        return 0;
    }


