// wuidbint.c - WUIMAN DataBase INTerface definitions
// Copyright (c) 1993 by Ron Burk

#include <string.h>

#include "wuistd.h"
#include "wuidbint.h"

#if !(defined(_INC_WINDOWS) || defined(__WINDOWS_H))
    #include <windows.h>
#endif

const char *SENTINEL = "$";

// constructor - just save database name (assume it's legal)
TWuiDatabase::TWuiDatabase(const char *Database)
    :   DatabaseName_(StringClone(Database))
    {   MEMBERASSERT();
    }

// destructor - free up database name
TWuiDatabase::~TWuiDatabase()
    {   MEMBERASSERT();
    if(DatabaseName_)
        delete[] DatabaseName_;
    }

// WriteItem - Set value for item under specified path.
int     TWuiDatabase::WriteItem(const char *Path,
                       TWuiName KeyName, const char *Buffer)
    {   MEMBERASSERT();
    char    Key[TWuiName::MAXNAME+1];
    KeyName.GetName(Key);
    char   *Copy = new char[strlen(Buffer)+2];
    strcpy(Copy, Buffer);
    strcat(Copy, SENTINEL);
    int Status = WritePrivateProfileString(Path, Key, Copy,
                                        DatabaseName_) != 0;
    delete[] Copy;
    return Status;
    }

// ReadItem - Get value for named item under specified path.
int     TWuiDatabase::ReadItem(const char *Path,
        TWuiName KeyName, char *Buffer, int Length)
    {   MEMBERASSERT();
    char    Key[TWuiName::MAXNAME+1];
    KeyName.GetName(Key);
    char   *DatabaseRecord = new char[Length+3];
    int BytesRead = GetPrivateProfileString(Path, Key, "",
                   DatabaseRecord, Length+3, DatabaseName_);
    // if not even sentinel was read, then record is
    // missing or garbaged
    if(BytesRead <= 0)
        BytesRead  = -1;
    else
        {
        if(BytesRead < Length+2) // if didn't overflow
            {
            ASSERT(DatabaseRecord[BytesRead-1]
                                            == SENTINEL[0]);
            BytesRead   = strlen(DatabaseRecord)-1;
            // trim sentinel byte
            DatabaseRecord[BytesRead] = '\0';
            }
        if(Buffer != NULL)
            memcpy(Buffer, DatabaseRecord, BytesRead+1);
        }
    delete[] DatabaseRecord;
    return BytesRead;
    }

// DeleteItem - Delete named item under specified path.
int     TWuiDatabase::DeleteItem(const char *Path,
                                            const char *Key)
    {   MEMBERASSERT();
    return WritePrivateProfileString(Path, Key, NULL,
                                        DatabaseName_) != 0;
    }

// DeletePath - Delete path itself (and any children)
int     TWuiDatabase::DeletePath(const char *Path)
    {   MEMBERASSERT();
    return WritePrivateProfileString(Path, NULL, NULL,
                                        DatabaseName_) != 0;
    }

// GetKeys - Get the children/attribute of specified path.
TWuiDatabaseKeys *TWuiDatabase::GetKeys(const char *Path)
    {   MEMBERASSERT();
    const int MAXKEYS   = (1024*4);
    char *Keys          = new char[MAXKEYS];

    // if no keys found, or too many to fit in buffer,
    // then fail by returning 0.

    int Status = GetPrivateProfileString(Path, NULL, "",
                   Keys, MAXKEYS, DatabaseName_);
//    int Status  = ReadItem(Path, NULL, Keys, MAXKEYS);
    if(Status <= 0 || (Status >= MAXKEYS-1))
        {
        delete[] Keys;
        return 0;
        }
    else
        return new TWuiDatabaseKeys(Keys);
    }

// constructor - Save string of keys.
TWuiDatabaseKeys::TWuiDatabaseKeys(char *Keys)
    :   Keys_(Keys), Current_(Keys)
    {   MEMBERASSERT();
    ASSERT(Keys != 0);
    }

// destructor - Free string of keys.
TWuiDatabaseKeys::~TWuiDatabaseKeys()
    {   MEMBERASSERT();
    if(Keys_)
        delete[] Keys_;
    }

// FirstKey - fetch first key of series.
const char *TWuiDatabaseKeys::FirstKey()
    {   MEMBERASSERT();
    Current_    = Keys_;
    if(*Current_ == '\0')
        return NULL;
    else
        return Current_;
    }

// NextKey - fetch next key in series, NULL if at end.
const char *TWuiDatabaseKeys::NextKey()
    {   MEMBERASSERT();
    ASSERT(Current_ != NULL);
    if(*Current_)
        Current_    += strlen(Current_)+1;
    if(*Current_)
        return Current_;
    else
        return NULL;
    }

