// w_hook.c - implement window subclassing code
// Copyright (c) 1993 by Ron Burk

#include "wuistd.h"
#include "w_atom.h"
#include "w_hook.h"

// Atom names and handles for fast window properties.
static TGlobalAtom FunctionLow("WUIMAN_FunctionLow");
static TGlobalAtom FunctionHigh("WUIMAN_FunctionHigh");

// TWindowSubclassDefault - an object of this class gets put
//      at the end of the chain, and calls the window's
//      previous window procedure.
class   TWindowSubclassDefault : public TWindowSubclass
    {
public:
    TWindowSubclassDefault(HWND Window, WNDPROC Old);
    virtual LRESULT Handler(HWND Window, UINT Message,
                              WPARAM Param1, LPARAM Param2)
        {   MEMBERASSERT();
        return OldHandler(Window, Message, Param1, Param2);
        }
private:
    // WindowHook wants to use OldHandler.
    friend LRESULT CALLBACK _export WindowHook(HWND,UINT,
                                             WPARAM,LPARAM);
    WNDPROC OldHandler;
    };

// constructor - take over window, start chain of handlers.
TWindowSubclassDefault::TWindowSubclassDefault(HWND Window,
                                                WNDPROC Old)
    :   OldHandler(Old)
    {   MEMBERASSERT();
    // make window point to me as first handler
    FunctionLow.SetProperty(Window, (HANDLE)LOWORD(this));
    FunctionHigh.SetProperty(Window,(HANDLE) HIWORD(this));
    }

// GetClassPointer() - Fetch my object from window handle.
inline
static  TWindowSubclass *GetClassPointer(HWND Window)
    {
    HANDLE  Low     = FunctionLow.GetProperty(Window);
    HANDLE  High    = FunctionHigh.GetProperty(Window);
    return (TWindowSubclass *)MAKELP(High, Low);
    }

inline
WNDPROC CurrentHandler(HWND Window)
    {
    return (WNDPROC)GetWindowLong(Window, GWL_WNDPROC);
    }

// Unhook() - delete all but last handler.
WNDPROC TWindowSubclass::Unhook(HWND Window,
                                  TWindowSubclass *Subclass)
    {
    // first, delete all the "real" subclass objects
    while(Subclass->Next != NULL)
        {
        TWindowSubclass *Temp = Subclass;
        Subclass    = Subclass->Next;
        delete Temp;
        }
    // we happen to know last in chain is an
    // object of type TWindowSubclassDefault
    TWindowSubclassDefault *Default =
                         (TWindowSubclassDefault *)Subclass;
    // second, remove our window properties
    ASSERT2(
        FunctionLow.RemoveProperty(Window) != NULL
        );
    ASSERT2(
        FunctionHigh.RemoveProperty(Window) != NULL
        );
    WNDPROC Current = CurrentHandler(Window);
    // put back previous window procedure (assume it's safe)
    SetWindowLong(Window, GWL_WNDPROC, (LONG)Current);
    delete  Default;
    return Current;
    }

// WindowHook() - Entry point for all subclassed windows.
LRESULT CALLBACK _export TWindowSubclass::WindowHook(
    HWND Window, UINT Message, WPARAM Param1, LPARAM Param2)
    {
    TWindowSubclass *Subclass   = GetClassPointer(Window);
    ASSERT(Subclass != NULL);

    // if it's not the end of the line
    if(Message != WM_DESTROY && Message != WM_NCDESTROY)
        // then just call first subclass function in chain
        return Subclass->Handler(Window, Message, Param1, Param2);
    // else, try to unsubclass the window

    LRESULT Result      = 0;
    // if it's not safe on the way down
    if(CurrentHandler(Window) != WindowHook)
        {   // then pass the message on
        Result  = Subclass->Handler(Window, Message, Param1,
                                                    Param2);
        // and see if it's safe on the way back
        if(CurrentHandler(Window) == WindowHook)
            {   // then unhook and we are out of loop
            Unhook(Window, Subclass);
            return Result;
            }
        else if(Message == WM_NCDESTROY)
            {   // if last chance!
            Unhook(Window, Subclass);
            return Result;
            }
        else    // else try again on WM_NCDESTROY
            return Result;
        }
    else
        {   // else it is ok to unhook right now
        WNDPROC OldHandler = Unhook(Window, Subclass);
        return OldHandler(Window, Message, Param1, Param2);
        }
    }

TWindowSubclass::TWindowSubclass(HWND Window)
    :   Next(NULL)
    {   MEMBERASSERT();
    ASSERT(Window != (HWND)NULL);
    // might already be a linked list of handlers
    TWindowSubclass *CurrentHandler =
                                    GetClassPointer(Window);
    if(CurrentHandler == NULL)
        {   // then we have not poked window proc yet
        WNDPROC OldHandler = (WNDPROC)SetWindowLong(Window,
                             GWL_WNDPROC, (LONG)WindowHook);
        ASSERT2(
            new TWindowSubclassDefault(Window, OldHandler)
                    != NULL
            );
        CurrentHandler  = GetClassPointer(Window);
        }
    ASSERT(CurrentHandler != NULL);
    // point me to previous first handler
    this->Next  = CurrentHandler;
    }

// constructor called only by TWindowSubclassDefault
TWindowSubclass::TWindowSubclass()
    :   Next(NULL)
    {   MEMBERASSERT();
    }

TWindowSubclass::~TWindowSubclass()
    {   MEMBERASSERT();
    }

// Handler() - by default, just pass to next handler in chain
LRESULT TWindowSubclass::Handler(HWND Window, UINT Message,
                               WPARAM Param1, LPARAM Param2)
    {   MEMBERASSERT();
    ASSERT(Next != NULL);
    return Next->Handler(Window, Message, Param1, Param2);
    }


