/* ----------------------------------------------------------
 *  Filename:           makelib.c
 *  Summary:            Object library maintenance utility.
 *  Author:             T.W. Nelson
 *  Compiler:           BORLAND
 *  Compile options:
 *  Date:               10-Oct-1993
 *  Version:            2.00
 *  Notes:
 *
 *  Compiler must enforce struct byte-packing
 *
 *  Source code Copyright (c) 1993 T.W. Nelson.
 *  Released to Public Domain. Not for Resale.
 * ------------------------------------------------------- */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <dir.h>
#include <dos.h>

#define FOUND   0           //findfirst/next() return value
#define OBJ_LIMIT   100     //maximum # .OBJ files added
                            //during one run (default)

// Global switch variables for buildargs, cmdargs ....
char *Compiler      = "";    //Compiler path name
char *CompExtension = "C";   //default source file ext
char *Assembler     = "";    //Assembler pathname
char *AsmExtension  = "ASM"; //default source file ext
char *LibManager    = "";    //Library manager pathname
char *TargetLibrary = "";    //library name to modify
char *ObjectPath    = "";    //.OBJ(s) deposited here
char *ErrorFile     = "";    //error msgs re-directed
char *LibListFile   = "";    //library module list file
char *AsmCmdLine    = "";    //assembler command line
char *CompCmdLine   = "";    //compiler command line
char *LibCmdLine    = "";    //lib manager command line
char *RespCmdLine   = "";    //response file command line
char *Source        = "%s";  //for SOURCE keyword in .BLD file
int  ObjLimit       = OBJ_LIMIT;     //see above
int  BatchMode      = 0;     //feed source in batch style
int  KeepObjs       = 0;     //keep .OBJs after run
int  DeleteBak      = 0;     //delete library .BAK file
int  ForceBuild     = 0;     //defeat dependency check
int  ListOptions    = 0;     //list cmd/build args

struct buildargs   {
    char    *swarg;     // pointer to switch string
    int     type;       // argument type (boolean or text)
    int     *swvar;     // switch variable to set
};

#define BOOLARG     0
#define TEXTARG     1
#define NUMARG      2   //integer argument

struct buildargs Buildtab[] = {
{ "COMPILER",      TEXTARG, (int *) &Compiler       },
{ "COMPEXTENSION", TEXTARG, (int *) &CompExtension  },
{ "ASSEMBLER",     TEXTARG, (int *) &Assembler      },
{ "ASMEXTENSION",  TEXTARG, (int *) &AsmExtension   },
{ "LIBMANAGER",    TEXTARG, (int *) &LibManager     },
{ "TARGETLIBRARY", TEXTARG, (int *) &TargetLibrary  },
{ "OBJECTPATH",    TEXTARG, (int *) &ObjectPath     },
{ "ERRORFILE",     TEXTARG, (int *) &ErrorFile      },
{ "LIBLISTFILE",   TEXTARG, (int *) &LibListFile    },
{ "ASMCMDLINE",    TEXTARG, (int *) &AsmCmdLine     },
{ "COMPCMDLINE",   TEXTARG, (int *) &CompCmdLine    },
{ "LIBCMDLINE",    TEXTARG, (int *) &LibCmdLine     },
{ "RESPCMDLINE",   TEXTARG, (int *) &RespCmdLine    },
{ "SOURCE",        TEXTARG, (int *) &Source         },
{ "OBJLIMIT",      NUMARG,  &ObjLimit               },
{ "BATCHMODE",     BOOLARG, &BatchMode              },
{ "KEEPOBJS",      BOOLARG, &KeepObjs               },
{ "DELETEBAK",     BOOLARG, &DeleteBak              },
{ "FORCEBUILD",    BOOLARG, &ForceBuild             },
{ "LISTOPTIONS",   BOOLARG, &ListOptions            },
};

#define TABSIZE     sizeof(Buildtab)
#define BUILDTABSIZE (TABSIZE/sizeof(struct buildargs))
#define NONTEXTARGS     6       //---!!CAUTION!!-----

/* TEXTARGs are saved here after being read from
 * the .BLD file.... */
char Argspace[BUILDTABSIZE-NONTEXTARGS] [65];

struct cmdargs   {
    int     swarg;      //switch character
    int     type;       //argument type (boolean or text)
    int     *swvar;     //switch variable to set
};

char *BuildFile = "";
char BuildFileBuf[65];      //in case not on cmd line
char *ResponseFile = "MAKELIB$.ARF"; // Auto-response file

/* This used for cmd-line switches. Switches can be specified
 * in either lower or upper case ..... */
struct cmdargs Cmdtab[] = {
    {   'B',    BOOLARG,    &BatchMode          },
    {   'K',    BOOLARG,    &KeepObjs           },
    {   'D',    BOOLARG,    &DeleteBak          },
    {   'F',    BOOLARG,    &ForceBuild         },
    {   'N',    TEXTARG,    (int *) &BuildFile  },
    {   'L',    BOOLARG,    &ListOptions        },
};

#define CMDTABSIZE (sizeof(Cmdtab)/sizeof(struct cmdargs))

/* This points to source file names specified on MAKELIB
 * command line (may include wild cards) ..... */
char *Sargv[20];
int Sargc = 0;              //#source filespecs

char **Argv;
int  Argc;
unsigned long LibraryAge = 0L;
char break_flag = 0;        //set by int 1bh handler
void interrupt (*oldint_1b)();  //original 1bh handler

char *sign_on[] = {
 "",
 "MAKELIB Library Maintenance Utility, Version 2.00",
 "Copyright (c) 1993 T.W. Nelson. Public Domain, Not for resale.",
 "", 0
 };

char *Usage[] = {
"Syntax:",
"   MAKELIB [switches] [path]source [[path]source]...",
"",
"Switches:",
"/F         Suppress time-related dependency checking",
"/B         Feed source files in batch style",
"/K         Keep .OBJ files after modifying library",
"/D         Delete .BAKup file after modifying library",
"/L         List all cmd line and .BLD file arguments",
"/N<file>   Name of alternate build file to use",
"",
"Switches are introduced with either '/' or '-' and may be",
"typed in lower or upper case. All but the 'N' switch may",
"also be specified in the build configuration file. Source",
"files may include wildcard characters '?' and '*'.",
"", 0 };

/* Structure used by file search functions. This is used
 * in preference to struct 'ffblk' in dir.h to treat both
 * date and time as a single number (i.e. time stamp) ...*/
struct file_search {
    char   reserved[21];       //DOS work area
    char   file_attr;          //attr of file found
    long   stamp;              //file's date/time stamp
    long   filsize;            //file size in bytes
    char   filename[13];       //file name & ext.
} fsrch;

//Determine string equality, case insensitive ......
#define strequ(a,b)         (strcmpi(a,b) == 0 ? 1 : 0)

//Functions in file 'chelp.asm' .......
int redirect_stdout( char *tofile );
void restore_stdout(void);
int exec_prog( char *progname, char *cmdline );

/* ------------------------------------------------------- */
void    print_strtab( char **strtab )
{
    char    **p ;

    for( p = strtab ; *p ; )
            fprintf( stderr, "%s\n", *p++ );
}

void abort_run(int ecode)
{
    restore_stdout();
    setvect( 0x1b, oldint_1b );
    if( ecode != 0)
        fprintf(stderr, "\n**** MAKELIB terminated\n" );
    exit(ecode);
}

void interrupt newint_1b(void)
{
   /* Receives control from int 09h keybd handler whenever
    * someone presses Ctrl-Break. This means we get ^Break
    * notification instead of DOS.
    */

    break_flag = 1;
}

void set_cmdarg( struct cmdargs *tabp, int n )
{
   /* Set a switch variable from the command line .... */
    if( tabp->type == BOOLARG )
            *tabp->swvar = 1;
    else // tabp->type == TEXTARG
            *(char **)tabp->swvar = &Argv[n][2];
}

struct cmdargs *find_cmdarg( int arg )
{
   /* Find an argument in Cmdtab[]. Return -> table member.
    * Aborts program if argument not found ...
    */

    register int i;

    arg = toupper(arg);

    for( i = 0 ; i < CMDTABSIZE ; i++ )         {
             if( Cmdtab[i].swarg == arg )
                      return &Cmdtab[i];
    }

    fprintf(stderr,"**** Invalid command line switch <%c>\n",
                               arg);
    print_strtab(Usage);
    abort_run(1);

    return (struct cmdargs *) 0;
}

void read_command_line(void)
{
   /* Read and set switches and source file specs from the
    * command line. Cmd line switches are introduced with
    * either '/' or '-'. Source file specs are re-assigned
    * to an argv[]-like pointer array. If no source files
    * are specified, print usage message and exit.
    */

    register int i;

    for( i = 1 ; i < Argc ; i++ )   {
        if( Argv[i][0] == '/' || Argv[i][0] == '-' )   {
                /* set command line switches .... */
                set_cmdarg(find_cmdarg( (int) Argv[i][1] ),i);
        }
        else    /* Assume it's a source filespec .... */
                Sargv[Sargc++] = Argv[i];
    }

    if( Sargc == 0 )    {
            print_strtab(Usage);
            abort_run(0);
    }

    Sargv[Sargc] = (char *) NULL;   //mark array end
}

void access_buildfile(void)
{
   /* Attempt to access the .BLD file on either the current
    * directory (default) or the name specified with the /N
    * switch. Program aborts if no .BLD file was found. Build
    * files specified with the /N switch need not have a
    * ".BLD" extension.
    */

    if( *BuildFile )    {
        if( findfirst( BuildFile,
                   (struct ffblk *) &fsrch, 0 ) != FOUND )  {
            fprintf(stderr,"Unable to find %s\n", BuildFile );
            abort_run(1);
        }
    }
    else    {
        if( findfirst( "*.BLD",
                   (struct ffblk *) &fsrch, 0  ) == FOUND )
            BuildFile = strcpy( BuildFileBuf,fsrch.filename);
        else    {
                fprintf(stderr,"Unable to find .BLD file\n");
                abort_run(1);
        }
    }
}

void print_buildtab(void)
{
    register int i;

    fprintf(stderr, "Valid switches are:\n" );

    for( i = 0 ; i < BUILDTABSIZE ; i++ )
            fprintf(stderr, "\"%s\"\n", Buildtab[i].swarg);
}

int find_buildarg(char *arg)
{
   /* Find an argument in Buildtab[]. Return index of table
    * member. Program aborts if the argument was not found.
    */

    register int i;

    for( i = 0 ; i < BUILDTABSIZE ; i++ )         {
            if( strequ( Buildtab[i].swarg,arg ) )
                    return i;
    }

    fprintf( stderr,"Invalid build file switch '%s'\n\n",arg);
    print_buildtab();
    abort_run(1);

    return -1;
}

void set_buildarg( char *sw,        //switch value to set
                   int n     )      //index into Buildtab[]
{
   /* Set switch variables to arguments found in the build
    * file. BOOLARGs are simply set to non-zero. TEXTARGs are
    * first copied to global data area and the switch variable
    * is repointed to it.
    */

    struct buildargs *tabp = &Buildtab[n];
    char *p;

    if( tabp->type == BOOLARG )
            *tabp->swvar = 1;
    else if( tabp->type == NUMARG )
            *tabp->swvar = atoi( sw);
    else    {     // tabp->type == TEXTARG
            p = strchr( sw, '\0' );
            while( isspace( *--p ) );
            *++p = '\0';  // strip trailing whitespace
            strcpy( Argspace[n], sw );  //save it and ...
            *(char **)tabp->swvar = Argspace[n]; //repoint
    }
}

void read_buildfile(void)
{
   /* Read each line of the .BLD file and set TEXTARG or
    * BOOLARG switches. Incorrect switch syntax will abort
    * the program.
    */

    FILE  *pbuild;
    char ln[150];
    register char *tok;
    int i;

    if( !(pbuild = fopen(BuildFile, "r" )) )    {
        fprintf(stderr,"Unable to access %s\n", BuildFile );
        abort_run(1);
    }

    // read each line from buildfile and set switches ...
    while( fgets( ln, 140, pbuild) )   {
        tok = strtok( ln, " =\t\n" );  //get switch string

        if( *tok == '\0' ||
            *tok == '#'  ||
            *tok == '\n'   )
                    continue;   //skip empty lines

        i = find_buildarg( tok );        //find table index
        tok = strtok( NULL, "#\t\n" );   //get switch value
        set_buildarg( tok, i );          //NULL for BOOLARG
    }

    fclose(pbuild);
}

void open_errorfile(void)
{
   /* Open a stdout redirection file. */

    if( !*ErrorFile )
            return;

    if( redirect_stdout( ErrorFile ) == -1 )     {
        fprintf(stderr,"Unable to redirect STDOUT to %s\n",
                        ErrorFile);
        abort_run(1);
    }
}

void check_args(void)
{
   /* Check for required arguments before proceeding ... */
    int i;

    if( ForceBuild && BatchMode )
            ForceBuild = 0;    //use BatchMode only

    if( !*TargetLibrary ) {
        fprintf(stderr,"Please specify a TARGETLIBRARY\n" );
        abort_run(1);
    }
    else if( !*LibManager ) {
        fprintf(stderr,"Please specify a LIBMANAGER\n" );
        abort_run(1);
    }
    else if( !*LibCmdLine ) {
        fprintf(stderr,"Please specify a LIBCMDLINE\n" );
        abort_run(1);
    }
    else if( !*RespCmdLine )    {
        fprintf(stderr,"Please specify a RESPCMDLINE\n" );
        abort_run(1);
    }
    else if( ObjLimit == 0 )     {
        fprintf(stderr,"Invalid OBJLIMIT parameter.\n" );
        abort_run(1);
    }
    else if( ListOptions )   {
        printf("Current MAKELIB options:\n\n");

        for( i = 0 ; i < BUILDTABSIZE; i++ )     {
            printf("%-16s- ", Buildtab[i].swarg );

            if( Buildtab[i].type == BOOLARG )
                printf("%s\n",*Buildtab[i].swvar ? "YES":"NO");
            else if( Buildtab[i].type == NUMARG )
                printf("%d\n", (int) *Buildtab[i].swvar);
            else    //TEXTARG ......
                printf("%s\n", (char *) *Buildtab[i].swvar);
        }

        printf("\n");
    }
}

void access_library_file(void)
{
   /* Access the specified library file and determine its age
    * (date/time stamp). If no library was found, its age is
    * 0. Source files are processed only if they have been
    * updated since the target library was last modified,
    * unless the FORCEBUILD or BATCHMODE options were
    * specified.
    */

    int rval;

    rval = findfirst( TargetLibrary,
                      (struct ffblk *) &fsrch, 0 );

    if( rval == 3 )   {
            fprintf(stderr,"Invalid TARGETLIBRARY path\n");
            abort_run(1);
    }

    LibraryAge = ( ( rval != FOUND ) ? 0L : fsrch.stamp );
}

void verify_object_path(void)
{
    char *p;

    if( *ObjectPath )   {
        p = strchr( ObjectPath, '\0' );

        if( *--p == '\\' )
                *p = '\0';  /* delete trailing '\' for DOS */

        if( findfirst( ObjectPath,
                    (struct ffblk *) &fsrch, FA_DIREC ) )  {
            fprintf(stderr,"Invalid OBJECTPATH specified\n");
            abort_run(1);
        }

        /* append trailing '\' for compiler\assembler ... */
        strcat( ObjectPath, "\\" );
    }
}

void run_program( char *prog, char *cmdline )
{
   /* Load and execute a program. 'cmdline' must have a count
    * byte at cmdline[0] and terminate with a carriage return.
    * Function aborts if program name was not specified with
    * an explicit path name or did not have an ".EXE" or
    * ".COM" extension. The exec_prog() function does not
    * search the PATH environment variable.  Function uses a
    * string-format cmdline to avoid parsing arguments into a
    * pointer array for execl() or spawn() functions.....
    */

    printf( "\nExec: %s%s\n\n", prog, &cmdline[1] );

    if( (int) cmdline[0] > 127 )    {
            printf("Command line for %s > 127 bytes. ", prog);
            printf("%s not executed\n", prog);
            return;
    }

    if( exec_prog( prog, cmdline ) != 0 )   {
            fprintf(stderr, "Unable to execute %s\n", prog );
            abort_run(1);
    }
}

void expand_arguments( char *argstr,   //argument list
                       char *target  ) //expanded result
{
   /* Expand embedded parameters in 'argstr' by substituting
    * arguments denoted by '$()' indicators with string switches
    * read from the buildfile, placing the result in 'target'.
    * The $(SOURCE) argument is simply replaced by a '%s' since
    * it will not be expanded until an actual argument (filename)
    * is supplied later.
    */

    int i;
    char *pa, *pt, *pw, *ps;
    char wk[50];     //work space

    for( pa = argstr, pt = target; *pa; pa++ )      {
        if( *pa == '$' )    {
            pa += 2;        //point past left parens
            pw = wk;

            while( *pa != ')' )
                    *pw++ = *pa++;  //copy arg to workspace

            *pw = '\0';     //make asciiz
            i = find_buildarg( wk );
            ps = (char *) *Buildtab[i].swvar;

            while( *ps )
                *pt++ = *ps++;  //copy switch from Buildtab[]

            continue;       //set pa past right parens
        }

        *pt++ = *pa;
    }

}

char *build_cmdline( const char *srcfile,   //source argument
                     char *cmdline    )     //"pre-formatted" cmdline
{
   /* Merge a source argument into a "pre-formatted"
    * command line, complete with leading byte count
    * and trailing CR.  Return pointer to built string.
    */

    static char dest[150];
    char worksp[150];
    sprintf( worksp, "  %s\r", cmdline );     //create space for
                                              //count and CR bytes
    sprintf( dest, worksp, srcfile );
    *dest = (char) strlen(dest) - 2;    //fill in byte count
    return dest;
}

void assemble_source(char *fname)
{
   /* Format the command line and run the assembler */
    static char cmdline[150];   //preformatted command line

    if( !*Assembler ) {
        fprintf(stderr,"Please specify an ASSEMBLER\n" );
        abort_run(1);
    }
    else if( !*AsmCmdLine )     {
        fprintf(stderr,"Please specify an ASMCMDLINE\n" );
        abort_run(1);
    }

    if( AsmCmdLine != cmdline )     {
        expand_arguments( AsmCmdLine, cmdline );
        AsmCmdLine = cmdline;
    }

    run_program( Assembler, build_cmdline( fname, cmdline ) );
}

void compile_source(char *fname)
{
   /* Format the command line and run the compiler .... */
    static char cmdline[150];   //preformatted command line

    if( *Compiler == '\0' ) {
            fprintf(stderr,"Please specify a COMPILER\n" );
            abort_run(1);
    }
    else if( !*CompCmdLine )     {
        fprintf(stderr,"Please specify a COMPCMDLINE\n" );
        abort_run(1);
    }

    if( CompCmdLine != cmdline )     {
        expand_arguments( CompCmdLine, cmdline );
        CompCmdLine = cmdline;
    }

    run_program( Compiler, build_cmdline( fname, cmdline ) );
}

void process_source_files(void)
{
    register int i;
    char *p;

    for( i = 0; i < Sargc; i++ ) {
        if( BatchMode )     {
            /* Batchmode also turns off dependency checking */
            p = strchr( Sargv[i], '.' );

            if( strequ( ++p, AsmExtension ) )
                    assemble_source( Sargv[i] );
            else if( strequ( p, CompExtension ) )
                    compile_source( Sargv[i] );

            continue;
        }

        if( findfirst( Sargv[i],
                (struct ffblk *) &fsrch, 0 ) != FOUND )  {
                printf("'%s' not found\n", Sargv[i] );
                continue;
        }

        do {
                if( break_flag )
                        abort_run(1);

                if( ForceBuild || fsrch.stamp > LibraryAge ) {
                    p = strchr( fsrch.filename, '.' );

                    if( strequ( ++p, AsmExtension ) )
                            assemble_source( fsrch.filename );
                    else if( strequ( p, CompExtension ) )
                            compile_source( fsrch.filename );
                }
        } while(findnext((struct ffblk *) &fsrch ) == FOUND );
    }
}

void invoke_libmanager(void)
{
   /* Invoke library manager with "@ResponseFile" in the
    * command line. The ResponseFile is always on the
    * current directory.
    */

    static char cmdline[150];   //preformatted command line

    if( break_flag )
            abort_run(1);

    if( LibCmdLine != cmdline )     {
        expand_arguments( LibCmdLine, cmdline );
        LibCmdLine = cmdline;
    }

    run_program( LibManager,
                 build_cmdline( ResponseFile, cmdline ) );
}

void modify_library(void)
{
   /* Build .ARF file from all .OBJ files found on OBJECTPATH.
    * Up to 'ObjLimit' files are added to the response file at
    * one time. If necessary, function invokes the library
    * manager multiple times until all objects have been
    * processed.
    */

    FILE    *arf;
    int num_added = 0;
    char    worksp[100];
    static char cmdline[100];   //preformatted command line

    if( (arf = fopen( ResponseFile, "w+t" )) == NULL )  {
            fprintf(stderr,"Unable to open response file\n");
            abort_run(1);
    }

    expand_arguments( RespCmdLine, cmdline );
    sprintf( worksp, "%s%s", ObjectPath, "*.OBJ" );

    if( findfirst( worksp,
                 (struct ffblk *) &fsrch, 0 ) != FOUND) {
            printf("No object files found\n");
            abort_run(1);
    }

    do  {
            fprintf( arf, cmdline, fsrch.filename);  //expand SOURCE arg
            fprintf( arf, "\n" );

            if( ++num_added == ObjLimit )  {
                    fclose(arf);
                    invoke_libmanager();
                    arf = fopen( ResponseFile, "w+t" );
                    num_added = 0;
            }
    } while( findnext((struct ffblk *) &fsrch) == FOUND );

    fclose(arf);

    if( num_added > 0 )
            invoke_libmanager();
}

void delete_object_files(void)
{
   /* Delete all .OBJ files on the ObjectPath. */
    char worksp[60];

    if( KeepObjs )
            return;

    sprintf( worksp, "%s%s", ObjectPath, "*.OBJ" );
    findfirst( worksp, (struct ffblk *) &fsrch, 0 );

    do {
            unlink( fsrch.filename );
    } while( findnext((struct ffblk *) &fsrch) == FOUND );
}

void delete_bak_file(void)
{
   /* Delete the library .BAK file if specified. Code assumes
    * that this file is on the same path as the target library.
    */

    char    worksp[60], *p;

    if( DeleteBak )     {
            strcpy( worksp, TargetLibrary );

            if( (p = strchr( worksp, '.' )) != NULL )
                    strcpy( ++p, "BAK" );
            else    {
                    p = strchr( worksp, '\0' );
                    strcpy( p, ".BAK" );
            }

            unlink( worksp );
    }
}

main( int argc, char **argv )
{
        print_strtab( sign_on );
        Argv = argv;
        Argc = argc;
        oldint_1b = getvect( 0x1b );
        setvect( 0x1b, newint_1b );
        read_command_line();
        access_buildfile();
        read_buildfile();
        open_errorfile();
        access_library_file();
        check_args();
        verify_object_path();
        process_source_files();
        modify_library();
        unlink( ResponseFile );
        delete_object_files();
        delete_bak_file();
        restore_stdout();
        setvect( 0x1b, oldint_1b );
        return 0;
}

/* ----- End of file -------------------------------------- */
