/******************************************************************************
*                                                                             *
*     Filename:      makepar.c                                    94/08/02    *
*                                                                             *
*     Description:   Create the SPART.PAR and 386SPART.PAR files              *
*                                                                             *
*     Usage:	     makepar						      *
*                                                                             *
*     Principle:     Get the requested partition disk and size from SYSTEM.INI*
*                    Get the disk geometry from DOS                           *
*                    Compute the track size                                   *
*                    Compute the number of tracks needed for the partition    *
*                    Scan the FAT for a big enough non-fragmented area        *
*                    Create the swap file, size 0                             *
*                    Allocate the clusters directly from the FAT              *
*                    Update the directory entry for the swap file             *
*                    Create the SPART.PAR descriptor file                     *
*                                                                             *
*     Notes:	     Assumes MS-Windows is installed in directory C:\WINDOWS. *
*		     Only supports swap files on hard disk drives C and D.    *
*									      *
*     Author:	     Jean-Francois LARVOIRE				      *
*		     Hewlett-Packard					      *
*		     Grenoble PC Division				      *
*		     Email: larvoire@hp6300.desk.hp.com 		      *
*                                                                             *
*     History:								      *
*	94/07/26 JFL Created this file, with bits and pieces reused from      *
*		      various other projects.				      *
*                                                                             *
*               COPYRIGHT Hewlett-Packard Company, 1987-1994.                 *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <direct.h>

#include <sys\types.h>     // for _chmod
#include <sys\stat.h>      // for _chmod
#include <io.h>            // for _chmod

#define streq(s1, s2) (!strcmp(s1, s2)) /* For the main test routine only */

typedef unsigned char BYTE;
typedef unsigned short WORD;
typedef unsigned long DWORD;

typedef int BOOL;

#define FALSE 0
#define TRUE 1

#pragma pack(1)   // IMPORTANT: All DOS structures are packed!

/* Constants converted back to C from DDK\386\PAGEFILE\SPART.INC
              and DDK\386\PAGEFILE\SPOEM.INC */

// WARNING WARNING DANGER DANGER!!!!!!!!!!!!!!!!!!!!!!!
//
// THESE DEFINITIONS MUST AGREE WITH SPART.H and SPOEM.H
// WHICH ARE IN THE SPART PROJECT OF WIN386.EXE.

#define OEM_INFO_MAX 128
#define WFP_MAX   128
#define FREEMIN   (1024L*1024L*2)
#define PARTMIN   (1024L*1024L)
#define PARTMINMARGINAL (1024L*1024L*8)
#define INIT_OVER_COMMIT 4

// This is the aproximate size of the WIN386 fixed kernel

#define WIN386VMMSIZE   200

#define DOSPARTFILENM   "X:\\386SPART.PAR\0"
#define DOSPARTDIRNM "386SPARTPAR\0"

#define SPARTNAME "\\spart.par\0"

#define PARTCURVERSION  0x00000300L

typedef struct tagPFileForm
{
   DWORD PFileVersion;     /* Version of partition file */
   //
   // The following two fields are provided so that the location of the
   //      swap partition can be verified via reads and writes through DOS.
   //
   //    if PFileForm.PFileName[0] != 0
   //      DOS Open PFileForm.PFileName
   //          if open fails, swap partition file is corrupted
   //      Write Random Pattern Using DOS, 512 bytes at PFileForm.DOSEndOffset
   //          if write fails or returns differnt count written,
   //        swap partition file is corrupted
   //      Write Random Pattern Using DOS, 512 bytes at PFileForm.DOSStartOffset
   //          if write fails or returns differnt count written,
   //        swap partition file is corrupted
   //
   //      Read using direct DISK call (INT 13, etc.) at location that should
   //          correspond to DOSStartOffset (first sector of partition) based
   //          on partition info in OEMField.
   //         Verify that pattern matches. If Not,
   //         swap partition file is corrupted.
   //
   //      Read using direct DISK call (INT 13, etc.) at location that should
   //          correspond to DOSEndOffset (last sector of partition) based
   //          on partition info in OEMField.
   //         Verify that pattern matches. If Not,
   //         Swap partition file is corrupted.
   //
   // NOTE: The pattern needs to be random (use a date/time seed) because
   //        the contents of the swap partition may be randomly left from the
   //        the last time the verification of the partition was performed.
   //
   DWORD DOSStartOffset;   /* if PFileForm.PFileName is not NUL */
                           /* this is the offset in that file */
                           /* of the start sector of the partition. */
   DWORD DOSEndOffset;     /* if PFileForm.PFileName is not NUL */
                           /* this is the offset in that file */
                           /* of the end sector of the partition. */
   char PFileName[WFP_MAX];
   char OEMField[OEM_INFO_MAX];
} PFileForm;

// Structure of the OEM part of spart.par file

#define OEMCURVERSION   0x0300
#define OEMINT13TYPE 1

typedef struct tagPFileOEMForm
{
   WORD PVersion;          /* Version of OEM part */
   WORD PType;             /* Type of OEM part */
                           /*    1 == INT 13 drive type */
   WORD StartCyln;         /* The first partition cylinder */
   WORD EndCyln;           /* The last partition cylinder */
   WORD HeadNumStart;      /* Heads >= this on StartCyln are in partition */
   WORD HeadNumEnd;        /* Heads <= this on EndCyln are in partition */
   WORD INT13DrvNum;       /* INT 13 DL value of drive */
   WORD INT13NumHeads;     /* total # heads */
   WORD INT13SecPerTrk;    /* sec/trk */
} PFileOEMForm;

/* Definitions adapted from the DOS programmers reference */

typedef struct tagBOOTSECTOR
{
   BYTE   bsJump[3];       /* E9 XX XX or EB XX 90                */
   char    bsOemName[8];   /* OEM name and version                */
                           /* Start of BIOS parameter block (BPB) */
   WORD  bsBytesPerSec;    /* Bytes per sector                    */
   BYTE  bsSecPerClust;    /* Sectors per cluster                 */
   WORD  bsResSectors;     /* Number of reserved sectors          */
   BYTE  bsFATs;           /* Number of file allocation tables    */
   WORD  bsRootDirEnts;    /* Number of root directory entries    */
   WORD  bsSectors;        /* Total number of sectors             */
   BYTE  bsMedia;          /* Media descriptor                    */
   WORD  bsFATsecs;        /* Number of sectors per FAT           */
   WORD  bsSecPerTrack;    /* Sectors per track                   */
   WORD  bsHeads;          /* Number of heads                     */
   DWORD bsHiddenSecs;     /* Number of hidden sectors            */
   DWORD bsHugeSectors;    /* Number of sectors if bsSectors = 0  */
                           /* End of BIOS parameter block (BPB)   */
   BYTE  bsDriveNumber;    /* Drive number (80h)                  */
   BYTE  bsReserved1;      /* Reserved                            */
   BYTE  bsBootSignature;  /* Extended boot signature (29h)       */
   DWORD bsVolumeID;       /* Volume ID number                    */
   char  bsVolumeLabel[11];/* Volume label                        */
   char  bsFileSysType[8]; /* File system type                    */
}
BOOTSECTOR;

typedef struct tagDEVICEPARAMS
{
   BYTE  dpSpecFunc;       /* Special functions                   */
   BYTE  dpDevType;        /* Device type                         */
   WORD  dpDevAttr;        /* Device attribute                    */
   WORD  dpCylinders;      /* Number of cylinders                 */
   BYTE  dpMediaType;      /* Media type                          */
                           /* Start of BIOS parameter block (BPB) */
   WORD  dpBytesPerSec;    /* Bytes per sector                    */
   BYTE  dpSecPerClust;    /* Sectors per cluster                 */
   WORD  dpResSectors;     /* Number of reserved sectors          */
   BYTE  dpFATs;           /* Number of file allocation tables    */
   WORD  dpRootDirEnts;    /* Number of root directory entries    */
   WORD  dpSectors;        /* Total number of sectors             */
   BYTE  dpMedia;          /* Media descriptor                    */
   WORD  dpFATsecs;        /* Number of sectors per FAT           */
   WORD  dpSecPerTrack;    /* Sectors per track                   */
   WORD  dpHeads;          /* Number of heads                     */
   DWORD dpHiddenSecs;     /* Number of hidden sectors            */
   DWORD dpHugeSectors;    /* Number of sectors if dpSectors = 0  */
}                          /* End of BIOS parameter block (BPB)   */
DEVICEPARAMS;

typedef struct tagRWBLOCK
{
   BYTE  rwSpecFunc;       /* Special function (must be 0)        */
   WORD  rwHead;           /* Head to read/write                  */
   WORD  rwCylinder;       /* Cylinder to read/write              */
   WORD  rwFirstSector;    /* First sector to read/write          */
   WORD  rwSectors;        /* Number of sectors to read/write      */
   DWORD rwBuffer;         /* Address of buffer for read/write data */
}
RWBLOCK;

typedef struct tagDISKIO
{
   DWORD diStartSector;    /* Sector number to start              */
   WORD  diSectors;        /* Number of sectors                   */
   char far *diBuffer;     /* Address of buffer                   */
}
DISKIO;

typedef struct tagDIRENTRY
{
   char  deName[8];        /* File name                           */
   char  deExtension[3];   /* File name extension                 */
   BYTE  deAttributes;     /* Attributes                          */
   char  deReserved[10];   /* Reserved. Do not use                */
   WORD  deTime;           /* Time of last change                 */
   WORD  deDate;           /* Date of last change                 */
   WORD  deStartCluster;   /* Number of the starting cluster      */
   DWORD deFileSize;       /* Size                                */
}
DIRENTRY;

typedef struct tagFCB      /* File control block */
{
   BYTE  fcbDriveID;       /* Drive no (0=default, 1=A, etc...)   */
   char  fcbFileName[8];   /* File name                           */
   char  fcbExtent[3];     /* File extension                      */
   WORD  fcbCurBlockNo;    /* Current block number                */
   WORD  fcbRecSize;       /* Record size                         */
   DWORD fcbFileSize;      /* File size                           */
   WORD  fcbDate;          /* Date last modified                  */
   WORD  fcbTime;          /* Time last modified                  */
   char  fcbReserved[8];   /* Reserved                            */
   BYTE  fcbCurRecNo;      /* Current record number               */
   DWORD fcbRandomRecNo;   /* Random record number                */
}
FCB;

/* Prototypes */

void usage(void);

FILE *findIni(char *name); /* Locate and open the .INI file */
int findSection(FILE *f, char *name);  /* Find a section in it */
int findItem(FILE *f, char *name);  /* Find an item in the curr. section */
int GetIniString(FILE *f, char *section, char *item, char *buf, int len);
int GetIniInt(FILE *f, char *section, char *item, int *pi);
int GetIniHex(FILE *f, char *section, char *item, unsigned int *pi);
int SetIniString(FILE *f, char *section, char *item, char *buf);
int SetIniInt(FILE *f, char *section, char *item, int i, char *format);

int GetPrivateProfileString(char *pszSection, char *pszEntry,
                            char *pszDefault, char *pszBuffer,
                            int iBufferLength, char *pszFilename);
int GetPrivateProfileInt(char *pszSection, char *pszEntry,
                         int iDefault, char *pszFilename);

DEVICEPARAMS *GetDeviceParams(int drive);
int Log2PhysSector(int iDrive, DWORD dwLogSect,
                   WORD *pwCyl, WORD *pwHead, WORD *pwSect);
WORD AbsDiskRead(int iDrive, DWORD first, WORD n, char far *buffer);
WORD AbsDiskWrite(int iDrive, DWORD first, WORD n, char far *buffer);
WORD NextCluster(int iDrive, WORD wCluster0);
#define INVALID_CLUSTER 1  // Returned by NextCluster if arguments are invalid
#define FIRST_VALID_CLUSTER 2
#define LAST_VALID_CLUSTER 0xFFEF
WORD NextCluster(int iDrive, WORD wCluster0);
WORD SetNextCluster(int iDrive, WORD wCluster0, WORD wNextCluster);
long Cluster2Sector(int iDrive, WORD wCluster);
void BeginCriticalSection(void);
void EndCriticalSection(void);

/* Global variables */

int debug = FALSE;

/******************************************************************************
*                                                                             *
*                                Main routine                                 *
*                                                                             *
******************************************************************************/

void main(int argc, char *argv[])
{
   char *pszIni = "C:\\WINDOWS\\SYSTEM.INI";
   char *pszSpart = "C:\\WINDOWS\\SPART.PAR";
   char *psz386Spart = "C:\\386SPART.PAR";      // Swap file canonic name
   char *pszName11 = "386SPARTPAR";             // Swap file name in 11 characters format
   int err;
   PFileForm spart = {0};        // Image of file SPART.PAR
   PFileOEMForm *pOEM;           // Pointer to OEM structure within spart
   FILE *f;
   int i, j;
   char szBuffer[FILENAME_MAX];
   DEVICEPARAMS *pdp;
   int iDrive;                    // 1 = A, 2 = B, 3 = C, ...
   DIRENTRY *pde;
   char *pszBuf;                 // Pointer to a dynamically allocated buffer
   int i0, imax;                 // First sector and above limit of root directory
   int found;                    // TRUE if swap file directory entry found
   WORD w;
   DWORD dw;
   DWORD dwSwapFileSize;
   DWORD dwTrackSize;
   DWORD dwClusterSize;
   WORD wFreeTracks;             // Count of consecutive free tracks
   WORD wMaxFreeTracks;          // Count of maximum consecutive free tracks
   WORD wStartCluster;           // Free cluster at the beginning of the current free tracks
   WORD wCluster0;               // First cluster for our swap file
   WORD wCluster1;               // Last cluster for our swap file
   WORD wNeededTracks;           // Number of tracks needed for the swap file
   WORD wCyl0, wHead0, wSect0;   // First sector location
   WORD wCyl1, wHead1, wSect1;   // Last sector location

   printf("\nMakePar version 0.1\n\n");

   /* Get the command line arguments */

   for (i=1; i<argc; i++)
   {
      if ((argv[i][0] == '-') || (argv[i][0] == '/')) /* It's a switch */
      {
         strlwr(argv[i]);
         if (   streq(argv[i]+1, "help")
         || streq(argv[i]+1, "h")
         || streq(argv[i]+1, "?"))
         {
            usage();
         }
         if (streq(argv[i]+1, "v"))
         {
            debug = TRUE;
            continue;
         }
         printf("Unrecognized switch %s. Ignored.\n", argv[i]);
         continue;
      }
      printf("Unexpected argument: %s\nIgnored.\n", argv[i]);
      break;  /* Ignore other arguments */
   }

   /* Get permanent swap file settings in SYSTEM.INI */

   GetPrivateProfileString("386Enh", "PermSwapDOSDrive", "not found",
                           szBuffer, sizeof(szBuffer), pszIni);
   printf("PermSwapDOSDrive = %s\n", szBuffer);
   if (szBuffer[1])           // If not found
   {
      printf("PermSwapDOSDrive not specified in SYSTEM.INI.\n");
      exit(1);
   }
   iDrive = toupper(szBuffer[0]) - '@';    // 1 = A, 2 = B, 3 = C, ...

   dwSwapFileSize = GetPrivateProfileInt("386Enh", "PermSwapSizeK", 0, pszIni);
   if (!dwSwapFileSize)
   {
      printf("PermSwapSizeK not specified in SYSTEM.INI.\n");
      exit(1);
   }
   if (debug) printf("PermSwapSizeK = %ld\n", dwSwapFileSize);
   dwSwapFileSize *= 1024;    // Convert size to bytes

   /* Get the swap disk geometry */

   pdp = GetDeviceParams(iDrive);
   if (!pdp)
   {
      printf("\nDrive %c not accessible!\n", iDrive + '@');
      exit(1);
   }

   dwTrackSize = pdp->dpSecPerTrack;
   dwTrackSize *= pdp->dpBytesPerSec;    // Track size, in bytes

   dwClusterSize = pdp->dpSecPerClust;
   dwClusterSize *= pdp->dpBytesPerSec;   // Cluster size, in bytes

   if (debug)
   {
      printf("\nDrive %c geometry:\n", iDrive + '@');
      printf("Cylinders: %04X\n", pdp->dpCylinders);
      printf("Heads: %04X\n", pdp->dpHeads);
      w = pdp->dpSecPerTrack;
      dw = dwTrackSize;
      printf("Sectors/track: %04X (%ld%s KB)\n", w, dw/1024, (dw%1024)?".5":"");
      w = pdp->dpBytesPerSec;
      printf("Sector size: %04X (%d%s KB)\n", w, w/1024, (w%1024)?".5":"");
      w = pdp->dpBytesPerSec * pdp->dpSecPerClust;
      printf("Cluster size: %04X (%d%s KB)\n", w, w/1024, (w%1024)?".5":"");
      printf("Hidden sectors: %04lX\n", pdp->dpHiddenSecs);
      printf("Reserved sectors: %04X\n", pdp->dpResSectors);
      printf("FATs: %d * %04X sectors\n", pdp->dpFATs, pdp->dpFATsecs);
   }

   /* Make sure files SPART.PAR and 386SPART.PAR don't already exist */

   f = fopen(pszSpart, "rb");
   if (f)
   {
      printf("Error: The %s file already exists!\n", pszSpart);
      exit(1);
   }

   f = fopen(psz386Spart, "rb");
   if (f)
   {
      printf("Error: The %s file already exists!\n", psz386Spart);
      exit(1);
   }

   f = fopen(psz386Spart, "wb");
   if (!f)
   {
      EndCriticalSection();
      printf("Error: Can't create file %s!\n", psz386Spart);
      exit(1);
   }
   fclose(f);

   /* Compute the swap file requirements */

   wNeededTracks = (WORD)(dwSwapFileSize / dwTrackSize); // Integer number of tracks needed
   dwSwapFileSize = wNeededTracks * dwTrackSize;         // Round size to integer number of tracks
   printf("The swap file will be %ld KB\n", dwSwapFileSize / 1024);

   /* Scan the FAT for free tracks */

   BeginCriticalSection(); // Make sure nobody else runs while we play with the FAT!
   // Note: Don't forget to call EndCriticalSection before exiting

   _bdos(0x0D, 0, 0);      // Flush all DOS disk buffers to make sure the FAT is up to date
   // Note: Don't use DOS until the end of the critical section

   wFreeTracks = 0;        // Count of consecutive free tracks
   wMaxFreeTracks = 0;     // Count of maximum consecutive free tracks
   wStartCluster = 0;      // Free cluster at the beginning of the current free tracks
   wCluster0 = 0;          // First cluster for our swap file
   wCluster1 = 0;          // Last cluster for our swap file

   for (w = FIRST_VALID_CLUSTER; w <= LAST_VALID_CLUSTER; w++)  // For all valid clusters
   {
      WORD wNext;

      wNext = NextCluster(iDrive, w);
      if (wNext == INVALID_CLUSTER) break;   // End of FAT reached

      dw = Cluster2Sector(iDrive, w);        // First sector of the cluster
      Log2PhysSector(iDrive, dw, &wCyl0, &wHead0, &wSect0);
      dw += pdp->dpSecPerClust - 1;          // Last sector of the cluster
      Log2PhysSector(iDrive, dw, &wCyl1, &wHead1, &wSect1);

      if (   ((wNext == 0) && (wHead0 != wHead1))  // If the cluster is free and is across two tracks
          || (wSect0 == 0))                        // or the cluster is aligned at the beginning of a track
      {
         if (wStartCluster)                           // If previous tracks are free,
         {
            wFreeTracks += 1;                            // Count one more full free track
            if (   (wFreeTracks == wNeededTracks)        // If enough free tracks
                && (wCluster0 == 0))                     // and it's the first time
            {
               wCluster0 = wStartCluster;                // then memorize the limits.
               wCluster1 = wSect0 ? w : w-1;             // The end is the previous sector if the
            }                                            //  cluster is not split across two tracks.
         }
         else                                         // The previous track was not free
         {
            wStartCluster = w;                           // Remember 1st cluster of 1st free track
         }                                            // Endif previous track free
      }                                            // Endif it's the beginning of a new track
      if (wNext != 0)                              // If not a free cluster (used or bad)
      {
         if (wFreeTracks > wMaxFreeTracks)
         {
            wMaxFreeTracks = wFreeTracks;
         }
         wFreeTracks = 0;                          // Not in a series of free tracks
         wStartCluster = 0;
      }
   } // End for all valid clusters

   if (!wCluster0)
   {
      EndCriticalSection();
      printf("Not enough contiguous disk space.\nLargest possible swap file: %ld KB.\n",
               (wMaxFreeTracks * dwTrackSize) / 1024);
      exit(1);
   }
   if (debug) printf("The swap file will be stored from clusters %4X to %4X.\n", wCluster0, wCluster1);

   dw = Cluster2Sector(iDrive, wCluster0);         // First sector of the first cluster
   dw += pdp->dpSecPerClust - 1;                   // Last sector of the first cluster
   Log2PhysSector(iDrive, dw, &wCyl0, &wHead0, &wSect0);
   dw = Cluster2Sector(iDrive, wCluster1);         // First sector of the last cluster
   Log2PhysSector(iDrive, dw, &wCyl1, &wHead1, &wSect1);

   /* Create the swap file */

   pszBuf = malloc(pdp->dpBytesPerSec);  // Allocate memory for 1 sector
   // Index of the 1st root directory sector:
   i0 = pdp->dpResSectors + (pdp->dpFATs * pdp->dpFATsecs);
   // 1st sector not in root = 1st sector + sizeof root / sizeof sector
   imax = i0 + (pdp->dpRootDirEnts * sizeof(DIRENTRY)) / pdp->dpBytesPerSec;
   for (found=FALSE, i=i0; (i<imax) && (!found); i++)
   {
      err = AbsDiskRead(iDrive, i, 1, pszBuf);
      if (err)
      {
         EndCriticalSection();
         printf("Cannot read the FAT at sector %04X. Error %d\n", i, err);
         exit(1);
      }
      pde = (DIRENTRY *)pszBuf;
      for (j=0; j<(int)(pdp->dpBytesPerSec/sizeof(DIRENTRY)); j++, pde++)
      {
         if (!(strncmp(pszName11, pde->deName, 11)))
         {
            found = i;
            break;
         }
      }
   }
   if (!found)
   {
      EndCriticalSection();
      printf("Entry %s not found in %c:\\.\n", pszName11, iDrive+'@');
      exit(1);
   }
   pde->deFileSize = (wCluster1 + 1 - wCluster0) * dwClusterSize;
   pde->deStartCluster = wCluster0;
   pde->deAttributes = 0x26;  // Archive, System, Hidden
   if (debug) printf("Updating sector %04X in root directory.\n", found);
   AbsDiskWrite(iDrive, found, 1, pszBuf);

   for (w = wCluster0; w < wCluster1; w++)
   {
      SetNextCluster(iDrive, w, w+1);        // Link each cluster to the next
   }
   SetNextCluster(iDrive, wCluster1, 0xFFFF);// End of file
   SetNextCluster(-1, 0, 0);                 // Flush the buffer

   EndCriticalSection();               // Can use DOS again to access the disk

   /* Create the descriptor file */

   spart.PFileVersion = 0x300L;        // Version of partition file. Must be 00000300.
   dw = wSect0 + 1;                    // Part of the first cluster on the 1st track, in sectors
   dw *= pdp->dpBytesPerSec;           // Idem, in bytes
   dw = dwClusterSize - dw;            // Part of the first cluster before the 1st track, in bytes
   spart.DOSStartOffset = dw;          // Offset in the swap file of the start sector of the partition
   dw += dwSwapFileSize;               // Offset in the swap file of the first secter after the partition
   dw -= pdp->dpBytesPerSec;           // Offset in swap file of the end sector of the partition
   spart.DOSEndOffset = dw;            // Offset in swap file of the end sector of the partition
   strcpy(spart.PFileName, psz386Spart);    // Swap file name.

   pOEM = (PFileOEMForm *)&spart.OEMField;
   pOEM->PVersion = 0x300;                   // Version of OEM part. Must be 0300.
   pOEM->PType = 1;                          // Type of OEM part. 1 == INT 13 drive type.
   pOEM->StartCyln = wCyl0;                  // First partition cylinder
   pOEM->EndCyln = wCyl1;                    // Last partition cylinder
   pOEM->HeadNumStart = wHead0;              // First head in first cylinder
   pOEM->HeadNumEnd = wHead1;                // Last head in last cylinder
   pOEM->INT13DrvNum = 0x80 + iDrive - 3;    // INT 13 DL value of drive
   pOEM->INT13NumHeads = pdp->dpHeads;       // Total # heads
   pOEM->INT13SecPerTrk = pdp->dpSecPerTrack;// Sectors per track

   f = fopen(pszSpart, "wb");
   if (!f)
   {
      printf("Error: Can't create file %s!\n", pszSpart);
      exit(1);
   }

   if (!fwrite(&spart, sizeof(PFileForm), 1, f))  // If can't write the structure
   {
      printf("Error: Can't write to file %s.\n", pszSpart);
      exit(1);
   }

   fclose(f);

   _chmod(pszSpart, _S_IREAD);   // Make the file read only

   exit(0);
}

void usage(void)
{
   printf("\
Create a Windows permanent swap file. Assumes Windows is in C:\\WINDOWS.\n\
\n\
Usage: makepar [switches]\n\
\n\
Switches:\n\
  -v    Display verbose information\n\
");
   exit(0);
}

/******************************************************************************
*                                                                             *
*                      .INI file manipulation routines.                       *
*                                                                             *
******************************************************************************/

FILE *findIni(char *name)
{
   FILE *f;
   char buf[128];
   char *pInit;

   f = fopen(name, "r+");
   if (f) return f;              /* Found as specified */

   /* Is may be possible to look somewhere else */

   if (strchr(name, '\\') || strchr(name, ':'))
      return NULL;               /* Failure if the path is specified */

   /* The path is not specified. Try other places */

//   pInit = getenv("INIT");                    // Use this for DOS programs
   pInit = "C:\\WINDOWS;C:\\WINDOWS\\SYSTEM";   // Use this for Windows programs
   while (pInit && *pInit)       /* Scan all paths */
   {
      int l;

      l = strcspn(pInit, " \t;");   /* Next separation character */
      strncpy(buf, pInit, l);       /* Copy the path before it */
      if (l && (buf[l-1] != '\\')) buf[l++] = '\\';   /* Add a \ if needed */
      strcpy(buf+l, name);          /* And append the name */

      f = fopen(buf, "r+");
      if (f) return f;           /* Found with the path */

      pInit += l;                /* Skip the path */
      pInit += strspn(pInit, " \t;"); /* Skip the separation characters */
   }

   return f;
}

int findSection(FILE *f, char *name)
{
   char line[256];
   char *pc;
   size_t l;

   strupr(name);
   l = strlen(name);

   while (fgets(line, 255, f))
   {
      strupr(line);

      pc = line;
      pc += strspn(line, " \t");    // Skip blanks
      if (strlen(pc) < (l+2)) continue;   // If too short, try next line
      if (*(pc++)!='[') continue;      // If no [, try next line
      if (strncmp(pc, name, l)) continue; // If no name, try next line
      pc += l;          // Skip the name
      if (*(pc++)!=']') continue;      // If no ],try next line
      return TRUE;            // Found!
   }
   return FALSE;          // Not found
}

int findItem(FILE *f, char *name)
{
   char line[256];
   char *pc;
   size_t l;
   long pos;

   strupr(name);
   l = strlen(name);

   while (TRUE)
   {
      pos = ftell(f);
      if (!fgets(line, 255, f)) break; // End of file. Item not found

      strupr(line);

      pc = line;
      pc += strspn(line, " \t");    // Skip blanks
      if (*pc=='[') return FALSE;      // If next section, not found
      if (strlen(pc) < (l+1)) continue;   // If too short, try next line
      if (strncmp(pc, name, l)) continue; // If no name, try next line
      pc += l;          // Skip the name
      pc += strspn(pc, " \t");      // Skip blanks
      if (*(pc++)!='=') continue;      // If no =,try next line

      /* Found. Now move the file pointer back to the char. after the '=' */

      fseek(f, pos + (pc - line), SEEK_SET);
      return TRUE;            // Found!
   }

   return FALSE;
}

int GetIniString(FILE *f, char *section, char *item, char *buf, int len)
{
   char *p;

   rewind(f);

   if (!findSection(f, section)) return FALSE;
   if (!findItem(f, item)) return FALSE;
   fgets(buf, len, f);
   p = strchr(buf, ';');               // Search the comment
   if (p)                              // If found...
      *p = '\0';                          // Remove it
   else                                // Else
      p = buf + strlen(buf);              // Point to the end of the string
   p -= 1;                             // Back to the last character
   if ((*p == '\n') || (*p != '\r'))   // If it is a trailing new line
      *(p--) = '\0';                      // Remove it and go back 1 char.
   for (; p >= buf; --p)               // Remove trailing spaces
   {
      if ((*p != ' ' ) && (*p != '\t')) break;
      *p = '\0';
   }
   return TRUE;
}

int GetIniInt(FILE *f, char *section, char *item, int *pi)
{
   char buf[32];

   if (!GetIniString(f, section, item, buf, 32)) return FALSE;
   if (!sscanf(buf, "%d", pi)) return FALSE;
   return TRUE;
}

int GetIniHex(FILE *f, char *section, char *item, unsigned int *pi)
{
   char buf[32];

   if (!GetIniString(f, section, item, buf, 32)) return FALSE;
   if (!sscanf(buf, "%x", pi)) return FALSE;
   return TRUE;
}

int SetIniString(FILE *f, char *section, char *item, char *buf)
{
   rewind(f);

   if (!findSection(f, section)) return FALSE;
   if (!findItem(f, item)) return FALSE;
   if (!fwrite(buf, strlen(buf), 1, f)) return FALSE;
   return TRUE;
}

int SetIniInt(FILE *f, char *section, char *item, int i, char *format)
{
   char buf[32];

   sprintf(buf, format, i);
   return SetIniString(f, section, item, buf);
}

/* ------------------------------------------------------------------------- *\
|                         Windows-compatible routines                         |
\* ------------------------------------------------------------------------- */

int GetPrivateProfileString(char *pszSection, char *pszEntry,
                            char *pszDefault, char *pszBuffer,
                            int iBufferLength, char *pszFilename)
{
   FILE *f;

   f = findIni(pszFilename);
   if (!f || !GetIniString(f, pszSection, pszEntry, pszBuffer, iBufferLength))
   {  // String not found. Copy the default, making sure it fits.
      strncpy(pszBuffer, pszDefault, iBufferLength);
      pszBuffer[iBufferLength - 1] = '\0';
   }

   if (f) fclose(f);

   return strlen(pszBuffer);
}

int GetPrivateProfileInt(char *pszSection, char *pszEntry,
                         int iDefault, char *pszFilename)
{
   FILE *f;
   int i;

   f = findIni(pszFilename);
   if (!f) return iDefault;

   if (!GetIniInt(f, pszSection, pszEntry, &i))
      i = iDefault;

   fclose(f);

   return i;
}

/******************************************************************************
*                                                                             *
*                            Low level disk access                            *
*                                                                             *
******************************************************************************/

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   GetDeviceParams                                                |
|                                                                             |
|  Purpose:    Get a drive parameters                                         |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Pointer to the DEVICEPARAMS structure. NULL = Failure.         |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

DEVICEPARAMS *GetDeviceParams(
int iDrive)        /* 1=A, 2=B, 3=C, etc... */
{
   static DEVICEPARAMS *pdp[27] = {0}; /* Room for 26 drives + index 0 */
   DEVICEPARAMS *p;
   WORD wErrCode;

   if (!pdp[iDrive])
   {
      p = (DEVICEPARAMS *)malloc(sizeof(DEVICEPARAMS));
      if (!p) return NULL;       // Not enough memory
      _asm
      {
         mov   bx, iDrive
         mov   dx, p             ; Structure is malloc-ed, so DS is alreay OK.
         mov   ax, 440DH         ; Function IOCTL for block devices
         mov   cx, 0860H         ; Category 08, subfunction 60: Get device params
         int   21H
         jc    params_error      ; Jump if error
         xor   ax, ax
params_error:
         mov   wErrCode, ax      ; Save the error code
      }
      if (!wErrCode)
      {
         pdp[iDrive] = p;         // Validate the structure
      }
      else
      {
         free(p);                // Invalid structure. Free the RAM.
      }
   }

   return pdp[iDrive];
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   Log2PhysSector                                                 |
|                                                                             |
|  Purpose:    Convert a logical sector number into cylinder/head/sector      |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Error code. 0 = Success.                                       |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

int Log2PhysSector(
int iDrive,             /* 1=A, 2=B, 3=C, etc... */
DWORD dwLogSect,        /* Logical sector number, ie. index in the partition */
WORD *pwCyl,            /* Cylinder */
WORD *pwHead,           /* Head */
WORD *pwSect)           /* Sector (0-based) */
{
   DEVICEPARAMS *pdp;

   pdp = GetDeviceParams(iDrive);
   if (!pdp) return 0xFFFF;

   dwLogSect += pdp->dpHiddenSecs;
   *pwSect = (WORD)(dwLogSect % pdp->dpSecPerTrack);
   dwLogSect /= pdp->dpSecPerTrack;
   *pwHead = (WORD)(dwLogSect % pdp->dpHeads);
   dwLogSect /= pdp->dpHeads;
   *pwCyl = (WORD)dwLogSect;
   return 0;
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   AbsDiskRead                                                    |
|                                                                             |
|  Purpose:    Read absolute disk sectors                                     |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Error code. 0 = Success.                                       |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

WORD AbsDiskRead(
int iDrive,             /* 1=A, 2=B, 3=C, etc... */
DWORD dwFirst,          /* Logical number of the first sector */
WORD wNumber,           /* Number of sectors to read */
char far *lpcBuffer)
{
   WORD wErrCode;
   DEVICEPARAMS *pdp;
   RWBLOCK rw;

   pdp = GetDeviceParams(iDrive);
   if (!pdp) return 0xFFFF;

   /* Convert an absolute sector number into a cylinder/head/sector triplet */
   Log2PhysSector(iDrive, dwFirst, &rw.rwCylinder, &rw.rwHead, &rw.rwFirstSector);

#if NEEDED  // Use for debug if needed
   printf("Reading cyl=%d, head=%d, sect=%d.\n",
           rw.rwCylinder, rw.rwHead, rw.rwFirstSector);
#endif

#if NEEDED  // This does not work, and I don't know why. This looks like a DOS bug!
   rw.rwSpecFunc = 0;
   rw.rwSectors = wNumber;
   rw.rwBuffer = (long)lpcBuffer;

   _asm
   {
      push  ds
      mov   bx, iDrive
      lea   dx, word ptr rw
      push  ss
      pop   ds             ; rw is on stack, so copy ss -> ds
      mov   ax, 440DH      ; Function IOCTL for block devices
      mov   cx, 0861H      ; Category 08, subfunction 61: Read track
      int   21H
      pop   ds
      jc    error_abs_dr   ; Jump if error
      xor   ax, ax
error_abs_dr:
      mov   wErrCode, ax   ; Save the error code
   }
#endif

   /* Because of the bug with the MS-DOS IOCTL function 440D/0861,
      we have to use the BIOS instead. The drawback is that only
      hard disks are supported */

   _asm     // Use int 13 fct 2
   {
      push  es
      mov   ah, 2
      mov   al, byte ptr wNumber
      mov   cx, rw.rwCylinder
      xchg  ch, cl               ; Move bits <7:0> of cylinder to CH
      shl   cl, 6                ; Move bits <9:8> of cylinder to bits <7:6> of CL
      or    cl, byte ptr (rw.rwFirstSector)  ; Sector in CL<5:0>
      inc   cl                   ; Int 13 counts sectors starting from index 1
      mov   dx, iDrive
      add   dl, 80H - 3          ; Hard disk C is number 80H for the BIOS
      mov   dh, byte ptr (rw.rwHead)
      les   bx, lpcBuffer        ; Buffer address
      int   13h                  ; Call BIOS
      mov   al, ah               ; BIOS returns error code in AH
      xor   ah, ah               ; AX = BIOS error code
      mov   wErrCode, ax         ; Save the error code
      pop   es
   }

   return wErrCode;
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   AbsDiskWrite                                                   |
|                                                                             |
|  Purpose:    Write absolute disk sectors                                    |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Error code. 0 = Success.                                       |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

WORD AbsDiskWrite(
int iDrive,             /* 1=A, 2=B, 3=C, etc... */
DWORD dwFirst,          /* Logical number of the first sector */
WORD wNumber,           /* Number of sectors to write */
char far *lpcBuffer)
{
   WORD wErrCode;
   DEVICEPARAMS *pdp;
   RWBLOCK rw;

   pdp = GetDeviceParams(iDrive);
   if (!pdp) return 0xFFFF;

   /* Convert an absolute sector number into a cylinder/head/sector triplet */
   Log2PhysSector(iDrive, dwFirst, &rw.rwCylinder, &rw.rwHead, &rw.rwFirstSector);

#if NEEDED  // Use for debug if needed
   printf("Reading cyl=%d, head=%d, sect=%d.\n",
           rw.rwCylinder, rw.rwHead, rw.rwFirstSector);
#endif

#if NEEDED  // This does not work, and I don't know why. This looks like a DOS bug!
   rw.rwSpecFunc = 0;
   rw.rwSectors = wNumber;
   rw.rwBuffer = (long)lpcBuffer;

   _asm
   {
      push  ds
      mov   bx, iDrive
      lea   dx, word ptr rw
      push  ss
      pop   ds             ; rw is on stack, so copy ss -> ds
      mov   ax, 440DH      ; Function IOCTL for block devices
      mov   cx, 0841H      ; Category 08, subfunction 41: Write track
      int   21H
      pop   ds
      jc    error_abs_dr   ; Jump if error
      xor   ax, ax
error_abs_dr:
      mov   wErrCode, ax   ; Save the error code
   }
#endif

   /* Because of the bug with the MS-DOS IOCTL function 440D/0841,
      we have to use the BIOS instead. The drawback is that only
      hard disks are supported */

   _asm     // Use int 13 fct 3
   {
      push  es
      mov   ah, 3
      mov   al, byte ptr wNumber
      mov   cx, rw.rwCylinder
      xchg  ch, cl               ; Move bits <7:0> of cylinder to CH
      shl   cl, 6                ; Move bits <9:8> of cylinder to bits <7:6> of CL
      or    cl, byte ptr (rw.rwFirstSector)  ; Sector in CL<5:0>
      inc   cl                   ; Int 13 counts sectors starting from index 1
      mov   dx, iDrive
      add   dl, 80H - 3          ; Hard disk C is number 80H for the BIOS
      mov   dh, byte ptr (rw.rwHead)
      les   bx, lpcBuffer        ; Buffer address
      int   13h                  ; Call BIOS
      mov   al, ah               ; BIOS returns error code in AH
      xor   ah, ah               ; AX = BIOS error code
      mov   wErrCode, ax         ; Save the error code
      pop   es
   }

   return wErrCode;
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   NextCluster                                                    |
|                                                                             |
|  Purpose:    Find the next cluster of a file in the FAT                     |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Next cluster. 0xFFF0 to 0 mean not found.                      |
|                                                                             |
|  Notes:      Always force to reread the buffer if the routine SetNextCluster|
|              was called between two consecutive calls to NextCluster, or    |
|              even if a lot of time passed, allowing other TSRs to modify    |
|              the FAT.                                                       |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

#define NO_SECTOR 0xFFFFFFFFL    // Invalid sector number

WORD NextCluster(int iDrive, WORD wCluster0)
{
   WORD wErrCode;
   static DEVICEPARAMS *pdp;
   static int iDrive0 = -1;    /* >= 1 if dp is valid */
   static char *pBuf = NULL;
   static DWORD dwLastSector = NO_SECTOR;
   DWORD dw;
   DWORD dwSector;
   int i;

   if (iDrive != iDrive0)
   {
      iDrive0 = -1;   /* Assume failure reading the drive parameters */
      pdp = GetDeviceParams(iDrive);
      if (!pdp) return INVALID_CLUSTER;
      iDrive0 = iDrive;   /* Flag success: Drive parameters are valid. */
      if (pBuf)
      {
         free(pBuf);    // The sector size may have changed.
         pBuf = NULL;   // Force a reallocation of the buffer.
      }
      dwLastSector = NO_SECTOR;   // Force reading the new drive
   }

   if (!pBuf) pBuf = malloc(pdp->dpBytesPerSec);

   /* Find in which sector is the cluster link */
   // Assume FAT 16: 2 bytes per cluster link. Floppys have FAT 12!
   dw = wCluster0 * 2L;                // Index in FAT, in bytes
   dwSector = dw / pdp->dpBytesPerSec; // Index in FAT, in sectors
   if (dwSector >= pdp->dpFATsecs) return INVALID_CLUSTER;
   dwSector += pdp->dpResSectors;      // Index in partition of sector
   if (dwSector != dwLastSector)       // Don't read twice the same sector
   {
      wErrCode = AbsDiskRead(iDrive, dwSector, 1, pBuf);
      if (wErrCode) return INVALID_CLUSTER;
      dwLastSector = dwSector;
   }
   i = (int)(dw % pdp->dpBytesPerSec); // Index in sector of link
   return *(WORD *)(pBuf+i);
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   SetNextCluster                                                 |
|                                                                             |
|  Purpose:    Set the next cluster of a file in the FAT                      |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Next cluster. 0xFFF0 to 0 mean not found.                      |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

WORD SetNextCluster(int iDrive, WORD wCluster0, WORD wNextCluster)
{
   WORD wErrCode;
   static DEVICEPARAMS *pdp;
   static int iDrive0 = -1;    /* >= 1 if dp is valid */
   static char *pBuf = NULL;
   static DWORD dwLastSector = NO_SECTOR;
   DWORD dw;
   DWORD dwSector;
   int i;

   if (iDrive != iDrive0)
   {
      if (dwLastSector != NO_SECTOR)
      {
         AbsDiskWrite(iDrive0, dwLastSector, 1, pBuf); // Flush the last sector
         if (pdp->dpFATs > 1)
         {
            AbsDiskWrite(iDrive0, dwLastSector + pdp->dpFATsecs, 1, pBuf); // Flush the last sector
         }
         dwLastSector = NO_SECTOR;
      }
      iDrive0 = -1;   /* Assume failure reading the drive parameters */
      pdp = GetDeviceParams(iDrive);
      if (!pdp) return INVALID_CLUSTER;
      iDrive0 = iDrive;   /* Flag success: Drive parameters are valid. */
      dwLastSector = NO_SECTOR;     // Force reading the new drive
      if (pBuf)
      {
         free(pBuf);    // The sector size may have changed.
         pBuf = NULL;   // Force a reallocation of the buffer.
      }
   }

   if (!pBuf) pBuf = malloc(pdp->dpBytesPerSec);

   /* Find in which sector is the cluster link */
   // Assume FAT 16: 2 bytes per cluster link. Floppys have FAT 12!
   dw = wCluster0 * 2L;                // Index in FAT, in bytes
   dwSector = dw / pdp->dpBytesPerSec; // Index in FAT, in sectors
   if (dwSector >= pdp->dpFATsecs) return INVALID_CLUSTER;
   dwSector += pdp->dpResSectors;      // Index in partition of sector
   if (dwSector != dwLastSector)       // Don't read twice the same sector
   {
      if (dwLastSector != NO_SECTOR)
      {
         AbsDiskWrite(iDrive0, dwLastSector, 1, pBuf); // Flush the last sector
         if (pdp->dpFATs > 1)
         {
            AbsDiskWrite(iDrive0, dwLastSector + pdp->dpFATsecs, 1, pBuf); // Flush the last sector
         }
         dwLastSector = NO_SECTOR;
      }
      wErrCode = AbsDiskRead(iDrive, dwSector, 1, pBuf);
      if (wErrCode) return INVALID_CLUSTER;
      dwLastSector = dwSector;
   }
   i = (int)(dw % pdp->dpBytesPerSec); // Index in sector of link
   *(WORD *)(pBuf+i) = wNextCluster;
   return 0;
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   Cluster2Sector                                                 |
|                                                                             |
|  Purpose:    Convert a cluster number to a logical sector number            |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     Sector number. -1 = Failure.                                   |
|                                                                             |
|  Creation:   28-Jul-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|  94/08/01 JFL      The first usable cluster after the root directory is #2! |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

long Cluster2Sector(int iDrive, WORD wCluster)
{
   long l;
   DEVICEPARAMS *pdp;

   pdp = GetDeviceParams(iDrive);
   if (!pdp) return -1;

   l = wCluster - 2;                   // Cluster relative to cluster 2
   l *= pdp->dpSecPerClust;            // Sector number relative to cluster 2
   l += pdp->dpResSectors;                   // + Reserved sectors
   l += pdp->dpFATs * pdp->dpFATsecs;        // + FATs sectors
   l += (pdp->dpRootDirEnts * sizeof(DIRENTRY)) / pdp->dpBytesPerSec; // + Root

   return l;
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   BeginCriticalSection                                           |
|                                                                             |
|  Purpose:    Prevent multitasking operating systems from switching tasks    |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     None                       .                                   |
|                                                                             |
|  Creation:   02-Aug-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

void BeginCriticalSection(void)
{
   _asm                    // Beginning of a critical section
   {
      mov   ax, 1681H
      int   2FH
   }
}

/*---------------------------------------------------------------------------*\
*                                                                             *
|  Function:   EndCriticalSection                                             |
|                                                                             |
|  Purpose:    Reallow multitasking operating systems to switch tasks         |
|                                                                             |
|  Parameters: See below                                                      |
|                                                                             |
|  Return:     None                       .                                   |
|                                                                             |
|  Creation:   02-Aug-1994 by JFL                                             |
|                                                                             |
|  Modification history:                                                      |
|  Date     Author   Description                                              |
|  -------- -------- ------------------------------------------------------   |
|                                                                             |
*                                                                             *
\*---------------------------------------------------------------------------*/

void EndCriticalSection(void)
{
   _asm                    // Beginning of a critical section
   {
      mov   ax, 1682H
      int   2FH
   }
}

/******************************************************************************
*                                                                             *
*                                   The End                                   *
*                                                                             *
******************************************************************************/
