// w_dycall.c - implement dynamic calling class
// Copyright (c) 1994 by Ron Burk

#include <string.h>
#include "wuistd.h"
#include "w_dycall.h"

/***********************************************************

TDynaCall::Call() - execute dynamic function call.

Description:
  The input is an array of pointers to arguments and a
parallel array of argument widths (the latter has an
extra element equal to zero, to denote the last argument).
You can also optionally supply a value to set the DS (and
AX) to before executing the call.

  The Intel 80x86 stack grows downward, and SP points to
the last used stack cell, not the next available cell.

***********************************************************/

TDynaCall::RESULT TDynaCall::Call(FARPROC Function,
                    int NArgs, int *Widths, void **Pointers,
                                        int Reverse, int Ds)
    {
    ASSERT(Function != NULL);
    ASSERT(NArgs    >= 0);
    if(NArgs > 0)
        {
        ASSERT(Widths   != NULL);
        ASSERT(Pointers != NULL);
        ASSERT(NArgs <= 32);
        }

    int     AdjustedWidths[MAXARGS];
    RESULT  Result;
    int     StackSize   = 0;

    // first, figure total stack required
    for(int i=0; i < NArgs; ++i)
        {
        // Sanity checks
        ASSERT(Widths[i] > 0 && Widths[i] < MAXARGLEN);

        int ArgWidth    = Widths[i];
        int Remainder   = ArgWidth % sizeof(int);
        if(Remainder != 0)  // if not even with alignment
            // then round up to alignment size
            ArgWidth       += sizeof(int)-Remainder;
        AdjustedWidths[i]   = ArgWidth;
        StackSize          += ArgWidth;
        }

    // now make contiguous copy of args
    char *Args  = new char[StackSize];
    // first stack loc is at highest mem loc
    // (because stack grows down)
    char *Stack = Args + StackSize;

    int Start = 0;
    int Step  = 1;
    if(Reverse) // if C calling sequence
        {
        Start   = NArgs-1;
        Step    = -1;
        }
    // construct image of stack
    for(i=Start; i >= 0 && i < NArgs; i+=Step)
        {
        ASSERT(Pointers[i] != NULL);
        // back up (stack grows down)
        Stack  -= AdjustedWidths[i];
        memcpy(Stack, Pointers[i], Widths[i]);
        }

    // now let subfunction do the assembly work
    Result.Long = Call(Function, Args+StackSize, StackSize, Ds);
    delete[] Args;
    return Result;
    }


/*
 *  Call - Handle assembly language part of call.
 *
 *  Description:
 *    The goal is to keep all the assembly language in this
 *  function and to keep it as simple as possible.  It saves
 *  and restores some registers across the call (most importantly,
 *  SP and DS) rather than depending on the caller.
 */

#define WORDSIZE (2)

long  TDynaCall::Call(FARPROC Function,
                 const char *Args, int StackSize, int DsReg)
    {
    int     SaveSp, SaveDs;
    // need explicit return value to appease some compilers
    int     ResultLo, ResultHi;

    // Save current value of stack pointer, so we can restore
    // it later (pray that caller does not trash bp!).
    __asm   mov ax,sp
    __asm   mov SaveSp,sp
    __asm   mov ax,ds
    __asm   mov SaveDs,ax

    __asm lds si,Args   // ds:si <- ptr to source
    __asm mov ax,ss
    __asm mov es,ax
    __asm mov di,sp         // es:di <- ptr to destination
    __asm sub sp,StackSize  // adjust stack pointer now, else
                            // stack space we write to is
                            // fair game for interrupts

    __asm sub di,WORDSIZE   // due to the nature of sp
    __asm sub si,WORDSIZE

    __asm mov cx,StackSize
    __asm shr cx,(WORDSIZE/2)   // # of words to move

    __asm jcxz CopyDone
    __asm std
    __asm rep movsw
    __asm cld

    CopyDone:


    __asm mov ax,DsReg        // install desired DS
    __asm mov ds,ax

    __asm call Function   // finally(!) call function

    __asm mov bx,SaveDs   // restore DS reg
    __asm mov ds,bx

    __asm mov sp,SaveSp   // restore SP reg

    __asm mov ResultLo,ax
    __asm mov ResultHi,dx

    return ResultLo | (ResultHi << 16);
    }

