/****************************************************
 * Listing 2 - populate.c
 *   written by: Charles Leamon
 ***************************************************/
#include <windows.h>
#include "populate.h"

/****************************************************
 * private function prototypes
 ***************************************************/
static WORD PASCAL FAR 
   ProcessResource(HWND hWnd, 
                   VOID FAR *lp);
                   
static VOID PASCAL FAR 
   AddString(HWND hWnd, 
             WORD wCtrlID, 
             LPSTR lpszString);
             
static WORD PASCAL FAR 
   GetCtrlType(HWND hWnd);

/****************************************************
 * supported control types
 ***************************************************/
struct tagCtrlType {
   WORD msg;      /* message to 'settext' */
   char *name;    /* window class name    */
} CtrlType[] = {
   LB_ADDSTRING, "listbox",
   CB_ADDSTRING, "combobox",
   WM_SETTEXT,   "static",
   WM_SETTEXT,   "edit" ,
   WM_SETTEXT,   "button"
};
   
#define MAX_CTRLTYPE \
   (sizeof(CtrlType)/sizeof(CtrlType[0]))
   
#define MAXSTR  (1024)

/****************************************************
 * PopulateDlgControls - this is the entry point for 
 *   populating the dialog. It's responsibilities are 
 *   to find, load and lock the specified resource.
 *   It then delegates the remaining work to 
 *   ProcessResource(), passing the dialog window 
 *   handle and a pointer to the locked resource.
 ***************************************************/
WORD PASCAL FAR _export PopulateDlgControls(
   HWND hWnd,          /* dialog window handle     */
   HINSTANCE hInst,    /* inst handle for resource */
   LPSTR lpszResource) /* RCDATA name              */
                       /* (or MAKEINTRESOURCE(x))  */
{
   WORD     ret = PDERR_NOERROR;
   HRSRC    hRes;
   VOID FAR *lp;
   /* find, load and lock the resource */
   hRes = FindResource(hInst,lpszResource,RT_RCDATA);
   if (hRes) {
      HGLOBAL hMem = LoadResource(hInst, hRes);
      if (hMem) {
         lp = LockResource(hMem);
         if (lp) {
            ret = ProcessResource(hWnd, lp);
            UnlockResource(hMem);
         } else ret = PDERR_CANTLOCKRES;
         FreeResource(hMem);
      } else ret = PDERR_CANTLOADRES;
   } else ret = PDERR_CANTFINDRES;
   return(ret);
}

/****************************************************
 * SetTextLimits - this is the entry point for setting
 *   the text limit of a group of edit contols.
 ***************************************************/
WORD PASCAL FAR _export SetTextLimits(
   HWND hWnd,          /* dialog window handle     */
   HINSTANCE hInst,    /* inst handle for resource */
   LPSTR lpszResource) /* RCDATA name              */
                       /* (or MAKEINTRESOURCE(x))  */
{
   WORD     ret = PDERR_NOERROR;
   HRSRC    hRes;
   WORD FAR *wp;
   /* find, load and lock the resource */
   hRes = FindResource(hInst,lpszResource,RT_RCDATA);
   if (hRes) {
      HGLOBAL hMem = LoadResource(hInst, hRes);
      if (hMem) {
         wp = (WORD FAR *)LockResource(hMem);
         if (wp) {
            while(*wp) {
               SendDlgItemMessage(hWnd, *wp,
                  EM_LIMITTEXT, *(wp+1), 0L);
               wp += 2;
            }
            UnlockResource(hMem);
         } else ret = PDERR_CANTLOCKRES;
         FreeResource(hMem);
      } else ret = PDERR_CANTLOADRES;
   } else ret = PDERR_CANTFINDRES;
   return(ret);
}

/****************************************************
 * ProcessResource - enumerates the resource, calling
 *   AddString for each id/string pair found.
 ***************************************************/
static WORD PASCAL FAR ProcessResource(
   HWND hWnd,        /* dialog window handle       */
   VOID FAR *lp)     /* pointer to locked resource */
{  
   WORD ret = PDERR_NOERROR;
   WORD FAR *wp;
   WORD wCtrlID;
   WORD wLen;
   HGLOBAL hMem;
   LPSTR lpszString, lpResString, lpWork;

   hMem = GlobalAlloc(GMEM_MOVEABLE|GMEM_ZEROINIT, MAXSTR+1);
   if (hMem) {
      /* allocate a working buffer  */
      lpszString = (LPSTR)GlobalLock(hMem);
      if (lpszString) {
         /* if successfully allocated */
         /* point to first ctrl id */
         wp = (WORD FAR *)lp;
         while(*wp) {
            /* walk the resource with pointer wp */
            /* the WORD here is the control ID   */
            wCtrlID = *wp;
            /* set target ptr to start of buffer */
            lpWork = lpszString;               
            /* set source ptr at 1st byte after id */
            lpResString =  (LPSTR)wp+sizeof(WORD);
            wLen = 0;
            /* copy bytes until a NULL is found  */
            /* or maximum length is reached      */
            while(*lpResString && wLen < MAXSTR) {
               *lpWork++ = *lpResString++;
               wLen++;
            }
            if (wLen < MAXSTR)
               /* copy the null too */
               *lpWork++ = *lpResString++;
            else {
               /* null terminate it here, then skip */
               /* the remainder so we point to the  */
               /* next ctrl id                      */
               *lpWork = 0;
               while(*lpResString++)
                  /* empty loop */
                  ;
            }
            AddString(hWnd, wCtrlID, lpszString);
            /* move word pointer to new position    */
            wp = (WORD FAR *)lpResString;
         }
         GlobalUnlock(hMem);
      } else ret = PDERR_CANTLOCKMEMORY;
      GlobalFree(hMem);
   } else ret = PDERR_CANTALLOCMEMORY;
   return(ret);
}

/****************************************************
 * AddString - Sends appropriate message to control
 ***************************************************/
static VOID PASCAL FAR AddString(
   HWND hWnd,           /* dialog window handle */
   WORD wCtrlID,        /* id of target control */
   LPSTR lpszString)    /* string to set into ctrl */
{
   HWND hWndCtrl = GetDlgItem(hWnd, wCtrlID);
   if (hWndCtrl) {
      WORD msg = GetCtrlType(hWndCtrl);
      if (msg) {
         SendMessage(hWndCtrl, msg, 0, 
                     (LONG)lpszString);
      }
   }
}

/****************************************************
 * GetCtrlType - determines class of window and what 
 *   message should be used for setting the control's
 *   text (ie. CB_ADDSTRING, WM_SETTEXT)
 ***************************************************/
static WORD PASCAL FAR GetCtrlType(
      HWND hWnd)        /* control window handle */
{
   static char szClassName[40];
   WORD ret;
   
   if (GetClassName(hWnd, szClassName, 
                    sizeof(szClassName))) {
      int which;
      for(which=0; which < MAX_CTRLTYPE; which++) {
         if (lstrcmpi(CtrlType[which].name, 
                      szClassName) == 0) {
            ret =  CtrlType[which].msg;
            break;
         }
      }
      if (which == MAX_CTRLTYPE) ret = 0;
   } else ret = 0;
   return(ret);
}

