// wuiobjdb.c - implement WUIMAN object database.
// Copyright (c) 1993 by Ron Burk

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>      // sscanf()

#include "wuidbint.h"
#include "wuierror.h"
#include "wuistd.h"
#include "wuiobjdb.h"

// Hmm, these should probably be in wuiobjdb.h
static  const   TWuiName    M_DELETE(       "m_Delete");
static  const   TWuiName    M_ISA(          "m_IsA");
static  const   TWuiName    M_RENAME(       "m_Rename");
static  const   TWuiName    M_CLASS(        "m_Class");
static  const   TWuiName    M_PARENT(       "m_ParentClass");
static  const   TWuiName    M_CHILD(        "m_Child");
static  const   TWuiName    M_ATTRIBUTE(    "m_Attribute");

static  const   TWuiName    P_CLASS(        "p_Class");
static  const   TWuiName    P_PARENTCLASS(  "p_ParentClass");
static  const   TWuiName    P_COMPILER(     "p_Compiler");
static  const   TWuiName    P_WUIMANVERSION("p_WuimanVersion");


static  const   TWuiName    S_CLASSES("Classes");
static  const   TWuiName    S_ROOT("/");

static  const   TWuiName    S_INHERITED("P_INHERITED");
static  const   TWuiName    S_INHERITABLE("P_INHERITABLE");
static  const   TWuiName    S_PERSISTENT("P_PERSISTENT");
static  const   TWuiName    S_READONLY("P_READONLY");

static  const   TWuiName    C_ROOT("/");
static  const   TWuiName    C_TCONTAINER("TContainer");
static  const   TWuiName    C_TROOT("TRoot");
static  const   TWuiName    C_TATTRIBUTE("TAttribute");
static  const   TWuiName    C_TATTRIBUTEMETHOD("TAttributeMethod");
static  const   TWuiName    C_TATTRIBUTESTRING("TAttributeString");
static  const   TWuiName    C_TATTRIBUTEINT("TAttributeInt");

////////////////////////////////////////////////////////////
// TWuiObjects - Provide arrays of AWuiObject pointers

// TWuiObjects - initialize object with desired array size.
TWuiObjects::TWuiObjects(int Size)
    :   TWuiGenericList(Size)
    {   MEMBERASSERT();
    }

// (private)
// Find - Return position of named child

int     TWuiObjects::Find(const TWuiName Which)
    {   MEMBERASSERT();

    for(int iChild = 0; iChild < NObjects(); ++iChild)
        if(Get(iChild)->Name() == Which)
            return iChild;
    return -1;
    }

void   *TWuiObjects::Insert(AWuiObject *NewObject, const TWuiName To)
    {   MEMBERASSERT();
    ASSERT(NewObject != 0);
    return Insert(NewObject, Find(To));
    }

void   *TWuiObjects::Insert(AWuiObject *NewObject, int Position)
    {   MEMBERASSERT();
    ASSERT(NewObject != 0);
    return TWuiGenericList::Insert((void *)NewObject, Position);
    }

AWuiObject  *TWuiObjects::Get(const TWuiName Which)
    {   MEMBERASSERT();
    int     Position = Find(Which);
    if(Position >= 0)
        return Get(Position);
    else
        return 0;
    }

AWuiObject *TWuiObjects::Remove(const TWuiName Which)
    {   MEMBERASSERT();
    int     Position = Find(Which);
    if(Position >= 0)
        return Remove(Position);
    else
        return 0;
    }

/////////////////////////////////////////////////
// AWuiObject - supply minimal object functions

AWuiObject     *AWuiObject::Root            = 0;
int             AWuiObject::TraceFlag       = 1;
TWuiDatabase   *AWuiObject::Database        = 0;
TWuiObjects    *AWuiObject::MasterObjects   = 0;

AWuiObject::AWuiObject(const TWuiName ObjectName,
                                      const char * /*Value*/)
    :   Name_(ObjectName)
    {   MEMBERASSERT();
    }

// WUIMAN Object destructor - delete objects in list.
AWuiObject::~AWuiObject()
    {   MEMBERASSERT();
    AWuiObject  *Object = 0;

    TWuiObjects *List = Children();
    if(List)
        {
        while((Object = List->Remove()) != NULL)
            delete Object;
        ASSERT(List->NObjects() == 0);
        delete List;
        }

    List    = Attributes();
    if(List)
        {
        while((Object = List->Remove()) != NULL)
            delete Object;
        ASSERT(List->NObjects() == 0);
        delete List;
        }
    }

void    AWuiObject::DumpObject()
    {   MEMBERASSERT();
    TWuiObjects *List = Children();
    AWuiObject *Item;
    int i;
    DEBUG_TRACE("**Children**");
    int n = List->NObjects();
    if(List)
        for(i=0; i < n; ++i)
            {
            Item = List->Get(i);
            char    Buffer[TWuiName::MAXNAME];
            Item->Name().GetName(Buffer);
            DEBUG_MORE("    '%s'", Buffer);
            }
    List    = Attributes();
    DEBUG_TRACE("**Attributes**");
    n = List->NObjects();
    if(List)
        for(i=0; i < n; ++i)
            {
            Item = List->Get(i);
            char    Buffer[TWuiName::MAXNAME];
            Item->Name().GetName(Buffer);
            DEBUG_MORE("    '%s'", Buffer);
            }
    }

void    AWuiObject::Trace(const char *Function,
                              TWuiPath Path, TWuiArgs *Args)
    {
    char    AttributeName[TWuiName::MAXNAME];
    char    MyName[TWuiName::MAXNAME];
    Args->AttributeName().GetName(AttributeName);
    this->Name().GetName(MyName);
    const char *AbsolutePath = Path.AbsolutePath()+1;
    for(int Count = 0; AbsolutePath != NULL
                && AbsolutePath < Path.RelativePath(); ++Count)
        AbsolutePath = strchr(AbsolutePath+1, '/');
    Count   *= 4;
    DEBUG_TRACE("%*.*s[%s][%s] AbsPath = '%s', RelPath = '%s'"
                " Attribute='%s'",
                    Count, Count, "",
                                            Function, MyName,
                    Path.AbsolutePath(), Path.RelativePath(),
                    AttributeName);
    }

#if 0
// GetSetAttribute() - handle m_Attribute() messages.
//  - get of "m_Attribute"
//      returns number of attributes
//  - get of "m_Attribute(n)"
//      returns name of nth attribute.
//  - get of "m_Attribute('name','bitname')"
//      returns value of named property bit.
//  - set of "m_Attribute('name','bitname')"
//      sets value of named property bit.
static
long    GetSetAttribute(AWuiObject *Node, TWuiArgs *Args,
                             char *Buffer, size_t MaxLength)
    {
    TWuiObjects *MyAttributes = Node->Attributes();
    TWuiName AttributeName    = Args->AttributeName();

    TWuiBaseAttribute *Child;
    if(Args->NArgs() <= 0)
        return MyAttributes ? MyAttributes->NObjects() : 0;
    else if(Args->NArgs() == 1)
        {
        // if "m_Attribute(n)
        if(Args->GetType(0) == TWuiArgs::LONG_ARG)
            {
            long    ArgNumber = Args->GetLong(0);
            if(ArgNumber >= 0 && ArgNumber
                         < MyAttributes->NObjects())
                {
                Child = MyAttributes
                            ->Get(int(ArgNumber));
                Child->Name().GetName(Buffer, MaxLength);
                return 0;
                }
            }
        else // else attr arg is wrong type
            return WUIMAN_ERR_BAD_ATTR_ARG;
        }
    // else if "m_Attribute('AttrName', 'AttrBitName')"
    else if(Args->NArgs() == 2)
        {
        if(   Args->GetType(0) == TWuiArgs::STRING_ARG
           && Args->GetType(1) == TWuiArgs::STRING_ARG)
            {
            Child = (TWuiBaseAttribute *)
                      MyAttributes->Get(Args->GetString(0));
            if(Child == NULL)
                return WUIMAN_ERR_BAD_ATTR;
            TWuiName Property = Args->GetString(1);
            int Bit;
            if(Property == S_NULL)
                Bit = Child->GetFlags(TWuiBaseAttribute
            }
        else // else attr arg is wrong type
            return WUIMAN_ERR_BAD_ATTR_ARG;
        return WUIMAN_ERR_BAD_ATTR_ARG;
        }
    }
#endif


// Get - default get attribute handler
//      NOTE: the default is to forward the message, or to return
//            FALSE if this object is the destination.
long    AWuiObject::Get(TWuiPath Path, TWuiArgs *Args,
                        AWuiObject *Attribute, char *Buffer,
                                           size_t MaxLength)
    {   MEMBERASSERT();
    ASSERT(Args != NULL);
    if(AWuiObject::TraceFlag)
        Trace("AWuiObject::Get", Path, Args);
    TWuiObjects *MyAttributes = Attributes();
    long    Result  = TRUE; // assume success
    if(Path.Descend())      // if message is for a child node
        {
        // then try to locate child and pass it message
        TWuiName ChildName      = Path.CurrentName();
        TWuiObjects *MyChildren;
        if(Path.RelativePath()[0] == '.')
            MyChildren = Attributes();
        else
            MyChildren = Children();
        AWuiObject  *Child      = NULL;
        if(MyChildren)
            Child   = MyChildren->Get(ChildName);
        if(Child)
            Result  = Child->Get(Path, Args, Attribute,
                                         Buffer, MaxLength);
        else    // else, no child by that name!
            {
            char    Name[TWuiName::MAXNAME];
            ChildName.GetName(Name);
            DEBUG_ERROR("'%s' no such child", Name);
            DumpObject();
            Result  = WUIMAN_ERR_NO_SUCH_OBJECT;
            }
        }
    else        // else message is for me
        {
        TWuiName AttributeName = Args->AttributeName();
        // M_CHILD() returns # of children
        // M_CHILD(n) returns name of nth child
        if(AttributeName == M_CHILD
            || AttributeName == M_ATTRIBUTE)
            {
            TWuiObjects *MyChildren;
            if(AttributeName == M_ATTRIBUTE)
                MyChildren  = MyAttributes;
            else
                MyChildren  = Children();
            AWuiObject  *Child = 0;
            // m_xxxx() == # of children/attributes
            if(Args->NArgs() <= 0)
                return MyChildren
                             ? MyChildren->NObjects() : 0;
            // m_xxxx(n) == return nth child/attribute
            else if(Args->NArgs() == 1 && MyChildren)
                {
                if(Args->GetType(0) == TWuiArgs::LONG_ARG)
                    {
                    long    ArgNumber = Args->GetLong(0);
                    if(ArgNumber >= 0 && ArgNumber
                                 < MyChildren->NObjects())
                        {
                        Child = MyChildren
                                    ->Get(int(ArgNumber));
                        Child->Name().GetName(Buffer, MaxLength);
                        return 0;
                        }
                    }
                else // else attr arg is wrong type
                    return WUIMAN_ERR_BAD_ATTR_ARG;
                return WUIMAN_ERR_BAD_ATTR_ARG;
                }
            else if(AttributeName == M_CHILD)
                return WUIMAN_ERR_BAD_ATTR_ARG;
            // m_xxxx('attr','prop') == return property bit
            else if(Args->NArgs() == 2 && MyAttributes
                && Args->GetType(0) == TWuiArgs::STRING_ARG)
                {
                Child = MyAttributes->Get(
                                        Args->GetString(0));
                if(Child)
                    return Child->Get(Path, Args,
                              Attribute, Buffer, MaxLength);
                else
                    {
                    DEBUG_ERROR("No such attribute '%s'.", Args->GetString(0));
                    DumpObject();
                    return WUIMAN_ERR_BAD_ATTR;
                    }
                }
            }
        else if(AttributeName == M_CLASS)
            {
            // ??? need error checking ???
            ClassName().GetName(Buffer, MaxLength);
            return strlen(Buffer);
            }
        else if(AttributeName == M_PARENT)
            {
            // ??? need error checking ???
            ParentClassName().GetName(Buffer, MaxLength);
            return strlen(Buffer);
            }
        else if(MyAttributes)
            {
            AWuiObject  *Target =
                   MyAttributes->Get(AttributeName);
            if(Target)
                return Target->Get(Path, Args, Attribute, Buffer, MaxLength);
            }
        Result  = WUIMAN_ERR_BAD_ATTR;
        }
    return Result;
    }

// Set - default Set attribute handler
//      NOTE: the default is to forward the message, or to return
//            FALSE if this object is the destination
long    AWuiObject::Set(TWuiPath Path,
                      TWuiArgs *Args, AWuiObject *Attribute,
                                         const char *Buffer)
    {   MEMBERASSERT();
    if(AWuiObject::TraceFlag)
        Trace("AWuiObject::Set", Path, Args);
    TWuiObjects *MyAttributes = Attributes();
    long    Result  = TRUE; // assume success
    if(Path.Descend())   // if message is for a child
        {
        TWuiName ChildName      = Path.CurrentName();
        TWuiObjects *MyChildren;
        if(Path.RelativePath()[0] == '.')
            MyChildren = Attributes();
        else
            MyChildren = Children();
        AWuiObject  *Child      = NULL;
        if(MyChildren)
            Child   = MyChildren->Get(ChildName);
        if(Child)
            {
            Result  = Child->Set(Path, Args, Attribute,
                                                    Buffer);
            // if error or grandchild, just pass back result
            if(Result < WUIMAN_ERRORS || Path.Descend())
                ;
            // else, no error and message was for child
            else if(Args->AttributeName() == M_DELETE)
                {
                // Since tree has no upward pointers,
                // parent of target node must do deleting
                delete MyChildren->Remove(ChildName);
                
                if(MyChildren->NObjects() == 0)
                    {
                    }
                }
            }
        else    // else, no child by that name!
            Result  = WUIMAN_ERR_NO_SUCH_OBJECT;
        }
    else        // else message is for me
        {
        if(Args->AttributeName() == M_DELETE)
            ;   // parent will delete
        else if(Args->AttributeName() == M_ISA)
            return FALSE; // we ain't IsA nothin'!
        else if(Args->AttributeName() == M_RENAME)
            Name_   = Buffer;
        // m_Attribute('attr','bitname', 0|1)
        else if(Args->AttributeName() == M_ATTRIBUTE)
            {
            if(    Args->NArgs() != 3
                || Attributes() == NULL
                || Args->GetType(0) != TWuiArgs::STRING_ARG
                || Args->GetType(1) != TWuiArgs::STRING_ARG
                || Args->GetType(2) != TWuiArgs::LONG_ARG
              )
                return WUIMAN_ERR_BAD_ATTR_ARG;
            AWuiObject *Child = MyAttributes->Get(
                                        Args->GetString(0));
            if(Child)
                return Child->Set(Path, Args,
                              Attribute, Buffer);
            else
                {
                DEBUG_ERROR("No such attribute '%s'.", Args->GetString(0));
                DumpObject();
                return WUIMAN_ERR_BAD_ATTR;
                }
            }
        else
            {
            TWuiObjects *MyAttributes = Attributes();
            if(MyAttributes)
                {
                AWuiObject *Target = MyAttributes->Get(Args->AttributeName());
                if(Target)
                    return Target->Set(Path, Args, Attribute, Buffer);
                }
            DEBUG_TRACE("AWuiObject detects unknown attribute name");
            Result  = WUIMAN_ERR_BAD_ATTR;
            }
        }
    return Result;
    }

AWuiObject *AWuiObject::New(const TWuiName Type,
                                        const TWuiName Name)
    {
    ASSERT(AWuiObject::MasterObjects != NULL);
    AWuiObject *Master = AWuiObject::MasterObjects
                                                ->Get(Type);
    char Buffer[TWuiName::MAXNAME];
    if(Master == NULL)
        {
        DEBUG_ERROR("Attempt to create object of undefined "
                "type '%s'", Type.GetName(Buffer));
        return NULL;
        }
    // Found master object, new create new one
    AWuiObject *Object = Master->New(Name);
    if(Object == NULL)
        {
        DEBUG_ERROR("Could not allocate new object of type '%s'",
                Type.GetName(Buffer));
        return NULL;
        }
    else
        return Object;
    }

TWuiBaseAttribute *AWuiObject::Attribute(const TWuiName Name)
    {   MEMBERASSERT();
    TWuiObjects *List = Attributes();
    if(List)
        return (TWuiBaseAttribute *)List->Get(Name);
    else
        return NULL;
    }

// IsA - Due to the design of WUIMAN, much of 'IsA' is class-independent.
int     AWuiObject::IsA(const TWuiName What, const TWuiName Method)
    {   MEMBERASSERT();

    if(What == ClassName())
        {
        if(Method == "")
            return TRUE;
        else if(Attribute(Method) != NULL)
            return TRUE;
        }
    return FALSE;
    }


#if 0
// default version of LoadAttribute creates/loads attribute
int    AWuiObject::LoadAttribute(AWuiObject *Master,
                           TWuiName Name, const char *Value)
    {
    TWuiObjects *List = Attributes();
    if(List != NULL)
        {
        AWuiObject  *New = Master->New(Name, Value);
        if(New != NULL)
            {
            List->Insert(New);
            return 1;
            }
        else
            return 0;
        }
    else
        return 0;
    }

// default version of LoadChild creates/loads child node
int    AWuiObject::LoadChild(AWuiObject *Master,
                           TWuiName Name, const char *Value)
    {
    TWuiObjects *List = Children();
    if(List != NULL)
        {
        AWuiObject  *New = Master->New(Name, Value);
        if(New != NULL)
            {
            List->Insert(New);
            return 1;
            }
        else
            return 0;
        }
    else
        return 0;
    }
#endif

TWuiObjects *AWuiObject::Children()
    {   MEMBERASSERT();
    // default is to not support children
    return NULL;
    }

TWuiObjects *AWuiObject::Attributes()
    {   MEMBERASSERT();
    // default is to not support attributes
    return NULL;
    }


////////////
// 11/06/93

static TWuiRegister RegisterContainer(
    new TWuiContainer(C_TCONTAINER));

// TWuiContainer - An AWuiObject that supports children and attributes.
TWuiContainer::TWuiContainer(const TWuiName ObjectName)
    :   AWuiObject(ObjectName), Children_(0), Attributes_(0)
    {   MEMBERASSERT();
    }

long    TWuiContainer::Initialize(const char *Value)
    {   MEMBERASSERT();
    ASSERT(Children_ == 0);
    ASSERT(Attributes_ == 0);
    int     NChildren = 0, NAttributes = 0;
    if(Value)
        {
        if(sscanf(Value, "%d,%d", &NChildren, &NAttributes) != 2)
            return WUIMAN_ERR_CANT_INIT_ATTR;
        }
    Children_   = new TWuiObjects(NChildren);
    Attributes_ = new TWuiObjects(NAttributes);
    Attributes_->Insert(new TWuiMethod(M_ISA));
    Attributes_->Insert(new TWuiMethod(M_DELETE));
    Attributes_->Insert(new TWuiMethod(M_RENAME));
    return NChildren+NAttributes;
    }

TWuiName    TWuiContainer::ClassName()
    {   MEMBERASSERT();
    return C_TCONTAINER;
    }
TWuiName    TWuiContainer::ParentClassName()
    {   MEMBERASSERT();
    return C_ROOT;
    }

#if 0
AWuiObject  *TWuiContainer::Clone()
    {   MEMBERASSERT();
    char    InitialValue[32];
    sprintf(InitialValue, "%d,%d", Children()->NObjects(),
        Attributes()->NObjects());
    return new TWuiContainer(Name(), InitialValue);
    }
#endif
AWuiObject  *TWuiContainer::New(const TWuiName Name)
    {   MEMBERASSERT();
    return new TWuiContainer(Name);
    }
TWuiObjects *TWuiContainer::Children()
    {   MEMBERASSERT();
    return Children_;
    }
TWuiObjects *TWuiContainer::Attributes()
    {   MEMBERASSERT();
    return Attributes_;
    }

// TWuiRoot - Special class for root node.
TWuiRoot::TWuiRoot(const TWuiName ObjectName)
    :   TWuiContainer(S_ROOT)
    {   MEMBERASSERT();
    }

long TWuiRoot::Initialize(const char *Value)
    {   MEMBERASSERT();
    long Status = TWuiContainer::Initialize(Value);
    if(Status > WUIMAN_ERRORS)
        {
#if defined(__SC__)
        char *Compiler = "Symantec C++ v6.1";
#elif defined(__BORLANDC__)
        char *Compiler = "Borland C++ v4.0";
#elif defined(_MSC_VER) 
        char *Compiler = "Microsoft C++ v8.00c";
#else
        char *Compiler = "<Unknown compiler>";
#endif

        ASSERT(Attributes() != NULL);
        TWuiStringAttribute *Temp = new
            TWuiStringAttribute(P_COMPILER, Compiler,
                             TWuiBaseAttribute::P_READONLY);
        ASSERT(Temp != NULL);
        Attributes()->Insert(Temp);
        Temp    = new TWuiStringAttribute(P_WUIMANVERSION,
                "0.00a", TWuiBaseAttribute::P_READONLY);
        ASSERT(Temp != NULL);
        Attributes()->Insert(Temp);
        }
    return Status;
    }
TWuiName    TWuiRoot::ClassName()
    {   MEMBERASSERT();
    return C_TROOT;
    }
TWuiName    TWuiRoot::ParentClassName()
    {   MEMBERASSERT();
    return C_TCONTAINER;
    }

// TWuiRegister - Register a new WUIMAN class.
TWuiRegister::TWuiRegister(AWuiObject *MasterCopy)
    {   MEMBERASSERT();
    if(AWuiObject::TraceFlag)
        {
        char Name[32];
        MasterCopy->Name().GetName(Name);
        DEBUG_TRACE("TWuiRegister(\"%s\")", Name);
        }

    AWuiObject  *Classes = 0;
    if(AWuiObject::Root == NULL)    // if not initialized
        {
        // create root container object
        AWuiObject::Root    = new TWuiRoot(S_ROOT);
        ASSERT(AWuiObject::Root != NULL);
        AWuiObject::Root->Initialize("5,10");
        // add "/Classes", to hold registered classes
        Classes             = new TWuiContainer(S_CLASSES);
        Classes->Initialize("5,10");
        ASSERT(Classes != NULL);
        ASSERT(AWuiObject::Root->Children() != NULL);
        AWuiObject::Root->Children()->Insert(Classes);
        // Initialize 'MasterObjects', master list
        ASSERT(Classes->Children() != NULL);
        AWuiObject::MasterObjects = Classes->Children();
        }
    ASSERT(AWuiObject::MasterObjects != NULL);
    // Place object of class "X" in "/Classes/X".
    AWuiObject::MasterObjects->Insert(MasterCopy);
    }

// TWuiPath - Handle moving up and down the database "path"
TWuiPath::TWuiPath(char *TotalPath)
    :   AbsolutePath_(TotalPath), RelativePath_(TotalPath)
    {   MEMBERASSERT();
    ASSERT(TotalPath != NULL);
    ASSERT(TotalPath[0] == '/');
    }

// Descend() - Move relative path down one node.
int TWuiPath::Descend()
    {
    char    *Child = strchr(RelativePath_, '/');
    if(!Child)   // if no children
        return FALSE;
    // ignore trailers (e.g. "/a/b/c/")
    else if(Child[1] == '\0')
        return FALSE;
    // else there's *some* kind of child name
    else
        {
        RelativePath_   = Child+1;
        return TRUE;
        }
    }

// Ascend() - Move relative path up one node.
int TWuiPath::Ascend()
    {
    switch(RelativePath_ - AbsolutePath_)
        {
        case    0:
            return FALSE;
        case    1:
            RelativePath_   = AbsolutePath_;
            return TRUE;
        default:
            RelativePath_   -= 2;
            while(RelativePath_ > AbsolutePath_)
                if(*RelativePath_ == '/')
                    break;
                else
                    --RelativePath_;
            ASSERT(*RelativePath_ == '/');
            ++RelativePath_;
            return TRUE;
        }
    }

// CurrentName() - return first name of relative path.
TWuiName    TWuiPath::CurrentName()
    {   MEMBERASSERT();
    char        Name[TWuiName::MAXNAME+1];
    const char *Rover = strchr(RelativePath(), '/');
    int         Length;
    if(Rover)
        Length  = Rover-RelativePath();
    else
        Length  = strlen(RelativePath());
    const char *Start = RelativePath();
    // AWuiObject::Get/Set needs this
    if(*Start == '.')
        ++Start;
    strncpy(Name, Start, Length);
    Name[Length]    = '\0';
    return TWuiName(Name);
    }

// Push() - Append to the path.
int     TWuiPath::Push(const char *SubPath)
    {   MEMBERASSERT();
    char   *Rover = AbsolutePath_ + strlen(AbsolutePath_);
    if(Rover > AbsolutePath_)
        --Rover;    // point to char before EOS
    if(*Rover != '/')
        strcat(AbsolutePath_, "/");
    strcat(AbsolutePath_, SubPath);
    return TRUE;    // Need to check for overflow!
    }

// Pop() - Pop the last node off the path.
int TWuiPath::Pop()
    {   MEMBERASSERT();
    char   *Rover = AbsolutePath_ + strlen(AbsolutePath_);
    if(Rover > AbsolutePath_)
        --Rover;    // point to char before EOS
    if(Rover > AbsolutePath_ && *Rover == '/')
        --Rover;    // skip any bogus trailing '/'
    while(Rover > AbsolutePath_ && *Rover != '/')
        --Rover;
    if(Rover <= AbsolutePath_)
        return FALSE;
    else    // else Rover points at '/'
        {
        *Rover  = '\0';
        return TRUE;
        }
    }

#if 0
static TWuiRegister RegisterAttributeMethod(
    new TWuiMethod("TAttributeMethod"));

TWuiMethod::TWuiMethod(const TWuiName ObjectName)
    :   AWuiObject(ObjectName)
    {   MEMBERASSERT();
    }

AWuiObject *TWuiMethod::New(const TWuiName ObjectName,
                                     const char * /*Value*/)
    {   MEMBERASSERT();
    return new TWuiMethod(ObjectName);
    }


#endif


#if 0
// TWuiAttribute - Base class for attributes.

TWuiAttribute::TWuiAttribute(const TWuiName ObjectName,
                                int Value, int InitialFlags)
    :   AWuiObject(ObjectName), Type(T_INT),
        Flags(InitialFlags)
    {   MEMBERASSERT();
    IntValue    = Value;
    }
TWuiAttribute::TWuiAttribute(const TWuiName ObjectName,
                             int *Address, int InitialFlags)
    :   AWuiObject(ObjectName), Type(T_INT),
        Flags(InitialFlags), IntAddress(Address)
    {   MEMBERASSERT();
    }
TWuiAttribute::TWuiAttribute(const TWuiName ObjectName,
                        const char *Value, int InitialFlags)
    :   AWuiObject(ObjectName), Type(T_STRING),
        Flags(InitialFlags), StringValue(0)
    {   MEMBERASSERT();
    if(Value)
        StringValue = StringClone(Value);
    }
TWuiAttribute::TWuiAttribute(const TWuiName ObjectName,
                           char **Address, int InitialFlags)
    :   AWuiObject(ObjectName), Type(T_STRING),
        Flags(InitialFlags), StringAddress(Address)
    {   MEMBERASSERT();
    }
#if 0
AWuiObject  *TWuiAttribute::Clone()
    {   MEMBERASSERT();
    return new TWuiAttribute(Name(), Value_);
    }
#endif

long    TWuiAttribute::Set(const char *Value)
    {   MEMBERASSERT();
    int Indirect = (Flags & P_INDIRECT);
    if(Type == T_INT || Type == T_LONG)
        {
        char   *End;
        long    Number = strtol(Value, &End, 0);
        if(End == Value || *End != '\0')
            {
            DEBUG_ERROR("Not a legal value for a %s"
                " attribute: '%s'", Type==T_INT?"int":"long", Value);
            return WUIMAN_SET_BAD_INTEGRAL;
            }
        if(Type == T_INT)
            {
            if(Number > INT_MAX || Number < INT_MIN)
                {
                DEBUG_ERROR("Integer attribute exceeds integer range: '%s'", Value);
                return WUIMAN_SET_INT_OVERFLOW;
                }
            if(Indirect)
                *IntAddress = int(Number);
            else
                IntValue    = int(Number);
            return Number;
            }
        else // else it is T_LONG
            {
            if(Indirect)
                *LongAddress    = Number;
            else
                IntValue        = Number;
            if(Number < WUIMAN_ERRORS)
                return Number;
            else
                return 1;
            }
        }
    else if(Type == T_STRING)
        {
        char    **Address = (Indirect ? StringAddress : &StringValue);
        if(*Address)
            delete[] *Address;
        *Address    = StringClone(Value);
        return strlen(Value);
        }
    else
        return -1; //???????
    }


// Initialize basic attribute from a database string. Format of
// database string is : <bits>,<type>,<value>
int TWuiAttribute::Initialize(const char *Data)
    {   MEMBERASSERT();
    ASSERT(Data != NULL);
    int     NewFlags, NewType, Offset;
    if(sscanf(Data, "%x,%d,%n", &NewFlags, &NewType, &Offset) != 2)
        {
        DEBUG_ERROR("Bad attribute value creation"
                                        " string: '%s'", Data);
        return FALSE;
        }
    if(NewType < T_INT || NewType > T_STRING)
        {
        DEBUG_ERROR("Bad attribute creation type: %d", NewType);
        return FALSE;
        }
    else
        Type    = char(NewType);
    if(NewFlags & ~(0x3F))
        {
        DEBUG_ERROR("Bad attribute creation flags: %04X",
                                                  NewFlags);
        return FALSE;
        }
    else
        Flags   = char(NewFlags);
    return Set(Data+Offset);
    }

TWuiAttribute::TWuiAttribute(const TWuiName Name, char NewFlags, char NewType)
    :   AWuiObject(Name), Flags(NewFlags), Type(NewType)
    {   MEMBERASSERT();
    }

// Handle primitive attribute get.
long    TWuiAttribute::Get(TWuiPath Path, TWuiArgs *Args,
      AWuiObject *Attribute, char *Buffer, size_t MaxLength)
    {   MEMBERASSERT();
    if(AWuiObject::TraceFlag)
        Trace("TWuiAttribute::Get", Path, Args);
    if(Args->AttributeName() != Name())
{
DEBUG_TRACE("TWuiAttribute::Get passes off to AWuiObject");
        return AWuiObject::Get(Path, Args, Attribute, Buffer, MaxLength);
}

    int Indirect = (Flags & P_INDIRECT);
    char    Temp[64];
    char   *Value = Temp;
    switch(Type)
        {
        case    T_INT   :
            sprintf(Temp, "%d", Indirect?*IntAddress:IntValue);
            break;
        case    T_LONG  :
            sprintf(Temp, "%ld", Indirect?*LongAddress:LongValue);
            break;
        case    T_FLOAT  :
            sprintf(Temp, "%f", Indirect?*FloatAddress:FloatValue);
            break;
        case    T_DOUBLE  :
            sprintf(Temp, "%g", Indirect?*DoubleAddress:DoubleValue);
            break;
        case    T_STRING :
            Value   = Indirect?*StringAddress:StringValue;
        }

    // else try to copy value to caller's buffer
    size_t  ValueLength = strlen(Value) + 1;
    long    Status = ValueLength;
    if(ValueLength > MaxLength)
        {
        ValueLength = MaxLength;
        Status      = WUIMAN_ERR_ATTR_TOO_LONG;
        }
    memcpy(Buffer, Value, ValueLength);
    return Status;
    }

long    TWuiAttribute::Set(TWuiPath Path,
            TWuiArgs *Args, AWuiObject * /*Attribute*/,
                                         const char *Buffer)
    {
    if(AWuiObject::TraceFlag)
        Trace("TWuiAttribute::Set", Path, Args);
    return Set(Buffer);
    }

///////////////////////////////////////////////////
// String attributes

static TWuiRegister RegisterStringAttribute(
    new TWuiStringAttribute("TAttributeString", ""));

TWuiStringAttribute::TWuiStringAttribute(const TWuiName Name, const char *Value)
    :   TWuiAttribute(Name, Value)
    {   MEMBERASSERT();
    }

TWuiStringAttribute::TWuiStringAttribute(const TWuiName Name, char **Address)
    :   TWuiAttribute(Name, Address)
    {   MEMBERASSERT();
    }
AWuiObject *TWuiStringAttribute::New(const TWuiName Name,
                                          const char *Value)
    {   MEMBERASSERT();
    ASSERT(Value != NULL);
    TWuiStringAttribute *Result = new TWuiStringAttribute(Name,"");
    ASSERT(Result != NULL);
    if(Result->Initialize(Value))
        return Result;
    else
        {
        delete Result;
        return NULL;
        }
    }

#endif

#if 0
// TWuiPERSISTENTAttribute - Base class for attributes.
TWuiPERSISTENTAttribute::TWuiPERSISTENTAttribute(
               const TWuiName ObjectName, const char *Value)
    :   TWuiAttribute(ObjectName, Value)
    {   MEMBERASSERT();
    }
#if 0
AWuiObject  *TWuiPERSISTENTAttribute::Clone()
    {   MEMBERASSERT();
    return new TWuiPERSISTENTAttribute(Name(), Value());
    }
#endif
AWuiObject  *TWuiPERSISTENTAttribute::New(
                     const TWuiName Name, const char *Value)
    {   MEMBERASSERT();
    return new TWuiPERSISTENTAttribute(Name, Value);
    }
// Here's where we implement persistance
long    TWuiPERSISTENTAttribute::Set(TWuiPath Path,
                      TWuiArgs *Args, AWuiObject *Attribute,
                                         const char *Buffer)
    {
    long    Status = TWuiAttribute::Set(Path, Args,
                                         Attribute, Buffer);
    Database->WriteItem(Path.AbsolutePath(),
                            Args->AttributeName(), Value());
    return Status;
    }

#endif



////////////////////////////////////////////////////////
// TWuiBaseAttribute - code common to all attributes.

TWuiBaseAttribute::TWuiBaseAttribute(
                const TWuiName ObjectName, int InitialFlags)
    :   AWuiObject(ObjectName), Flags(InitialFlags&0x003F)
    {   MEMBERASSERT();
    
    }
long TWuiBaseAttribute::Set(TWuiPath Path, TWuiArgs *Args,
        AWuiObject * /*Attribute*/, const char *Buffer)
    {   MEMBERASSERT();
    if(AWuiObject::TraceFlag)
        Trace("TWuiBaseAttribute::Set", Path, Args);
    // if m_Attribute('attrname','bitname',0|1)
    if(Args->AttributeName() == M_ATTRIBUTE)
        {
        if( Args->NArgs() != 3
        ||  Args->GetType(0) != TWuiArgs::STRING_ARG
        ||  Args->GetType(1) != TWuiArgs::STRING_ARG
        ||  Args->GetType(2) != TWuiArgs::LONG_ARG
          )
            return WUIMAN_ERR_BAD_ATTR_ARG;
        TWuiName Property = Args->GetString(1);
        int On = FALSE;
        int Bit;
        if(Buffer)
            On = atol(Buffer) != 0;
        if(Property == S_INHERITED)
            Bit = SetFlags(P_INHERITED, On);
        else if(Property == S_INHERITABLE)
            Bit = SetFlags(P_INHERITABLE, On);
        else if(Property == S_PERSISTENT)
            Bit = SetFlags(P_PERSISTENT, On);
        else if(Property == S_READONLY)
            Bit = SetFlags(P_READONLY, On);
        else
            return WUIMAN_ERR_BAD_ATTR_ARG;
        return Bit != 0;
        }
    // we should not get called unless "set" was successful
    // Since "set" succeeded, turn off INHERITED bit
    ClearFlags(P_INHERITED);
    // if PERSISTENT bit is set, save new value to database
    if(GetFlags(P_PERSISTENT))
        {
        char    SmallBuffer[32];
        // Get value to save by calling derived class function
        // (it will either create string version in buffer we
        //  supply, or return pointer to a string value)
        // We could just use "Buffer", but the derived class may
        // have sanitized the string in some way.
        char *Value = SmallBuffer;
        long    Result = AttributeValue(&Value);
        if(Result < WUIMAN_ERRORS)
            DEBUG_FATAL("Can't happen: AttributeValue() failed");
        TCharBuffer Temp(32+strlen(Value));
        char  ThisClass[TWuiName::MAXNAME];
        sprintf(Temp, "%s,%0X,%s", ClassName().GetName(ThisClass),
            Flags&0x00FF, Value);
        if(!Database->WriteItem(Path.AbsolutePath(),
                            Args->AttributeName(), (char *)Temp))
            return WUIMAN_ERR_WRITE_FAILED;
        else
            return Result;
        }
    else
        return TRUE;
    }

long TWuiBaseAttribute::Get(TWuiPath Path, TWuiArgs *Args,
      AWuiObject *Attribute, char *Buffer, size_t MaxLength)
    {   MEMBERASSERT();
    if(AWuiObject::TraceFlag)
        Trace("TWuiBaseAttribute::Get", Path, Args);
    // if this is result of "m_Attribute('AttrName',...)"
    if(Args->AttributeName() == M_ATTRIBUTE)
        {
        // we expect "m_Attribute('AttrName','BitName')"
        // must be two args
        if(Args->NArgs() != 2)
            return WUIMAN_ERR_BAD_ATTR_ARG;
        // 2nd arg must be string
        else if(Args->GetType(1) != TWuiArgs::STRING_ARG)
            return WUIMAN_ERR_BAD_ATTR_ARG;
        else
            {
            TWuiName Property = Args->GetString(1);
            int Bit;
            if(Property == S_INHERITED)
                Bit = GetFlags(P_INHERITED);
            else if(Property == S_INHERITABLE)
                Bit = GetFlags(P_INHERITABLE);
            else if(Property == S_PERSISTENT)
                Bit = GetFlags(P_PERSISTENT);
            else if(Property == S_READONLY)
                Bit = GetFlags(P_READONLY);
            else
                return WUIMAN_ERR_BAD_ATTR_ARG;
            if(Buffer && MaxLength > 1)
                sprintf(Buffer, "%d", Bit != 0);
            return Bit != 0;
            }
        }
    // else if we got called as a node under "/Classes"...
    else if(Args->AttributeName() != Name())
        return AWuiObject::Get(Path, Args, Attribute, Buffer, MaxLength);
    // else this is just a "normal" attribute get value call
    char    SmallBuffer[64];
    SmallBuffer[0]  = '\0';
    // Call derived class to obtain value as string
    char   *Value   = SmallBuffer;
    long    Result  = AttributeValue(&Value);
    if(Buffer != NULL)
        {
        if(Value == NULL)
            Value   = "";
        size_t  ValueLength = strlen(Value) + 1;
        if(ValueLength > MaxLength)
            {
            ValueLength = MaxLength;
            Result      = WUIMAN_ERR_ATTR_TOO_LONG;
            }
        memcpy(Buffer, Value, ValueLength);
        }
    return Result;
    }

long    TWuiBaseAttribute::Initialize(const char *Value)
    {   MEMBERASSERT();
    int LoadFlag;
    int Offset;

    if(sscanf(Value, "%i,%n", &LoadFlag, &Offset) != 1)
        {
        DEBUG_ERROR("'%s' Bad base attribute value.", Value);
//???
DEBUG_ERROR("LoadFlag=%d, Offset=%d", LoadFlag, Offset);
        return WUIMAN_ERR_CANT_INIT_ATTR;
        }
    else
        {
        Flags   = LoadFlag & 0x003F;
        return Offset;
        }
    }

int TWuiBaseAttribute::SetFlags(int Bits, int On)
    {   MEMBERASSERT();
    if(On)
        return Flags   |= (Bits & ~P_INDIRECT) & 0x00FF;
    else
        return ClearFlags(Bits);
    }
int TWuiBaseAttribute::ClearFlags(int Bits)
    {   MEMBERASSERT();

    return Flags   &= ~((Bits & ~P_INDIRECT) & 0x00FF);
    }
int TWuiBaseAttribute::GetFlags(int Bits)
    {   MEMBERASSERT();
    return (Flags & Bits) & 0x00FF;
    }

///////////////////////////////////////////////////
// Integer attributes

static TWuiRegister RegisterIntAttribute(
    new TWuiIntAttribute("TAttributeInt"));

TWuiIntAttribute::TWuiIntAttribute(const TWuiName Name)
    :   IntValue(0), TWuiBaseAttribute(Name,
        P_PERSISTENT|P_INHERITED|P_INHERITABLE)
    {   MEMBERASSERT();
    }

TWuiIntAttribute::TWuiIntAttribute(const TWuiName Name, int *Address)
    :   IntAddress(Address), TWuiBaseAttribute(Name,
        P_INDIRECT|P_PERSISTENT|P_INHERITED|P_INHERITABLE)
    {   MEMBERASSERT();
    }
AWuiObject *TWuiIntAttribute::New(const TWuiName Name)
    {   MEMBERASSERT();
    return new TWuiIntAttribute(Name);
    }
long    TWuiIntAttribute::Initialize(const char *Value)
    {   MEMBERASSERT();
    ASSERT(Value != NULL);
    long    Status = TWuiBaseAttribute::Initialize(Value);
    if(Status < WUIMAN_ERRORS)
        return Status;
    // else status is offset of our value
    char *End;
    Status = strtol(Value+Status, &End, 0);
    if(End == (Value+Status) || *End != '\0')
        {
        DEBUG_ERROR("Not a legal value for an integer "
                    "attribute: '%s'", Value);
        return WUIMAN_ERR_SET_BAD_INTEGRAL;
        }
    if(Status > INT_MAX || Status < INT_MIN)
        {
        DEBUG_ERROR("Integer attribute exceeds integer "
                    "range: '%s'", Value);
        return WUIMAN_ERR_SET_INT_OVERFLOW;
        }
    if(GetFlags(P_INDIRECT))
        *IntAddress = int(Status);
    else
        IntValue    = int(Status);
    return Status;
    }

long    TWuiIntAttribute::AttributeValue(char **Buffer)
    {   MEMBERASSERT();
    int *Address;
    if(GetFlags(P_INDIRECT))
        Address = IntAddress;
    else
        Address = &IntValue;
    sprintf(*Buffer, "%d", *Address);
    return *Address;
    }

TWuiName TWuiIntAttribute::ClassName()
    {   MEMBERASSERT();
    return C_TATTRIBUTEINT;
    }
TWuiName TWuiIntAttribute::ParentClassName()
    {   MEMBERASSERT();
    return C_TATTRIBUTE;
    }

///////////////////////////////////////////////////
// String attributes

static TWuiRegister RegisterStringAttribute(
    new TWuiStringAttribute("TAttributeString"));

TWuiStringAttribute::TWuiStringAttribute(const TWuiName Name)
    :   StringValue(0), TWuiBaseAttribute(Name,
        P_PERSISTENT|P_INHERITED|P_INHERITABLE)
    {   MEMBERASSERT();
    }

TWuiStringAttribute::TWuiStringAttribute(
                    const TWuiName Name, char *InitialValue,
                                           int InitialFlags)
    :   StringValue(0), TWuiBaseAttribute(Name,
        InitialFlags)
    {   MEMBERASSERT();
    if(InitialValue != NULL)
        StringValue = StringClone(InitialValue);
    }

TWuiStringAttribute::TWuiStringAttribute(
      const TWuiName Name, char **Address, int InitialFlags)
    :   StringAddress(Address), TWuiBaseAttribute(Name,
        InitialFlags)
    {   MEMBERASSERT();
    ASSERT(Address != NULL);
    }

TWuiStringAttribute::~TWuiStringAttribute()
    {   MEMBERASSERT();
    char    **Address = &StringValue;
    if(GetFlags(P_INDIRECT))
        Address = StringAddress;
    if(*Address != NULL)
        {
        delete[] *Address;
        *Address    = NULL;
        }
    }
AWuiObject *TWuiStringAttribute::New(const TWuiName Name)
    {   MEMBERASSERT();
    return new TWuiStringAttribute(Name);
    }

long    TWuiStringAttribute::Initialize(const char *Value)
    {   MEMBERASSERT();
    ASSERT(Value != NULL);
    long Status = TWuiBaseAttribute::Initialize(Value);
    if(Status < WUIMAN_ERRORS)
        return Status;
    // else status is offset of our value
    char *MyValue = StringClone(Value+Status);
    if(GetFlags(P_INDIRECT))
        *StringAddress = MyValue;
    else
        StringValue    = MyValue;
    return Status;
    }

long    TWuiStringAttribute::AttributeValue(char **Buffer)
    {   MEMBERASSERT();
    char **Address;
    if(GetFlags(P_INDIRECT))
        Address = StringAddress;
    else
        Address = &StringValue;
    *Buffer = *Address;
    return *Address ? strlen(*Address) : 0;
    }
TWuiName TWuiStringAttribute::ClassName()
    {   MEMBERASSERT();
    return C_TATTRIBUTESTRING;
    }
TWuiName TWuiStringAttribute::ParentClassName()
    {   MEMBERASSERT();
    return C_TATTRIBUTE;
    }


TWuiMethod::TWuiMethod(const TWuiName ObjectName)
    :   TWuiBaseAttribute(ObjectName, P_READONLY)
    {   MEMBERASSERT();
    }

AWuiObject *TWuiMethod::New(const TWuiName ObjectName)
    {   MEMBERASSERT();
    return new TWuiMethod(ObjectName);
    }

long TWuiMethod::AttributeValue(char **Buffer)
    {   MEMBERASSERT();
    **Buffer    = '\0';
    return 0;
    }

TWuiName    TWuiMethod::ClassName()
    {   MEMBERASSERT();
    return C_TATTRIBUTEMETHOD;
    }
TWuiName    TWuiMethod::ParentClassName()
    {   MEMBERASSERT();
    return C_TATTRIBUTE;
    }

