/*****************************************************/
/* keymap.c                                          */
/* -- DLL remaps keyboard keys.                      */
/* -- To compile: cc -wd -DSTRICT keymap.c           */
/*****************************************************/
#include <windows.h>
#include "keymap.h"

typedef struct KRN /* Key code Remap Node. */
    {
    struct KRN *pkrnNext;
    KRI kri;
    } KRN;

HINSTANCE hins;      /* This DLL's instance. */
HHOOK     hhok;      /* Message hook handle. */
KRN       *pkrnHead; /* Map list head. */

int CALLBACK LibMain(HINSTANCE hinsThis, WORD wDS,
  WORD cbHeap, LPSTR lpsz);
LRESULT CALLBACK __export LwMsgHook(int code,
  WPARAM wParam, LPMSG lpmsg);
int CALLBACK _export WEP(int wExitCode);
KRN **PpkrnFind(LPKEY lpkey);
BYTE BSetVkc(UINT vkc, BYTE bSet);

int CALLBACK LibMain(HINSTANCE hinsThis, WORD wDS,
  WORD cbHeap, LPSTR lpsz)
/*****************************************************/
/* -- Initialize the key map array.                  */
/*****************************************************/
    {
    hins = hinsThis;
    return FMapEnable(TRUE);
    }

BOOL WINAPI __export FMapEnable(BOOL fEnable)
/*****************************************************/
/* -- Enable (or disable) all keyboard mappings.     */
/* -- Return the new state.                          */
/*****************************************************/
    {
    if (fEnable && NULL == hhok)
        hhok = SetWindowsHookEx(WH_GETMESSAGE,
          (HOOKPROC)LwMsgHook, hins, NULL);
    else if (!fEnable && NULL != hhok)
        {
        UnhookWindowsHookEx(hhok);
        hhok = NULL;
        }
    return NULL != hhok;
    }

int CALLBACK _export WEP(int wExitCode)
/*****************************************************/
/* -- Remove the message hook.                       */
/*****************************************************/
    {
    FMapEnable(FALSE);
    return TRUE;
    }

LRESULT CALLBACK __export LwMsgHook(int code,
  WPARAM wParam, LPMSG lpmsg)
/*****************************************************/
/* -- GetMessage() hook maps keyboard messages.      */
/*****************************************************/
    {
    KRN **ppkrn;
    KEY key;

    if (code < 0)
        goto LCallNext; /* SDK says leave alone. */
    if (lpmsg->message != WM_KEYDOWN &&
      lpmsg->message != WM_KEYUP &&
      lpmsg->message != WM_SYSKEYDOWN &&
      lpmsg->message != WM_SYSKEYUP)
        goto LCallNext; /* Not a keyboard message. */
    if (lpmsg->wParam >= 256)
        goto LCallNext; /* Bogus key code. */

    GetKeyMsg(key, lpmsg->wParam, lpmsg->lParam);
    if (NULL == (ppkrn = PpkrnFind(&key)))
        goto LCallNext; /* Key is not mapped. */

    if (WM_KEYDOWN == lpmsg->message ||
      WM_SYSKEYDOWN == lpmsg->message)
        BSetVkc((*ppkrn)->kri.keyTo.vkc,
          BSetVkc(key.vkc, 0));
    else
        BSetVkc((*ppkrn)->kri.keyTo.vkc, 0);

    lpmsg->wParam = (*ppkrn)->kri.keyTo.vkc;
    lpmsg->lParam = (lpmsg->lParam & ~0x01ff0000) |
      ((long)(*ppkrn)->kri.keyTo.wExt << 16);

LCallNext:
    return CallNextHookEx(hhok, code, wParam,
      (LPARAM)lpmsg);
    }

KRN **PpkrnFind(LPKEY lpkey)
/*****************************************************/
/* -- Find the key mapping.                          */
/* -- lpkey : Source key for mapping to find.        */
/*****************************************************/
    {
    KRN **ppkrn;

    for (ppkrn = &pkrnHead; NULL != *ppkrn;
      ppkrn = &(*ppkrn)->pkrnNext)
        if ((*ppkrn)->kri.keyFrom.vkc == lpkey->vkc &&
          (*ppkrn)->kri.keyFrom.wExt == lpkey->wExt)
            return ppkrn;
    return NULL;
    }

BYTE BSetVkc(UINT vkc, BYTE bSet)
/*****************************************************/
/* -- Set the given key to the given state.          */
/* -- Return the previous state.                     */
/*****************************************************/
    {
    BYTE b, rgb[256];

    GetKeyboardState(rgb);
    b = rgb[vkc];
    rgb[vkc] = bSet;
    SetKeyboardState(rgb);
    return b;
    }

BOOL WINAPI __export FSetKri(LPKRI lpkri)
/*****************************************************/
/* -- Add the given key mapping, alloc may fail.     */
/* -- lpkri : Mapping to add.                        */
/*****************************************************/
    {
    KRN **ppkrn, *pkrn;

    if (VK_MENU == lpkri->keyFrom.vkc ||
      VK_MENU == lpkri->keyTo.vkc)
        return FALSE;

    if (NULL != (ppkrn = PpkrnFind(&lpkri->keyFrom)))
        pkrn = *ppkrn; /* Already mapped? */
    else
        {
        pkrn = (KRN *)LocalAlloc(LPTR, sizeof(KRN));
        if (NULL == pkrn)
            return FALSE; /* No memory. */
        pkrn->pkrnNext = pkrnHead; /* Link. */
        pkrnHead = pkrn;
        }
    pkrn->kri = *lpkri;
    return TRUE;
    }

BOOL WINAPI __export FGetKri(LPKRI lpkri)
/*****************************************************/
/* -- Return the key mapping, if found.              */
/* -- lpkri : keyFrom is key to find, fill rest.     */
/*****************************************************/
    {
    KRN **ppkrn;

    if (NULL != (ppkrn = PpkrnFind(&lpkri->keyFrom)))
        {
        *lpkri = (*ppkrn)->kri;
        return TRUE;
        }
    return FALSE;
    }

BOOL WINAPI __export FNextKri(LPKRI lpkri, LPKEY lpkey)
/*****************************************************/
/* -- Return the mapping following given, else head. */
/* -- lpkey : Find first if null, else following.    */
/* -- lpkri : Filled on output.                      */
/*****************************************************/
    {
    KRN **ppkrn;

    if (NULL == pkrnHead)
        return FALSE; /* No mappings at all. */
    if (NULL == lpkey ||
      NULL == (ppkrn = PpkrnFind(lpkey)) ||
      NULL == *(ppkrn = &(*ppkrn)->pkrnNext))
        ppkrn = &pkrnHead;
    *lpkri = (*ppkrn)->kri;
    return TRUE;
    }

BOOL WINAPI __export FRemoveKey(LPKEY lpkey)
/*****************************************************/
/* -- Remove the given key map, if it exists.        */
/* -- lpkri : Map to remove.                         */
/*****************************************************/
    {
    KRN **ppkrn, *pkrn;

    if (NULL != (ppkrn = PpkrnFind(lpkey)))
        {
        pkrn = *ppkrn;
        *ppkrn = (*ppkrn)->pkrnNext; /* Unlink. */
        LocalFree((HLOCAL)pkrn);
        return TRUE;
        }
    return FALSE;
    }

void WINAPI __export ResetMap(void)
/*****************************************************/
/* -- Remove all key mappings.                       */
/*****************************************************/
    {
    KRN *pkrn, *pkrnNext;

    for (pkrn = pkrnHead; NULL != pkrn;
      pkrn = pkrnNext)
        {
        pkrnNext = pkrn->pkrnNext;
        LocalFree((HLOCAL)pkrn);
        }
    pkrnHead = NULL;
    }
